﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <pthread.h>
#include <unistd.h>
#include <ctime>
#include <sys/time.h>
#include <cstdlib>

#include <cerrno>

namespace NetTest {

#define NETTEST_DECLARE_THREAD_FUNC(THREAD_NAME, PARAM_NAME) void* THREAD_NAME(void* PARAM_NAME)
#define NETTEST_THREAD_RETURN \
do{\
pthread_exit(0);\
return NULL; \
}while(false)

#define NN_NOEXCEPT
#define NN_OVERRIDE

#define nnMain main
#define NETTEST_MAIN_RETURN() return 0
#define TEST(X, Y) int main(int argc, const char** argv)

#define NETTEST_GET_ARGS(ARGC, ARGV) \
do { \
    ARGC = argc; \
    ARGV = argv; \
} while(false)


#define EXPECT_EQ(LHS, RHS) \
do { \
if( LHS == RHS ) \
{ \
    printf("\n * Test PASSED!\n\n"); \
} \
else \
{ \
    printf("\n * Test FAILED!\n\n"); \
} \
} while( false )

typedef void* (*ThreadFn)(void*);

struct Thread
{
    pthread_t thread;
    pthread_attr_t threadAttr;
    ThreadFn threadFunc;
    void* pParam;
};

/* Types */
typedef pthread_mutex_t Mutex;

struct Event
{
    pthread_cond_t condition;
    pthread_mutex_t mutex;
    bool flag;
};

struct Time
{
    timespec m_time;

    void operator= (const Time& rhs)
    {
        m_time.tv_sec  = rhs.m_time.tv_sec;
        m_time.tv_nsec = rhs.m_time.tv_nsec;
    }

    Time operator+ (const Time& rhs) const
    {
        Time time;
        time.m_time.tv_nsec = m_time.tv_nsec + rhs.m_time.tv_nsec;
        if( time.m_time.tv_nsec >= 1000000000 )
        {
            time.m_time.tv_nsec -= 1000000000;
            time.m_time.tv_sec = 1 + m_time.tv_sec + rhs.m_time.tv_sec;
        }
        else
        {
            time.m_time.tv_sec = m_time.tv_sec + rhs.m_time.tv_sec;
        }

        return time;
    }

    Time operator- (const Time& rhs) const
    {
        Time time;
        time.m_time.tv_sec = m_time.tv_sec - rhs.m_time.tv_sec;

        if( m_time.tv_nsec < rhs.m_time.tv_nsec )
        {
            --time.m_time.tv_sec;
            time.m_time.tv_nsec = 1000000000 - (rhs.m_time.tv_nsec - m_time.tv_nsec);
        }
        else
        {
            time.m_time.tv_nsec = m_time.tv_nsec - rhs.m_time.tv_nsec;
        }

        return time;
    }

    uint64_t GetSeconds() const
    { return m_time.tv_sec; }

    uint64_t GetMilliSeconds() const
    { return m_time.tv_sec * 1000 + (m_time.tv_nsec / 1000000); }

    uint64_t GetMicroSeconds() const
    { return m_time.tv_sec * 1000000 + (m_time.tv_nsec / 1000); }
};

typedef Time Tick;

/* APIs */
inline void ClearEvent(Event*)
{ return; }

inline void LockMutex(Mutex* pMutex)
{ pthread_mutex_lock(pMutex); }

inline void UnlockMutex(Mutex* pMutex)
{ pthread_mutex_unlock(pMutex); }

inline void YieldThread()
{ pthread_yield(); }

inline void DestroyThread(Thread*)
{ return; }

inline void WaitEvent(Event* pEvent)
{ pthread_cond_wait(&pEvent->condition, &pEvent->mutex); }

inline Time TickToTime(Tick tick)
{ return ((Time)(tick)); }

inline Tick TimeToTick(Time time)
{ return ((Tick)(time)); }

inline uint32_t GetRandom()
{
    int value = random();
    return *(uint32_t*)&value;
}

inline void FinalizeEvent(Event* pEvent)
{
    pthread_cond_destroy(&pEvent->condition);
    pthread_mutex_destroy(&pEvent->mutex);
}

inline void SignalEvent(Event* pEvent)
{
    pEvent->flag = true;
    pthread_cond_signal(&pEvent->condition);
}

inline Tick GetTick()
{
    Time time;
    clock_gettime(CLOCK_REALTIME, &time.m_time);
    return static_cast<Tick>(time);
}

inline bool TimedWaitEvent(Event* pEvent, Time time)
{
    Time absaluteTime;
    Time now;
    int rval = 0;
    clock_gettime(CLOCK_REALTIME, &now.m_time);
    absaluteTime = now + time;

    pthread_mutex_lock(&pEvent->mutex);
    while( pEvent->flag == false && rval != ETIMEDOUT )
    {
        int rval = pthread_cond_timedwait(&pEvent->condition, &pEvent->mutex, &absaluteTime.m_time);
        if( rval != 0 && rval != ETIMEDOUT )
        {
            break;
            NN_NETTEST_LOG("\npthread_cond_timedwait: Unexpected error: %d\n\n", rval);
        }
    }

    pEvent->flag = false;
    pthread_mutex_unlock(&pEvent->mutex);
    return ( rval == 0);
}

inline Time TimeFromSec(uint32_t timeSec)
{
    Time time;
    time.m_time.tv_nsec = 0;
    time.m_time.tv_sec = timeSec;
    return time;
}

inline Time TimeFromMs(uint32_t timeMs)
{
    Time time;
    time.m_time.tv_nsec = (timeMs % 1000) * 1000000;
    time.m_time.tv_sec = timeMs / 1000;
    return time;
}

inline void SleepMs(uint32_t timeMs)
{ usleep(timeMs * 1000); }

inline void InitMutex( Mutex* pMutex )
{ pthread_mutex_init(pMutex, NULL); }

inline void InitEvent( Event* pEvent )
{
    pEvent->flag = false;
    pthread_cond_init(&pEvent->condition, NULL);
    pthread_mutex_init(&pEvent->mutex, NULL);
}

inline bool CreateThread( Thread* pThread, ThreadFn pThreadFn, void* pParam, unsigned char* pStack, uint32_t stackSize)
{
    pthread_attr_init(&pThread->threadAttr);
    pthread_attr_setstack(&pThread->threadAttr, pStack, stackSize);
    pThread->threadFunc = pThreadFn;
    pThread->pParam = pParam;

    return true;
}

inline bool StartThread(Thread* pThread)
{
    int rval = pthread_create(&pThread->thread, &pThread->threadAttr, pThread->threadFunc, pThread->pParam);
    pthread_attr_destroy(&pThread->threadAttr);

    return (rval != 0);
}

inline void WaitThread(Thread* pThread)
{
    pthread_join(pThread->thread, NULL);
}

} // namespace NetTest
