﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/account.h>
#include <nn/account/account_ApiForSystemServices.h>
#include <nn/nd/nd_Api.h>
#include <nn/nd/nd_ApiForDebug.h>
#include <nn/nd/nd_ApiForSystem.h>
#include <nn/nd/detail/nd_DataConverter.h>
#include <nn/ndd.h>
#include <nn/nifm.h>
#include <nn/nifm/nifm_ApiForSystem.h>
#include <nn/nifm/nifm_ApiRequest.h>
#include <nn/nifm/nifm_NetworkConnection.h>
#include <nn/os.h>
#include <nn/time.h>
#include <nn/util/util_ScopeExit.h>
#include "testNd_Util.h"

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>

using namespace nn;

namespace
{

// ndd の受信バッファ及び nd のカウンタ値を削除し、Add..ForDebug 類で本体に追加したデータが受信できる状態にする。
void ClearNeighborDetectionData()
{
    ndd::ClearReceiveData();
    for( int i = 0; i < account::UserCountMax; i++ )
    {
        nn::account::Uid uid;
        if( nnt::nd::GetAccountUid(&uid, 0) )
        {
            nd::ClearReceiveCounterForDebug(uid);
        }
    }
}

void AddAsReceiveData(const ndd::SendDataDescription& sendDataDescription)
{
    ndd::ReceiveDataDescription recvDataDescription;
    std::memcpy(recvDataDescription.data, sendDataDescription.data, sendDataDescription.dataSize);
    recvDataDescription.dataSize = sendDataDescription.dataSize;
    ndd::AddReceiveData(recvDataDescription);
}

class NeighborDetectionUtilTest : public testing::Test
{
protected:
    static void SetUpTestCase()
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::Initialize());
    }
    static void TearDownTestCase()
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::Finalize());
    }
    virtual void SetUp()
    {
    }
    virtual void TearDown()
    {
    }
};

class NeighborDetectionSystemTest : public testing::Test
{
protected:
    static void SetUpTestCase()
    {
        account::InitializeForSystemService();
        NN_ABORT_UNLESS_RESULT_SUCCESS(time::Initialize());
        nd::InitializeForSystem();
        ndd::Initialize();
    }
    static void TearDownTestCase()
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::Finalize());
    }
    virtual void SetUp()
    {
        ClearNeighborDetectionData();
    }
    virtual void TearDown()
    {
        ClearNeighborDetectionData();
    }
};

} // ~<anonymous>

// Util のテスト（とりあえずここで）

TEST_F(NeighborDetectionUtilTest, MakeDummyNeighborInfoForsystemWithRandomData)
{
    for( int i = 0; i < 100; i++ )
    {
        nd::NeighborInfoForSystem info0, info1;
        nnt::nd::MakeDummyNeighborInfoForsystemWithRandomData(&info0);
        nnt::nd::MakeDummyNeighborInfoForsystemWithRandomData(&info1);

        // nnt::nd::DumpNeighborInfoForSystem(info0);
        // nnt::nd::DumpNeighborInfoForSystem(info1);

        ASSERT_TRUE(nnt::nd::AreEqualNeighborInfoForSystemExceptReceviedTimePoint(info0, info0));
        ASSERT_TRUE(nnt::nd::AreEqualNeighborInfoForSystemExceptReceviedTimePoint(info1, info1));
        ASSERT_FALSE(nnt::nd::AreEqualNeighborInfoForSystemExceptReceviedTimePoint(info0, info1));
    }
}

// SystemTest

/*
    - 設定したデータがそのユーザーの送信データに反映される。
    - 設定したデータがそのユーザーの送信データを受信したユーザーに届く。
*/
TEST_F(NeighborDetectionSystemTest, SetAndClearSystemData)
{
    account::Uid uid;
    ASSERT_TRUE(nnt::nd::GetAccountUid(&uid, 0));

    // Set/Clear 時に送信データが更新されることを確認するために送信者に設定。
    nd::SetSender(uid);
    // Set
    {
        Bit8 data[nd::SystemDataSizeMax];
        os::GenerateRandomBytes(data, sizeof(data));
        nd::SetSystemData(uid, data, sizeof(data));

        // 送信データの確認
        ndd::SendDataDescription sendDataDescription;
        nd::GetSendDataForDebug(&sendDataDescription, uid);
        auto pPayload = reinterpret_cast<nd::detail::Payload*>(sendDataDescription.data);
        ASSERT_EQ(sizeof(data), pPayload->systemDataSize);
        ASSERT_EQ(0, std::memcmp(data, pPayload->systemData, pPayload->systemDataSize));

        // TODO: nd から送信データ更新検知イベントを出す。
        os::SleepThread(TimeSpan::FromSeconds(1));
        auto count = ndd::GetSendData(&sendDataDescription, 0, 1);
        ASSERT_EQ(1, count);
        pPayload = reinterpret_cast<nd::detail::Payload*>(sendDataDescription.data);
        ASSERT_EQ(sizeof(data), pPayload->systemDataSize);
        ASSERT_EQ(0, std::memcmp(data, pPayload->systemData, pPayload->systemDataSize));

        // 受信データの確認
        auto e = nd::GetNeighborInfoUpdateEventForSystem();
        e->Clear();
        AddAsReceiveData(sendDataDescription);
        ASSERT_TRUE(e->TimedWait(TimeSpan::FromSeconds(3)));
        nd::NeighborInfoForSystem info;
        count = nd::ReceiveNeighborInfoForSystem(&info, 1, uid);
        // nnt::nd::DumpNeighborInfoForSystem(info);
        ASSERT_EQ(1, count);
        ASSERT_EQ(sizeof(data), info.systemDataSize);
        ASSERT_EQ(0, std::memcmp(data, info.systemData, info.systemDataSize));
    }

    // Clear
    {
        // 送信データの確認
        nd::ClearSystemData(uid);
        ndd::SendDataDescription sendDataDescription;
        nd::GetSendDataForDebug(&sendDataDescription, uid);
        auto pPayload = reinterpret_cast<nd::detail::Payload*>(sendDataDescription.data);
        ASSERT_EQ(0u, pPayload->systemDataSize);

        // TODO: nd から送信データ更新検知イベントを出す。
        os::SleepThread(TimeSpan::FromSeconds(1));
        auto count = ndd::GetSendData(&sendDataDescription, 0, 1);
        ASSERT_EQ(1, count);
        pPayload = reinterpret_cast<nd::detail::Payload*>(sendDataDescription.data);
        ASSERT_EQ(0u, pPayload->systemDataSize);

        // 受信データの確認
        auto e = nd::GetNeighborInfoUpdateEventForSystem();
        e->Clear();
        AddAsReceiveData(sendDataDescription);
        ASSERT_TRUE(e->TimedWait(TimeSpan::FromSeconds(3)));
        nd::NeighborInfoForSystem info;
        count= nd::ReceiveNeighborInfoForSystem(&info, 1, uid);
        ASSERT_EQ(1, count);
        ASSERT_EQ(0, info.systemDataSize);
    }
}

/*
    - 設定したデータがそのユーザーの送信データに反映される。
    - 送信ユーザーを切り替えたときに、送信データがそのユーザーのものに切り替わること。
*/
TEST_F(NeighborDetectionSystemTest, SetSender)
{
    const int TestAccountCount = 2; // TORIAEZU: 2ユーザー
    account::Uid uid[TestAccountCount];
    size_t dataSize[TestAccountCount];
    Bit8 data[TestAccountCount][nd::SystemDataSizeMax];
    for( int i = 0; i < TestAccountCount; i++ )
    {
        ASSERT_TRUE(nnt::nd::GetAccountUid(&uid[i], i)) << "There is no account at index[" << i << "]";
        os::GenerateRandomBytes(&dataSize[i], sizeof(dataSize[i]));
        dataSize[i] = (dataSize[i] & 0xff);
        os::GenerateRandomBytes(data[i], sizeof(data[i]));
        nd::SetSystemData(uid[i], data[i], dataSize[i]);
    }

    for( int i = 0; i < TestAccountCount; i++ )
    {
        nd::SetSender(uid[i]);
        account::Uid sender;
        ASSERT_TRUE(nd::GetSender(&sender));
        ASSERT_EQ(uid[i], sender);

        // TODO: nd から送信データ更新検知イベントを出す。
        os::SleepThread(TimeSpan::FromSeconds(1));

        // 送信データの確認
        ndd::SendDataDescription sendDataDescription;
        auto sendDataCount = ndd::GetSendData(&sendDataDescription, 0, 1);
        ASSERT_EQ(1, sendDataCount);
        auto pPayload = reinterpret_cast<nd::detail::Payload*>(sendDataDescription.data);
        ASSERT_EQ(dataSize[i], pPayload->systemDataSize);
        ASSERT_EQ(0, std::memcmp(data[i], pPayload->systemData, pPayload->systemDataSize));
    }
}

/*
  - 新しいデータを受信したときにイベントがシグナルされること。
  - イベントシグナル後にデータを取得したとき、新しいデータが取得できること。
  - データが順番通りに取得できること。
*/
TEST_F(NeighborDetectionSystemTest, WaitEventAndReceive)
{
    const int ReceiveCount = 20;
    static nd::NeighborInfoForSystem dummyInfo[ReceiveCount];

    account::Uid uid; // TORIAEZU : 1ユーザー
    ASSERT_TRUE(nnt::nd::GetAccountUid(&uid, 0));

    // データ追加を別スレッドで動かす。
    auto threadId = nnt::nd::RunOnThread([](void*) {
        for( int i = 0; i < ReceiveCount; i++ )
        {
            nnt::nd::MakeDummyNeighborInfoForsystemWithRandomData(&dummyInfo[i]);
            // nnt::nd::DumpNeighborInfoForSystem(dummyInfo[i]);
            nd::AddReceivedNeighborInfoForSystemForDebug(&dummyInfo[i], 1);
            Bit8 waitMilliseconds;
            os::GenerateRandomBytes(&waitMilliseconds, 1u);
            os::SleepThread(nn::TimeSpan::FromMilliSeconds(waitMilliseconds)); // 0 ~ 255 ms 待つ。
        }
    }, nullptr);
    NN_UTIL_SCOPE_EXIT{ nnt::nd::DestroyThread(threadId); };

    auto e = nd::GetNeighborInfoUpdateEventForSystem();
    int totalReceivedCount = 0;
    while( totalReceivedCount < ReceiveCount )
    {
        if( !e->TryWait() )
        {
            os::SleepThread(nn::TimeSpan::FromMilliSeconds(30));
            continue;
        }
        e->Clear();
        int receivableCount = nd::GetReceivableNeighborInfoCountForSystem(uid);
        for( int i = 0; i < receivableCount; i++ )
        {
            nd::NeighborInfoForSystem info = {};
            int receivedCount = nd::ReceiveNeighborInfoForSystem(&info, 1, uid);
            // nnt::nd::DumpNeighborInfoForSystem(info);
            ASSERT_EQ(receivedCount, 1);
            ASSERT_TRUE(nnt::nd::AreEqualNeighborInfoForSystemExceptReceviedTimePoint(info, dummyInfo[totalReceivedCount + i])) << i;
        }
        totalReceivedCount += receivableCount;
    }
}

/*
  - 取得可能最大数のデータを取得できること。
*/
TEST_F(NeighborDetectionSystemTest, ReceiveMultipleData)
{
    // デフォルトの最大取得可能数。取得可能数が設定可能にした際にはその値を使うようにすること。
    const int ReceiveCount = 10;
    static nd::NeighborInfoForSystem dummyInfo[ReceiveCount];

    account::Uid uid;
    ASSERT_TRUE(nnt::nd::GetAccountUid(&uid, 0));

    for( int i = 0; i < ReceiveCount; i++ )
    {
        nnt::nd::MakeDummyNeighborInfoForsystemWithRandomData(&dummyInfo[i]);
    }
    // （nd::AddReceivedNeighborInfoForSystemForDebug が呼ぶ）nd::AddReceiveData を高頻度で呼ぶと ndd のキュー（最大数20）が溢れてアボートしてしまう。
    // テスト順によっては問題になるので、適当なスリープを置いて回避する。将来的には ndd 内でブロックするようになるので回避策は不要になる。
    os::SleepThread(TimeSpan::FromSeconds(1));
    nd::AddReceivedNeighborInfoForSystemForDebug(dummyInfo, ReceiveCount);

    while( NN_STATIC_CONDITION(true) )
    {
        int receivableCount = nd::GetReceivableNeighborInfoCountForSystem(uid);
        if( receivableCount == ReceiveCount )
        {
            break;
        }
        NN_LOG("Waiting for receivableCount to become %d. Current receivableCount is %d.\n", ReceiveCount, receivableCount);
        os::SleepThread(TimeSpan::FromMilliSeconds(100));
    }

    static nd::NeighborInfoForSystem receiveInfo[ReceiveCount];
    int receiveCount = nd::ReceiveNeighborInfoForSystem(receiveInfo, NN_ARRAY_SIZE(receiveInfo), uid);
    ASSERT_EQ(ReceiveCount, receiveCount);

    for( int i = 0; i < receiveCount; i++ )
    {
        EXPECT_TRUE(nnt::nd::AreEqualNeighborInfoForSystemExceptReceviedTimePoint(receiveInfo[i], dummyInfo[i])) << i;
    }
}


/*
  - 取得可能最大数より多くのデータ数が貯まっている際に、データの取得数が正しく制限されること。
*/
TEST_F(NeighborDetectionSystemTest, ReceiveDataLimit)
{
    // デフォルトの最大取得可能数。取得可能数が設定可能にした際にはその値を使うようにすること。
    const int ReceiveCount = 10;
    const int AddCount = 20;
    static nd::NeighborInfoForSystem dummyInfo[AddCount];

    account::Uid uid;
    ASSERT_TRUE(nnt::nd::GetAccountUid(&uid, 0));

    for( size_t i = 0; i < AddCount; i++ )
    {
        nnt::nd::MakeDummyNeighborInfoForsystemWithRandomData(&dummyInfo[i]);
    }
    // （nd::AddReceivedNeighborInfoForSystemForDebug が呼ぶ）nd::AddReceiveData を高頻度で呼ぶと ndd のキュー（最大数20）が溢れてアボートしてしまう。
    // テスト順によっては問題になるので、適当なスリープを置いて回避する。将来的には ndd 内でブロックするようになるので回避策は不要になる。
    os::SleepThread(TimeSpan::FromSeconds(1));
    nd::AddReceivedNeighborInfoForSystemForDebug(dummyInfo, AddCount);

    while( NN_STATIC_CONDITION(true) )
    {
        // nd で取得できる最大数は超えるため、ndd に貯まった数で判定する。
        int availableReceiveDataCount = ndd::GetAvailableReceiveDataCount(ndd::GetOldestReceiveDataCounter());
        if( availableReceiveDataCount == AddCount )
        {
            break;
        }
        NN_LOG("Waiting for availableReceiveDataCount to become %d. Current availableReceiveDataCount is %d.\n", AddCount, availableReceiveDataCount);
        os::SleepThread(TimeSpan::FromMilliSeconds(100));
    }

    // ndd からの取得可能数は ReceiveCount でキャップされる。
    ASSERT_EQ(ReceiveCount, nd::GetReceivableNeighborInfoCountForSystem(uid));
    static nd::NeighborInfoForSystem receiveInfo[ReceiveCount];
    int receiveCount = nd::ReceiveNeighborInfoForSystem(receiveInfo, NN_ARRAY_SIZE(receiveInfo), uid);
    ASSERT_EQ(ReceiveCount, receiveCount);

    for( int i = 0; i < receiveCount; i++ )
    {
        // データは新しい方から最大数分取得される。
        EXPECT_TRUE(nnt::nd::AreEqualNeighborInfoForSystemExceptReceviedTimePoint(receiveInfo[i], dummyInfo[AddCount - ReceiveCount + i])) << i;
    }
}

// TORIAEZU: サイズは適当。
static NN_ALIGNAS(os::MemoryPageSize) Bit8 scannerWorkBuffer[os::MemoryPageSize * 3];

TEST_F(NeighborDetectionSystemTest, ScanBasic)
{
    account::Uid uid;
    ASSERT_TRUE(nnt::nd::GetAccountUid(&uid, 0));

    NN_ABORT_UNLESS_RESULT_SUCCESS(nifm::InitializeSystem());
    nifm::NetworkConnection connection;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nifm::SetRequestRequirementPreset(connection.GetRequestHandle(), nn::nifm::RequirementPreset_NeighborDetectionForApplet));
    connection.SubmitRequestAndWait();
    ASSERT_TRUE(connection.IsAvailable());

    auto scanner = nd::CreateScannerForSystem(scannerWorkBuffer, sizeof(scannerWorkBuffer));
    for( int scanCount = 0; scanCount < 3; scanCount++ )
    {
        NN_LOG("Scan %d...\n", scanCount + 1);
        scanner.StartScan();
        scanner.GetSystemEvent()->Wait();
        // scanner.GetSystemEvent()->Clear();
        nn::nd::NeighborInfoForSystem info;
        auto count = scanner.GetScanResult(&info, 1, uid);
        NN_LOG("count = %d\n", count);
        for( int i = 0; i < count; i++ )
        {
            nnt::nd::DumpNeighborInfoForSystem(info);
        }
    }
}

TEST_F(NeighborDetectionSystemTest, ScanMany)
{
    account::Uid uid;
    ASSERT_TRUE(nnt::nd::GetAccountUid(&uid, 0));

    NN_ABORT_UNLESS_RESULT_SUCCESS(nifm::InitializeSystem());
    nifm::NetworkConnection connection;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nifm::SetRequestRequirementPreset(connection.GetRequestHandle(), nn::nifm::RequirementPreset_NeighborDetectionForApplet));
    connection.SubmitRequestAndWait();
    ASSERT_TRUE(connection.IsAvailable());

    auto scanner = nd::CreateScannerForSystem(scannerWorkBuffer, sizeof(scannerWorkBuffer));
    // 大きな受信データ数を指定。
    scanner.StartScan();
    scanner.GetSystemEvent()->Wait();
    nn::nd::NeighborInfoForSystem info[50];
    auto count = scanner.GetScanResult(info, NN_ARRAY_SIZE(info), uid);
    NN_LOG("count = %d\n", count);
    for( int i = 0; i < count; i++ )
    {
        nnt::nd::DumpNeighborInfoForSystem(info[i]);
    }
}

TEST_F(NeighborDetectionSystemTest, ScanCancel)
{
    account::Uid uid;
    ASSERT_TRUE(nnt::nd::GetAccountUid(&uid, 0));

    auto scanner = nd::CreateScannerForSystem(scannerWorkBuffer, sizeof(scannerWorkBuffer));
    for( int scanCount = 0; scanCount < 3; scanCount++ )
    {
        NN_LOG("Scan %d...\n", scanCount + 1);
        scanner.StartScan();
        scanner.CancelScan();
        scanner.GetSystemEvent()->Wait();
        // scanner.GetSystemEvent()->Clear();
        nn::nd::NeighborInfoForSystem info;
        auto count = scanner.GetScanResult(&info, 1, uid);
        NN_LOG("count = %d\n", count);
        for( int i = 0; i < count; i++ )
        {
            nnt::nd::DumpNeighborInfoForSystem(info);
        }
    }
}

// 死ぬテストなのでコメントアウト。
//TEST_F(NeighborDetectionSystemTest, CreateSecondScanner)
//{
//    auto scanner = nd::CreateScannerForSystem(scannerWorkBuffer, sizeof(scannerWorkBuffer));
//    auto scanner2 = nd::CreateScannerForSystem(scannerWorkBuffer, sizeof(scannerWorkBuffer));
//}


TEST_F(NeighborDetectionSystemTest, CreateScannerRepeatedly)
{
    // Scanner の作成、イベントの取得と破壊の繰り返し。メモリリークで死んだりしないかどうか。
    for( int i = 0; i < 10000; i++ )
    {
        auto scanner = nd::CreateScannerForSystem(scannerWorkBuffer, sizeof(scannerWorkBuffer));
        scanner.GetSystemEvent();
    }
}

TEST_F(NeighborDetectionSystemTest, Move)
{
    account::Uid uid;
    ASSERT_TRUE(nnt::nd::GetAccountUid(&uid, 0));
    nn::nd::NeighborInfoForSystem info;

    for( int i = 0; i < 1000; i++ )
    {
        // 作成
        auto scanner = nd::CreateScannerForSystem(scannerWorkBuffer, sizeof(scannerWorkBuffer));
        scanner.StartScan();
        scanner.CancelScan();
        scanner.GetSystemEvent()->Wait();
        scanner.GetScanResult(&info, 1, uid);

        // ムーブコンストラクタ
        nd::ScannerForSystem scanner2(std::move(scanner));
        scanner2.StartScan();
        scanner2.CancelScan();
        scanner2.GetSystemEvent()->Wait();
        scanner2.GetScanResult(&info, 1, uid);

        // ムーブ代入演算子
        nd::ScannerForSystem scanner3;
        scanner3 = std::move(scanner2);
        scanner3.StartScan();
        scanner3.CancelScan();
        scanner3.GetSystemEvent()->Wait();
        scanner3.GetScanResult(&info, 1, uid);

        // swap
        nd::ScannerForSystem scanner4;
        scanner4.swap(scanner3);
        scanner4.StartScan();
        scanner4.CancelScan();
        scanner4.GetSystemEvent()->Wait();
        scanner4.GetScanResult(&info, 1, uid);
    }
}

// TODO: エージング用のテストを別に作る。
//TEST_F(NeighborDetectionSystemTest, ScanAging)
//{
//    account::Uid uid;
//    ASSERT_TRUE(nnt::nd::GetAccountUid(&uid, 0));
//
//    auto scanner = nd::CreateScannerForSystem(scannerWorkBuffer, sizeof(scannerWorkBuffer));
//    for( int scanCount = 0; scanCount < 1000; scanCount++ )
//    {
//        NN_LOG("Scan %d...\n", scanCount + 1);
//        scanner.StartScan();
//        scanner.GetSystemEvent()->Wait();
//        nn::nd::NeighborInfoForSystem info;
//        auto count = scanner.GetScanResult(&info, 1, uid);
//        NN_LOG("count = %d\n", count);
//        for( int i = 0; i < count; i++ )
//        {
//            nnt::nd::DumpNeighborInfoForSystem(info);
//        }
//    }
//}
