﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
/* Copyright (c) 2015-16, NVIDIA CORPORATION.  All rights reserved.
*
* NVIDIA Corporation and its licensors retain all intellectual property
* and proprietary rights in and to this software, related documentation
* and any modifications thereto.  Any use, reproduction, disclosure or
* distribution of this software and related documentation without an
* express license agreement from NVIDIA Corporation is strictly prohibited.
*/

//#define MOUNT_SDCARD
#include "nvnflinger_service.h"
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/fs.h>
#include <nn/tma/tma.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/mem/mem_StandardAllocator.h>
#include <nv/nv_MemoryManagement.h>
#include <nv/nv_ServiceName.h>
#include <binder/Parcel.h>
#include "jpeglib.h"
#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <map>
#ifdef MOUNT_SDCARD
#include <nn/fs/fs_SdCardForDebug.h>
#endif
#include "mm_MemoryManagement.h"

using namespace android;
using namespace nn::nvjpg;

#define LOG(fmt, ...) NN_LOG("%s <%s:%d> " fmt, "NvJpgScoop:", __FUNCTION__, __LINE__, ##__VA_ARGS__)
#define ARRAY_SIZE(x) (sizeof(x)/sizeof((x)[0]))

enum JpegSamplingRatio
{
    JpegSamplingRatio_444 = 0,
    JpegSamplingRatio_422 = 1,
    JpegSamplingRatio_420 = 2
};

typedef struct JpegEncoder {
    uint8_t                 *pReadYuv;
    uint8_t                 *pJpegData;
    uint8_t                 *pWorkBuf;
    jpeg_compress_struct    cinfo;
    jpeg_workbuf            wbMgr = {};
    struct jpeg_error_mgr   jerr;
    uint32_t                Width;
    uint32_t                Height;
    unsigned long           codeSize;
    JSAMPROW                *lines; //This is actually not used by the HW path.
    JpegSamplingRatio       kSample;
    int                     rgb;
    float                   BytesPerPixel;
    uint32_t                kWorkBufSize;
    uint32_t                JpegDataSize;
} JpegEnc;

typedef struct Crc_t {
    uint32_t Value;
    bool   Enabled;
} CrcData;

static const int g_FsHeapSize = 512 * 1024;
static const int mmHeapSize = 200 * 1024 * 1024;
static const int mmFirmwareMemorySize = 8 * 1024 * 1024;
static uint8_t g_FsHeapBuffer[g_FsHeapSize];
static nn::lmem::HeapHandle g_FsHeap;
static char                        g_mmHeapBuffer[mmHeapSize];
static nn::mem::StandardAllocator  g_MultimediaAllocator(g_mmHeapBuffer, sizeof(g_mmHeapBuffer));
static char                        g_mmFirmwareMemory[mmFirmwareMemorySize] __attribute__((aligned(4096)));

static const gralloc_module_t *g_grmodule = NULL;
static alloc_device_t *g_gralloc = NULL;

void JpegEncInit(JpegEnc **pJpegEnc, uint16_t width, uint16_t height, int rgb);
void JpegGetBuffer(JpegEnc *pJpegEnc);
void JpegEncStart(JpegEnc *pJpegEnc);
void JpegEncFinish(JpegEnc *pJpegEnc);
void JpegEncSaveFile(const char *pJpegFile);
void JpegEncClose(JpegEnc *pJpegEnc);

static JpegEnc *pJpegEncoder;

const unsigned long int CRC32_POLYNOMIAL = 0xEDB88320L;
static NvU32 CRCTable[256];
static void BuildCRCTable()
{
    NvU16 i;
    NvU16 j;
    NvU32 crc;
    for (i = 0; i <= 255; i++)
    {
        crc = i;
        for (j = 8; j > 0; j--)
        {
            if (crc & 1)
            {
                crc = (crc >> 1) ^ CRC32_POLYNOMIAL;
            }
            else
            {
                crc >>= 1;
            }
        }
        CRCTable[i] = crc;
    }
}


static int32_t CalculateBufferCRC(uint32_t count, uint32_t crc, uint8_t *buffer)
{
    uint8_t *p;
    uint32_t temp1;
    uint32_t temp2;
    p = (uint8_t*)buffer;
    while (count-- != 0)
    {
        temp1 = (crc >> 8) & 0x00FFFFFFL;
        temp2 = CRCTable[((uint32_t)crc ^ *p++) & 0xFF];
        crc = temp1 ^ temp2;
    }
    return crc;
}

void JpegEncInit(JpegEnc **pJpegEncoder, uint16_t width, uint16_t height, int rgb)
{
    jpeg_compress_struct *pCinfo;
    JpegEnc *pJpegEnc;

    *pJpegEncoder = (JpegEnc *)NvOsAlloc(sizeof(JpegEnc));

    pJpegEnc = *pJpegEncoder;
    pCinfo = &pJpegEnc->cinfo;
    pJpegEnc->Width = width;
    pJpegEnc->Height = height;
    pJpegEnc->kSample = JpegSamplingRatio_420;
    pJpegEnc->BytesPerPixel = rgb ? 4 : 1.5;
    pJpegEnc->kWorkBufSize = pJpegEnc->Width * 30 +     // Large memory allocation used by libjpeg
                             pJpegEnc->Width * 16 +     // For strip buffer used by libjpeg
                             1600 +                     // For first PERMANENT pool
                             16000 +                    // For first IMAGE pool
                             1024;                      // For alignment
    pJpegEnc->JpegDataSize  = pJpegEnc->Width * pJpegEnc->Height * pJpegEnc->BytesPerPixel;
    pJpegEnc->pJpegData = (uint8_t *)NvOsAlloc(pJpegEnc->JpegDataSize);
    pJpegEnc->pWorkBuf  = (uint8_t *)NvOsAlloc(pJpegEnc->kWorkBufSize);

    pJpegEnc->wbMgr.ptr = reinterpret_cast<JSAMPLE*>(pJpegEnc->pWorkBuf);
    pJpegEnc->wbMgr.total = pJpegEnc->kWorkBufSize;
    pCinfo->workbuf = &pJpegEnc->wbMgr;

    pCinfo->err = jpeg_std_error(&pJpegEnc->jerr);

    jpeg_create_compress(pCinfo);

    pJpegEnc->codeSize = pJpegEnc->JpegDataSize;
    jpeg_mem_dest(pCinfo, reinterpret_cast<uint8_t**>(&(pJpegEnc->pJpegData)), &pJpegEnc->codeSize);

    if (rgb)
    {
        pCinfo->in_color_space = JCS_RGB;
        pCinfo->tegra_acceleration = false;
    }
    else
    {
        pCinfo->in_color_space = JCS_YCbCr;
        pCinfo->tegra_acceleration = true;
    }

    jpeg_set_defaults(pCinfo);
    jpeg_set_colorspace(pCinfo, JCS_YCbCr);

    if (!rgb)
    {
        pJpegEnc->kSample = JpegSamplingRatio_420;
        switch (pJpegEnc->kSample)
        {
            case JpegSamplingRatio_444:
                pCinfo->comp_info[0].h_samp_factor = 1;
                pCinfo->comp_info[0].v_samp_factor = 1;
                pCinfo->comp_info[1].h_samp_factor = pCinfo->comp_info[2].h_samp_factor = 1;
                pCinfo->comp_info[1].v_samp_factor = pCinfo->comp_info[2].v_samp_factor = 1;
                break;
            case JpegSamplingRatio_422:
                pCinfo->comp_info[0].h_samp_factor = 2;
                pCinfo->comp_info[0].v_samp_factor = 1;
                pCinfo->comp_info[1].h_samp_factor = pCinfo->comp_info[2].h_samp_factor = 1;
                pCinfo->comp_info[1].v_samp_factor = pCinfo->comp_info[2].v_samp_factor = 1;
                break;
            case JpegSamplingRatio_420:
                pCinfo->comp_info[0].h_samp_factor = 2;
                pCinfo->comp_info[0].v_samp_factor = 2;
                pCinfo->comp_info[1].h_samp_factor = pCinfo->comp_info[2].h_samp_factor = 1;
                pCinfo->comp_info[1].v_samp_factor = pCinfo->comp_info[2].v_samp_factor = 1;
                break;
            default:
                NN_UNEXPECTED_DEFAULT;
        }
    }

    jpeg_set_quality(pCinfo, 100, true);
    pCinfo->input_components = 3;
    if (rgb)
        pCinfo->image_width = pJpegEnc->Width;
    else
        pCinfo->image_width = pJpegEnc->Width;
    pCinfo->image_height = pJpegEnc->Height;
    pCinfo->outputBuffSize = pJpegEnc->Width * pJpegEnc->Height * pJpegEnc->BytesPerPixel;
    pCinfo->dct_method = JDCT_ISLOW;
    pCinfo->optimize_coding = false;
    pCinfo->do_fancy_downsampling = false;
}

void JpegGetBuffer(JpegEnc *pJpegEnc)
{
    jpeg_start_compress(&pJpegEnc->cinfo, TRUE);
}

void JpegEncStart(JpegEnc *pJpegEnc)
{
    jpeg_compress_struct *pCinfo = &pJpegEnc->cinfo;

    (void)jpeg_write_scanlines(pCinfo, pJpegEnc->lines, (pJpegEnc->Height - pCinfo->next_scanline));
}

void JpegEncFinish(JpegEnc *pJpegEnc)
{
    jpeg_finish_compress(&pJpegEnc->cinfo);
    jpeg_destroy_compress(&pJpegEnc->cinfo);
}

void JpegEncSaveFile(const char *pJpegFile)
{
    NvError Status;
    NvOsFileHandle fpInputBuffer;
    Status = NvOsFopen(pJpegFile, NVOS_OPEN_CREATE | NVOS_OPEN_WRITE | NVOS_OPEN_READ, &fpInputBuffer);
    NvOsFwrite(fpInputBuffer, pJpegEncoder->pJpegData, pJpegEncoder->codeSize);
    NvOsFclose(fpInputBuffer);
}

void JpegEncClose(JpegEnc *pJpegEnc)
{
    NvOsFree(pJpegEnc->pJpegData);
    NvOsFree(pJpegEnc->pWorkBuf);
    NvOsFree(pJpegEncoder);
}

static void CopySurfaceToTegraBuffer(
    jpeg_compress_struct *pCinfo,
    struct android_ycbcr *ycbcr, int32_t width, int32_t height)
{
    const uint8_t *src = (const uint8_t *)ycbcr->y;
    const uint8_t *srcU = (const uint8_t *)ycbcr->cb;
    const uint8_t *srcV = (const uint8_t *)ycbcr->cr;
    uint8_t *dstY = pCinfo->jpegTegraMgr->buff[0];
    uint8_t *dstU = pCinfo->jpegTegraMgr->buff[2];
    uint8_t *dstV = pCinfo->jpegTegraMgr->buff[1];

    for (size_t y = height; y > 0; --y) {
        memcpy(dstY, src, width);
        dstY += pCinfo->jpegTegraMgr->pitch[0];
        src += ycbcr->ystride;
    }

    if (ycbcr->cstride == ycbcr->ystride >> 1 && ycbcr->chroma_step == 1)
    {
        // planar
        for (size_t y = height >> 1; y > 0; --y) {
            memcpy(dstU, srcU, width >> 1);
            dstU += pCinfo->jpegTegraMgr->pitch[2];
            srcU += ycbcr->cstride;
            memcpy(dstV, srcV, width >> 1);
            dstV += pCinfo->jpegTegraMgr->pitch[1];
            srcV += ycbcr->cstride;
        }
    }
}

static bool
getDisplaySize(sp<SurfaceComposerClient> &client,
               int32_t id,
               int &width,
               int &height)
{
    sp<IBinder> dpy = client->getBuiltInDisplay(id);
    if (dpy == NULL)
        return false;

    DisplayInfo info;
    status_t err = client->getDisplayInfo(dpy, &info);
    if (err != NO_ERROR)
        return false;

    width = info.w;
    height = info.h;
    return true;
}

static void
takeScreenshot(const char *filename, CrcData *pCrc)
{
    sp<SurfaceComposerClient> client;
    sp<IBinder> dpy;
    sp<IGraphicBufferProducer> producer;
    sp<IGraphicBufferConsumer> consumer;
    buffer_handle_t buffers[2];
    struct android_ycbcr ycbcr;
    status_t err;
    nn::TimeSpan interval;
    nn::os::TimerEventType retryTimer;
    nn::os::TimerEventType failTimer;
    bool ok;
    int width = 256;
    int height = 256;
    uint32_t format = HAL_PIXEL_FORMAT_YV12; //PIXEL_FORMAT_RGBA_8888;
    uint32_t usage = GRALLOC_USAGE_HW_COMPOSER
                   | GRALLOC_USAGE_HW_VIDEO_ENCODER;
    uint32_t Crc;

    for (size_t i = 0; i < ARRAY_SIZE(buffers); i++)
    {
        buffers[i] = NULL;
    }

    client = new SurfaceComposerClient();
    dpy = client->createDisplay(String8("screenshot"), false);
    if (dpy == NULL)
    {
        LOG("Failed to create virtual display\n");
        goto end;
    }
    ok = getDisplaySize(client,
                        ISurfaceComposer::eDisplayIdHdmi,
                        width,
                        height);
    if (ok == false)
    {
        getDisplaySize(client,
                       ISurfaceComposer::eDisplayIdMain,
                       width,
                       height);
    }
    err = client->getDisplaySurface(dpy, producer, consumer);
    if (err)
    {
        LOG("Failed to get virtual display buffer queue: %d (%s)\n",
            err, strerror(-err));
        goto end;
    }
    if (producer == NULL || consumer == NULL)
    {
        LOG("Failed to virtual display buffer queue (invalid)\n");
        goto end;
    }

    SurfaceComposerClient::openGlobalTransaction();
    client->setDisplaySize(dpy, width, height);
    SurfaceComposerClient::closeGlobalTransaction();

    consumer->setConsumerUsageBits(usage);
    for (size_t i=0; i<ARRAY_SIZE(buffers); i++)
    {
        int stride;
        int result;

        result = g_gralloc->alloc(g_gralloc,
                                  width,
                                  height,
                                  format,
                                  usage,
                                  &buffers[i],
                                  &stride);

        sp<GraphicBuffer> gbuf;
        gbuf = new GraphicBuffer(width,
                                 height,
                                 format,
                                 usage,
                                 stride,
                                 const_cast<native_handle_t *>(buffers[i]),
                                 false);

        producer->setPreallocatedBuffer(i, gbuf);
    }

    consumer->setDefaultBufferSize(width, height);
    consumer->setDefaultBufferFormat(format);

    nn::os::InitializeTimerEvent(&retryTimer, nn::os::EventClearMode_AutoClear);
    nn::os::InitializeTimerEvent(&failTimer, nn::os::EventClearMode_AutoClear);

    interval = nn::TimeSpan::FromSeconds(10);
    nn::os::StartOneShotTimerEvent(&failTimer, interval);

    JpegEncInit(&pJpegEncoder, width, height, 0);

    LOG("Taking screenshot at %dx%d\n", width, height);

    while (true)
    {
        LOG("waiting for screenshot...\n");
        IGraphicBufferConsumer::BufferItem item;
        err = consumer->acquireBuffer(&item, 0);

        if (err == NO_ERROR)
        {
            if (item.mFence != NULL)
                item.mFence->waitForever("screenshot");

            const NvRmSurface *planes;
            size_t numPlanes;

            nvgr_get_surfaces(buffers[item.mBuf], &planes, &numPlanes);

            status_t res;

            res = g_grmodule->lock_ycbcr(g_grmodule,
                        buffers[item.mBuf], GRALLOC_USAGE_SW_READ_OFTEN | GRALLOC_USAGE_SW_WRITE_NEVER,
                        0, 0, width, height,
                        &ycbcr);
            if (res != OK) {
                ALOGE("Unable to lock image buffer %p for access", buffers[item.mBuf]);
                return;
            }

            JpegGetBuffer(pJpegEncoder);

            CopySurfaceToTegraBuffer(&pJpegEncoder->cinfo, &ycbcr, width, height);

            if (g_grmodule->unlock(g_grmodule, buffers[item.mBuf]) != OK) {
                ALOGE("Unable to unlock image buffer %p for access", buffers[item.mBuf]);
            }

            consumer->releaseBuffer(item.mBuf,
                                    item.mFrameNumber,
                                    EGL_NO_DISPLAY,
                                    EGL_NO_SYNC_KHR,
                                    Fence::NO_FENCE);

            // Do JPEG encoding here
            JpegEncStart(pJpegEncoder);

            JpegEncFinish(pJpegEncoder);

            if (pCrc->Enabled)
            {
                Crc = CalculateBufferCRC(pJpegEncoder->codeSize, 0, pJpegEncoder->pJpegData);
                if (pCrc->Value == Crc)
                    NN_SDK_LOG("%s:%d Test PASSED\n", __func__, __LINE__);
                else
                    NN_SDK_LOG("%s:%d Test FAILED\n", __func__, __LINE__);
            }

            JpegEncSaveFile(filename);

            JpegEncClose(pJpegEncoder);

            break;
        }

        if (err != BufferQueue::NO_BUFFER_AVAILABLE)
        {
            LOG("Failed to get virtual display buffer: %d (%s)\n",
                err, strerror(-err));
            break;
        }

        if (nn::os::TryWaitTimerEvent(&failTimer))
        {
            LOG("Failed to get screenshot after 10 seconds\n");
            break;
        }

        interval = nn::TimeSpan::FromSeconds(1);
        nn::os::StartOneShotTimerEvent(&retryTimer, interval);
        nn::os::WaitTimerEvent(&retryTimer);
    }

    nn::os::FinalizeTimerEvent(&retryTimer);
    nn::os::FinalizeTimerEvent(&failTimer);

end:
    if (dpy != NULL)
        client->destroyDisplay(dpy);
    dpy = NULL;
    producer = NULL;
    consumer = NULL;
    client = NULL;

    for (size_t i=0; i<ARRAY_SIZE(buffers); ++i)
    {
        if (buffers[i] != NULL)
        {
            g_gralloc->free(g_gralloc, buffers[i]);
            buffers[i] = NULL;
        }
    }
}//NOLINT(impl/function_size)

static void
NVNFlinger_Util(const char *outFile, CrcData *pCrc)
{
    int result;
    const hw_module_t *module;
    alloc_device_t *alloc;

    result = hw_get_module(GRALLOC_HARDWARE_MODULE_ID, &module);
    if (result != 0)
    {
        LOG("Failed to open gralloc module: %d (%s)\n",
             result, strerror(-result));
        return;
    }

    result = gralloc_open(module, &alloc);
    if (result != 0)
    {
        LOG("Failed to open gralloc allocator: %d (%s)\n",
             result, strerror(-result));
        return;
    }

    g_grmodule = (const gralloc_module_t *) module;
    g_gralloc = alloc;

    takeScreenshot(outFile, pCrc);

    gralloc_close(g_gralloc);
    g_gralloc = NULL;
    g_grmodule = NULL;
}

static void *
FsAllocate(size_t size)
{
    return nn::lmem::AllocateFromExpHeap(g_FsHeap, size);
}

static void
FsDeallocate(void *ptr,
             size_t size)
{
    NN_UNUSED(size);
    return nn::lmem::FreeToExpHeap(g_FsHeap, ptr);
}

std::map<void*, size_t> g_Allocs;
nn::os::Mutex g_AllocMutex(true);
static void* MultimediaAllocate(size_t size, size_t alignment, void *userPtr)
{
    g_AllocMutex.Lock();
    void *ptr = g_MultimediaAllocator.Allocate(size, alignment);
    g_Allocs[ptr] = size;
    g_AllocMutex.Unlock();

    return ptr;
}

static void MultimediaFree(void *addr, void *userPtr)
{
    if(!addr)
        return;

    g_AllocMutex.Lock();
    std::map<void*, size_t>::iterator it = g_Allocs.find(addr);
    NN_ASSERT(it != g_Allocs.end());
    g_Allocs.erase(it);
    g_MultimediaAllocator.Free(addr);
    g_AllocMutex.Unlock();
}

static void *MultimediaReallocate(void* addr, size_t newSize, void *userPtr)
{
    g_AllocMutex.Lock();

    if(addr)
    {
        std::map<void*, size_t>::iterator it = g_Allocs.find(addr);
        NN_ASSERT(it != g_Allocs.end());
        g_Allocs.erase(it);
    }

    void *ptr = g_MultimediaAllocator.Reallocate(addr, newSize);
    g_Allocs[ptr] = newSize;
    g_AllocMutex.Unlock();

    return ptr;
}

extern "C" void
nninitStartup()
{
    const size_t heapSize = 100 * 1024 * 1024;
    const size_t mallocSize = 84 * 1024 * 1024;
    uintptr_t address;
    nn::Result result;

    result = nn::os::SetMemoryHeapSize(heapSize);
    NN_ASSERT(result.IsSuccess());

    result = nn::os::AllocateMemoryBlock(&address, mallocSize);
    NN_ASSERT(result.IsSuccess());

    nn::init::InitializeAllocator(reinterpret_cast<void*>(address), mallocSize);

    g_FsHeap = nn::lmem::CreateExpHeap(g_FsHeapBuffer,
                                       g_FsHeapSize,
                                       nn::lmem::CreationOption_DebugFill);
    nn::fs::SetAllocator(FsAllocate, FsDeallocate);
}

static void displayUsage()
{
    LOG("usage: %s --screenshot <filename> --crc <RefValue> [--crc is optional]\n", nn::os::GetHostArgv()[0]);
}

void ScoopImageNvjpeg_test()
{
    tma::Initialize();

    int argc = nn::os::GetHostArgc();
    char **argv = nn::os::GetHostArgv();
#ifdef MOUNT_SDCARD
    bool sdcardMounted = false;
#endif
    const char *outFile = NULL;
    CrcData Crc;

    Crc.Enabled = false;

    for (int i = 1; i < argc; i++)
    {
        if(argc - i < 2)
            break;

        else if(!strcmp(argv[i], "--screenshot"))
            outFile = argv[++i];
        else if(!strcmp(argv[i], "--crc"))
        {
            Crc.Enabled = true;
            sscanf(argv[++i], "%8x\n", &Crc.Value);
        }
        else
        {
            displayUsage();
            return;
        }
    }

#ifdef MOUNT_SDCARD
    nn::Result resultSdcardMount = nn::fs::MountSdCardForDebug("sdcard");
    if( resultSdcardMount.IsFailure() )
    {
        NN_SDK_LOG( "\n nn::fs::SD card mount failure. Module:%d, Description:%d\n" ,
                resultSdcardMount.GetModule(),
                resultSdcardMount.GetDescription());
        return;
    }
    sdcardMounted = true;
#else
    NN_ASSERT(nn::fs::MountHostRoot().IsSuccess());
    NN_ASSERT(nn::fs::MountHost("host", "c:/HosHostFs").IsSuccess());
#endif

    BuildCRCTable();

    NVNFlinger_Util(outFile, &Crc);

#ifdef MOUNT_SDCARD
    if( sdcardMounted == true )
    {
        nn::fs::Unmount("sdcard");
    }
#else
    nn::fs::UnmountHostRoot();
    nn::fs::Unmount("host");
#endif
    tma::Finalize();
}

TEST(ScoopImageNvjpeg, ScoopImage)
{
    /* Set allocator callback functions */
    nv::mm::SetAllocator(MultimediaAllocate, MultimediaFree, MultimediaReallocate, NULL);
    nv::SetGraphicsServiceName("nvdrv:s");
    nv::SetGraphicsAllocator(MultimediaAllocate, MultimediaFree, MultimediaReallocate, NULL);
    nv::SetGraphicsDevtoolsAllocator(MultimediaAllocate, MultimediaFree, MultimediaReallocate, NULL);
    nv::InitializeGraphics(g_mmFirmwareMemory, sizeof(g_mmFirmwareMemory));

    ScoopImageNvjpeg_test();

    LOG("Memory left: %i allocations\n", g_Allocs.size());
    int memory = 0;
    for(std::map<void*, size_t>::iterator it = g_Allocs.begin(); it != g_Allocs.end(); ++it)
    {
        NN_SDK_LOG("address: %p size: %i\n", it->first, it->second);
        memory += it->second;
    }
    LOG("Total not deallocated: %i\n", memory);
    SUCCEED();
}
