﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <cinttypes>
#include <memory>
#include <string>
#include <tuple>
#include <mm_MemoryManagement.h>
#include <nn/fs.h>
#include <nn/init.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/mem/mem_StandardAllocator.h>
#include <movie/Player.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_SdkLog.h>
#include <nnt/nntest.h>
#include <nn/os.h>
#include <nv/nv_MemoryManagement.h>
#include <nv/nv_ServiceName.h>

#ifdef MOUNT_SDCARD
#include <nn/fs/fs_SdCardForDebug.h>
#endif

#ifdef ENABLE_SOCKET
#include <nn/nifm.h>
#include <nn/socket.h>
#endif

const int64_t actionDelayMS = 300;
const int64_t seekDelayMS = 1000;
const int64_t seekEpsilonUS = 4 * 1000 * 1000;
const int64_t playbackRateEpsilonBaseUS = 100 * 1000;
const float volumeEpsilon = 0.2;

// System heap setup
namespace // 'unnamed'
{
nn::lmem::HeapHandle g_FsHeap;
}

void *FsAllocate(size_t size)
{
    return nn::lmem::AllocateFromExpHeap(g_FsHeap, size);
}

void FsDeallocate(void *p, size_t size)
{
    NN_UNUSED(size);
    return nn::lmem::FreeToExpHeap(g_FsHeap, p);
}

void FsCreateHeap()
{
    const int heapSize = 512 * 1024;
    static uint8_t s_FsHeapBuffer[heapSize];
    g_FsHeap = nn::lmem::CreateExpHeap(s_FsHeapBuffer, sizeof(s_FsHeapBuffer), nn::lmem::CreationOption_DebugFill);
}

void FsDestroyHeap()
{
    nn::lmem::DestroyExpHeap(g_FsHeap);
    g_FsHeap = 0;
}

namespace // 'unnamed'
{
nn::mem::StandardAllocator g_MultimediaAllocator;
}

void MMCreateAllocator()
{
    const int allocatorSize = 512 << 20;
    static char s_MMAllocatorBuffer[allocatorSize];
    g_MultimediaAllocator.Initialize(s_MMAllocatorBuffer, sizeof(s_MMAllocatorBuffer));
}

void MMDestroyAllocator()
{
    g_MultimediaAllocator.Finalize();
}

void *MMAlloc(size_t size, size_t alignment, void*)
{
    return g_MultimediaAllocator.Allocate(size, alignment);
}

void MMFree(void *p, void*)
{
    g_MultimediaAllocator.Free(p);
}

void *MMRealloc(void *p, size_t newSize, void*)
{
    return g_MultimediaAllocator.Reallocate(p, newSize);
}

// Process startup setup
extern "C" void nninitStartup()
{
    const size_t heapSize = 128 << 20;
    nn::Result result = nn::os::SetMemoryHeapSize(heapSize);
    //ASSERT(result.IsSuccess());

    uintptr_t address;
    const size_t mallocSize = 32 << 20;
    result = nn::os::AllocateMemoryBlock(&address, mallocSize);
    //ASSERT(result.IsSuccess());

    nn::init::InitializeAllocator(reinterpret_cast<void*>(address), mallocSize);
    nn::fs::SetAllocator(FsAllocate, FsDeallocate);
}

extern "C" int *__errno()
{
    static int i = 0;
    return &i;
}

class MediaPlayerSanity : public movie::PlayerObserver
{
public:
    MediaPlayerSanity();
    virtual ~MediaPlayerSanity();

    void WaitForPlaybackCompletion(nn::TimeSpan waitTime);
    void WaitForSeekCompletion(nn::TimeSpan waitTime);
    bool playbackComplete;

public:
    virtual void OnError(movie::Status status) NN_NOEXCEPT;
    virtual void OnStateChange(movie::PlayerState state) NN_NOEXCEPT;
    virtual void OnBufferingUpdate(float startTime, float endTime) NN_NOEXCEPT;
    virtual void OnAudioOutputFrameAvailable(movie::AudioFrameInfo *frameInfo) NN_NOEXCEPT;
    virtual void OnVideoOutputFrameAvailable(movie::VideoFrameInfo *frameInfo) NN_NOEXCEPT;
    virtual int32_t OnHttpRequest(CURL *easyRequest, const char *uri) NN_NOEXCEPT;
    virtual int32_t OnMultiConfig(CURLM *multiRequest, const char *uri) NN_NOEXCEPT;
    virtual int32_t OnHttpResponse(CURLM *multiRequest, const char *uri) NN_NOEXCEPT;
    virtual void OnOutputBufferAvailable(int trackNumber, movie::TrackType eTrackType) NN_NOEXCEPT;
    virtual void OnOutputBufferAvailable(int trackNumber, movie::TrackType eTrackType, int64_t presentationTimeUs, int32_t index) NN_NOEXCEPT;
    virtual void OnFormatChanged(movie::TrackType eTrackType);

private:
    nn::os::Event playBackCompleteEvent{nn::os::EventClearMode_AutoClear};
    nn::os::Event seekCompleteEvent{nn::os::EventClearMode_AutoClear};
};

MediaPlayerSanity::MediaPlayerSanity()
{
    playbackComplete = false;
}

MediaPlayerSanity::~MediaPlayerSanity()
{
}

void MediaPlayerSanity::WaitForPlaybackCompletion(nn::TimeSpan waitTime)
{
    if(playBackCompleteEvent.TimedWait(waitTime))
    {
        playBackCompleteEvent.Clear();
    }
}

void MediaPlayerSanity::WaitForSeekCompletion(nn::TimeSpan waitTime)
{
    if(seekCompleteEvent.TimedWait(waitTime))
    {
        seekCompleteEvent.Clear();
    }
}

// Called when there is any error
void MediaPlayerSanity::OnError(movie::Status status) NN_NOEXCEPT
{
    NN_UNUSED(status);
    NN_SDK_LOG("\n OnError\n");
}

const char *PlayerStateToStr(movie::PlayerState state)
{
    const char *ret = "PlayerState UNKNOWN";
    switch(state)
    {
    case movie::PlayerState_UnInitialized:
        ret = "PlayerState_UnInitialized";
        break;
    case movie::PlayerState_Initialized:
        ret = "PlayerState_Initialized";
        break;
    case movie::PlayerState_Preparing:
        ret = "PlayerState_Preparing";
        break;
    case movie::PlayerState_Prepared:
        ret = "PlayerState_Prepared";
        break;
    case movie::PlayerState_Started:
        ret = "PlayerState_Started";
        break;
    case movie::PlayerState_Stopped:
        ret = "PlayerState_Stopped";
        break;
    case movie::PlayerState_Paused:
        ret = "PlayerState_Paused";
        break;
    case movie::PlayerState_Seeking:
        ret = "PlayerState_Seeking";
        break;
    case movie::PlayerState_SeekCompleted:
        ret = "PlayerState_SeekCompleted";
        break;
    case movie::PlayerState_PlaybackCompleted:
        ret = "PlayerState_PlaybackCompleted";
        break;
    case movie::PlayerState_Error:
        ret = "PlayerState_Error";
        break;
    default:
        break;
    }
    return ret;
}

// Called when player state changes
void MediaPlayerSanity::OnStateChange(movie::PlayerState state) NN_NOEXCEPT
{
    NN_SDK_LOG("\n %s\n", PlayerStateToStr(state));
    switch(state)
    {
    case movie::PlayerState_PlaybackCompleted:
        playbackComplete = true;
        playBackCompleteEvent.Signal();
        break;
    case movie::PlayerState_SeekCompleted:
        seekCompleteEvent.Signal();
        break;
    default:
        break;
    }
}

// Called when the user pressed Play/Pause or the player rebuffers
void MediaPlayerSanity::OnBufferingUpdate(float startTime, float endTime) NN_NOEXCEPT
{
    NN_UNUSED(startTime);
    NN_UNUSED(endTime);
    NN_SDK_LOG("\n OnBufferingUpdate\n");
}

// Called when decoded audio data is available and client has requested data back
void MediaPlayerSanity::OnAudioOutputFrameAvailable(movie::AudioFrameInfo *frameInfo) NN_NOEXCEPT
{
    NN_UNUSED(frameInfo);
    NN_SDK_LOG("\n OnAudioOutputFrameAvailable\n");
}

// Called when decoded video data is available and client has requested data back
void MediaPlayerSanity::OnVideoOutputFrameAvailable(movie::VideoFrameInfo *frameInfo) NN_NOEXCEPT
{
    NN_UNUSED(frameInfo);
    NN_SDK_LOG("\n OnVideoOutputFrameAvailable\n");
}

// Called before curl_easy_perform() for an HTTP request. Modify any CURL option (proxy, cookies, authentication…) and return 0 to proceed
int32_t MediaPlayerSanity::OnHttpRequest(CURL *easyRequest, const char *uri) NN_NOEXCEPT
{
    NN_UNUSED(easyRequest);
    NN_UNUSED(uri);
    NN_SDK_LOG("\n OnHttpRequest\n");
    return 0;
}

// Called before curl_multi_perform() for an HTTP request. Modify any CURLM option (sockets cache…) and return 0 to proceed
int32_t MediaPlayerSanity::OnMultiConfig(CURLM *multiRequest, const char *uri) NN_NOEXCEPT
{
    NN_UNUSED(multiRequest);
    NN_UNUSED(uri);
    NN_SDK_LOG("\n OnMultiConfig\n");
    return 0;
}

// Called when an error was detected on an HTTP request managed with curl.
int32_t MediaPlayerSanity::OnHttpResponse(CURLM *multiRequest, const char *uri) NN_NOEXCEPT
{
    NN_UNUSED(multiRequest);
    NN_UNUSED(uri);
    NN_SDK_LOG("\n OnHttpResponse\n");
    return 0;
}

// Called when client provided memory is used for output buffers.
void MediaPlayerSanity::OnOutputBufferAvailable(int trackNumber, movie::TrackType eTrackType, int64_t presentationTimeUs, int32_t index)  NN_NOEXCEPT
{
    NN_UNUSED(trackNumber);
    NN_UNUSED(eTrackType);
    NN_UNUSED(index);
    NN_SDK_LOG("\n OnOutputBufferAvailable with index\n");
}

// Called when client provided memory is used for output buffers.
void MediaPlayerSanity::OnOutputBufferAvailable(int trackNumber, movie::TrackType eTrackType)  NN_NOEXCEPT
{
    NN_UNUSED(trackNumber);
    NN_UNUSED(eTrackType);
    NN_SDK_LOG("\n OnOutputBufferAvailable\n");
}

// Called when audio or video output format is changed.
void MediaPlayerSanity::OnFormatChanged(movie::TrackType eTrackType)
{
    NN_UNUSED(eTrackType);
    NN_SDK_LOG("\n OnFormatChanged\n");
}

namespace // 'unnamed'
{
movie::Player *CreatePlayer()
{
    movie::Player *ret = 0;
    movie::PlayerConfig conf;
    conf.returnAudioDataToClient = false;
    conf.returnVideoDataToClient = false;
    conf.returnSyncedData = false;
    conf.useNativeBuffers = false;
    conf.audioFormat = movie::OutputFormat_None;
    conf.videoFormat = movie::OutputFormat_None;
    EXPECT_TRUE((movie::Status_Success == movie::Player::Create(&ret, &conf)));
    return ret;
}
}

// Test "fixture"
struct SanityTest : public ::testing::Test
{
    static void SetUpTestCase();
    static void TearDownTestCase();

    virtual void SetUp();
    virtual void TearDown();

    static bool isNetworkFile;
    static bool sdcardMounted;
    static const char *inputFileName;
    int64_t trackDuration = 0;
    std::unique_ptr<MediaPlayerSanity> mediaPlayerSanity{new MediaPlayerSanity};
    std::unique_ptr<movie::Player, decltype(&movie::Player::Destroy)> player{CreatePlayer(), &movie::Player::Destroy};
};

bool SanityTest::isNetworkFile = false;
bool SanityTest::sdcardMounted = false;
const char *SanityTest::inputFileName = 0;
#ifdef ENABLE_SOCKET
NN_ALIGNAS(4096) uint8_t g_SocketMemoryPoolBuffer[nn::socket::DefaultSocketMemoryPoolSize];
#endif

void SanityTest::SetUpTestCase()
{
    FsCreateHeap();
    MMCreateAllocator();
    NN_SDK_LOG("Calling nv::mm::SetAllocator\n");
    nv::mm::SetAllocator(MMAlloc, MMFree, MMRealloc, nullptr);
    NN_SDK_LOG("Calling nv::SetGraphicsAllocator\n");
    nv::SetGraphicsAllocator(MMAlloc, MMFree, MMRealloc, nullptr);
    nv::SetGraphicsServiceName("nvdrv:t");
    NN_SDK_LOG("Calling nv::SetGraphicsDevtoolsAllocator\n");
    nv::SetGraphicsDevtoolsAllocator(MMAlloc, MMFree, MMRealloc, nullptr);
    const int mmFirmwareMemorySize = 8 << 20;
    NN_ALIGNAS(4096) static char s_mmFirmwareMemory[mmFirmwareMemorySize];
    NN_SDK_LOG("Calling nv::InitializeGraphics\n");
    nv::InitializeGraphics(s_mmFirmwareMemory, sizeof(s_mmFirmwareMemory));
    const int argc = nn::os::GetHostArgc();
    const char * const * const argv = nn::os::GetHostArgv();
    NN_SDK_LOG("\n MediaPlayerSanity Entering Main\n");
    if(argc <= 1)
    {
        NN_SDK_LOG("\n MediaPlayerSanity::Usage - %s  <inputFile - Complete Path >\n", argv[0]);
        return;
    }
    inputFileName = argv[1];
    const std::string token(inputFileName, 0, std::string(inputFileName).find(':'));
    if(token == "sdcard")
    {
#ifdef MOUNT_SDCARD
        nn::Result resultSdcardMount = nn::fs::MountSdCardForDebug("sdcard");
        if(resultSdcardMount.IsFailure())
        {
            NN_SDK_LOG("\n nn::fs::SD card mount failure. Module:%d, Description:%d\n",
                resultSdcardMount.GetModule(),
                resultSdcardMount.GetDescription());
            return;
        }
        sdcardMounted = true;
#else
        NN_SDK_LOG("\n SD card Not supported\n");
        return;
#endif
    }
    else if(token == "http" || token == "https")
    {
        isNetworkFile = true;
#ifdef ENABLE_SOCKET
        nn::nifm::Initialize();
        nn::nifm::SubmitNetworkRequest();
        while(nn::nifm::IsNetworkRequestOnHold())
        {
            NN_SDK_LOG("Network request on hold\n");
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        }
        if(!nn::nifm::IsNetworkAvailable())
        {
            NN_SDK_LOG("Network initialization failed\n");
            nn::nifm::CancelNetworkRequest();
            return;
        }
        else
        {
            nn::Result res = nn::socket::Initialize(g_SocketMemoryPoolBuffer,
                                                    nn::socket::DefaultSocketMemoryPoolSize,
                                                    nn::socket::DefaultSocketAllocatorSize,
                                                    nn::socket::DefaultConcurrencyLimit);
            if(res.IsFailure())
            {
                nn::nifm::CancelNetworkRequest();
                NN_SDK_LOG("nn::socket::Initialize failed!\n");
                return;
            }
        }
#endif
    }
    else
    {
        const nn::Result resultHostMount = nn::fs::MountHostRoot();
        if(resultHostMount.IsFailure())
        {
            NN_SDK_LOG("\n nn::fs::Host root mount failure. Module:%d, Description:%d\n",
                resultHostMount.GetModule(),
                resultHostMount.GetDescription());
            return;
        }
    }
    // Check whether media exists if it is a local file
    if(!isNetworkFile && token != "file")
    {
        nn::fs::FileHandle fileHandle{};
        const nn::Result result = nn::fs::OpenFile(&fileHandle, inputFileName, nn::fs::OpenMode_Read);
        if(result.IsFailure())
        {
            NN_SDK_LOG("\n Failed to open %s\n\n Exiting MediaPlayerSanity Test\n", inputFileName);
            return;
        }
        else
        {
            nn::fs::CloseFile(fileHandle);
        }
    }
    NN_SDK_LOG("\n MediaPlayerSanity:: Input File Name : %s\n", inputFileName);
}

void SanityTest::TearDownTestCase()
{
#ifdef ENABLE_SOCKET
    if(isNetworkFile == true)
    {
        nn::socket::Finalize();
        nn::nifm::CancelNetworkRequest();
    }
#endif

    if(sdcardMounted == true)
    {
#ifdef MOUNT_SDCARD
        nn::fs::Unmount("sdcard");
#endif
    }
    else
    {
        nn::fs::UnmountHostRoot();
    }
    MMDestroyAllocator();
    FsDestroyHeap();
}

void SanityTest::SetUp()
{
    ASSERT_TRUE(player.get());

    NN_SDK_LOG("\n MediaPlayerSanity:: Set SetObserver\n");
    EXPECT_TRUE((movie::Status_Success == player->SetObserver(mediaPlayerSanity.get())));

    movie::PlayerObserver *observer = 0;
    EXPECT_TRUE((movie::Status_Success == player->GetObserver(&observer)));
    EXPECT_EQ(observer, mediaPlayerSanity.get());

    NN_SDK_LOG("\n MediaPlayerSanity :: Set data Source\n");
    EXPECT_TRUE((movie::Status_Success == player->SetDataSource(inputFileName)));

    NN_SDK_LOG("\n MediaPlayerSanity:: Prepare\n");
    EXPECT_TRUE((movie::Status_Success == player->Prepare()));

    EXPECT_TRUE((movie::Status_Success == player->GetPlaybackDuration(&trackDuration)));
    NN_SDK_LOG("\n MediaPlayerSanity:: GetPlaybackDuration: %lld\n", trackDuration);
}

void SanityTest::TearDown()
{
    player.reset();
    NN_SDK_LOG("\n MediaPlayerSanity:: Player destroyed\n");

    mediaPlayerSanity.reset();
}

TEST_F(SanityTest, NormalPlayback)
{
    NN_SDK_LOG("\n MediaPlayerSanity:: Start\n");
    EXPECT_TRUE((movie::Status_Success == player->Start()));

    NN_SDK_LOG("\n MediaPlayerSanity Wait till playback is complete\n");
    mediaPlayerSanity->WaitForPlaybackCompletion(nn::TimeSpan::FromMicroSeconds(trackDuration) + nn::TimeSpan::FromMilliSeconds(actionDelayMS));

    NN_SDK_LOG("\n MediaPlayerSanity:: Stop\n");
    EXPECT_TRUE((movie::Status_Success == player->Stop()));
}

TEST_F(SanityTest, RepeatedSeeking) // interfered with by SIGLONTD-3452
{
    NN_SDK_LOG("\n MediaPlayerSanity:: Start\n");
    EXPECT_TRUE((movie::Status_Success == player->Start()));

    const int64_t early = (trackDuration) / 4;
    const int64_t middle = (trackDuration) / 2;
    const int64_t late = (3 * trackDuration) / 4;
    const int64_t seekTimesUS[] =
    {
        // Fast-forward three times
        early, middle, late,
        // Rewind three times
        middle, early, 0,
        // Alternate between rewind and fast-forward five times
        early, 0, middle, early, late, middle, early, 0
    };
    for(int i = 0; i < sizeof(seekTimesUS) / sizeof(*seekTimesUS); ++i)
    {
        const int64_t targetTime = seekTimesUS[i];
        NN_SDK_LOG("\n MediaPlayerSanity:: SeekTo\n");
        movie::Status movieStatus = player->SeekTo(targetTime);
        if (movieStatus == movie::Status_SeekNotAllowed)
        {
            break;
        }
        EXPECT_TRUE((movie::Status_Success == movieStatus));
        mediaPlayerSanity->WaitForSeekCompletion(nn::TimeSpan::FromMilliSeconds(seekDelayMS));
        movie::PlayerState state{};
        while(state != movie::PlayerState_Started){
            EXPECT_TRUE((movie::Status_Success == player->GetState(&state)));
        }
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(seekDelayMS));
        int64_t currentTime = 0;
        EXPECT_TRUE((movie::Status_Success == player->GetCurrentPlaybackPosition(&currentTime)));
        EXPECT_NEAR(currentTime, targetTime, seekEpsilonUS);
    }

    NN_SDK_LOG("\n MediaPlayerSanity Wait till playback is complete\n");
    mediaPlayerSanity->WaitForPlaybackCompletion(nn::TimeSpan::FromMicroSeconds(trackDuration) + nn::TimeSpan::FromMilliSeconds(actionDelayMS));
}

TEST_F(SanityTest, RepeatedPausing)
{
    NN_SDK_LOG("\n MediaPlayerSanity:: Start\n");
    EXPECT_TRUE((movie::Status_Success == player->Start()));

    // Alternate between play and pause at least 10 times
    for(int i = 0; i < 10; ++i)
    {
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(actionDelayMS));
        if(!mediaPlayerSanity->playbackComplete){
            NN_SDK_LOG("\n MediaPlayerSanity:: Pause on\n");
            EXPECT_TRUE((movie::Status_Success == player->Pause(true)));
        }
        else
            break;
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(actionDelayMS));
        if(!mediaPlayerSanity->playbackComplete){
            NN_SDK_LOG("\n MediaPlayerSanity:: Pause off\n");
            EXPECT_TRUE((movie::Status_Success == player->Pause(false)));
        }
        else
            break;
    }
    if(!mediaPlayerSanity->playbackComplete){
        NN_SDK_LOG("\n MediaPlayerSanity Wait till playback is complete\n");
        mediaPlayerSanity->WaitForPlaybackCompletion(nn::TimeSpan::FromMicroSeconds(trackDuration) + nn::TimeSpan::FromMilliSeconds(actionDelayMS));
    }
}

TEST_F(SanityTest, ManipulateVolume)
{
    NN_SDK_LOG("\n MediaPlayerSanity:: Start\n");
    EXPECT_TRUE((movie::Status_Success == player->Start()));

    const float desiredVolume = 0.5;
    NN_SDK_LOG("\n MediaPlayerSanity:: SetVolume\n");
    EXPECT_TRUE((movie::Status_Success == player->SetVolume(desiredVolume)));
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(actionDelayMS));
    float volume = 0;
    NN_SDK_LOG("\n MediaPlayerSanity:: GetVolume\n");
    EXPECT_TRUE((movie::Status_Success == player->GetVolume(&volume)));
    EXPECT_NEAR(desiredVolume, volume, volumeEpsilon);

    NN_SDK_LOG("\n MediaPlayerSanity:: Stop\n");
    EXPECT_TRUE((movie::Status_Success == player->Stop()));
}

TEST_F(SanityTest, ManipulateTracks)
{
    NN_SDK_LOG("\n MediaPlayerSanity:: Start\n");
    EXPECT_TRUE((movie::Status_Success == player->Start()));

    NN_SDK_LOG("\n MediaPlayerSanity:: GetTrackCount\n");
    int32_t trackCount = 0;
    EXPECT_TRUE((movie::Status_Success == player->GetTrackCount(&trackCount)));
    NN_SDK_LOG("\n MediaPlayerSanity:: has %d tracks\n", trackCount);
    for(int32_t i = 0; i < trackCount; ++i)
    {
        movie::TrackType type = movie::TrackType_Unknown;
        NN_SDK_LOG("\n MediaPlayerSanity:: GetTrackType\n");
        EXPECT_TRUE((movie::Status_Success == player->GetTrackType(i, &type)));
        NN_SDK_LOG("\n MediaPlayerSanity:: track %d is type %d\n", i, type);
        NN_SDK_LOG("\n MediaPlayerSanity:: SelectTrack\n");
        EXPECT_TRUE((movie::Status_Success == player->SelectTrack(i)));
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(actionDelayMS));
        EXPECT_TRUE((movie::Status_Success == player->DeSelectTrack(i)));
        movie::TrackInfo info;
        std::memset(&info, 0, sizeof(info));
        EXPECT_TRUE((movie::Status_Success == player->GetTrackInfo(i, &info)));
    }
    NN_SDK_LOG("\n MediaPlayerSanity Wait till playback is complete\n");
    mediaPlayerSanity->WaitForPlaybackCompletion(nn::TimeSpan::FromMicroSeconds(trackDuration) + nn::TimeSpan::FromMilliSeconds(actionDelayMS));
}

TEST_F(SanityTest, Reset)
{
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(actionDelayMS));
    NN_SDK_LOG("\n MediaPlayerSanity:: Reset\n");
    EXPECT_TRUE((movie::Status_Success == player->Reset()));
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(actionDelayMS));
}

TEST_F(SanityTest, Looping)
{
    NN_SDK_LOG("\n MediaPlayerSanity:: SetLooping\n");
    EXPECT_TRUE((movie::Status_Success == player->SetLooping(true)));
    bool loop = false;
    EXPECT_TRUE((movie::Status_Success == player->GetLooping(&loop)));
    EXPECT_TRUE(loop);
}

TEST_F(SanityTest, PlaybackRate)
{
    movie::PlaybackRate rate = movie::PlaybackRate_Normal;
    NN_SDK_LOG("\n MediaPlayerSanity:: GetPlaybackRate\n");
    EXPECT_TRUE((movie::Status_Success == player->GetPlaybackRate(&rate)));
    NN_SDK_LOG("\n MediaPlayerSanity:: SetPlaybackRate\n");
    EXPECT_TRUE((movie::Status_Success == player->SetPlaybackRate(movie::PlaybackRate_0_5x)));
    const nn::os::Tick begin = nn::os::GetSystemTick();
    NN_SDK_LOG("\n MediaPlayerSanity:: Start\n");
    EXPECT_TRUE((movie::Status_Success == player->Start()));
    NN_SDK_LOG("\n MediaPlayerSanity Wait till playback is complete\n");
    mediaPlayerSanity->WaitForPlaybackCompletion(nn::TimeSpan::FromMicroSeconds(4 * trackDuration) + nn::TimeSpan::FromMilliSeconds(actionDelayMS));
    const nn::os::Tick end = nn::os::GetSystemTick();
    const nn::TimeSpan duration = nn::os::ConvertToTimeSpan(end - begin);
    NN_SDK_LOG("\n MediaPlayerSanity playback took %" PRId64 " milliseconds\n", duration.GetMilliSeconds());
    const int64_t playbackRateEpsilonUS = trackDuration / 5 + playbackRateEpsilonBaseUS;
    EXPECT_GE(2 * trackDuration + playbackRateEpsilonUS, duration.GetMicroSeconds());
    EXPECT_LE(2 * trackDuration - playbackRateEpsilonUS, duration.GetMicroSeconds());
}
