﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/os.h>
#include <nn/fs.h>

#include "AudioOutRenderer.h"
#include "VideoRenderer.h"

#include <nn/nn_Result.h>
#include <movie/Utils.h>
#include <movie/Extractor.h>
#include <movie/Decoder.h>

#include <vector>

#include "ByteStreamReader.h"

void AudioAndVideoInputReadThreadFunction(void *arg);
void AudioVideoWorkerThreadFunction(void *arg);

class MovieDecoderPlayer
{
public:

    MovieDecoderPlayer();

    movie::Status Initialize(uint64_t coreMask);

    void Finalize();

    movie::Status CreateExtractor(const char* uri, movie::ContainerType container, movie::Extractor** extractor, bool use_buffer_as_datasource = false);

    movie::Status CreateDecoders(movie::Extractor* extractor);

    movie::Status CreateDecoder(movie::Decoder** decoder, movie::DecoderType decoderType, movie::DecoderOutputFormat decoderOutputFormat);

    movie::Status StartDecoder(movie::Decoder* decoder);

    movie::Status StopDecoder(movie::Decoder* decoder);

    movie::Status FlushDecoder(movie::Decoder* decoder);

    void DestroyDecoder(movie::Decoder* decoder);

    movie::Status CreateAudioVideoRenderer();

    void StopAudioVideoRenderer();

    void DestroypAudioVideoRenderer();

    void DestroyExtractor(movie::Extractor* extractor);

    void SetAudioPlaybackComplete() { m_AudioPlayBackDone = true; }

    void ResetAudioPlaybackComplete() { m_AudioPlayBackDone = false; }

    bool IsAudioPlaybackComplete() { return m_AudioPlayBackDone; }

    void SetVideoPlaybackComplete() { m_VideoPlayBackDone = true; }

    void ResetVideoPlaybackComplete() { m_VideoPlayBackDone = false; }

    bool IsVideoPlaybackComplete() { return m_VideoPlayBackDone; }

    bool IsVideoDecoderCreated() { return m_VideoDecoderCreated; }

    bool IsAudioDecoderCreated() { return m_AudioDecoderCreated; }

    void ResetPlaybackComplete();

    bool IsPlaybackComplete();

    movie::Status ReadInputDataFromExtractorSendTodecoder(int index, movie::Decoder* decoder);

    movie::Status ReadInputDataFromExtractorSendTodecoder(ByteStreamReader* byteStreamReader, int index, movie::Decoder* decoder);

#ifdef ENABLE_METADATA_READER
    movie::Status ReadMetaDataFromExtractor();
#endif

    movie::Status ConfigureDecoder(movie::Decoder* decoder, movie::DecoderEvents* decoderEvents, movie::MediaData* configuration);

    movie::Status SendEosToDecoder();

    movie::Status CheckForInputBuffersAndReadInputdataDecoder();

    movie::Status AddBufferToAudioIndexList(int32_t bufferIndex);

    movie::Status RemoveBufferFromAudioIndexList(int32_t* bufferIndex);

    movie::Status AddBufferToVideoIndexList(int32_t bufferIndex);

    movie::Status RemoveBufferFromVideoIndexList(int32_t* bufferIndex);

    movie::Status VideoInputBufferAvailableEvent();

    movie::Status VideoOutputBufferAvailableEvent();

    movie::Status VideoFormatChangedEvent();

    movie::Status AudioInputBufferAvailableEvent();

    movie::Status AudioOutputBufferAvailableEvent();

    movie::Status AudioFormatChangedEvent();

    movie::Status SignalInputBufferAvailable();

    movie::Status SignalAudioAndVideoInputReadThreadExit();

    movie::Status SignalAudioVideoWorkerThreadExit();

    movie::Status CreateThreadForAudioAndVideoInputReader();
    movie::Status CreateThreadForAudioVideoWorker();

    void StartAudioAndVideoInputReaderThread();
    void StartAudioVideoWorkerThread();

    void StopAndDestroyAudioAndVideoInputReaderThread();
    void StopAndDestroyAudioVideoWorkerThread();

    friend void AudioAndVideoInputReadThreadFunction(void *arg);
    friend void AudioVideoWorkerThreadFunction(void *arg);

    void Reset();

    ~MovieDecoderPlayer() {};

public:
    movie::Extractor* m_Extractor;
    movie::Decoder* m_AudioDecoder;
    movie::Decoder* m_VideoDecoder;

    struct DecoderEventTypes
    {
        nn::os::EventType inputBufferAvailableEvent;
        nn::os::EventType outputBufferAvailableEvent;
        nn::os::EventType formatChangedEvent;
        nn::os::EventType errorEvent;
    };

    struct DecoderMultiWaitHolderTypes
    {
        nn::os::MultiWaitHolderType inputBufferAvailableEventHolder;
        nn::os::MultiWaitHolderType outputBufferAvailableEventHolder;
        nn::os::MultiWaitHolderType formatChangedEventHolder;
        nn::os::MultiWaitHolderType errorEventHolder;
    };

    nn::os::EventType m_AudioVideoWorkerThreadExitEvent;
    nn::os::MultiWaitHolderType m_AudioVideoWorkerThreadExitEventHolder;

    nn::os::MultiWaitType m_AudioVideoWorkerMultiWait;

    movie::DecoderEvents m_AudioDecoderEvents;
    movie::DecoderEvents m_VideoDecoderEvents;

    DecoderEventTypes m_AudioDecoderEventTypes;
    DecoderEventTypes m_VideoDecoderEventTypes;

    DecoderMultiWaitHolderTypes m_AudioDecoderMultiWaitHolderTypes;
    DecoderMultiWaitHolderTypes m_VideoDecoderMultiWaitHolderTypes;

    AudioOutRenderer *m_AudioOutRenderer;
    VideoRenderer *m_VideoRenderer;

    int32_t m_Tracks;
    int32_t m_Width;
    int32_t m_Height;
    int32_t m_NumChannels;
    int32_t m_SampleRate;
    double m_VideoFrameRate;
    int64_t m_TrackDurationUs;

    int32_t m_AudioTrackIndex;
    int32_t m_VideoTrackIndex;

#ifdef ENABLE_METADATA_READER
    int32_t m_MetaTrackIndex;
#endif

    size_t m_ThreadStackSize;

    char* m_AudioVideoInputReadThreadStack;
    bool m_AudioVideoInputReadThreadDone;
    bool m_AudioVideoInputReadThreadCreated;
    nn::os::ThreadType m_AudioVideoInputReadThreadType;

    char* m_AudioVideoWorkerThreadStack;
    bool m_AudioVideoWorkerThreadDone;
    bool m_AudioVideoWorkerThreadCreated;
    nn::os::ThreadType m_AudioVideoWorkerThreadType;

    std::vector<int> m_AudioBufferIndexList;
    std::vector<int> m_VideoBufferIndexList;

    nn::os::MutexType m_AudioListMutex;
    nn::os::MutexType m_VideoListMutex;

    size_t m_BufferSize;
    uintptr_t *m_MessageBuffer;

    nn::os::MessageQueueType m_MessageQueue;

    enum InputThreadMessages
    {
        InputReadThreadDone = 0,
        InputBufferAvailable = 1,
    };

    ByteStreamReader* m_ByteStreamReader;
    movie::DecoderType m_AudioDecoderType;
    movie::DecoderType m_VideoDecoderType;
    bool m_AudioDecoderCreated;
    bool m_VideoDecoderCreated;
    bool m_LoopPlayback;
    bool m_suppressRendering;

    int32_t m_LoopCount;
    movie::MediaData m_TrackFormat[2];

private:

    std::vector<char> m_SetDataSourceBuffer;

    bool m_AudioPlayBackDone;
    bool m_VideoPlayBackDone;
    bool m_EndOfStream;
};

