﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

#include <nn/vi.h>
#include <nvn/nvn.h>
#include <nvnTool/nvnTool_GlslcInterface.h>
#include <nn/mem/mem_StandardAllocator.h>

unsigned Align(unsigned value, unsigned alignment);

struct DataSection
{
    const void *data;
    size_t size;
};

class GlslCompiler
{
public:
    GlslCompiler();
    ~GlslCompiler();
    void Init();
    void Destroy();
    void CompileShader(const char *src, NVNshaderStage stage);

    //the pointers returned in DataSection are invalid after new compilations
    DataSection GetOutput() const;

private:
    GLSLCcompileObject m_Compiler;
    DataSection m_OutputData;
    bool m_Init;
};

class NvnShaderSerialize
{
public:
    NvnShaderSerialize();
    ~NvnShaderSerialize();
    void LoadNvnShaderMemory(const uint8_t *shaderBinary);

    DataSection GetDataSection();
    DataSection GetControlSection();
    unsigned GetScratchSize();

private:
    void *m_ShaderData;
    DataSection m_DataSection;
    DataSection m_ControlSection;
    unsigned m_ScratchSize;
};

class NvnMemoryPool
{
public:
    NvnMemoryPool();
    ~NvnMemoryPool();
    void Init(NVNdevice &device, int32_t flags, unsigned size, void *pMemory);
    void Destroy();
    ptrdiff_t GetNewMemoryChunkOffset(unsigned size, unsigned alignment);
    NVNmemoryPool &GetMemoryPool();

private:
    NVNmemoryPool m_MemoryPool;
    unsigned m_Size;
    void *m_pMemory;
    unsigned m_Offset;
};

class NvnBuffer
{
public:
    NvnBuffer();
    ~NvnBuffer();
    void Init(NVNdevice &device, NVNmemoryPool &pPool, unsigned offset, unsigned size);
    void Destroy();
    NVNbufferAddress GetAddress() const;
    NVNbuffer &GetBuffer();

private:
    NVNbuffer m_Buffer;
    NVNdevice *m_pDevice;
};

class NvnCommandBuffer
{
public:
    NvnCommandBuffer();
    ~NvnCommandBuffer();
    void Init(class NvnDevice &device, nn::mem::StandardAllocator &allocator, NvnMemoryPool &memoryPool);
    void Destroy();
    void Reset();
    size_t GetCommandMemoryUsed() const;
    size_t GetControlMemoryUsed() const;
    NVNcommandBuffer &GetCommandBuffer();

private:
    NVNcommandBuffer m_CommandBuffer;
    NvnDevice *m_pDevice;
    nn::mem::StandardAllocator *m_pAllocator;
    NvnMemoryPool *m_pMemoryPool;
    ptrdiff_t m_CommandOffset;
    void *m_pControlPool;

    static const size_t m_CommandSize = 1024;
    static const size_t m_ControlSize = 512;
};

class NvnDevice
{
public:
    NvnDevice();
    ~NvnDevice();
    void Init(NVNnativeWindow nativeWindow, nn::mem::StandardAllocator &allocator);
    void Destroy();
    void ResizeRenderTargets(unsigned width, unsigned height);
    NVNdevice &GetDevice();
    NVNwindow &GetWindow();
    NVNtexture &GetRenderTarget(int index);
    int GetDeviceInfo(NVNdeviceInfo name);

private:
    static void DebugLayerCallback(
        NVNdebugCallbackSource source,
        NVNdebugCallbackType type,
        int id,
        NVNdebugCallbackSeverity severity,
        const char * message,
        void* pUser);

    NVNdevice m_Device;

    nn::mem::StandardAllocator *m_pAllocator;

    NVNtextureBuilder m_RenderTargetBuilder;
    NVNtexture m_RenderTargets[2];

    NvnMemoryPool m_RenderTargetMemoryPool;
    void *m_pRenderTargetMemory;

    NVNwindow m_pWindow;
    NVNwindowBuilder m_WindowBuilder;
    size_t m_ColorTargetSize;

    bool m_FirstResize;

    static const int m_NumColorBuffers = 2;
};

class NvnProgram
{
public:
    NvnProgram();
    ~NvnProgram();
    void Init(NVNdevice &device, const NVNshaderData *stageData, unsigned stageCount);
    void Destroy();
    NVNprogram &GetProgram();

private:
    NVNprogram m_Program;
    NVNdevice *m_pDevice;
};

class NvnQueue
{
public:
    NvnQueue();
    ~NvnQueue();
    void Init(NVNdevice &device);
    void Destroy();
    void WaitForAllCommands();
    NVNqueue &GetQueue();

private:
    NVNqueue m_Queue;
    NVNsync m_CommandBufferSync;
    NVNdevice *m_pDevice;
};

class NvnSampler
{
public:
    NvnSampler();
    ~NvnSampler();
    void Init(NVNdevice &device);
    void Destroy();
    NVNsampler &GetSampler();

private:
    NVNsampler m_Sampler;
    NVNdevice *m_pDevice;
};

class NvnTexture
{
public:
    NvnTexture();
    ~NvnTexture();
    void Init(NVNdevice &device, NvnMemoryPool &memoryPool, unsigned width, unsigned height, NVNformat format, int32_t flags, unsigned stride);
    void Destroy();
    NVNtexture &GetTexture();

private:
    NVNtexture m_Texture;
    NVNdevice *m_pDevice;
};

class NvnTextureSamplerPool
{
public:
    NvnTextureSamplerPool();
    ~NvnTextureSamplerPool();
    void Init(NVNdevice &device, nn::mem::StandardAllocator &allocator);
    void Destroy();
    NVNtexturePool &GetTexturePool();
    NVNsamplerPool &GetSamplerPool();
    unsigned Register(NVNtexture &pTexture, NVNsampler &pSampler);

private:
    NVNtexturePool m_TexturePool;
    NVNsamplerPool m_SamplerPool;
    NvnMemoryPool m_MemoryPool;
    void *m_pMemory;
    nn::mem::StandardAllocator *m_pAllocator;
    NVNdevice *m_pDevice;
    unsigned m_Id;
};

class NvnWindow
{
public:
    NvnWindow();
    ~NvnWindow();
    void Init();
    void Destroy();
    nn::vi::NativeWindowHandle GetNativeWindow();
    int GetWidth() const;
    int GetHeight() const;

private:
    void InitMemory();
    void InitWindow();

    nn::vi::NativeWindowHandle  m_NativeWindow;
    nn::vi::Display*            m_pDisplay;
    nn::vi::Layer*              m_pLayer;
    int                         m_WindowWidth;
    int                         m_WindowHeight;
};
