﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
//Siglo networking is unstable, causing many false problems
//#define ENABLE_SOCKET
#include <algorithm>
#include <cinttypes>
#include <string>
#include <vector>
#include <nn/fs.h>
#include <nn/init.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/nn_SdkLog.h>
#include <nn/os.h>
#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>

#ifdef MOUNT_SDCARD
#include <nn/fs/fs_SdCardForDebug.h>
#endif

#ifdef ENABLE_SOCKET
#include <nn/nifm.h>
#include <nn/socket.h>
#endif

// System heap setup
namespace // 'unnamed'
{
nn::lmem::HeapHandle g_FsHeap;
#ifdef ENABLE_SOCKET
NN_ALIGNAS(4096) uint8_t g_SocketMemoryPoolBuffer[nn::socket::DefaultSocketMemoryPoolSize];
#endif
}

void *FsAllocate(size_t size)
{
    return nn::lmem::AllocateFromExpHeap(g_FsHeap, size);
}

void FsDeallocate(void *p, size_t size)
{
    NN_UNUSED(size);
    return nn::lmem::FreeToExpHeap(g_FsHeap, p);
}

void FsCreateHeap()
{
    const int heapSize = 512 * 1024;
    static uint8_t s_FsHeapBuffer[heapSize];
    g_FsHeap = nn::lmem::CreateExpHeap(s_FsHeapBuffer, sizeof(s_FsHeapBuffer), nn::lmem::CreationOption_DebugFill);
}

void FsDestroyHeap()
{
    nn::lmem::DestroyExpHeap(g_FsHeap);
    g_FsHeap = 0;
}

// Process startup setup
extern "C" void nninitStartup()
{
    const size_t heapSize = 512 << 20;
    nn::Result result = nn::os::SetMemoryHeapSize(heapSize);
    EXPECT_TRUE(result.IsSuccess());

    uintptr_t address;
    const size_t mallocSize = 32 << 20;
    result = nn::os::AllocateMemoryBlock(&address, mallocSize);
    EXPECT_TRUE(result.IsSuccess());

    nn::init::InitializeAllocator(reinterpret_cast<void*>(address), mallocSize);
    nn::fs::SetAllocator(FsAllocate, FsDeallocate);
}

extern "C" int *__errno()
{
    static int i = 0;
    return &i;
}

// Test "fixture"
struct MediaCodecSanityTest : public ::testing::Test
{
    static void SetUpTestCase();
    static void TearDownTestCase();

    virtual void SetUp();
    virtual void TearDown();

    static bool sdcardMounted;
    static const char *mediaPath;
    static std::vector<char> data;
};

bool MediaCodecSanityTest::sdcardMounted = false;
const char *MediaCodecSanityTest::mediaPath = 0;
std::vector<char> MediaCodecSanityTest::data;

void MediaCodecSanityTest::SetUpTestCase()
{
    FsCreateHeap();
    const int argc = nn::os::GetHostArgc();
    const char * const * const argv = nn::os::GetHostArgv();
    NN_SDK_LOG("\n MediaCodecSanity Entering Main\n");
    if(argc <= 1)
    {
        NN_SDK_LOG("\n MediaCodecSanity::Usage - %s  <media-file-path>\n", argv[0]);
        return;
    }
    mediaPath = argv[1];
#ifdef MOUNT_SDCARD
    nn::Result resultSdcardMount = nn::fs::MountSdCardForDebug("sdcard");
    if(resultSdcardMount.IsFailure())
    {
        NN_SDK_LOG("\n nn::fs::SD card mount failure. Module:%d, Description:%d\n",
            resultSdcardMount.GetModule(),
            resultSdcardMount.GetDescription());
        return;
    }
    sdcardMounted = true;
#endif

#ifdef ENABLE_SOCKET
    nn::nifm::Initialize();
    nn::nifm::SubmitNetworkRequest();
    while(nn::nifm::IsNetworkRequestOnHold())
    {
        NN_SDK_LOG("Network request on hold\n");
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }
    if(!nn::nifm::IsNetworkAvailable())
    {
        NN_SDK_LOG("Network initialization failed\n");
        nn::nifm::CancelNetworkRequest();
        return;
    }
    else
    {
        nn::Result res = nn::socket::Initialize(g_SocketMemoryPoolBuffer,
                                                nn::socket::DefaultSocketMemoryPoolSize,
                                                nn::socket::DefaultSocketAllocatorSize,
                                                nn::socket::DefaultConcurrencyLimit);
        if(res.IsFailure())
        {
            nn::nifm::CancelNetworkRequest();
            NN_SDK_LOG("nn::socket::Initialize failed!\n");
            return;
        }
    }
    NN_SDK_LOG("Network initialized (supposedly)\n");
#endif

    const nn::Result resultHostMount = nn::fs::MountHostRoot();
    if(resultHostMount.IsFailure())
    {
        NN_SDK_LOG("\n nn::fs::Host root mount failure. Module:%d, Description:%d\n",
            resultHostMount.GetModule(),
            resultHostMount.GetDescription());
        return;
    }

    const std::string token(mediaPath, 0, std::string(mediaPath).find(':'));
    if((token == "http") || (token == "https") || (token == "file"))
    {
    }
    // Check whether media exists if it is a local file
    else
    {
        nn::fs::FileHandle fileHandle{};
        const nn::Result result = nn::fs::OpenFile(&fileHandle, mediaPath, nn::fs::OpenMode_Read);
        if(result.IsFailure())
        {
            NN_SDK_LOG("\n Failed to open %s\n\n Exiting MediaCodecSanity Test\n", mediaPath);
        }
        else
        {
            int64_t fileSize = 0;
            nn::fs::GetFileSize(&fileSize, fileHandle);
            data.resize(fileSize);
            nn::fs::ReadFile(fileHandle, 0, &data[0], fileSize);
            nn::fs::CloseFile(fileHandle);
            NN_SDK_LOG("\n MediaCodecSanity:: data (%" PRIi64 ") : %x%x%x%x\n", fileSize, data[0], data[1], data[2], data[3]);
        }
    }
    NN_SDK_LOG("\n MediaCodecSanity:: Input File Name : %s\n", mediaPath);
}

void MediaCodecSanityTest::TearDownTestCase()
{
#ifdef ENABLE_SOCKET
    nn::socket::Finalize();
    nn::nifm::CancelNetworkRequest();
#endif

#ifdef MOUNT_SDCARD
    nn::fs::Unmount("sdcard");
#endif

    nn::fs::UnmountHostRoot();

    FsDestroyHeap();
}

void MediaCodecSanityTest::SetUp()
{
}

void MediaCodecSanityTest::TearDown()
{
    decltype(data) swaptrick{};
    std::swap(data, swaptrick);
}

TEST_F(MediaCodecSanityTest, Encode)
{
    NN_SDK_LOG("Testing encode\n");
}
