﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "MediaPlayerObserver.h"
#include "MediaPlayerUtilities.h"
//#include "HidHandler.h"
#include "HeapTracker.h"
#include "CommandLineOptions.h"
#include "PausableLoop.h"
#include "FileHandling.h"
#include "NXMSESource.h"

#include <experimental/memory_resource>

#define SOL_USE_EXPERIMENTAL_PMR_STRING

#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wunused-private-field"
#pragma clang diagnostic ignored "-Wnull-dereference"
#pragma clang diagnostic ignored "-Wunused-variable"
#include <sol/sol.hpp>
#pragma clang diagnostic pop

#include <mutex>
#include <chrono>
#include <string>
#include <cinttypes>
#include <thread>
#include <random>
#include <utility>

#include <movie/Utils.h>
#include <movie/Player.h>

#include "ThreadWrapper.h"

#include <nn/nn_SdkLog.h>
#include <nn/nn_Abort.h>
#include <nn/util/util_ScopeExit.h>

#include <nv/nv_MemoryManagement.h>
#include <nv/nv_ServiceName.h>

#include <nn/lmem/lmem_ExpHeap.h>

#include <libxml/xmlreader.h>
#include <libxml/parser.h>
#include <libxml/tree.h>

#include <curl/curl.h>

#ifdef MOUNT_SDCARD
#include <nn/fs/fs_SdCardForDebug.h>
#endif

#ifdef ENABLE_PROFILER
#include <nn/profiler.h>
#endif


//////////////////////////

constexpr size_t g_GraphicsSystemMemorySize = 8 * 1024 * 1024;
uint8_t g_GraphicsHeap[g_GraphicsSystemMemorySize] __attribute__((aligned(4096)));
bool USE_HEAP_TRACKING{};

//////////////////////////

struct TrackInfo {
    std::string initSegment;
    std::vector<std::string> dataSegments;
    uint32_t initSegmentSize;
};

struct MPDInfo {
    std::vector<TrackInfo> trackList;
};


void print_element_names(xmlNode * a_node, TrackInfo& trackInfo)
{
    xmlNode *cur_node = NULL;

    for (cur_node = a_node; cur_node; cur_node = cur_node->next) {
        if (cur_node->type == XML_ELEMENT_NODE) {
            NN_SDK_LOG("node type: Element, name: %s,\n", cur_node->name);
            if (!strcmp((const char*)cur_node->name, "Initialization"))
            {
                xmlAttr *cur_attr = NULL;
                for (cur_attr = cur_node->properties; cur_attr; cur_attr = cur_attr->next)
                {
                    if (!strcmp((const char*)cur_attr->name, "sourceURL"))
                    {
                        trackInfo.initSegment = (const char*)cur_attr->children->content;
                    }
                }
            }
            if (!strcmp((const char*)cur_node->name, "SegmentURL"))
            {
                xmlAttr *cur_attr = NULL;
                for (cur_attr = cur_node->properties; cur_attr; cur_attr = cur_attr->next)
                {
                    if (!strcmp((const char*)cur_attr->name, "media"))
                    {
                        trackInfo.dataSegments.push_back((const char*)cur_attr->children->content);
                    }
                }
            }
            if (!strcmp((const char*)cur_node->name, "SegmentList"))
            {
                //clear the filename list if a new SegmentList is reached
                //fileNames.clear();
            }
        }

        print_element_names(cur_node->children, trackInfo);
    }
}

extern "C" void nninitStartup()
{
    constexpr size_t heapSize = 256 * 1024 * 1024;
    constexpr size_t fsHeapSize = 512 * 1024;

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::SetMemoryHeapSize(heapSize));

    static uint8_t g_FsHeapBuffer[fsHeapSize];
    static nn::lmem::HeapHandle g_FsHeap = nn::lmem::CreateExpHeap(g_FsHeapBuffer, fsHeapSize, nn::lmem::CreationOption_DebugFill);
    nn::fs::SetAllocator([](size_t sz) { return nn::lmem::AllocateFromExpHeap(g_FsHeap, sz); },
        [](void* p, size_t) { return nn::lmem::FreeToExpHeap(g_FsHeap, p); });
}

extern "C" void nnMain()
{
    NN_UTIL_SCOPE_EXIT{
        if (USE_HEAP_TRACKING) {
            NN_LOG("mm heap\n");     MMHeap().OutputUsage();
            NN_LOG("core heap\n");   CoreHeap().OutputUsage();
            NN_LOG("malloc heap\n"); MallocHeap().OutputUsage();
            NN_LOG("new heap\n");    NewHeap().OutputUsage();
        } };

    { // We use an embedded scope to make sure local destructors have been called
      // before we print our final heap tracking statistics

#ifdef ENABLE_PROFILER
        std::vector<char> profiler_buffer__(nn::profiler::MinimumBufferSize);
        nn::profiler::Initialize(profiler_buffer__.data(), profiler_buffer__.size());
        NN_UTIL_SCOPE_EXIT{ nn::profiler::Finalize(); };
#endif

        // The default sol2 library uses static local std::strings in various places, and they are not
        // destructed at application exit in our environment; so they appear as leaks to our heap tracker.
        // To work around this I've modified sol2 to be able use the polymorphic_allocator string in std::pmr::string.
        // This allows us to specify a default allocator (as we do below) that uses our own custom untracked allocator.
        // (This turns out to be useful for other things we don't want to track as well, like the pmr::vector<> playlist in
        // CommandLineOptions.)
        movie::sample::UntrackedAllocator sneaky_allocator____{ 2 };
        std::experimental::pmr::set_default_resource(&sneaky_allocator____);
        // ------------------

        movie::sample::CommandLineOptions options_{ nn::os::GetHostArgv(), nn::os::GetHostArgc(), sneaky_allocator____ };

        if (!options_.valid()) {
            NN_SDK_LOG("%s", options_.usage_string());
            return;
        }

        USE_HEAP_TRACKING = options_.useHeapTracking;

        // To handle various cleanup cases we keep a vector of calls that are executed at scope exit..
        std::vector<movie::sample::ScopeExit> exit_calls_;

        // nn::hid::InitializeDebugPad();

        //--------------------------
        movie::SetAllocator(mmAllocate, mmDeallocate, mmReallocate, &MMHeap());
        nv::SetGraphicsServiceName("nvdrv:t");
        nv::SetGraphicsAllocator(mmAllocate, mmDeallocate, mmReallocate, &CoreHeap());
        nv::SetGraphicsDevtoolsAllocator(mmAllocate, mmDeallocate, mmReallocate, &CoreHeap());
        nv::InitializeGraphics(g_GraphicsHeap, g_GraphicsSystemMemorySize);

        exit_calls_.push_back([] { NN_SDK_LOG("Cleanup: all done.. \n"); });
        exit_calls_.push_back([] { nv::FinalizeGraphics();
        NN_SDK_LOG("Cleanup: nv::FinalizeGraphics()\n");
        });

        //--------------------------
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountHostRoot());
        exit_calls_.push_back([] { nn::fs::UnmountHostRoot(); });

        std::vector<std::string> additional_filenames___;

        if (!options_.playlist_file_.empty()) {
            if (options_.playlist_file_.find(':') == options_.playlist_file_.npos) {
                additional_filenames___ = movie::sample::get_lines_from_string(movie::sample::get_file_as_string(
                    std::string{ options_.working_folder_ }.append(options_.playlist_file_).c_str()));
            } else {
                additional_filenames___ = movie::sample::get_lines_from_string(movie::sample::get_file_as_string(
                    std::string{ options_.playlist_file_ }.c_str()));
            }
            for (auto&& s : additional_filenames___) {
                if (s.find(':') == s.npos)
                    s.insert(begin(s), begin(options_.working_folder_), end(options_.working_folder_));
            }

            options_.playlist_.insert(end(options_.playlist_),
                begin(additional_filenames___), end(additional_filenames___));
        }

        for (auto&& e : options_.playlist_) {
            NN_SDK_LOG("loaded up %s\n", std::string{ e }.c_str());
        }

        auto unique_uri_schemes_ = movie::sample::get_unique_uri_schemes(options_.playlist_);

        for (auto&& e : unique_uri_schemes_) {
            exit_calls_.push_back(movie::sample::get_mount_handler(e));
        }
        //--------------------------

        movie::PlayerConfig config;
        config.returnAudioDataToClient = false;
        config.returnVideoDataToClient = true;
        config.videoDecodeMode = options_.videoDecodeMode;
        config.returnSyncedData = true;
        config.useClientMemoryForOutput = true;
        config.autoReleaseOutputBuffers = true;
        config.audioFormat = movie::OutputFormat_AudioPcm16;
        config.videoFormat = options_.videoOutputFormat;
        config.maxResolution = options_.maxPlayerRes;

        movie::BrowserConfig browserConfig;
        browserConfig.coreMask = 0x00;
        browserConfig.sfThreadPriority = 16;
        browserConfig.audioRendererThreadPriority = 16;
        browserConfig.videoRendererThreadPriority = 16;

        //--------------------------

        MediaPlayerObserver mediaPlayerObserver{ config };

        ///////////////////////////////////////////////////////////////

        sol::state lua;

        lua.open_libraries();

        int64_t trackDuration{};
        int32_t trackCount{};
        int32_t width{};
        int32_t height{};

        movie::sample::pausable_loop rendering_thread_;

        //--------------------------

        std::mt19937 rng_{ options_.rng_seed_value_ };
        NN_SDK_LOG("Seeding rng with value == %u\n", options_.rng_seed_value_);

        auto iterz_ = additional_filenames___.begin();
        auto endz_ = additional_filenames___.end();


        auto get_filenames = [&]() {
            if (iterz_ == endz_) return std::make_tuple(false, "");

            auto k = iterz_++;
            return std::make_tuple(true, k->c_str());
        };

        auto get_duration = [](movie::BrowserPlayer& player_) { int64_t duration_{}; player_.GetPlaybackDuration(&duration_); return duration_; };

        auto prepare_ = [&](movie::BrowserPlayer& player_) {
            NN_SDK_LOG("LuaPlayer: preparing..\n");
            auto result = player_.Prepare();
            if (result == movie::Status_Success) {
                NN_SDK_LOG("LuaPlayer: waiting for prepare completion...\n");
                mediaPlayerObserver.WaitForPrepare();

                player_.GetPlaybackDuration(&trackDuration);    // duration is in microseconds
                trackDuration = trackDuration / 1000;           // convert to milliseconds

                player_.GetVideoDimensions(&width, &height);
                player_.GetTrackCount(&trackCount);

                for (int32_t t = 0; t < trackCount; t++)        // TODO more than two tracks?
                {
                    movie::TrackType trackType{};
                    player_.GetTrackType(t, &trackType);
                    switch (trackType)
                    {
                    case movie::TrackType_Audio:
                        mediaPlayerObserver.SetAudioTrackNumber(t);
                        break;
                    case movie::TrackType_Video:
                        if (options_.playVideo)
                            options_.isVideoTrack = true;
                        mediaPlayerObserver.SetVideoTrackNumber(t);
                        break;
                    case movie::TrackType_Subtitle: break;
                    case movie::TrackType_ClosedCaption: break;
                    default: break;
                    }
                }
            }
            return result == movie::Status_Success;
        };

        auto start_ = [&](movie::BrowserPlayer& player_) {
            NN_SDK_LOG("LuaPlayer: starting..\n");
            auto result = player_.Start();
            if (result == movie::Status_Success) {
                NN_SDK_LOG("LuaPlayer: waiting for start compeletion..\n");
                mediaPlayerObserver.WaitForStart();
                if (options_.isVideoTrack) {
                    player_.GetVideoDimensions(&width, &height);
                }
                rendering_thread_.resume();
            }
            return result == movie::Status_Success;
        };

        lua.new_usertype<movie::BrowserPlayer>("Player",
            "new", sol::factories([&]() {
            movie::BrowserPlayer* player{};
            movie::BrowserPlayer::Create(&player, &config, &browserConfig);
            mediaPlayerObserver.SetMoviePlayer(*player);
            player->SetObserver(&mediaPlayerObserver);
            return player; }),
            "SetDataSource", &movie::BrowserPlayer::SetDataSource,
            "SetMSESource", &movie::BrowserPlayer::SetMSESource,
            "Prepare", prepare_,
            "Reset", &movie::BrowserPlayer::Reset,
            "TerminateNetworkConnection", &movie::BrowserPlayer::TerminateNetworkConnection,
            "SeekTo", static_cast<movie::Status(movie::BrowserPlayer::*)(int64_t)>(&movie::BrowserPlayer::SeekTo), // This static_cast is needed to disambiguate SeekTo()
            "Stop", [&](movie::BrowserPlayer& player_) {
                player_.Stop();
                mediaPlayerObserver.WaitForStop();
                rendering_thread_.pause();
            },
            "Start", start_,
            "Pause", &movie::BrowserPlayer::Pause,
            "GetDuration", get_duration,
            "SeekRandom", [&](movie::BrowserPlayer& player_) {
                auto seek_target_ = std::uniform_int_distribution<int64_t>{ 0, trackDuration }(rng_);
                NN_SDK_LOG("Seeking to %" PRId64 "\n", seek_target_);
                player_.SeekTo(seek_target_);
            },
            "SetPlaybackRate", &movie::BrowserPlayer::SetPlaybackRate,
            sol::base_classes, sol::bases<movie::Player>{},
            sol::meta_function::garbage_collect, sol::destructor([](auto...) {}));

        lua.new_usertype<NXMSESourceBuffer>("NXMSESourceBuffer",
            "new", sol::constructors<NXMSESourceBuffer()>{},
            "appendData", &NXMSESourceBuffer::appendData,
            sol::base_classes, sol::bases<movie::DataSource>{});

        lua.new_usertype<NXMSESource>("NXMSESource",
            "new", sol::constructors<NXMSESource()>{},
            "AddSourceBuffer", &NXMSESource::AddSourceBuffer,
            "GetSourceBuffer", &NXMSESource::GetSourceBuffer,
            "RemoveSourceBuffer", &NXMSESource::RemoveSourceBuffer,
            "IsEndOfStreamReached", &NXMSESource::IsEndOfStreamReached,
            sol::base_classes, sol::bases<movie::MSESource>{});

        lua.set_function("get_filenames", get_filenames);

        lua.set_function("destroy", [](movie::BrowserPlayer* ptr_) { movie::BrowserPlayer::Destroy(ptr_); });

        lua.set_function("milliseconds", [](int value) { return std::chrono::milliseconds(value); });
        lua.set_function("microseconds", [](int value) { return std::chrono::microseconds(value); });

        lua.set_function("wait", sol::overload(
            [](std::chrono::milliseconds m) { nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(m.count())); },
            [](std::chrono::microseconds u) { nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(u.count())); }
        ));

        lua.set_function("play_until_complete",
        [&](movie::BrowserPlayer& player_, int delay_between, sol::variadic_args va_) {
            std::vector<sol::thread> runners_; for ([[maybe_unused]] auto&& e : va_) { runners_.emplace_back(sol::thread::create(lua.lua_state())); }
            std::vector<sol::state_view> views_; for (auto&& e : runners_) { views_.push_back(e.state()); }
            std::vector<sol::coroutine> coros_; for (int i = 0; i < va_.size(); ++i) {
                coros_.emplace_back(views_[i], va_[i].as<sol::function>());
            }

            for (auto&& e : coros_) {
                    NN_SDK_LOG("initializing coro\n");
                    e(player_);
            }
            std::this_thread::sleep_for(std::chrono::milliseconds(delay_between));

            while (!mediaPlayerObserver.IsPlaybackComplete()) {
                NN_SDK_LOG("processing coroutines..\n");
                for (auto&& e : coros_) {
                    if (e.valid()) {
                        NN_SDK_LOG("resuming coroutine..\n");
                        e();
                    }
                    std::this_thread::sleep_for(std::chrono::milliseconds(delay_between));
                }
            std::this_thread::sleep_for(std::chrono::milliseconds(delay_between));
            }
            NN_SDK_LOG("done with play_until\n");
        });


        movie::sample::thread_wrapper
            my_lua_thread_{
            [&]() {

            auto err_handler_ = [](lua_State*, sol::protected_function_result result) {
                NN_SDK_LOG("LuaPlayer: error handler called\n");
                for (auto&& e : result) {
                    NN_LOG("%s\n", e.as<std::string>().c_str());
                }
                return result;
            };

            if (options_.lua_file_.length() > 0)
            {
                //load in the script if provided
                int64_t fileLength = 0;
                char* scriptBuffer = nullptr;
                nn::fs::FileHandle fileHandle;
                fileHandle.handle = NULL;
                nn::Result result;

                result = nn::fs::OpenFile(&fileHandle, std::string{ options_.lua_file_ }.c_str(), nn::fs::OpenMode_Read);
                nn::fs::GetFileSize(&fileLength, fileHandle);

                scriptBuffer = new char[fileLength];

                nn::fs::ReadFile(fileHandle, 0, scriptBuffer, fileLength);
                nn::fs::CloseFile(fileHandle);

                lua.script(scriptBuffer, err_handler_);

                if(scriptBuffer)
                {
                    delete [] scriptBuffer;
                    scriptBuffer = nullptr;
                }
            }
            else
            {
                lua.script(
                    //------------------------------------
                    R"(
function change_rate(my_player)
    print("entering change_rate coroutine") coroutine.yield()
    --print("Setting rate to 0.5x") my_player:SetPlaybackRate(0.5) coroutine.yield()
    --print("Setting rate to 2x") my_player:SetPlaybackRate(2.0) coroutine.yield()
    print("Setting rate to 1x") my_player:SetPlaybackRate(1.0) coroutine.yield()
    print("Done setting rates..")
end

function seek_around(my_player)
    print("entering seek_around coroutine") coroutine.yield()
    for i=0,10 do
        print("seeking " .. i .. " -- ")
        --my_player:SeekRandom() coroutine.yield()
    end
end

function start_playback(msesource)
    print("Lua playing MSE")
    local my_player = Player.new()
    my_player:SetMSESource(msesource)
    if my_player:Prepare() then
        if my_player:Start() then
            play_until_complete( my_player, 1000--[[, change_rate, seek_around]])
            my_player:Stop()
            destroy(my_player)
            print("LuaPlayer: exiting start_playback() ")
        else
            print("LuaPlayer: Failed to start")
            destroy(my_player)
        end
    else
        print("LuaPlayer: Failed to prepare")
        destroy(my_player)
    end
end
)"
                    //------------------------------------
                    , err_handler_);
            }

            NXMSESourceBuffer mseSourceBuffer;
            NXMSESource mseSource;
            NN_SDK_LOG("\n BrowserPlayerSample :: Create MSESource \n");

            int64_t fileLength = 0;
            char* fileBuffer = nullptr;
            char* directoryPath = nullptr;
            char* file = nullptr;
            char* segmentName = nullptr;
            nn::fs::FileHandle fileHandle;
            fileHandle.handle = NULL;
            nn::Result result;

            int64_t segmentLength = 0;
            char* segmentBuffer = nullptr;

            bool initializationSegmentSet = false;
            bool bailTest = false;

            result = nn::fs::OpenFile(&fileHandle, options_.inputFileName, nn::fs::OpenMode_Read);
            if (result.IsFailure())
            {
                NN_SDK_LOG("\n Failed to open %s \n \n Exiting BrowserPlayerSample Test \n", options_.inputFileName);
                bailTest = true;
            }
            else
            {
                // Read the file into a buffer
                nn::fs::GetFileSize(&fileLength, fileHandle);

                fileBuffer = new char[fileLength];

                nn::fs::ReadFile(fileHandle, 0, fileBuffer, fileLength);
                nn::fs::CloseFile(fileHandle);

                // Determine the path of the file
                int dirPos = 0;
                int directoryLength = 0;
                while (options_.inputFileName[dirPos] != '\0')
                {
                    if (options_.inputFileName[dirPos] == '\\')
                    {
                        directoryLength = dirPos + 1; //Include '\'
                    }
                    ++dirPos;
                }

                // Copy the path
                if (directoryLength)
                {
                    directoryPath = new char[directoryLength + 1];
                    memcpy(directoryPath, options_.inputFileName, directoryLength);
                    directoryPath[directoryLength] = 0;
                }

                xmlDoc *doc = NULL;
                xmlNode *root_element = NULL;
                std::vector<std::string> fileNames;
                doc = xmlReadMemory(fileBuffer, fileLength, options_.inputFileName, NULL, 0);

                root_element = xmlDocGetRootElement(doc);

                TrackInfo trackInfo;

                print_element_names(root_element, trackInfo);

                xmlFreeDoc(doc);

                xmlCleanupParser();

                // Get each segment in the MPD file
                std::string dirPath = directoryPath;
                std::string catPath;

                catPath = dirPath + trackInfo.initSegment;
                result = nn::fs::OpenFile(&fileHandle, catPath.c_str(), nn::fs::OpenMode_Read);

                if (result.IsFailure())
                {
                    NN_SDK_LOG("\n Failed to open Segment File %s \n \n Exiting BrowserPlayerSample Test \n", catPath.c_str());
                    bailTest = true;
                }
                else
                {
                    // Read the Segment file into a buffer
                    nn::fs::GetFileSize(&segmentLength, fileHandle);

                    if (segmentBuffer)
                    {
                        delete[] segmentBuffer;
                    }
                    segmentBuffer = new char[segmentLength];

#if VERBOSE_LOGGING
                    NN_SDK_LOG("Reading segment file %s\n", file);
                    NN_SDK_LOG("Reading segment file size %d\n", segmentLength);
#endif /* VERBOSE_LOGGING */

                    nn::fs::ReadFile(fileHandle, 0, segmentBuffer, segmentLength);
                    nn::fs::CloseFile(fileHandle);

                    initializationSegmentSet = true;
                    mseSourceBuffer.appendData(segmentBuffer, segmentLength, MSEDataType::INITIALIZATION_SEGMENT);
                    NN_SDK_LOG("\n BrowserPlayerSample:: getSegmentDuration(%lld): %lld \n", 0, mseSourceBuffer.getSegmentDuration(0));
                }

                if (!bailTest) {
                    for (int i = 0; i < trackInfo.dataSegments.size(); i++)
                    {
                        if (!bailTest) {
                            catPath = dirPath + trackInfo.dataSegments[i];
                            // Open the Data Segment file
                            result = nn::fs::OpenFile(&fileHandle, catPath.c_str(), nn::fs::OpenMode_Read);
                            if (result.IsFailure())
                            {
                                NN_SDK_LOG("\n Failed to open Segment File %s \n \n Exiting BrowserPlayerSample Test \n", catPath.c_str());
                                bailTest = true;
                            }
                            else
                            {
                                // Read the Segment file into a buffer
                                nn::fs::GetFileSize(&segmentLength, fileHandle);

                                if (segmentBuffer)
                                {
                                    delete[] segmentBuffer;
                                }
                                segmentBuffer = new char[segmentLength];

#if VERBOSE_LOGGING
                                NN_SDK_LOG("Reading segment file %s\n", file);
                                NN_SDK_LOG("Reading segment file size %d\n", segmentLength);
#endif /* VERBOSE_LOGGING */

                                nn::fs::ReadFile(fileHandle, 0, segmentBuffer, segmentLength);
                                nn::fs::CloseFile(fileHandle);

                                // Just set the first segment as the intialization segment
                                mseSourceBuffer.appendData(segmentBuffer, segmentLength, MSEDataType::DATA_SEGMENT);
                            }
                        }
                    }

                    mseSourceBuffer.appendData(0, 0, MSEDataType::END_OF_FILE_SEGMENT);
                }
            }

            NN_SDK_LOG("\n BrowserPlayerSample:: MSE Source buffer filled\n");

            if (directoryPath)
            {
                delete[] directoryPath;
                directoryPath = nullptr;
            }

            if (file)
            {
                delete[] file;
                file = nullptr;
            }

            if (segmentName)
            {
                delete[] segmentName;
                segmentName = nullptr;
            }

            if (segmentBuffer)
            {
                delete[] segmentBuffer;
                segmentBuffer = nullptr;
            }

            if (fileBuffer)
            {
                delete[] fileBuffer;
                fileBuffer = nullptr;
            }

            if (!bailTest) {
                mseSource.AddSourceBuffer(&mseSourceBuffer);

                sol::thread runner = sol::thread::create(lua);
                sol::state_view view(runner.state());
                sol::coroutine cr = view["start_playback"];

                NN_SDK_LOG("about to start lua..\n");

                cr(mseSource);
            }

            rendering_thread_.stop();
            NN_SDK_LOG("Stopping render thread; Lua thread exiting..\n");
        } };

        ///////////////////////////////////////////////////////////////

        auto const start_time_
            = std::chrono::high_resolution_clock::now();

        ///////////////////////////////////////////////////////////////

        rendering_thread_.loop(
            [&] {
            if (options_.isVideoTrack)
            {
                int yOffset = 0;
                int uvOffset = 0;
                int ystride = 0;
                int nativeWidth = 0;
                int nativeHeight = 0;
                int cropWidth = 0;
                int cropHeight = 0;
                int colorSpace = 0;
                int32_t bufferIndex = -1;
                int32_t trackNumber = -1;
                int32_t bufferIndexForProperties = -1;

                if ((config.returnVideoDataToClient == true) && (config.useClientMemoryForOutput == true))
                {
                    nn::os::LockMutex(&mediaPlayerObserver.m_VideoOutputBuffersListMutex);
                    uint32_t videoListSize = mediaPlayerObserver.m_VideoOutputBuffersList.size();

                    if (videoListSize > 0)
                    {
                        bool videoAvailableForRendering = false;
                        MediaPlayerObserver::OutputBufferInfo videoBuffer;

                        videoBuffer = mediaPlayerObserver.m_VideoOutputBuffersList.front();
                        mediaPlayerObserver.m_VideoOutputBuffersList.erase(mediaPlayerObserver.m_VideoOutputBuffersList.begin());

                        nn::os::UnlockMutex(&mediaPlayerObserver.m_VideoOutputBuffersListMutex);

                        bufferIndex = videoBuffer.bufferIndex;
                        trackNumber = videoBuffer.trackNumber;

                        if (config.videoDecodeMode == movie::VideoDecodeMode_Cpu)
                        {
                            bufferIndexForProperties = bufferIndex;
                        }
                        else
                        {
                            //make sure the BufferIndex is one of the valid ones
                            for (int i = 0; i < mediaPlayerObserver.m_videoBufferCount; ++i)
                            {
                                if (bufferIndex == mediaPlayerObserver.m_videoBufferIndexArray[i])
                                {
                                    bufferIndexForProperties = bufferIndex;
                                    mediaPlayerObserver.m_PresentationIndex = bufferIndex;
                                    videoAvailableForRendering = true;
                                    break;
                                }
                            }
                        }

                        mediaPlayerObserver.GetOutputBufferProperties(bufferIndexForProperties, &mediaPlayerObserver.m_BufferProperty);
                        mediaPlayerObserver.m_BufferProperty.FindInt32("nv12-y-stride", &ystride);
                        mediaPlayerObserver.m_BufferProperty.FindInt32("nv12-y-offset", &yOffset);
                        mediaPlayerObserver.m_BufferProperty.FindInt32("nv12-uv-offset", &uvOffset);
                        mediaPlayerObserver.m_BufferProperty.FindInt32("width", &nativeWidth);
                        mediaPlayerObserver.m_BufferProperty.FindInt32("height", &nativeHeight);
                        mediaPlayerObserver.m_BufferProperty.FindInt32("crop-width", &cropWidth);
                        mediaPlayerObserver.m_BufferProperty.FindInt32("crop-height", &cropHeight);
                        mediaPlayerObserver.m_BufferProperty.FindInt32("nv12-colorspace", &colorSpace);
                        if (mediaPlayerObserver.m_CropWidth != cropWidth || mediaPlayerObserver.m_CropHeight != cropHeight)
                        {
                            mediaPlayerObserver.m_CropWidth = cropWidth;
                            mediaPlayerObserver.m_CropHeight = cropHeight;
                            if (config.videoFormat == movie::OutputFormat_VideoColorNv12)
                            {
                                NN_SDK_LOG("Video resolution changed, ResizeTextures : WxH %dx%d ystride %d\n", cropWidth, cropHeight, ystride);
                                if ((cropWidth > 0) && (cropHeight > 0))
                                {
                                    mediaPlayerObserver.ResizeTextures(cropWidth, cropHeight);
                                }
                            }
                        }

                        if (videoAvailableForRendering)
                        {
                            mediaPlayerObserver.DrawVideoNvn(mediaPlayerObserver.m_PresentationIndex, cropWidth, cropHeight, yOffset, uvOffset, ystride, colorSpace);
                        }
                    }
                    else
                    {
                        nn::os::UnlockMutex(&mediaPlayerObserver.m_VideoOutputBuffersListMutex);
                    }
                }
            }
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));
        });

        auto playback_duration_
            = std::chrono::duration_cast<std::chrono::milliseconds>(std::chrono::high_resolution_clock::now() - start_time_).count();
        NN_SDK_LOG(" \n Track Duration : %" PRId64 " msec ******* Total Playback Time : %" PRId64 " msec\n",
            trackDuration, playback_duration_);

    } // end of embedded scope
} //NOLINT(impl/function_size)




