﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <getopt.h>
#include "MediaPlayerObserver.h"
#include "HeapTracker.h"
#include <movie/Utils.h>
#include <nn/nn_Common.h>
#include <nn/nn_SdkLog.h>
#include <nn/init.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/hid.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>
#include <nn/util/util_ScopeExit.h>

#include <nv/nv_MemoryManagement.h>
#include <sys/time.h>

#include <nn/nifm.h>
#include <cinttypes>
#include <limits>
#include <cmath>

#include <nvn/nvn_FuncPtrInline.h>
#include <nn/util/util_Matrix.h>
#include <nn/util/util_Vector.h>

#include <curl/curl.h>

//extern bool USE_PROXY;
//extern std::string PROXY_SETTING_HOST;
//extern int PROXY_SETTING_PORT;
//extern std::string PROXY_SETTING_USERPWD;
//extern unsigned long PROXY_SETTING_AUTH;

bool USE_PROXY = false;
std::string PROXY_SETTING_HOST = "proxy-host.com";
int PROXY_SETTING_PORT = 8080;
std::string PROXY_SETTING_USERPWD = "UserName:Password";
unsigned long PROXY_SETTING_AUTH = CURLAUTH_BASIC;

constexpr auto MAX_NVN_TEXTURE_WIDTH = 3840;
constexpr auto MAX_NVN_TEXTURE_HEIGHT = 2160;
constexpr auto MAX_NVN_BUFFER_SIZE = 2 * MAX_NVN_TEXTURE_WIDTH * MAX_NVN_TEXTURE_HEIGHT;

MediaPlayerObserver::MediaPlayerObserver(movie::PlayerConfig const& config_)
    : m_PlayerConfig{config_}
{
    format_changed_.clear();

    nn::os::InitializeMutex(&m_VideoOutputBuffersListMutex, false, 0 );

    nn::os::InitializeEvent(&m_PlaybackCompleteEvent, false, nn::os::EventClearMode_AutoClear);
    nn::os::InitializeEvent(&m_StartPlaybackEvent, false, nn::os::EventClearMode_AutoClear);

    nn::os::InitializeEvent(&m_StopPlaybackEvent, false, nn::os::EventClearMode_AutoClear);
    nn::os::InitializeEvent(&m_PreparedEvent, false, nn::os::EventClearMode_AutoClear);

    m_PlayerConfig = config_;
    m_Player = NULL;

    m_AudioTrackNumber = -1;
    m_VideoTrackNumber = -1;
    m_VideoOutputBufferSize = 0;
    m_PresentationIndex = -1;
    m_CropWidth = 0;
    m_CropHeight = 0;

    if( ( m_PlayerConfig.returnVideoDataToClient == true ) && ( m_PlayerConfig.useClientMemoryForOutput == true ) )
    {
        m_PcmBufferSize = 1024 * 64;    // Calculate based on samplerate and alignment requirement
        m_PcmBuffer.reset( new char[m_PcmBufferSize] );
    }

    NvnVideoRenderer::NvnOutputFormat nvnOutputFormat = NvnVideoRenderer::NvnOutputFormat_YuvNv12NvnBuffer;
    if (m_PlayerConfig.videoFormat == movie::OutputFormat_VideoColorAbgr)
    {
        nvnOutputFormat = NvnVideoRenderer::NvnOutputFormat_AbgrNvnTexture;
    }
    m_NvnVideoRenderer = std::make_unique<NvnVideoRenderer>(nullptr, nvnOutputFormat);
    m_NvnVideoRenderer->Initialize();
}

MediaPlayerObserver::~MediaPlayerObserver()
{
    m_NvnVideoRenderer->Finalize();
    nn::os::FinalizeMutex( &m_VideoOutputBuffersListMutex );

    nn::os::FinalizeEvent(&m_StartPlaybackEvent);
    nn::os::FinalizeEvent(&m_PlaybackCompleteEvent);
    nn::os::FinalizeEvent(&m_StopPlaybackEvent);
    nn::os::FinalizeEvent(&m_PreparedEvent);

    if (m_ctxPtr) {
        m_ctxPtr->Destroy();
    }
}

void MediaPlayerObserver::SetMoviePlayer(movie::BrowserPlayer& player)
{
    m_Player = std::addressof(player);

    m_NvnVideoRenderer->ResizeTextures(MAX_NVN_TEXTURE_WIDTH, MAX_NVN_TEXTURE_HEIGHT);
    int videoBufferSize = MAX_NVN_BUFFER_SIZE;

    if ((m_PlayerConfig.videoDecodeMode == movie::VideoDecodeMode_NativeTexture) &&
        (m_PlayerConfig.videoFormat == movie::OutputFormat_VideoColorAbgr))
    {
        size_t textureSize = 0;

        m_videobufferArray[0].SetRange(0, videoBufferSize);
        void *VideoTexture = nullptr;
        m_NvnVideoRenderer->GetNvnVideoTexture(0, &VideoTexture, &textureSize);
        m_videobufferArray[0].SetDataAndCapacity(VideoTexture, videoBufferSize);
        m_Player->RegisterOutputBuffer(&m_videobufferArray[0], &m_videoBufferIndexArray[0]);
    }
    else if ((m_PlayerConfig.videoDecodeMode == movie::VideoDecodeMode_Cpu) &&
        (m_PlayerConfig.videoFormat == movie::OutputFormat_VideoColorNv12))
    {
        char *bufferMemory = nullptr;
        int yuvBufferIndex = 0;
        size_t yuvBufferSize = 0;
        m_NvnVideoRenderer->GetNvnVideoBuffer(yuvBufferIndex, &bufferMemory, &yuvBufferSize);

        m_videobufferArray[0].SetDataAndCapacity(bufferMemory, yuvBufferSize);
        m_videobufferArray[0].SetRange(0, yuvBufferSize);

        if (m_PlayerConfig.videoDecodeMode == movie::VideoDecodeMode_Cpu)
        {
            m_videoBufferIndexArray[0] = 0;
        }
    }
}

/* Called when player state changes */
void MediaPlayerObserver::OnStateChange(movie::PlayerState state)
{
    switch (state)
    {
    case movie::PlayerState_PlaybackCompleted:
        NN_LOG( "\n PlayerState_PlaybackCompleted\n" );
        // Signal event, in case was waiting at start
        playback_complete_.store(true);
        nn::os::SignalEvent(&m_PlaybackCompleteEvent);
        //nn::os::SignalEvent(&m_StartPlaybackEvent);
        break;

    case movie::PlayerState_Prepared:
        NN_SDK_LOG( "\n PlayerState_Prepared\n" );
        nn::os::SignalEvent(&m_PreparedEvent);
        break;

    case movie::PlayerState_Seeking:
        if( m_PlayerConfig.videoDecodeMode == movie::VideoDecodeMode_Cpu )
        {
            nn::os::LockMutex(&m_VideoOutputBuffersListMutex);
            m_VideoOutputBuffersList.clear();
            nn::os::UnlockMutex(&m_VideoOutputBuffersListMutex);
        }
        NN_SDK_LOG("\n PlayerState_Seeking\n");
        break;

    case movie::PlayerState_SeekCompleted:
        NN_SDK_LOG( "\n PlayerState_SeekCompleted\n" );
        break;

    case movie::PlayerState_Started:
        NN_SDK_LOG( "\n PlayerState_Started\n" );
        playback_complete_.store(false);
        nn::os::SignalEvent(&m_StartPlaybackEvent);
        break;

    case movie::PlayerState_Paused:
        NN_SDK_LOG( "\n PlayerState_Paused\n" );
        break;

    case movie::PlayerState_Stopped:
        NN_SDK_LOG( "\n PlayerState_Stopped\n" );
        nn::os::SignalEvent(&m_StopPlaybackEvent);
        break;

    default:
        break;
    }
}

void MediaPlayerObserver::OnError(movie::Status movieStatus)
{
    // Signal event, in case was waiting at start
    //nn::os::SignalEvent(&m_StartPlaybackEvent);
    NN_SDK_LOG("\n Player OnError: %s \n", movie::StatusToString(movieStatus));
}

void MediaPlayerObserver::OnAudioOutputFrameAvailable(movie::AudioFrameInfo* frameInfo)
{
    NN_SDK_LOG( "\n OnAudioOutputFrameAvailable size: %d \n", frameInfo->decodedSamplesSize);
}

void MediaPlayerObserver::OnVideoOutputFrameAvailable(movie::VideoFrameInfo* frameInfo)
{
    //NN_SDK_LOG( "\n OnVideoOutputFrameAvailable size: %d \n", frameInfo->decodedSamplesSize);
    if( frameInfo->decodedSamplesSize > 0 )
    {

    }
}

/* Called repeatedly after Player::Start() while the movie preloads */
void MediaPlayerObserver::OnBufferingUpdate(float startTime, float endTime)
{

}

/* Called before curl_easy_perform() for an HTTP request. Return 0 to proceed */
int32_t MediaPlayerObserver::OnCurlInit(CURL* easyRequest, const char* uri)
{
    return 0;
}

/* Called before curl_easy_perform() for an HTTP request. Return 0 to proceed */
int32_t MediaPlayerObserver::OnHttpRequest(CURL* easyRequest, const char* uri)
{
    if (!m_ctxPtr) {
        static nn::ssl::Context ctx;
        nn::Result resultContextCreate = ctx.Create(nn::ssl::Context::SslVersion_Auto);
        if (resultContextCreate.IsFailure())
        {
            NN_SDK_LOG("\n nn::ssl::Context::create failed: %s\n", uri);
        }
        m_ctxPtr = &ctx;
    }
    CURLcode curlRes;
    curlRes = curl_easy_setopt(easyRequest, CURLOPT_SSL_CONTEXT, m_ctxPtr);
    if (curlRes != CURLE_OK)
        return curlRes;
    //Note: Turning off verification is not typical.  This is only for testing SSL in this limited context
    //See SSL samples and documentation for more information
    curlRes = curl_easy_setopt(easyRequest, CURLOPT_SSL_VERIFYHOST, 0L);
    if (curlRes != CURLE_OK)
        return curlRes;
    curlRes = curl_easy_setopt(easyRequest, CURLOPT_SSL_VERIFYPEER, 0L);
    if (curlRes != CURLE_OK)
        return curlRes;

    if(USE_PROXY)
    {
        curlRes = curl_easy_setopt(easyRequest, CURLOPT_PROXY, PROXY_SETTING_HOST.c_str());
        if (curlRes != CURLE_OK)
            return curlRes;
        curl_easy_setopt(easyRequest, CURLOPT_PROXYPORT, PROXY_SETTING_PORT);
        if (curlRes != CURLE_OK)
            return curlRes;
        curl_easy_setopt(easyRequest, CURLOPT_PROXYUSERPWD, PROXY_SETTING_USERPWD.c_str());
        if (curlRes != CURLE_OK)
            return curlRes;
        curl_easy_setopt(easyRequest, CURLOPT_PROXYAUTH, PROXY_SETTING_AUTH);
        if (curlRes != CURLE_OK)
            return curlRes;
    }

    return 0;
}

/* Called before curl_multi_perform() for an HTTP request. Return 0 to proceed */
int32_t MediaPlayerObserver::OnMultiConfig(CURLM* multiRequest, const char* uri)
{
    NN_SDK_LOG("\n MediaPlayerObserver:: OnMultiConfig: %s\n", uri);
    return 0;
}

/* Called after curl_multi_perform() for an HTTP request. Return 0 to proceed. */
int32_t MediaPlayerObserver::OnHttpResponse(CURLM* multiRequest, const char* uri)
{
    NN_SDK_LOG("\n MediaPlayerObserver:: OnHttpResponse: %s\n", uri);
    return 0;
}

void MediaPlayerObserver::OnOutputBufferAvailable(int trackNumber, movie::TrackType eTrackType)
{
    NN_ASSERT(true, "This should never be called..");
    // TODO nuke all these m_Player uses, remove the player reference entirely from this observer..
    //OutputBufferInfo outputBuffer;
    //int32_t bufferIndex = -1;
    //int64_t presentationTimeUs = 0;
    //int32_t remainingBufferIndices = -1;
    //uint32_t flags = 0;
    //
    //if( ( m_VideoTrackNumber == trackNumber) && ( eTrackType == movie::TrackType_Video ) )
    //{
    //    if( m_PlayerConfig.videoDecodeMode == movie::VideoDecodeMode_Cpu )
    //    {
    //        movie::Status movieStatus = m_Player->AcquireOutputBufferIndex(trackNumber, &bufferIndex, &presentationTimeUs, &remainingBufferIndices);
    //        if( movieStatus == movie::Status_Success )
    //        {
    //            nn::os::LockMutex( &m_VideoOutputBuffersListMutex );
    //            outputBuffer.trackNumber = trackNumber;
    //            outputBuffer.bufferIndex = bufferIndex;
    //            m_VideoOutputBuffersList.push_back(outputBuffer);
    //            nn::os::UnlockMutex( &m_VideoOutputBuffersListMutex );
    //        }
    //    }
    //}
    //else if( ( m_AudioTrackNumber == trackNumber) && ( eTrackType == movie::TrackType_Audio ) )
    //{
    //    movie::Status movieStatus = m_Player->AcquireOutputBufferIndex(trackNumber, &bufferIndex, &presentationTimeUs, &remainingBufferIndices);
    //    if( movieStatus == movie::Status_Success )
    //    {
    //        movie::Buffer buffer;
    //        buffer.SetDataAndCapacity(m_PcmBuffer, m_PcmBufferSize);
    //        buffer.SetRange(0, m_PcmBufferSize);
    //        m_Player->FillOutputBuffer(trackNumber, bufferIndex,  &buffer, &presentationTimeUs, &flags);
    //    }
    //}
    //
}

void MediaPlayerObserver::OnOutputBufferAvailable(int trackNumber, movie::TrackType eTrackType, int64_t presentationTimeUs, int32_t index)
{
    //NN_SDK_LOG("vid out %d", index);
    if( m_Player != NULL )
    {
        if( ( m_VideoTrackNumber == trackNumber) && ( eTrackType == movie::TrackType_Video ) )
        {
            if( m_PlayerConfig.videoDecodeMode == movie::VideoDecodeMode_NativeTexture )
            {
                OutputBufferInfo outputBuffer;
                outputBuffer.trackNumber = trackNumber;
                outputBuffer.bufferIndex = index;
                //NN_SDK_LOG("OnOutputBufferAvailable: TS %lld index %d\n", presentationTimeUs, index);
                nn::os::LockMutex( &m_VideoOutputBuffersListMutex );
                m_VideoOutputBuffersList.push_back(outputBuffer);
                nn::os::UnlockMutex( &m_VideoOutputBuffersListMutex );
            }
        }
    }
}

void MediaPlayerObserver::OnFormatChanged(movie::TrackType eTrackType)
{
    format_changed_.clear();
    NN_SDK_LOG("MediaPlayerObserver::OnFormatChanged event..\n");
}

void MediaPlayerObserver::OnVideoFrameDropped(int64_t presentationTimeUs)
{
    NN_SDK_LOG("MediaPlayerObserver::OnVideoFrameDropped, Dropped frame presentationTimeUs = %lld \n", presentationTimeUs);
}

void MediaPlayerObserver::ResizeTextures(int width, int height)
{
     if ((m_PlayerConfig.videoDecodeMode == movie::VideoDecodeMode_Cpu) &&
        (m_PlayerConfig.videoFormat == movie::OutputFormat_VideoColorNv12))
    {
        char *bufferMemory = nullptr;
        int yuvBufferIndex = 0;
        size_t yuvBufferSize = 0;

        m_NvnVideoRenderer->ResizeTextures(width, height);
        m_NvnVideoRenderer->GetNvnVideoBuffer(yuvBufferIndex, &bufferMemory, &yuvBufferSize);

        m_videobufferArray[0].SetDataAndCapacity(bufferMemory, yuvBufferSize);
        m_videobufferArray[0].SetRange(0, yuvBufferSize);

        if (m_PlayerConfig.videoDecodeMode == movie::VideoDecodeMode_Cpu)
        {
            m_videoBufferIndexArray[0] = 0;
        }
    }
}

void MediaPlayerObserver::DrawVideoNvn(int producedBufferIndex, int width, int height, int yOffset, int uvOffset, int yStride, int colorSpace)
{
    if(m_NvnVideoRenderer != nullptr)
    {
        m_NvnVideoRenderer->Draw(width, height, yOffset, yStride, uvOffset, colorSpace);
        m_Player->ReleaseOutputBufferIndex(m_VideoTrackNumber, producedBufferIndex);
    }
}
