﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "ScopeExit.h"

#include <nn/nn_Abort.h>
#include <nn/fs.h>

#include <nn/nn_SdkLog.h>
#include <nn/fs/fs_SdCardForDebug.h>

#include <nn/nifm.h>
#include <nn/socket.h>
#include <curl/curl.h>

#include <string>
#include <sstream>
#include <vector>

#include <algorithm>
#include <cctype>
#include <iterator>

namespace movie { namespace sample {

//------------------------------------------------------------------------------

inline
std::string get_file_as_string(const char* filename_)
{
    nn::fs::FileHandle handle_;
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::fs::OpenFile(&handle_, filename_, nn::fs::OpenMode_Read)
    );
    int64_t sz_{};
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::fs::GetFileSize(&sz_, handle_)
    );

    std::string file_contents_(sz_,'\0');

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::fs::ReadFile(handle_, 0, std::addressof(file_contents_[0]), sz_)
    );
    nn::fs::CloseFile(handle_);
    return file_contents_;
}

//------------------------------------------------------------------------------

inline
std::vector<std::string> get_lines_from_string(std::string string_)
{
    std::vector<std::string> ret_;
    std::stringstream stream_{std::move(string_)};
    std::string tmp_;

    while (stream_ >> tmp_) {
        if (!tmp_.empty()) {
            ret_.push_back(tmp_);
        }
    }

    return ret_;
}

//------------------------------------------------------------------------------

template <typename Container>
std::vector<std::string> get_unique_uri_schemes(Container const& files_)
{
    std::vector<std::string> ret_;
    for (auto&& v : files_) {
        std::string e{v};
        auto pos = e.find(":");

        NN_ABORT_UNLESS(pos != e.npos, "Invalid filename: %s", e.c_str());

        std::transform(begin(e), end(e), begin(e), [](auto c){ return std::tolower(c); });
        ret_.emplace_back(e.substr(0,pos));
    }
    std::for_each(begin(ret_), end(ret_), [](auto& s) { if (s == "http") s = "https"; });
    std::sort(begin(ret_),end(ret_));
    ret_.erase( std::unique(begin(ret_),end(ret_))
               ,end(ret_));
    return ret_;
}

//------------------------------------------------------------------------------

[[nodiscard]] inline
movie::sample::ScopeExit get_mount_handler(std::string uri_scheme_)
{
    static nn::socket::ConfigDefaultWithMemory g_SocketConfigWithMemory;

    if (uri_scheme_ == "sdcard") {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountSdCardForDebug("sdcard"));
        return []{ nn::fs::Unmount("sdcard"); };
    }

    if (uri_scheme_ == "http" || uri_scheme_ == "https") {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::nifm::Initialize());
        nn::nifm::SubmitNetworkRequest();
        while (nn::nifm::IsNetworkRequestOnHold())
        {
            NN_SDK_LOG("Network request on hold...\n");
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        }

        NN_ABORT_UNLESS(nn::nifm::IsNetworkAvailable());
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::socket::Initialize(g_SocketConfigWithMemory));
        NN_ABORT_UNLESS(curl_global_init(CURL_GLOBAL_ALL) == CURLE_OK);

        return []{ curl_global_cleanup();
                   nn::socket::Finalize(); };
    }

    return []{
        // empty -- any others to add?
    };
}

}}
