﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
// --------------------------------------------------------------------------
// Small_Time.h
//
//  Quick and Dirty timing package
// --------------------------------------------------------------------------

#pragma once
#include <nn/os.h>
#include <nn/os/os_Tick.h>
#include <nn/nn_TimeSpan.h>
#include <nn/nn_log.h>

typedef double f64;
/// Numeric format for the clock.   (s32 secs, s32 nsecs)
typedef int64_t TClock;

/// Returns the current clock.  (For linux, uses clock_gettime() and converts to a s64.  Sadly, this is slower than a PC.)
inline TClock GetClock()
{
    nn::os::Tick t = nn::os::GetSystemTick();
    TClock x =  t.GetInt64Value();
    return x;
}

/// Computes tEnd - tStart for a pair of clocks.  (Note tEnd comes first).
inline TClock DiffClock(TClock tEnd, TClock tStart)
{
    return tEnd - tStart;
}

/// Computes t1 + t2 for a pair of clocks  - it shouldn't have to be said that these should be DiffClock() times
inline TClock AddClock(TClock t1, TClock t2)
{
    return t1 + t2;
}

/// Converts clock to nanoseconds.   (For windows, uses QueryPerformanceFrequency());
inline int64_t ClockToNSec(TClock t) // nanoseconds
{
    nn::TimeSpan ts = nn::os::ConvertToTimeSpan(nn::os::Tick(t));
    int64_t tn = ts.GetNanoSeconds();
    return tn;
}

/// Converts clock to microseconds : integer return, may truncate.
inline int32_t ClockToUSec(TClock t) // microseconds
{
    return (int32_t) (ClockToNSec(t) / 1000);
}

/// Converts clock to microseconds : integer return, may truncate.
inline int32_t ClockToMSec(TClock t) // milliseconds
{
    return (int32_t) (ClockToNSec(t) / 1000000);
}

/// Converts clock to secs - note this returns an FType (double)
inline f64 ClockToSec(TClock t) // microseconds
{
    return ((f64)ClockToNSec(t)) / 1.0e9;
}

#define TS_START()      { TClock tS = GetClock();
#define TS_END(tag)     { TClock tE = GetClock();  NN_LOG("****** %-32s : %d uS \n", tag, ClockToUSec(tE-tS));}  }
