﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <nn/nn_log.h>
#include "BlitVicController.h"

bool BindDisplay(nvdcDisplayInfo* pOutInfo, nvdcHandle handle, nvdcDisplayType type)
{
    int numDisplays;
    nvdcDisplayHandle* pDisplays;

    if ( nvdcQueryDisplays(handle, &pDisplays, &numDisplays) != 0 )
    {
        NN_LOG("nvdcQueryDisplays failed.\n");
        return false;
    }

    nvdcDisplayHandle* pDisplayToBind = nullptr;
    for ( int i = 0; i < numDisplays; ++i )
    {
        nvdcQueryDisplayInfo(handle, pDisplays[i], pOutInfo);
        if ( pOutInfo->type == type )
        {
            pDisplayToBind = pDisplays + i;
            break;
        }
    }

    if ( pDisplayToBind == nullptr )
    {
        nvdcFreeDisplays(handle, pDisplays);
        NN_LOG("Requested display not found.\n");
        return false;
    }

    if ( nvdcDisplayBind(handle, *pDisplayToBind, pOutInfo->boundHead) != 0 )
    {
        nvdcFreeDisplays(handle, pDisplays);
        NN_LOG("nvdcDisplayBind failed.\n");
        return false;
    }

    nvdcFreeDisplays(handle, pDisplays);

    return true;
}

bool Flip(nvdcHandle handle, NvRmSurface* pSurface, const nvdcDisplayInfo& info)
{
    nvdcFlipWinArgs window;
    window.index = 0;
    window.surfaces[0] = pSurface;
    window.surfaces[1] = nullptr;
    window.surfaces[2] = nullptr;
    window.format = nvdcRGB_AUTO;
    window.blend = NVDC_BLEND_NONE;
    window.x = 0;
    window.y = 0;
    window.w = pSurface->Width << 12;
    window.h = pSurface->Height << 12;
    window.outX = 0;
    window.outY = 0;
    window.outW = pSurface->Width;
    window.outH = pSurface->Height;
    window.z = 0;
    window.swapInterval = 1;
    window.preSyncptId = -1;
    window.orientation = nvdcOrientation_0;
    window.reflection = nvdcReflection_none;

    nvdcFlipArgs flip;
    memset(&flip, 0, sizeof(flip));
    flip.win = &window;
    flip.numWindows = 1;

    bool isSuccess = true;
    if ( nvdcGetWindow(handle, info.boundHead, 0) == 0 )
    {
        if ( nvdcFlip(handle, info.boundHead, &flip) != 0 )
        {
            isSuccess = false;
        }

        nvdcPutWindow(handle, info.boundHead, 0);
    }

    return isSuccess;
}
