﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/migration/idc/migration_SocketConnection.h>
#include <nn/migration/idc/migration_Server.h>
#include <nn/migration/idc/detail/migration_EncryptionPolicy.h>
#include <nn/nifm/nifm_ApiIpAddress.h>
#include <nn/os.h>
#include <nn/socket/socket_Api.h>
#include <nn/socket/socket_SystemConfig.h>
#include <nn/time/time_Api.h>
#include <nn/util/util_StringUtil.h>

#include "testMigrationIdc_TestUserContext.h"
#include "testMigrationIdc_SequenceServer.h"

using namespace nn;

namespace
{
    void WaitConnection(migration::idc::SocketConnection* pOut) NN_NOEXCEPT
    {
        socket::SockAddrIn m_ServerSocketAddress;
        int m_ServerAcceptSocketDescriptor = -1;
        int m_ServerSocketDescriptor = -1;

        NN_LOG("-- Server startup --\n");

        if( (m_ServerAcceptSocketDescriptor = socket::Socket(socket::Family::Af_Inet, socket::Type::Sock_Stream, socket::Protocol::IpProto_Tcp)) < 0 )
        {
            NN_ABORT("server: Socket failed (error %d)\n", socket::GetLastError());
        }

        socket::SockAddrIn serverSocketAddress = { 0 };
        serverSocketAddress.sin_addr.S_addr = socket::InetHtonl(socket::InAddr_Any);
        serverSocketAddress.sin_port = socket::InetHtons(0);
        serverSocketAddress.sin_family = socket::Family::Af_Inet;

        if( socket::Bind(m_ServerAcceptSocketDescriptor, reinterpret_cast<socket::SockAddr *>(&serverSocketAddress), sizeof(serverSocketAddress)) < 0 )
        {
            NN_ABORT("server: Bind failed (error %d)\n", socket::GetLastError());
        }

        socket::SockLenT saLen = sizeof(m_ServerSocketAddress);
        if( socket::GetSockName(m_ServerAcceptSocketDescriptor, reinterpret_cast<socket::SockAddr *>(&m_ServerSocketAddress), &saLen) < 0 )
        {
            NN_ABORT("server: GetSockName failed (error %d)\n", socket::GetLastError());
        }

        auto result = nifm::GetCurrentPrimaryIpAddress(&m_ServerSocketAddress.sin_addr);
        if( result.IsSuccess() )
        {
            NN_LOG("server: listening for incoming messages at: %s:%d\n",
                socket::InetNtoa(m_ServerSocketAddress.sin_addr),
                static_cast<int>(socket::InetNtohs(m_ServerSocketAddress.sin_port)));
        }
        else
        {
            NN_ABORT("server: GetCurrentIpAddress failed (error 0x%08x)\n", result.GetInnerValueForDebug());
        }

        if( socket::Listen(m_ServerAcceptSocketDescriptor, 4) < 0 )
        {
            NN_ABORT("server: listen failed (error %d)\n", socket::GetLastError());
        }

        socket::SockAddrIn saClientAddress = { 0 };
        socket::SockLenT clientAddressSize = sizeof(saClientAddress);
        // メモ : Accept はタイムアウトしない。
        if( (m_ServerSocketDescriptor = socket::Accept(m_ServerAcceptSocketDescriptor, reinterpret_cast<socket::SockAddr *>(&saClientAddress), &clientAddressSize)) < 0 )
        {
            NN_LOG("server: Accept failed (error %d)\n", socket::GetLastError());
            return;
        }
        NN_LOG("server: connection accepted from %s\n", socket::InetNtoa(saClientAddress.sin_addr));

        new (pOut) migration::idc::SocketConnection(m_ServerSocketDescriptor);
    }

}

Result RunServer() NN_NOEXCEPT
{
    migration::idc::SocketConnection connection;
    WaitConnection(&connection);

    static migration::idc::ServerContext<migration::idc::detail::DefaultEncryptionPolicy> serverContext;
    static nnt::migration::idc::TestUserContext userContext;
    static Bit8 buffer[sizeof(migration::idc::Server::WaitUserCommandBuffer<decltype(serverContext)>)];

    NN_RESULT_DO(migration::idc::Server::WaitConnection(connection, serverContext, 0, buffer, sizeof(buffer), nullptr));
    NN_RESULT_DO(migration::idc::Server::WaitUserCommand(connection, serverContext, userContext, 0, buffer, sizeof(buffer), nullptr));
    NN_RESULT_DO(migration::idc::Server::WaitTermination(connection, serverContext, 0, buffer, sizeof(buffer), nullptr));

    NN_RESULT_SUCCESS;
}
