﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "nntMigration_ConnectionPolicy.h"

#include <mutex>
#include <nn/migration/idc/detail/migration_HandleSocketError.h>
#include <nn/migration/detail/migration_Result.h>
#include <nn/socket/socket_Api.h>


namespace nnt { namespace migration {

// ---------------------------------------------------------------------------------------------
// InprocessConnectionCreatorBase

SimpleRegistrar<int, InprocessConnectionCreatorBase> InprocessConnectionCreatorBase::s_Registrar;
nn::migration::detail::Mutex InprocessConnectionCreatorBase::s_Lock = NN_MIGRATION_DETAIL_MUTEX_INITIALIZER(false);
uint32_t InprocessConnectionCreatorBase::s_Counter = 0;
InprocessConnectionCreatorBase::ResourceStorage InprocessConnectionCreatorBase::s_ResourceStorage;
nn::migration::idc::SharedBufferConnectionManager* InprocessConnectionCreatorBase::s_pManager;
std::aligned_storage<sizeof(nn::migration::idc::SharedBufferConnectionManager), std::alignment_of<nn::migration::idc::SharedBufferConnectionManager>::value>::type InprocessConnectionCreatorBase::s_ManagerStorage;

void InprocessConnectionCreatorBase::InitializeAllocator() NN_NOEXCEPT
{
    std::lock_guard<nn::migration::detail::Mutex> lock(s_Lock);
    if (s_Counter == 0u)
    {
        s_pManager = new(&s_ManagerStorage)nn::migration::idc::SharedBufferConnectionManager(&s_ResourceStorage, sizeof(s_ResourceStorage));
    }
    ++ s_Counter;
}
void InprocessConnectionCreatorBase::FinalizeAllocator() NN_NOEXCEPT
{
    std::lock_guard<nn::migration::detail::Mutex> lock(s_Lock);
    NN_SDK_ASSERT(s_Counter > 0u);
    --s_Counter;
    if (s_Counter == 0u)
    {
        s_pManager->~SharedBufferConnectionManager();
    }
}

nn::Result InprocessConnectionCreatorBase::CreateConnectionBase(
    nn::migration::idc::SharedBufferConnection* pOut,
    const nn::migration::UserMigrationServerProfile& profile, const nn::migration::detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    // クライアントとの接続
    nn::migration::idc::SharedBufferConnection con;
    int serverId;
    NN_ABORT_UNLESS(s_pManager->CreateServer(&con, &serverId, PerConnectionBufferSize / 2));

    nn::util::Uuid id;
    NN_RESULT_THROW_UNLESS(
        s_Registrar.RegisterAdvertisement(&id, serverId, profile, GetMatchingHint()),
        nn::migration::detail::ResultResourceLimit());
    bool releaseRequired = true;
    NN_UTIL_SCOPE_EXIT
    {
        if (releaseRequired)
        {
            int ejected;
            s_Registrar.EjectServer(&ejected, id);
        }
    };
    while (!s_pManager->WaitConnection(serverId, 1))
    {
        NN_RESULT_THROW_UNLESS(!nn::migration::detail::IsCanceled(pCancellable), nn::migration::ResultCanceled());
    }

    *pOut = std::move(con);
    releaseRequired = false;
    NN_RESULT_SUCCESS;
}
nn::Result InprocessConnectionCreatorBase::ConnectBase(
    nn::migration::idc::SharedBufferConnection* pOut,
    const nn::util::Uuid& serverId) NN_NOEXCEPT
{
    // サーバーとの接続
    int server;
    NN_ABORT_UNLESS(s_Registrar.EjectServer(&server, serverId));

    nn::migration::idc::SharedBufferConnection con;
    NN_ABORT_UNLESS(s_pManager->Connect(&con, server));

    *pOut = std::move(con);
    NN_RESULT_SUCCESS;
}
nn::Result InprocessConnectionCreatorBase::ScanServers(
    size_t* pOutActual, nn::migration::UserMigrationServerInfo* pOut, size_t count,
    const nn::account::NetworkServiceAccountId& nsaId,
    const nn::migration::detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    NN_UNUSED(pCancellable);
    *pOutActual = s_Registrar.ScanServers(pOut, count, GetMatchingHint());
    NN_RESULT_SUCCESS;
}
void InprocessConnectionCreatorBase::Reset() NN_NOEXCEPT
{
    // NOP
}

// ---------------------------------------------------------------------------------------------
// LocalhostConnectionCreator

SimpleRegistrar<uint16_t, LocalhostConnectionCreator> LocalhostConnectionCreator::s_Registrar;

nn::Result LocalhostConnectionCreator::EnableInternet(const nn::migration::detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    NN_SDK_ASSERT(!m_IsConnected);
    NN_RESULT_DO(nn::migration::detail::ConnectToInternet(m_Connection, nn::nifm::RequirementPreset_InternetBestEffort, pCancellable));
    m_IsConnected = true;
    NN_RESULT_SUCCESS;
}

nn::Result LocalhostConnectionCreator::ScanServers(
    size_t* pOutActual, nn::migration::UserMigrationServerInfo* pOut, size_t count,
    const nn::account::NetworkServiceAccountId& nsaId,
    const nn::migration::detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    NN_UNUSED(pCancellable);
    *pOutActual = s_Registrar.ScanServers(pOut, count, GetMatchingHint());
    NN_RESULT_SUCCESS;
}

void LocalhostConnectionCreator::Reset() NN_NOEXCEPT
{
    if (m_IsConnected)
    {
        m_Connection.CancelRequest();
        NN_SDK_ASSERT(m_Connection.GetSystemEvent().TryWait());
        m_Connection.GetSystemEvent().Clear();
        m_IsConnected = false;
    }
}

}} // ~namespace nnt::migration
