﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/teamcity/testTeamcity_Logger.h>

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>

#include <nn/os.h>
#include <nn/mem.h>
#include <nn/mem/mem_NumberLineAllocator.h>

#define ASSERT_NULL(p) ASSERT_TRUE(p == nullptr)
#define ASSERT_NOT_NULL(p) ASSERT_TRUE(p != nullptr)

namespace {

const size_t ThreadStackSize = 16384;                                   // スレッドのスタックサイズ
const int MaxThreadCount = 8;                                           // 作成するスレッドの最大数
NN_ALIGNAS(4096) char g_ThreadStack[MaxThreadCount][ThreadStackSize];   // 各スレッドのスタック
nn::os::ThreadType g_Thread[MaxThreadCount];                            // スレッド構造体

/**
 * @brief   各スレッドに渡す情報を定義した構造体です。
 */
struct AllocateInfo
{
    nn::mem::NumberLineAllocator* pAllocator;
    int allocatePattern;       // 各スレッドに渡されるユニークなサイズとアライメントの組み合わせのパターン
};

/**
* @brief   管理領域確保用コールバックです。
* @details std::malloc() を利用します。
*/
void* ManagAlloc(size_t size, void* pParam)
{
    NN_UNUSED(pParam);
    return std::malloc(size);
}

/**
* @brief   管理領域解放用コールバックです。
* @details std::malloc() を利用します。
*/
void ManagFree(void* addr, void* pParam)
{
    NN_UNUSED(pParam);
    std::free(addr);
}

/**
 * @brief   テストで利用するテストフィクスチャです。
 */
class NumberLineAllocatorThreadSafeTest : public ::testing::Test
{
protected:

    /**
     * @brief   テスト開始時に毎回呼び出される関数です。
     */
    virtual void SetUp()
    {
        // 利用できるコア番号とコア数の取得
        // 最大 MaxThreadCount 個取得する
        nn::Bit64 coreMask = nn::os::GetThreadAvailableCoreMask();
        m_MaxCoreCount = 0;
        for(int i = 0; i < 64; ++i)
        {
            if(coreMask & (1ULL << i))
            {
                m_CoreNumber[m_MaxCoreCount] = static_cast<int>(i);
                m_MaxCoreCount++;
                if(m_MaxCoreCount >= MaxThreadCount)
                {
                    break;
                }
            }
        }

        NN_LOG("use %d core.\n", m_MaxCoreCount);

        // ヒープを初期化
        m_Allocator.Initialize(ManagAlloc, nullptr, ManagFree, nullptr, true);
        m_Allocator.AddRange(0, 0x1000000);
    }

    /**
     * @brief   テスト終了時に毎回呼び出される関数です。
     */
    virtual void TearDown()
    {
        m_Allocator.Finalize();
    }

    /**
     * @brief       スレッドの動作を開始します
     * @param[in]   functionCount   動作させるスレッド関数の数
     */
    void BeginThread(const int functionCount)
    {
        for(int i = 0; i < functionCount && i < m_MaxCoreCount; ++i)
        {
            nn::os::StartThread(&g_Thread[i]);
        }

        for(int i = 0; i < functionCount && i < m_MaxCoreCount; ++i)
        {
            nn::os::WaitThread(&g_Thread[i]);
        }

        for(int i = 0; i < functionCount && i < m_MaxCoreCount; ++i)
        {
            nn::os::DestroyThread(&g_Thread[i]);
        }
    }

    /**
     * @brief       マルチスレッドによるテストを開始します。
     * @param[in]   pFunctions      スレッド関数の配列
     * @param[in]   functionCount   pFunctions の長さ
     */
    void DoMultiThreadTest(nn::os::ThreadFunction* pFunctions, const int functionCount)
    {
        ASSERT_TRUE(functionCount <= MaxThreadCount);

        nn::Result result;

        // functions の関数を動作させる
        for(int i = 0; i < functionCount && i < m_MaxCoreCount; ++i)
        {
            pInfo[i].pAllocator = &m_Allocator;
            pInfo[i].allocatePattern = i + 1;
            result = nn::os::CreateThread(&g_Thread[i], pFunctions[i], &pInfo[i], &g_ThreadStack[i], ThreadStackSize, nn::os::DefaultThreadPriority, m_CoreNumber[i]);
            ASSERT_TRUE(result.IsSuccess());
        }

        BeginThread(functionCount);
    }

protected:
    nn::mem::NumberLineAllocator m_Allocator;
    int m_CoreNumber[MaxThreadCount];           // 利用できる CPU コアの番号
    int m_MaxCoreCount;                         // 利用できる CPU コア数
    AllocateInfo pInfo[MaxThreadCount];
};

// =============
// スレッド関数
// =============

/**
 * @brief   確保に失敗するまでメモリブロックの確保を行います。
 * @details アライメントとりません。
 */
void Allocate(void* arg)
{
    AllocateInfo* pInfo = reinterpret_cast<AllocateInfo*>(arg);

    int allocateSize = pInfo->allocatePattern * 128;
    int number;

    const int originSize = allocateSize;

    while (pInfo->pAllocator->Allocate(&number, allocateSize) != NULL)
    {
        // 確保サイズに幅を持たせる
        allocateSize++;
        if (allocateSize > originSize + 10)
        {
            allocateSize = originSize;
        }
    }
}

/**
 * @brief   確保に失敗するまでメモリブロックの確保を行います。
 * @details アライメントをとります。
 */
void AlignedAllocate(void* arg)
{
    AllocateInfo* pInfo = reinterpret_cast<AllocateInfo*>(arg);

    int allocateSize = pInfo->allocatePattern * 128;
    int alignment = pInfo->allocatePattern;
    int number;

    const int originSize = allocateSize;
    const int originAlignment = alignment;

    while(pInfo->pAllocator->Allocate(&number, allocateSize, alignment) != NULL)
    {
        // 確保サイズとアライメントの組み合わせに幅を持たせる
        allocateSize++;
        alignment++;
        if (allocateSize > originSize + 10)
        {
            allocateSize = originSize;
        }
        if (alignment > originAlignment + 11)
        {
            alignment = originAlignment;
        }
    }
}

/**
 * @brief   メモリブロックの確保と解放を行います。
 * @details また、確保済みのメモリブロックが破壊されていないか確認します。
 *          アライメントをとりません。
 */
void AllocateFree(void* arg)
{
    AllocateInfo* pInfo = reinterpret_cast<AllocateInfo*>(arg);

    const int repeatCount = 1000;
    int allocateSize = 3 * pInfo->allocatePattern;
    int number;

    const int originSize = allocateSize;

    for (int i = 0; i < repeatCount; ++i)
    {
        ASSERT_TRUE(pInfo->pAllocator->Allocate(&number, allocateSize));
        ASSERT_GE(pInfo->pAllocator->GetSizeOf(number), allocateSize);
        pInfo->pAllocator->Free(number);

        // 確保サイズに幅を持たせる
        allocateSize++;
        if (allocateSize > originSize + 10)
        {
            allocateSize = originSize;
        }
    }
}

/**
 * @brief   メモリブロックの確保と解放を行います。
 * @details また、確保済みのメモリブロックが破壊されていないか確認します。
 *          アライメントをとります。
 */
void AlignedAllocateFree(void* arg)
{
    AllocateInfo* pInfo = reinterpret_cast<AllocateInfo*>(arg);

    const int repeatCount = 1000;
    int allocateSize = 3 * pInfo->allocatePattern;
    int alignment = pInfo->allocatePattern;
    int number;

    const int originSize = allocateSize;
    const int originAlignment = alignment;

    for (int i = 0; i < repeatCount; ++i)
    {
        ASSERT_TRUE(pInfo->pAllocator->Allocate(&number, allocateSize, alignment));
        ASSERT_GE(pInfo->pAllocator->GetSizeOf(number), allocateSize);
        pInfo->pAllocator->Free(number);

        // 確保サイズとアライメントの組み合わせに幅を持たせる
        allocateSize++;
        alignment++;
        if (allocateSize > originSize + 10)
        {
            allocateSize = originSize;
        }
        if (alignment > originAlignment + 11)
        {
            alignment = originAlignment;
        }
    }
}

}   // unnamed namespace

// =======
// テスト
// =======

/**
 * @brief   複数スレッドを作成し、確保を繰り返します。
 */
TEST_F(NumberLineAllocatorThreadSafeTest, Allocate)
{
    const int FunctionCount = 8;    // 並列実行する関数の数 ( < MaxThreadCount)

    // 並列実行する関数
    // ただし、環境によっていくつのコアがあるかはわからないため、どの関数までが実行されるかは環境依存
    // 例えば、コア数が 4 なら、 functions[0] ～ functions[3] が並列実行される
    nn::os::ThreadFunction functions[FunctionCount] = { Allocate,           // コア0で実行したい関数
                                                        AlignedAllocate,    // コア1で実行したい関数
                                                        Allocate,           // コア2で実行したい関数
                                                        AlignedAllocate,    // コア3で実行したい関数
                                                        Allocate,           // コア4で実行したい関数
                                                        AlignedAllocate,    // コア5で実行したい関数
                                                        Allocate,           // コア6で実行したい関数
                                                        AlignedAllocate };  // コア7で実行したい関数
    DoMultiThreadTest(functions, FunctionCount);
}

/**
 * @brief   Allocate テストを解放も含めて行ったものです。
 */
TEST_F(NumberLineAllocatorThreadSafeTest, AllocateFree)
{
    const int FunctionCount = 8;
    nn::os::ThreadFunction functions[FunctionCount] = { AllocateFree,
                                                        AlignedAllocateFree,
                                                        AllocateFree,
                                                        AlignedAllocateFree,
                                                        AllocateFree,
                                                        AlignedAllocateFree,
                                                        AllocateFree,
                                                        AlignedAllocateFree };
    DoMultiThreadTest(functions, FunctionCount);
}

