﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <algorithm>
#include <random>

#include <nnt/nntest.h>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/mem.h>
#include <nn/mem/mem_NumberLineAllocator.h>

#define ASSERT_NULL(p) ASSERT_TRUE(p == nullptr)
#define ASSERT_NOT_NULL(p) ASSERT_TRUE(p != nullptr)

/**
 * @brief   管理領域確保用コールバックです。
 * @details std::malloc() を利用します。
 */
void* ManagAlloc(size_t size, void* pParam)
{
    NN_UNUSED(pParam);
    return std::malloc(size);
}

/**
 * @brief   管理領域解放用コールバックです。
 * @details std::malloc() を利用します。
 */
void ManagFree(void* addr, void* pParam)
{
    NN_UNUSED(pParam);
    std::free(addr);
}

/**
 * @brief   管理領域確保用コールバックです。
 * @details nn::mem::StandardAllocator を利用します。
 */
void* ManagAllocWithSa(size_t size, void* pParam)
{
    nn::mem::StandardAllocator* pAllocator = reinterpret_cast<nn::mem::StandardAllocator*>(pParam);
    return pAllocator->Allocate(size);
}

/**
 * @brief   管理領域解放用コールバックです。
 * @details nn::mem::StandardAllocator を利用します。
 */
void ManagFreeWithSa(void* addr, void* pParam)
{
    nn::mem::StandardAllocator* pAllocator = reinterpret_cast<nn::mem::StandardAllocator*>(pParam);
    pAllocator->Free(addr);
}

/**
 * @brief   テストに与えるパラメータです。
 */
enum NumberLineAllocatorTestParam
{
    NumberLineAllocatorTestParam_Malloc = 0,    //!< 管理領域のアロケータに std::malloc() を利用する
    NumberLineAllocatorTestParam_SA             //!< 管理領域のアロケータに nn::mem::StandardAllocator を用意して使う
};

/**
 * @brief   テストフィクスチャです。
 */
class NumberLineAllocatorTest : public testing::TestWithParam<int>
{
protected:

    NumberLineAllocatorTest() : m_ManagAllocatorSize(256 * 1024 * 1024), m_StartAddr(nullptr){}

    /**
     * @brief テスト開始時に毎回呼び出される関数です。
     */
    virtual void SetUp()
    {
        switch(GetParam())
        {
        case NumberLineAllocatorTestParam_Malloc:
            m_Allocator.Initialize(ManagAlloc, nullptr, ManagFree, nullptr);
            break;
        case NumberLineAllocatorTestParam_SA:
#if defined(NN_BUILD_CONFIG_OS_WIN)
            // Win32 で数百 MB の領域を std::malloc しようとすると失敗することがあるので nn::os を利用
            uintptr_t ptr;
            nn::os::AllocateMemoryBlock(&ptr, m_ManagAllocatorSize);
            m_StartAddr = reinterpret_cast<void*>(ptr);
#else
            m_StartAddr = std::malloc(m_ManagAllocatorSize);
#endif
            NN_ABORT_UNLESS(m_StartAddr != nullptr);
            m_StandardAllocator.Initialize(m_StartAddr, m_ManagAllocatorSize);
            m_Allocator.Initialize(ManagAllocWithSa, &m_StandardAllocator, ManagFreeWithSa, &m_StandardAllocator);
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    /**
     * @brief テスト終了時に毎回呼び出される関数です。
     */
    virtual void TearDown()
    {
        switch(GetParam())
        {
        case NumberLineAllocatorTestParam_Malloc:
            m_Allocator.Finalize();
            break;
        case NumberLineAllocatorTestParam_SA:
            m_Allocator.Finalize();
            m_StandardAllocator.Finalize();

#if defined(NN_BUILD_CONFIG_OS_WIN)
            nn::os::FreeMemoryBlock(reinterpret_cast<uintptr_t>(m_StartAddr), m_ManagAllocatorSize);
#else
            std::free(m_StartAddr);
#endif
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

protected:

    nn::mem::NumberLineAllocator    m_Allocator;

private:

    const size_t                    m_ManagAllocatorSize;   // 管理領域用アロケータのサイズ
    void*                           m_StartAddr;            // 管理領域用アロケータの先頭アドレス
    nn::mem::StandardAllocator      m_StandardAllocator;    // 管理領域用アロケータ

};

/**
 * @brief   単純な確保解放のテストです
 */
TEST_P(NumberLineAllocatorTest, SimpleTest)
{
    EXPECT_TRUE(m_Allocator.AddRange(1, 0x00FFFFFFu));
    EXPECT_TRUE(m_Allocator.RemoveRange(2, 0x00FFFFFEu));
    EXPECT_TRUE(m_Allocator.RemoveRange(1, 2));
    EXPECT_TRUE(m_Allocator.RemoveRange(0x00FFFFFEu, 0x00FFFFFFu));
    EXPECT_TRUE(m_Allocator.AddRange(0, 0x01000000u));

    int first0, first1;
    EXPECT_TRUE(m_Allocator.Allocate(&first0, 0x100));
    EXPECT_TRUE(m_Allocator.Allocate(&first1, 0x2));
    m_Allocator.Free(first0);
    m_Allocator.Dump(0, 1);
    m_Allocator.Free(first1);
}

/**
 * @brief   1 ～ 4096 までのサイズの確保とユーザ固有パラメータの取得をテストします。
 */
TEST_P(NumberLineAllocatorTest, AllocTest)
{
    const int length = 4096;

    int numberLine[length];
    for (int i = 0; i < length; ++i)
    {
        numberLine[i] = i + 1;
    }

    m_Allocator.AddRange(0, 0x01000000U);

    for (int j = 0; j < 100; ++j)
    {
        std::random_shuffle(&numberLine[0], &numberLine[0] + length);

        int first[length];

        for (int i = 0; i < length; ++i)
        {
            // ランダムなサイズのアロケート
            int x;
            EXPECT_TRUE(m_Allocator.Allocate(&x, numberLine[i]));
            first[i] = x;
        }

        for (int i = 0; i < length; ++i)
        {
            // ユーザ固有パラメータの取得のみ確認
            void* dummy;
            EXPECT_TRUE(m_Allocator.FindUserData(&dummy, first[i]));
        }
        for (int i = 0; i < length; ++i)
        {
            m_Allocator.Free(first[i]);
        }
    }
}

/**
 * @brief   メモリいっぱいまで確保しても問題ないことを確認します。
 * @details 管理領域が非常に大きくなるので動作するプラットフォームを制限します。
 */
#if defined(NN_BUILD_CONFIG_ADDRESS_64)
TEST(NumberLineAllocatorFullAllocTest, FullAllocTest)
{
    NN_LOG("This test may take a little time (approximately few minutes).\n");

    nn::mem::NumberLineAllocator allocator;
    const int total = 0x01000000u;

    allocator.Initialize(ManagAlloc, nullptr, ManagFree, nullptr);
    allocator.AddRange(0, total);

    int length[] = {1, 256};
    for(int i : length)
    {
        NN_LOG("length: %d\n", i);
        int first;
        for (int j = 0; j < total; j += i)
        {
            ASSERT_TRUE(allocator.Allocate(&first, i)) << "j: " << j;
        }
        EXPECT_FALSE(allocator.Allocate(&first, 1));
        for (int j = 0; j < total; j += i)
        {
            allocator.Free(j);
        }
        EXPECT_TRUE(allocator.Allocate(&first, total));
        EXPECT_FALSE(allocator.Allocate(&first, 1));
        allocator.Free(first);
    }
    allocator.Finalize();
}
#endif

/**
 * @brief   指定のアライメントでメモリ確保を行います。
 */
TEST_P(NumberLineAllocatorTest, Alignment)
{
    const int Seed = 12345;
    const int MaxRandomAllocateSize = 1024;
    const int AllocateCount = 1000;
    const int MaxAlign = 64;    // 確認する最大アライメント
    int pAlignment[MaxAlign];

    ASSERT_GE(AllocateCount, MaxAlign);

    for(int i = 0; i < MaxAlign; ++i)
    {
        pAlignment[i] = i + 1;
    }

    m_Allocator.AddRange(0, 0x01000000u);

    std::mt19937 engine(Seed);
    std::uniform_int_distribution<uint32_t> distributionForAllocate(1, MaxRandomAllocateSize - 1);
    std::uniform_int_distribution<uint32_t> distributionForAlign(0, MaxAlign - 1);

    NN_LOG("Seed : %d\n", Seed);

    // ランダムなサイズ、同じアライメントで繰り返し確保
    int pLine[AllocateCount];
    for(int alignment = 2; alignment <= MaxAlign; ++alignment)
    {
        for(int j = 0; j < AllocateCount; ++j)
        {
            int allocateSize = distributionForAllocate(engine);
            ASSERT_TRUE(m_Allocator.Allocate(&pLine[j], allocateSize, alignment)) << "pLine[" << j << "]";
            ASSERT_EQ( pLine[j] % alignment , 0 ) << "size: " << allocateSize << " alignment: " << alignment << " pLine[j]: " << pLine[j];
        }
        for(int j = 0; j < AllocateCount; ++j)
        {
            m_Allocator.Free(pLine[j]);
        }
    }
    // ランダムなサイズ、ランダムなアライメントで繰り返し確保
    int count = 0;
    const int SetNum = 3;
    for(int i = 0; i < SetNum; ++i)
    {
        // アライメントの値をシャッフル
        for(int j = 0; j < MaxAlign - 1; ++j)
        {
            int swapNum = distributionForAlign(engine);
            std::swap(pAlignment[j], pAlignment[swapNum]);
        }
        for(int j = 0; j < MaxAlign; ++j)
        {
            int allocateSize = distributionForAllocate(engine);
            ASSERT_TRUE(m_Allocator.Allocate(&pLine[count], allocateSize, pAlignment[j]));
            ASSERT_EQ( pLine[count] % pAlignment[j], 0 ) << "size: " << allocateSize << " alignment: " << pAlignment[j];
            count++;
        }
    }

    for(int i = 0; i < count; ++i)
    {
        m_Allocator.Free(pLine[i]);
    }
}

void TestCallback(void* ptr)
{
    *(reinterpret_cast<int*>(ptr)) += 1;
}

/**
 * @brief   ユーザ定義のパラメータが問題なく割り当てられるか確認します。
 */
TEST_P(NumberLineAllocatorTest, UserParam)
{
    const int length = 256;
    int pNumber[length];
    int pUserData[length];
    m_Allocator.AddRange(0, 0x01000000u);

    for(int i = 0; i < length; ++i)
    {
        pUserData[i] = i;
        ASSERT_TRUE(m_Allocator.Allocate(&pNumber[i], i + 1, &pUserData[i], TestCallback)) << "i: " << i;
    }
    for(int i = 0; i < length; ++i)
    {
        int* ptr;
        m_Allocator.FindUserData(reinterpret_cast<void**>(&ptr), pNumber[i]);
        EXPECT_EQ(*ptr, i);
        EXPECT_EQ(reinterpret_cast<uintptr_t>(ptr), reinterpret_cast<uintptr_t>(&pUserData[i]));
        m_Allocator.Free(pNumber[i]);
        EXPECT_EQ(*ptr, i + 1);
    }
}

/**
 * @brief   数直線の確保可能／不可能の設定を確認します。
 */
TEST_P(NumberLineAllocatorTest, AddRemoveRange)
{
    int first;

    // AddRange していないので false
    ASSERT_FALSE(m_Allocator.Allocate(&first, 1));
    ASSERT_TRUE(m_Allocator.AddRange(0, 10));
    ASSERT_TRUE(m_Allocator.Allocate(&first, 10));
    ASSERT_FALSE(m_Allocator.Allocate(&first, 1));
    // 確保済みの領域は Remove できない
    ASSERT_FALSE(m_Allocator.RemoveRange(0, 10));
    m_Allocator.Free(first);
    ASSERT_TRUE(m_Allocator.RemoveRange(0, 10));
    ASSERT_TRUE(m_Allocator.AddRange(100, 101));
    ASSERT_TRUE(m_Allocator.Allocate(&first, 1));
    ASSERT_EQ(first, 100);
}

/**
 * @brief   以前の確保解放状況が確保解放に影響を与えてないことの確認
 */
TEST_P(NumberLineAllocatorTest, Fragmentation)
{
    int first;
    ASSERT_TRUE(m_Allocator.AddRange(0, 20));
    ASSERT_TRUE(m_Allocator.Allocate(&first, 15));
    m_Allocator.Free(first);
    for(int i = 0; i < 20; ++i)
    {
        ASSERT_TRUE(m_Allocator.Allocate(&first, 1)) << "i: " << i;
    }
    ASSERT_FALSE(m_Allocator.Allocate(&first, 1));
    ASSERT_FALSE(m_Allocator.Allocate(&first, 1));
}

/**
 * @brief   確保できない大きさを指定した際の挙動
 */
TEST_P(NumberLineAllocatorTest, HugeAllocate)
{
    int first;
    ASSERT_TRUE(m_Allocator.AddRange(0, 20));
    ASSERT_FALSE(m_Allocator.Allocate(&first, 50));
    for(int i = 0; i < 20; ++i)
    {
        ASSERT_TRUE(m_Allocator.Allocate(&first, 1)) << "i: " << i;
    }
    ASSERT_FALSE(m_Allocator.Allocate(&first, 1));
    ASSERT_FALSE(m_Allocator.Allocate(&first, 1));
}

namespace {

// NumberLineAllocatorTest.RandomAllocate で使う単方向リスト
struct List
{
    int first;
    List* pPrev;
    List* pNext;
};

}

/**
 * @brief   ランダムに確保解放を繰り返しても影響がない
 */
TEST_P(NumberLineAllocatorTest, RandomAllocate)
{
    const int Seed = 12345;
    const int MaxAllocateSum = 4096;
    std::mt19937 engine(Seed);
    std::uniform_int_distribution<int> distributionForSize(1, 64);      // 確保するサイズの範囲
    std::uniform_int_distribution<int> distributionForOperation(0, 2);  // Allocate/Free のどちらをするか

    int allocateCount = 0;  // 確保した数
    List* pHead = reinterpret_cast<List*>(std::malloc(sizeof(List)));
    // 1 つ目のノードを追加
    List* pCurrent = reinterpret_cast<List*>(std::malloc(sizeof(List)));
    ASSERT_TRUE(m_Allocator.AddRange(0, MaxAllocateSum));
    ASSERT_TRUE(m_Allocator.Allocate(&pCurrent->first, 1));
    pHead->pNext = pCurrent;
    pHead->pPrev = nullptr;
    pHead->first = -1;
    pCurrent->pNext = nullptr;
    pCurrent->pPrev = pHead;

    // ランダムに確保・解放を繰り返しながら確保領域を少しずつ増やしていく
    for(;;)
    {
        int operation = distributionForOperation(engine);
        if(operation == 0 && allocateCount > 0)
        {
            // 0 なら Free する
            std::uniform_int_distribution<int> distributionForFree(1, allocateCount);
            List* pNode = pHead;
            int index = distributionForFree(engine);
            while(index > 0)
            {
                pNode = pNode->pNext;
                index--;
            }
            m_Allocator.Free(pNode->first);
            allocateCount--;
            pNode->pNext->pPrev = pNode->pPrev;
            pNode->pPrev->pNext = pNode->pNext;
            std::free(pNode);
        }
        else
        {
            // 1 か 2 なら Allocate する
            List* pNode = reinterpret_cast<List*>(std::malloc(sizeof(List)));
            int size = distributionForSize(engine);
            bool result = m_Allocator.Allocate(&pNode->first, size);
            if(!result)
            {
                // Allocate 失敗したら確保サイズを 1 にして再度チャレンジ
                result = m_Allocator.Allocate(&pNode->first, 1);
                if(!result)
                {
                    std::free(pNode);
                    break;
                }
                else
                {
                    ASSERT_EQ(m_Allocator.GetSizeOf(pNode->first), 1);
                }
            }
            else
            {
                ASSERT_EQ(m_Allocator.GetSizeOf(pNode->first), size);
            }
            allocateCount++;
            pCurrent->pNext = pNode;
            pNode->pPrev = pCurrent;
            pCurrent = pNode;
            pCurrent->pNext = nullptr;

        }
    }
    // すべて解放
    List* pNode = pHead->pNext;
    while(pNode != nullptr)
    {
        m_Allocator.Free(pNode->first);
        List* pNodeNext = pNode->pNext;
        std::free(pNode);
        pNode = nullptr;
        pNode = pNodeNext;
    }
    std::free(pHead);

    // 確保できる領域が減っていないこと(アロケータ内部がおかしくなっていないこと)を確認
    {
        int first;
        for(int i = 0; i < MaxAllocateSum; ++i)
        {
            ASSERT_TRUE(m_Allocator.Allocate(&first, 1));
        }
        ASSERT_FALSE(m_Allocator.Allocate(&first, 1));
    }
}

/**
 * @brief   サイズ取得関数が正しい値を返す
 */
TEST_P(NumberLineAllocatorTest, GetSize)
{
    EXPECT_TRUE(m_Allocator.AddRange(0, 0x00FFFFFFu));
    EXPECT_EQ(m_Allocator.GetAllocatableSize(), 0x00FFFFFFu);
    EXPECT_EQ(m_Allocator.GetTotalFreeSize(), 0x00FFFFFFu);
    EXPECT_TRUE(m_Allocator.RemoveRange(1, 0x00FFFFFFu));
    EXPECT_EQ(m_Allocator.GetAllocatableSize(), 1);
    EXPECT_EQ(m_Allocator.GetTotalFreeSize(), 1);
    EXPECT_TRUE(m_Allocator.AddRange(10, 15));
    EXPECT_EQ(m_Allocator.GetAllocatableSize(), 5);
    EXPECT_EQ(m_Allocator.GetTotalFreeSize(), 6);
    EXPECT_TRUE(m_Allocator.RemoveRange(10, 15));
    EXPECT_TRUE(m_Allocator.AddRange(1, 0x00FFFFFFu));

    const int allocatedCount = 1000;
    int allocatedSize = 0;
    int pAllocated[allocatedCount];
    for (int i = 1; i < allocatedCount; ++i)
    {
        EXPECT_TRUE(m_Allocator.Allocate(&pAllocated[i], i));
        allocatedSize += i;
        EXPECT_EQ(m_Allocator.GetAllocatableSize(), 0x00FFFFFFu - allocatedSize);
        EXPECT_EQ(m_Allocator.GetTotalFreeSize(), 0x00FFFFFFu - allocatedSize);
    }
    for (int i = allocatedCount - 1; i > 0; --i)
    {
        m_Allocator.Free(pAllocated[i]);
        allocatedSize -= i;
        EXPECT_EQ(m_Allocator.GetTotalFreeSize(), 0x00FFFFFFu - allocatedSize);
    }
}

/**
* @brief   Lerge 領域が枯渇した場合の挙動
*/
TEST_P(NumberLineAllocatorTest, RunOutLergeMemory)
{
    EXPECT_TRUE(m_Allocator.AddRange(0, 0x01000000u));

    int first;
    ASSERT_TRUE(m_Allocator.Allocate(&first, 0x01000000u));

    m_Allocator.Dump(0, 1);

    m_Allocator.Free(first);
}

INSTANTIATE_TEST_CASE_P(ManagAllocator,
                        NumberLineAllocatorTest,
                        testing::Values(NumberLineAllocatorTestParam_Malloc,
                                        NumberLineAllocatorTestParam_SA));

/**
 * @brief   管理領域が不足した場合の挙動
 * @details 管理領域のうち、 TrieNode 構造体の確保で失敗した場合と
 *          Span 構造体の確保で失敗した場合の両方を確認する。
 *          64 bit の場合 managSize が 2MB のとき長さ 1 の領域を連続で確保すると、
 *          最終的に TrieNode 構造体の確保に失敗する。
 *          managSize が 8MB のときは Span 構造体の確保に失敗する。
 */
TEST(NumberLineAllocatorManagTest, RunOutManagArea)
{
    const size_t pManagSize[2] = { 2 * 1024 * 1024, 8 * 1024 * 1024 };

    for (auto managSize : pManagSize)
    {
        nn::mem::NumberLineAllocator allocator;
        nn::mem::StandardAllocator managAllocator;
        void* ptr = std::malloc(managSize);

        managAllocator.Initialize(ptr, managSize);
        allocator.Initialize(ManagAllocWithSa, &managAllocator, ManagFreeWithSa, &managAllocator);

        EXPECT_TRUE(allocator.AddRange(0, 0x01000000));

        // 管理領域を使い果たすまで確保を繰り返す
        bool result = true;
        int count = 0;
        int first;
        while (result)
        {
            result = allocator.Allocate(&first, 1);
            if (result)
            {
                count++;
            }
        }

        EXPECT_EQ(0x01000000 - count, allocator.GetAllocatableSize());
        EXPECT_EQ(0x01000000 - count, allocator.GetTotalFreeSize());

        // 領域を解放したら、管理領域も解放され、確保できるようになる
        const int freeCount = 10;
        // 長さ 1 の確保を連続で行っただけなので以下の for 文で解放可能
        for (int i = 0; i < freeCount; ++i)
        {
            allocator.Free(i);
        }
        for (int i = 0; i < freeCount; ++i)
        {
            EXPECT_TRUE(allocator.Allocate(&first, 1));
        }
        for (int i = 0; i < count; ++i)
        {
            allocator.Free(i);
        }

        EXPECT_EQ(0x01000000, allocator.GetTotalFreeSize());

        EXPECT_TRUE(allocator.Allocate(&first, 0x01000000));
        allocator.Free(first);

        allocator.Finalize();
        managAllocator.Finalize();

        std::free(ptr);
    }
}

/**
 * @brief   以下の要素を複合的に入れたランダムな確保・解放
 * @details
 *          - アライメントを指定する
 *          - 管理領域が容易に枯渇する
 */
TEST(NumberLineAllocatorManagTest, RandomAllocate)
{
    const size_t managSize = 2 * 1024 * 1024;
    const int MaxRandomAllocateSize = 128;

    nn::mem::NumberLineAllocator allocator;
    nn::mem::StandardAllocator managAllocator;

    void* ptr = std::malloc(managSize);

    managAllocator.Initialize(ptr, managSize);
    allocator.Initialize(ManagAllocWithSa, &managAllocator, ManagFreeWithSa, &managAllocator);

    EXPECT_TRUE(allocator.AddRange(0, 0x01000000));

    const int Seed = 12345;
    const int MaxAlignment = 16;
    std::mt19937 engine(Seed);
    std::uniform_int_distribution<int> distributionForSize(1, MaxRandomAllocateSize);       // 確保するサイズの範囲
    std::uniform_int_distribution<int> distributionForOperation(0, 2);                      // Allocate/Free のどちらをするか
    std::uniform_int_distribution<int> distributionForAlignment(1, MaxAlignment);           // アライメントの範囲

    int allocateCount = 0;  // 確保した数
    List* pHead = reinterpret_cast<List*>(std::malloc(sizeof(List)));
    // 1 つ目のノードを追加
    List* pCurrent = reinterpret_cast<List*>(std::malloc(sizeof(List)));
    ASSERT_TRUE(allocator.Allocate(&pCurrent->first, 1));
    pHead->pNext = pCurrent;
    pHead->pPrev = nullptr;
    pHead->first = -1;
    pCurrent->pNext = nullptr;
    pCurrent->pPrev = pHead;

    auto FreeFromList = [&]()
    {
        std::uniform_int_distribution<int> distributionForFree(1, allocateCount);
        List* pNode = pHead;
        int index = distributionForFree(engine);
        while (index > 0)
        {
            pNode = pNode->pNext;
            index--;
        }
        allocator.Free(pNode->first);
        allocateCount--;
        pNode->pNext->pPrev = pNode->pPrev;
        pNode->pPrev->pNext = pNode->pNext;
        std::free(pNode);
    };

    const int allocFailBreak = 1024;
    int allocFailCount = 0;
    // ランダムに確保・解放を繰り返しながら確保領域を少しずつ増やしていく
    while (allocFailCount < allocFailBreak)
    {
        int operation = distributionForOperation(engine);
        if (operation == 0 && allocateCount > 0)
        {
            // 0 なら Free する
            FreeFromList();
        }
        else
        {
            // 1 か 2 なら Allocate する
            List* pNode = reinterpret_cast<List*>(std::malloc(sizeof(List)));
            int size = distributionForSize(engine);
            int alignment = distributionForAlignment(engine);
            bool result = allocator.Allocate(&pNode->first, size, alignment);
            if (!result)
            {
                // Allocate 失敗したらアライメントなしで確保できるかやってみる
                result = allocator.Allocate(&pNode->first, size);
                if (!result)
                {
                    // それでもだめならいくつかの領域を解放する
                    std::free(pNode);
                    allocFailCount++;

                    for (int i = 0; i < 64; ++i)
                    {
                        FreeFromList();
                    }
                    continue;
                }
            }
            else
            {
                EXPECT_TRUE(pNode->first % alignment == 0);
            }
            ASSERT_EQ(allocator.GetSizeOf(pNode->first), size);
            allocateCount++;
            pCurrent->pNext = pNode;
            pNode->pPrev = pCurrent;
            pCurrent = pNode;
            pCurrent->pNext = nullptr;
        }
    }
    // すべて解放
    List* pNode = pHead->pNext;
    while (pNode != nullptr)
    {
        allocator.Free(pNode->first);
        List* pNodeNext = pNode->pNext;
        std::free(pNode);
        pNode = nullptr;
        pNode = pNodeNext;
    }
    std::free(pHead);

    // 確保できる領域が減っていないこと(アロケータ内部がおかしくなっていないこと)を確認
    {
        int first;
        EXPECT_EQ(0x01000000, allocator.GetTotalFreeSize());

        EXPECT_TRUE(allocator.Allocate(&first, 0x01000000));
        allocator.Free(first);
    }

    allocator.Finalize();
    managAllocator.Finalize();

    std::free(ptr);
}
