﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/teamcity/testTeamcity_Logger.h>

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>

#include <nn/os.h>
#include <nn/mem.h>

#define ASSERT_NULL(p) ASSERT_TRUE(p == nullptr)
#define ASSERT_NOT_NULL(p) ASSERT_TRUE(p != nullptr)

namespace {

/**
 * @brief   テストに与えるパラメータです。
 */
enum StandardAllocatorTestParam
{
    StandardAllocatorTestParam_DisableThreadCache = 0,  //!< スレッドキャッシュを利用しない
    StandardAllocatorTestParam_EnableThreadCache        //!< スレッドキャッシュを利用する
};

// テストに使うメモリプールのサイズ
// Windows の場合は必ず確保できるヒープサイズの予測が困難なので、 512MB に絞る
size_t g_HeapSize = 512 * 1024 * 1024;

/**
 * @brief   テストで利用するテストフィクスチャです。
 */
class StandardAllocatorHugeAllocTest : public ::testing::TestWithParam<int>
{
protected:

    /**
     * @brief   テスト開始時に毎回呼び出される関数です。
     */
    virtual void SetUp()
    {
#if defined(NN_BUILD_CONFIG_OS_HORIZON) && defined(NN_BUILD_CONFIG_ADDRESS_64)
        g_HeapSize = 3 * 1024 * 1024 * 1024ul;
#endif
#if defined(NN_BUILD_CONFIG_OS_WIN)
        uintptr_t ptr;
        nn::os::AllocateMemoryBlock(&ptr, g_HeapSize);
        m_HeapStartAddr = reinterpret_cast<void*>(ptr);
#else
        m_HeapStartAddr = std::malloc(g_HeapSize);
#endif
        m_Param = GetParam();
        if (m_Param == StandardAllocatorTestParam_DisableThreadCache)
        {
            m_Allocator.Initialize(m_HeapStartAddr, g_HeapSize);
        }
        else
        {
            m_Allocator.Initialize(m_HeapStartAddr, g_HeapSize, true);
        }
    }

    /**
     * @brief   テスト終了時に毎回呼び出される関数です。
     */
    virtual void TearDown()
    {
        if (m_HeapStartAddr)
        {
            m_Allocator.Finalize();
#if defined(NN_BUILD_CONFIG_OS_WIN)
            nn::os::FreeMemoryBlock(reinterpret_cast<uintptr_t>(m_HeapStartAddr), g_HeapSize);
#else
            std::free(m_HeapStartAddr);
#endif
        }
    }

protected:
    nn::mem::StandardAllocator m_Allocator;
    void* m_HeapStartAddr;                  // グローバルヒープの開始アドレス
    int m_Param;                            // テストパラメータ
};

/**
 * @brief   ポインタを保持する単方向リストです。
 */
struct SinglyLinkedList
{
    void* ptr;
    SinglyLinkedList* next;
};

}   // unnamed namespace

// ======
// テスト
// ======
/**
 * @brief   確保可能な限界サイズの確保に成功するか確認します。
 */
TEST_P(StandardAllocatorHugeAllocTest, Alloc)
{
    void* ptr = m_Allocator.Allocate(m_Allocator.GetAllocatableSize());
    ASSERT_NOT_NULL(ptr);
    m_Allocator.Free(ptr);

    ptr = m_Allocator.Allocate(m_Allocator.GetAllocatableSize() + 1);
    ASSERT_NULL(ptr);
    m_Allocator.Free(ptr);
}

/**
 * @brief   アライメントを取った限界サイズの確保に成功するか確認します。
 */
TEST_P(StandardAllocatorHugeAllocTest, AlignAlloc)
{
    const int AlignCount = 16;
    // 128K アライメントまで確認
    int pAlignment[AlignCount] = {4, 8, 16, 32, 64, 128, 256, 512, 1024, 2048, 4096, 8192, 16384, 32768, 65536, 131072};

    for(int i = 0; i < AlignCount; ++i)
    {
        void* ptr = m_Allocator.Allocate(m_Allocator.GetAllocatableSize() - pAlignment[i], pAlignment[i]);
        ASSERT_NOT_NULL(ptr);
        ASSERT_EQ((reinterpret_cast<uintptr_t>(ptr) & (pAlignment[i] - 1)), static_cast<uintptr_t>(0));
        m_Allocator.Free(ptr);
    }
}

/**
 * @brief   再確保で限界サイズの確保に成功するか確認します。
 */
TEST_P(StandardAllocatorHugeAllocTest, ReAlloc)
{
    const size_t SmallMemory = NN_ALIGNOF(std::max_align_t);
    const size_t PageMemory = 4096;
    const size_t LargeMemory = 2 * 1024 * 1024;

    // 与えられたサイズから限界サイズに再確保します
    auto reallocateGain = [&](size_t size)
    {
        void* ptr = m_Allocator.Allocate(size);
        ASSERT_NOT_NULL(ptr);
        void* tmp = ptr;
        size_t allocatableSize = m_Allocator.GetAllocatableSize();
        ptr = m_Allocator.Reallocate(ptr, allocatableSize + 1);
        if(size == PageMemory)
        {
            // 後方へのメモリ伸長が可能な場合は確保が成功する
            ASSERT_NOT_NULL(ptr);
            ASSERT_EQ(ptr, tmp);
        }
        else
        {
            ASSERT_NULL(ptr);
        }
        ptr = m_Allocator.Reallocate(ptr, allocatableSize);
        ASSERT_NOT_NULL(ptr);
        EXPECT_GE(m_Allocator.GetSizeOf(ptr), allocatableSize);
        m_Allocator.Free(ptr);
    };
    reallocateGain(SmallMemory);
    reallocateGain(PageMemory);
    reallocateGain(LargeMemory);
}

/**
 * @brief   SmallMemory と LargeMemory の両方が大きなサイズのメモリ領域でも正常に確保しきれるかどうか確認します。
 */
TEST_P(StandardAllocatorHugeAllocTest, UsedUp)
{
    const size_t LargeMemory = 4 * 1024 * 1024;
    const int Set = 3;

    SinglyLinkedList* head;
    SinglyLinkedList* node;

    size_t totalFreeSize = 0;
    int allocateCount = 0;

    for (int i = 0; i < Set; ++i)
    {
        NN_LOG("Set %d\n", i);
        int count = 0;  // 確保した回数

        head = reinterpret_cast<SinglyLinkedList*>(m_Allocator.Allocate(sizeof(SinglyLinkedList)));
        ASSERT_NOT_NULL(head);
        count++;
        head->ptr = m_Allocator.Allocate(LargeMemory);
        if (head->ptr == nullptr)
        {
            m_Allocator.Dump();
        }
        ASSERT_NOT_NULL(head->ptr);
        count++;
        node = head;
        // 確保できなくなるまで確保
        for (;;)
        {
            node->next = NULL;
            SinglyLinkedList* ptr = reinterpret_cast<SinglyLinkedList*>(m_Allocator.Allocate(sizeof(SinglyLinkedList)));
            if (ptr == NULL)
            {
                break;
            }
            count++;
            ptr->ptr = m_Allocator.Allocate(LargeMemory);
            if (ptr->ptr != NULL)
            {
                count++;
            }
            node->next = ptr;
            node = node->next;
        }
#if defined(NN_BUILD_CONFIG_OS_WIN)
        if (m_Param == StandardAllocatorTestParam_DisableThreadCache)
        {
            // Windows の GoogleTest だと span に設定される cpu_id (nn::os::GetCurrentCoreNumber で取得)が変わり
            // 以前スレッドキャッシュ用として使っていた span が違う cpu_id 扱いで
            // 確保できなくなることがある
            EXPECT_EQ(m_Allocator.GetTotalFreeSize(), 0U);
        }
#else
        EXPECT_EQ(m_Allocator.GetTotalFreeSize(), 0U);
#endif
        EXPECT_EQ(m_Allocator.GetAllocatableSize(), 0U);
        if(i == 0)
        {
            allocateCount = count;
        }
        // 全て解放
        node = head;
        while(node != NULL)
        {
            if(node->ptr != NULL)
            {
                m_Allocator.Free(node->ptr);
            }
            SinglyLinkedList* ptr = node;
            node = node->next;
            m_Allocator.Free(ptr);
        }

        // 管理領域は全体の 1% 未満に収まっている
        EXPECT_GE(m_Allocator.GetTotalFreeSize(), g_HeapSize - g_HeapSize / 100 );

        if(i == 0)
        {
            totalFreeSize = m_Allocator.GetTotalFreeSize();
        }
        else
        {
            // アロケータの内部状況によって 4096 バイトの管理領域の増減があるが、 Set ごとにそれ以上のサイズ変化はないはず
            const size_t pageSize = 4096;
            EXPECT_TRUE(totalFreeSize <= m_Allocator.GetTotalFreeSize() &&
                        totalFreeSize + pageSize >= m_Allocator.GetTotalFreeSize());
#if defined(NN_BUILD_CONFIG_OS_WIN)
            if (m_Param == StandardAllocatorTestParam_DisableThreadCache)
            {
                // Windows の GoogleTest だと span に設定される cpu_id (nn::os::GetCurrentCoreNumber で取得)が変わり
                // 以前スレッドキャッシュ用として使っていた span が違う cpu_id 扱いで
                // 確保できなくなることがある
                EXPECT_TRUE(allocateCount <= count &&
                    allocateCount + static_cast<int>((pageSize / sizeof(SinglyLinkedList))) >= count);
            }
#else
            EXPECT_TRUE(allocateCount <= count &&
                allocateCount + static_cast<int>((pageSize / sizeof(SinglyLinkedList))) >= count);
#endif
        }
    }
}

#if defined(NN_BUILD_CONFIG_OS_WIN)
extern "C" void nninitStartup()
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::SetMemoryHeapSize(g_HeapSize));
}
#endif

INSTANTIATE_TEST_CASE_P(SwitchThreadCache,
                        StandardAllocatorHugeAllocTest,
                        testing::Values(StandardAllocatorTestParam_DisableThreadCache,
                                        StandardAllocatorTestParam_EnableThreadCache));
