﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <string>
#include <new>
#include <memory>
#include <random>
#include <vector>
#include <algorithm>

#include <nnt/nntest.h>

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Log.h>
#include <nn/diag.h>
#include <nn/os.h>
#include <nn/mem.h>
#include <nn/init.h>

#include <nn/nlib/UniquePtr.h>
#include <nn/nlib/MemoryInputStream.h>
#include <nn/nlib/exi/XmlStreamReader.h>

#define ASSERT_NOT_NULL(p) ASSERT_TRUE(p != nullptr)

extern "C" void nninitStartup()
{
    // malloc, new 用に 2GB
    nn::init::InitializeAllocator(nullptr, 2ull * 1024 * 1024 * 1024);
}

namespace {

    // NX では ExiChar は UTF-8
    // Windows だと wchar_t になるので NLIB_EXI_UTF8 等を利用する必要がある
    NN_STATIC_ASSERT(sizeof(nntest::nlib::exi::ExiChar) == sizeof(char));

    /**
     * @brief   テストに与えるパラメータです。
     */
    enum StandardAllocatorTestParam
    {
        StandardAllocatorTestParam_VammDisabled = 0, //!< 仮想アドレスメモリ機能を利用しない
        StandardAllocatorTestParam_VammEnabled     //!< 仮想アドレスメモリ機能を利用する
    };


    class StandardAllocatorDumpTest : public ::testing::TestWithParam<int>
    {
    public:

        /**
         * @brief   ログ出力される文字列を argument にも格納します。
         * @details ログオブザーバ用コールバック関数
         */
        static void GetString(const nn::diag::LogMetaData& metaInfo,
            const nn::diag::LogBody& logBody, void* argument)
        {
            std::string* pStr = reinterpret_cast<std::string*>(argument);
            *pStr += logBody.message;
        }

    protected:

        StandardAllocatorDumpTest() : m_ExiAllocatorBufSize(2 * 1024 * 1024),
            m_SaAllocatorBufSize(512 * 1024 * 1024),
            m_IsXmlParseReady(false), m_IsAnswerPrepared(false), m_IsRegistered(false) {}
        ~StandardAllocatorDumpTest() {}

        /**
         * @brief   テスト開始時に毎回呼び出される関数です。
         */
        virtual void SetUp()
        {
            m_Param = GetParam();

            // XML パーサ用アロケータの初期化
            m_pExiAllocatorBuf.reset(new char[m_ExiAllocatorBufSize]);
            ASSERT_NOT_NULL(m_pExiAllocatorBuf.get());
            ASSERT_TRUE(m_ExiAllocator.Init(m_pExiAllocatorBuf.get(), m_ExiAllocatorBufSize) == 0);

            // XML パーサの初期化オプションの設定
            m_Settings.processor = nlib_ns::exi::XmlProcessor::XML_PROCESSOR_TEXT;

            // ログオブザーバホルダの初期化
            nn::diag::InitializeLogObserverHolder(&m_Holder,
                StandardAllocatorDumpTest::GetString, &m_LogStr);

            // StandardAllocator の初期化
            if (m_Param == StandardAllocatorTestParam_VammDisabled)
            {
                m_SaAllocatorBuf = std::malloc(m_SaAllocatorBufSize);
                ASSERT_NOT_NULL(m_SaAllocatorBuf);
                m_SaAllocator.Initialize(m_SaAllocatorBuf, m_SaAllocatorBufSize);
            }
            else
            {
                m_SaAllocator.Initialize(nullptr, 4ull * 1024 * 1024 * 1024);
            }
        }

        /**
         * @brief   テスト終了時に毎回呼び出される関数です。
         */
        virtual void tearDown()
        {
            EndLogAcquisition();
            DestroyAnswer();
            FinalizeXmlParser();
            m_ExiAllocator.Finalize();
            m_SaAllocator.Finalize();
            std::free(m_SaAllocatorBuf);    // unique_ptr にすると Finalize 前に解放されてしまう
        }

        /**
         * @brief   ログ文字列の取得を開始します。
         * @details ログオブザーバを登録します。
         */
        void StartLogAcquisition()
        {
            if (m_IsRegistered)
            {
                return;
            }
            nn::diag::RegisterLogObserver(&m_Holder);
            m_IsRegistered = true;
        }

        /**
         * @brief   ログ文字列の取得を終了します。
         * @details 登録済みのログオブザーバを解除します。
         */
        void EndLogAcquisition()
        {
            if (!m_IsRegistered)
            {
                return;
            }
            nn::diag::UnregisterLogObserver(&m_Holder);
            m_IsRegistered = false;
        }

        /**
         * @brief   XML パーサの利用準備をします。
         * @details 取得したログ(XML)をパースして各エレメントを取得できる状態にします。
         *          StartLogAcquisition() でログを取得した後に呼び出してください。
         */
        void InitializeXmlParser()
        {
            if (m_IsXmlParseReady)
            {
                return;
            }

            errno_t e = m_Stream.Init(m_LogStr.c_str(), m_LogStr.length());
            ASSERT_FALSE(nlib_is_error(e));

            // XML オブジェクトの構築
            m_pReader.reset(nlib_ns::exi::XmlStreamReader::Create(&m_Stream, m_Settings,
                m_ExiAllocator));
            ASSERT_NOT_NULL(m_pReader.get());

            m_IsXmlParseReady = true;
        }

        /**
         * @brief   XML パーサの終了処理
         */
        void FinalizeXmlParser()
        {
            if (!m_IsXmlParseReady)
            {
                return;
            }
            ASSERT_FALSE(nlib_is_error(m_Stream.Close()));

            m_IsXmlParseReady = false;
        }

        /**
         * @brief   StandardAllocator::Dump() の構造的に正しい文字列を作成します。
         * @details nlib_ns::exi::XmlStreamReader で読み取れる形にします。
         */
        void PrepareAnswer()
        {
            if (m_IsAnswerPrepared)
            {
                return;
            }
            m_AnswerStr = "<heapinfo>";
            m_AnswerStr += "<total_pages>0</total_pages>";   // 構造のテストなので値は考慮しない
            m_AnswerStr += "<pagesize>0</pagesize>";
            m_AnswerStr += "<max_allocatable_size>0</max_allocatable_size>";
            m_AnswerStr += "<free_memory>0</free_memory>";
            m_AnswerStr += "<system_memory>0</system_memory>";
            m_AnswerStr += "<allocated_memory>0</allocated_memory>";
            if (m_Param == StandardAllocatorTestParam_VammDisabled)
            {
                m_AnswerStr += "<page_summary>+</page_summary>";
            }
            m_AnswerStr += "<spans><span start='252c2800000' numPages='145' cls='0' status='in_use_sys'/></spans>";
            m_AnswerStr += "<allocs><alloc ptr='252c2893000' size='8'/></allocs>";
            m_AnswerStr += "</heapinfo>";

            errno_t e = m_StreamForAnswer.Init(m_AnswerStr.c_str(), m_AnswerStr.length());
            ASSERT_FALSE(nlib_is_error(e));

            // XML オブジェクトの構築
            m_pReaderForAnswer.reset(nlib_ns::exi::XmlStreamReader::Create(
                &m_StreamForAnswer, m_Settings, m_ExiAllocator));
            ASSERT_NOT_NULL(m_pReaderForAnswer.get());

            m_IsAnswerPrepared = true;
        }

        /**
         * @brief   正解データの破棄
         */
        void DestroyAnswer()
        {
            if (!m_IsAnswerPrepared)
            {
                return;
            }
            ASSERT_FALSE(nlib_is_error(m_StreamForAnswer.Close()));

            m_IsAnswerPrepared = false;
        }

        /**
         * @brief   constants の出力
         * @details デバッグ用出力です。
         */
        void DispConstants(nlib_ns::exi::XmlStreamReader::XmlStreamConstants constants,
            nlib_ns::exi::XmlStreamReader* pReader)
        {
            switch (constants)
            {
            case nlib_ns::exi::XmlStreamReader::kNone:
                NN_LOG("kNone\n");
                break;
            case nlib_ns::exi::XmlStreamReader::kStartElement:
                NN_LOG("kStartElement: %s\n", pReader->GetLocalName());
                break;
            case nlib_ns::exi::XmlStreamReader::kEndElement:
                NN_LOG("kEndElement:   %s\n", pReader->GetLocalName());
                break;
            case nlib_ns::exi::XmlStreamReader::kProcessingInstruction:
                NN_LOG("kProcessingInstruction\n");
                break;
            case nlib_ns::exi::XmlStreamReader::kCharacters:
                NN_LOG("kCharacters: %s\n", pReader->GetText());
                break;
            case nlib_ns::exi::XmlStreamReader::kComment:
                NN_LOG("kComment\n");
                break;
            case nlib_ns::exi::XmlStreamReader::kStartDocument:
                NN_LOG("kStartDocument\n");
                break;
            case nlib_ns::exi::XmlStreamReader::kEndDocument:
                NN_LOG("kEndDocument\n");
                break;
            case nlib_ns::exi::XmlStreamReader::kCdata:
                NN_LOG("kCdata\n");
                break;
            default:
                break;
            }
        }

        /**
         * @brief       XML の構造が StandardAllocator::Dump() として正しいか確認します。
         * @param[in]   isVirtualMemoryEnabled  仮想アドレスメモリ機能を有効にしているなら true
         * @pre         m_IsXmlParseReady == true
         */
        void CheckXmlSchema()
        {
            ASSERT_TRUE(m_IsXmlParseReady);

            // 正解データの準備
            PrepareAnswer();

            // 正解データとログを同時に辿り、構造が一致しているか確認する
            while (m_pReader->HasNext() || m_pReaderForAnswer->HasNext())
            {
                nlib_ns::exi::XmlStreamReader::XmlStreamConstants log = m_pReader->Next();
                nlib_ns::exi::XmlStreamReader::XmlStreamConstants answer = m_pReaderForAnswer->Next();

                ASSERT_EQ(log, answer);

                // pagesize は 4096 のはず
                if (log == nlib_ns::exi::XmlStreamReader::kStartElement &&
                    std::string(m_pReader->GetLocalName()) == "pagesize")
                {
                    log = m_pReader->Next();
                    answer = m_pReaderForAnswer->Next();
                    ASSERT_EQ(log, nlib_ns::exi::XmlStreamReader::kCharacters);
                    ASSERT_EQ(answer, nlib_ns::exi::XmlStreamReader::kCharacters);
                    ASSERT_EQ(std::string(m_pReader->GetText()), "4096");
                }

                // page_summary は以下の記号のみを使って表される
                // _.:-=+*#%@
                if (log == nlib_ns::exi::XmlStreamReader::kStartElement &&
                    std::string(m_pReader->GetLocalName()) == "page_summary")
                {
                    // 仮想アドレスメモリ機能が有効の時は表示されない
                    ASSERT_EQ(m_Param, StandardAllocatorTestParam_VammDisabled);

                    log = m_pReader->Next();
                    answer = m_pReaderForAnswer->Next();
                    ASSERT_EQ(log, nlib_ns::exi::XmlStreamReader::kCharacters);
                    ASSERT_EQ(answer, nlib_ns::exi::XmlStreamReader::kCharacters);
                    std::string summary = std::string(m_pReader->GetText());

                    const int validSize = 10;
                    const char valid[validSize] = {
                        '_', '.', ':', '-', '=', '+', '*', '#', '%', '@'
                    };

                    for (size_t i = 0; i < summary.length(); ++i)
                    {
                        bool isValid = false;
                        for (int j = 0; j < validSize; ++j)
                        {
                            if (valid[j] == summary[i])
                            {
                                isValid = true;
                                break;
                            }
                        }
                        ASSERT_TRUE(isValid);
                    }
                }

                // spans と allocs の構造チェック
                auto SpanOrAllocCheck =
                    [&](std::string parentTagName, std::string childTagName)
                {
                    if (log == nlib_ns::exi::XmlStreamReader::kStartElement &&
                        std::string(m_pReader->GetLocalName()) == parentTagName)
                    {
                        log = m_pReader->Next();
                        answer = m_pReaderForAnswer->Next();
                        if (log != nlib_ns::exi::XmlStreamReader::kEndDocument)
                        {
                            ASSERT_EQ(log, nlib_ns::exi::XmlStreamReader::kStartElement);
                            ASSERT_EQ(answer, nlib_ns::exi::XmlStreamReader::kStartElement);
                            ASSERT_EQ(std::string(m_pReader->GetLocalName()), childTagName);

                            while (log != nlib_ns::exi::XmlStreamReader::kEndElement ||
                                std::string(m_pReader->GetLocalName()) != parentTagName)
                            {
                                ASSERT_TRUE(log == nlib_ns::exi::XmlStreamReader::kStartElement ||
                                    log == nlib_ns::exi::XmlStreamReader::kEndElement);
                                if (log == nlib_ns::exi::XmlStreamReader::kStartElement)
                                {
                                    // 属性が全てあるか確認
                                    for (size_t i = 0; i < m_pReader->GetAttributeCount(); ++i)
                                    {
                                        if (i < 4)
                                        {
                                            ASSERT_EQ(std::string(m_pReader->GetAttributeLocalName(i)),
                                                std::string(m_pReaderForAnswer->GetAttributeLocalName(i)));
                                        }
                                        // span には 5 つめ以降があり得る
                                        // span の 5 つめ以降は　objects_count か、 color と name が入っている
                                        else if(i == 4)
                                        {
                                            ASSERT_TRUE( (std::string(m_pReader->GetAttributeLocalName(i)) == "objects_count") ||
                                                (std::string(m_pReader->GetAttributeLocalName(i)) == "color"));
                                        }
                                        else if (i == 5)
                                        {
                                            ASSERT_STREQ(m_pReader->GetAttributeLocalName(i), "name");
                                        }
                                        else
                                        {
                                            ASSERT_TRUE(0);
                                        }
                                    }
                                }
                                log = m_pReader->Next();
                            }
                        }
                        ASSERT_EQ(std::string(m_pReader->GetLocalName()), parentTagName);
                        // 実際のログと正解データのエレメント数が異なった場合のため
                        // 帳尻合わせ
                        while (answer != nlib_ns::exi::XmlStreamReader::kEndElement ||
                            std::string(m_pReaderForAnswer->GetLocalName()) != parentTagName)
                        {
                            answer = m_pReaderForAnswer->Next();
                            ASSERT_TRUE(answer == nlib_ns::exi::XmlStreamReader::kStartElement ||
                                answer == nlib_ns::exi::XmlStreamReader::kEndElement);
                        }
                    }
                };

                // spans の中は 1つ以上の span タグ
                // span タグは 4 つの属性を持つ
                SpanOrAllocCheck("spans", "span");

                // allocs の中は 0 つ以上の alloc タグ
                // alloc タグは 2 つの属性を持つ
                SpanOrAllocCheck("allocs", "alloc");
            }
            ASSERT_FALSE(nlib_is_error(*m_pReader));  // ここで引っ掛かる場合は、途中でエラーが発生している
            ASSERT_FALSE(nlib_is_error(*m_pReaderForAnswer));
        }   // NOLINT(impl/function_size)

    protected:
        const size_t m_ExiAllocatorBufSize;     // XMLパーサ 用アロケータのサイズ
        const size_t m_SaAllocatorBufSize;      // StandardAllocator に与えるメモリサイズ

        // テスト対象アロケータ
        nn::mem::StandardAllocator m_SaAllocator;
        void* m_SaAllocatorBuf;   // StandardAllocator 用バッファ

        bool m_IsXmlParseReady;                 // XML パーサの初期化が済んでいるなら true
        // XML パーサ用アロケータバッファ
        std::unique_ptr<char[]> m_pExiAllocatorBuf;
        // XML パーサ(ログパース用)
        std::unique_ptr<nlib_ns::exi::XmlStreamReader> m_pReader;
        // XML パーサ(正解データ用)
        std::unique_ptr<nlib_ns::exi::XmlStreamReader> m_pReaderForAnswer;
        // ログパース用メモリ読み込みストリーム
        nlib_ns::MemoryInputStream m_Stream;
        // 正解データパース用メモリ読み込みストリーム
        nlib_ns::MemoryInputStream m_StreamForAnswer;
        nlib_ns::exi::XmlStreamReaderSettings m_Settings;   // XML パース設定
        nlib_ns::exi::ExiAllocatorEx m_ExiAllocator;        // XML パーサ用アロケータ

        bool m_IsAnswerPrepared;    // 正解データ準備済みフラグ
        std::string m_AnswerStr;    // StandardAllocator::Dump() のログ構造正解データ

        bool m_IsRegistered;        // ログオブザーバが登録済みなら true
        std::string m_LogStr;       // ログオブザーバを使って取得した NN_LOG の文字列
        nn::diag::LogObserverHolder m_Holder;   // ログオブザーバホルダ

        int m_Param;                            // テストパラメータ
    };

}   // unnamed namespace

// Dump のフォーマットが正しいかの確認
TEST_P(StandardAllocatorDumpTest, Schema)
{
    // StandardAllocator::Dump() のログを取得
    m_SaAllocator.Allocate(4096);    // alloc タグを作るため

    StartLogAcquisition();
    m_SaAllocator.Dump();
    EndLogAcquisition();

    InitializeXmlParser();  // ログのパース準備

    // ログが Dump として正しい構造かチェック
    CheckXmlSchema();
}

// alloc の数が想定通りであることを確認
TEST_P(StandardAllocatorDumpTest, AllocCount)
{
    const int allocNum = 32;
    void* address[allocNum] = {};
    for (int i = 0; i < allocNum; ++i)
    {
        address[i] = m_SaAllocator.Allocate(4096);
    }
    // ログ取得
    StartLogAcquisition();
    m_SaAllocator.Dump();
    EndLogAcquisition();

    InitializeXmlParser();

    // Large メモリの確保なので、 alloc タグの出現回数は allocNum になるはず
    int allocElemCount = 0;
    while (m_pReader->HasNext())
    {
        nlib_ns::exi::XmlStreamReader::XmlStreamConstants constant = m_pReader->Next();

        if (constant == nlib_ns::exi::XmlStreamReader::kStartElement &&
            std::string(m_pReader->GetLocalName()) == "alloc")
        {
            allocElemCount++;
        }
    }

    ASSERT_EQ(allocElemCount, allocNum);

    // XML パース情報を一旦破棄
    FinalizeXmlParser();
    m_LogStr = "";

    // メモリ解放して、 alloc タグが 0 個になっているか再確認
    for (int i = 0; i < allocNum; ++i)
    {
        m_SaAllocator.Free(address[i]);
    }

    StartLogAcquisition();
    m_SaAllocator.Dump();
    EndLogAcquisition();

    InitializeXmlParser();

    while (m_pReader->HasNext())
    {
        nlib_ns::exi::XmlStreamReader::XmlStreamConstants constant = m_pReader->Next();

        if (constant == nlib_ns::exi::XmlStreamReader::kStartElement ||
            constant == nlib_ns::exi::XmlStreamReader::kEndElement)
        {
            ASSERT_STRNE(m_pReader->GetLocalName(), "alloc");
        }
    }
}

// ランダムに確保・解放を繰り返し、
// その後でも Dump() の構文が崩れていないことを確認
TEST_P(StandardAllocatorDumpTest, Random)
{
    const int seed = 12345;
    const int AllocCount = 10000;
    const int ptrSize = 4096;
    void* ptr[ptrSize] = {};

    std::mt19937 engine(seed);

    // 0: small 1: large
    // 0: alloc 1: free
    std::uniform_int_distribution<int> allocateTypeDist(0, 1);
    std::uniform_int_distribution<size_t> smallAllocateDist(8, 1024);
    std::uniform_int_distribution<size_t> largeAllocateDist(2048, 64 * 1024);

    std::vector<int> freeIndex;      // ptr の空いている要素
    std::vector<int> allocatedIndex; // ptr の確保済みの要素

    // v 中の value のインデックスを検索
    auto FindIndex = [](std::vector<int>& v, int value) -> int
    {
        auto it = find(v.begin(), v.end(), value);
        size_t index = distance(v.begin(), it);
        if (index == v.size()) { return -1; }
        return index;
    };

    // メモリ確保処理
    // index : ptr のインデックス
    auto Allocate = [&](int index)
    {
        // 確保サイズとアライメントをランダムに決定
        size_t size = (allocateTypeDist(engine)) ? smallAllocateDist(engine) :
            largeAllocateDist(engine);

        ASSERT_TRUE(ptr[index] == nullptr);
        ptr[index] = m_SaAllocator.Allocate(size);
        ASSERT_NOT_NULL(ptr[index]);

        // freeIndex から確保済みの ptr のインデックスを削除
        int freeIndexPos = FindIndex(freeIndex, index);
        freeIndex.erase(freeIndex.begin() + freeIndexPos);

        // allocatedIndex に確保済みの ptr のインデックスを追加
        allocatedIndex.push_back(index);
    };

    // メモリ解放処理
    auto Free = [&]()
    {
        // 解法先をランダムに決定
        std::uniform_int_distribution<int> indexDist(0, allocatedIndex.size() - 1);
        int allocatedIndexPos = indexDist(engine);
        int index = allocatedIndex[allocatedIndexPos];

        ASSERT_NOT_NULL(ptr[index]);
        m_SaAllocator.Free(ptr[index]);
        ptr[index] = nullptr;

        // allocatedIndex から解法済みの ptr のインデックスを削除
        allocatedIndex.erase(allocatedIndex.begin() + allocatedIndexPos);

        // freeIndex に解放済みの ptr のインデックスを追加
        freeIndex.push_back(index);
    };

    // ptr を半分埋める
    for (int i = 0; i < ptrSize; ++i)
    {
        freeIndex.push_back(i);
        if (i <= ptrSize / 2)
        {
            Allocate(i);
        }
    }
    ASSERT_EQ(ptrSize, freeIndex.size() + allocatedIndex.size());

    // ランダムな確保・解放
    for (int i = 0; i < AllocCount; ++i)
    {
        // 配列が全部埋まっているなら解放
        if (freeIndex.empty())
        {
            Free();
        }
        else
        {
            // 確保するか解放するか決める
            if (allocateTypeDist(engine))
            {
                // 確保する
                int index = freeIndex[0];
                Allocate(index);
            }
            else
            {
                Free();
            }
        }
        ASSERT_EQ(ptrSize, freeIndex.size() + allocatedIndex.size());
    }


    StartLogAcquisition();
    m_SaAllocator.Dump();
    EndLogAcquisition();

    InitializeXmlParser();  // ログのパース準備

    // ログが Dump として正しい構造かチェック
    CheckXmlSchema();
}

INSTANTIATE_TEST_CASE_P(VirtualAddressManagementMode,
                        StandardAllocatorDumpTest,
                        testing::Values(StandardAllocatorTestParam_VammDisabled,
                                        StandardAllocatorTestParam_VammEnabled));

