﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nn/mbuf/mbuf_Mbuf.h>
#include <nn/mbuf/mbuf_MbufInit.h>
#include <nn/os/os_SystemEvent.h>

namespace
{
    // mbuf クラスタのサイズです。
    const size_t UnitSize = nn::mbuf::MbufUnitSizeMin;

    // mbuf クラスタに格納できるデータのサイズです。
    const size_t DataSize = UnitSize - nn::mbuf::MbufHeaderSize;

    // mbuf の Type (デフォルト値) です。
    const int DefaultType = 1;

    // mbuf プールの容量です。
    const int PoolCapacity = 16;

    // mbuf の初期化に使用するバッファです。
    nn::Bit8 g_MbufBuffer[8 * 1024 * 1024];

    /**
     * @brief       テストの開始・終了時に実行する共通の処理を定義します。
     */
    class MbufBasic : public ::testing::Test
    {
    protected:

        virtual void SetUp() NN_OVERRIDE
        {
            nn::mbuf::Type types[nn::mbuf::TypeCountMax];
            for (int i = 0; i < nn::mbuf::TypeCountMax; ++i)
            {
                nn::os::CreateSystemEvent(
                    &m_SystemEventTypes[i], nn::os::EventClearMode_AutoClear, false);
                types[i].pEvent = &m_SystemEventTypes[i];
            }
            nn::mbuf::Config configs[nn::mbuf::TypeCountMax];
            for (int i = 0; i < nn::mbuf::TypeCountMax; ++i)
            {
                configs[i].unitSize = UnitSize;
                configs[i].count = PoolCapacity;
                configs[i].type = i;
            }
            nn::mbuf::Initialize(
                types, nn::mbuf::TypeCountMax, configs, nn::mbuf::TypeCountMax,
                g_MbufBuffer, sizeof(g_MbufBuffer));
        }

        virtual void TearDown() NN_OVERRIDE
        {
            nn::mbuf::Finalize();
            for (int i = 0; i < nn::mbuf::TypeCountMax; ++i)
            {
                nn::os::DestroySystemEvent(&m_SystemEventTypes[i]);
            }
        }

    private:

        nn::os::SystemEventType m_SystemEventTypes[nn::mbuf::TypeCountMax];
    };

    /**
     * @brief       MbufGetm のテストに使用するフィクスチャです。
     */
    class MbufGetm : public MbufBasic
    {
    };

    /**
     * @brief       MbufAdj のテストに使用するフィクスチャです。
     */
    class MbufAdj : public MbufBasic
    {
    };

    /**
     * @brief       MbufAppend のテストに使用するフィクスチャです。
     */
    class MbufAppend : public MbufBasic
    {
    };

    /**
     * @brief       MbufDup のテストに使用するフィクスチャです。
     */
    class MbufDup : public MbufBasic
    {
    };

    /**
     * @brief       MbufPrepend のテストに使用するフィクスチャです。
     */
    class MbufPrepend : public MbufBasic
    {
    };

    /**
     * @brief       MbufReserve のテストに使用するフィクスチャです。
     */
    class MbufReserve : public MbufBasic
    {
    };
}

TEST_F(MbufGetm, DontWaitZeroData)
{
    // 0 Byte の mbuf は確保できません。
    nn::mbuf::Mbuf* pMbuf = nn::mbuf::MbufGetm(
        nullptr, 0, nn::mbuf::MbufAllocationMode_DontWait, DefaultType);
    ASSERT_EQ(nullptr, pMbuf);
}

TEST_F(MbufGetm, WaitZeroData)
{
    // 0 Byte の mbuf は確保できません。
    nn::mbuf::Mbuf* pMbuf = nn::mbuf::MbufGetm(
        nullptr, 0, nn::mbuf::MbufAllocationMode_Wait, DefaultType);
    ASSERT_EQ(nullptr, pMbuf);
}

TEST_F(MbufGetm, DontWaitMinimalMbuf)
{
    // 正しく mbuf を解放できていることを検証するため 2 回テストを実行します。
    for (int k = 0; k < 2; ++k)
    {
        // 最大数の mbuf を割り当てます。
        nn::mbuf::Mbuf* mbufs[PoolCapacity];
        for (int i = 0; i < PoolCapacity; ++i)
        {
            nn::mbuf::Mbuf* pMbuf = nn::mbuf::MbufGetm(
                nullptr, 1, nn::mbuf::MbufAllocationMode_DontWait, DefaultType);
            ASSERT_NE(nullptr, pMbuf);
            ASSERT_EQ(nullptr, pMbuf->_pPrev);
            ASSERT_EQ(nullptr, pMbuf->_pNext);
            ASSERT_EQ(nullptr, pMbuf->_pGeneral);
            ASSERT_EQ(0, pMbuf->_len);
            ASSERT_EQ(DataSize, pMbuf->_capacity);
            ASSERT_EQ(0, pMbuf->_top);
            ASSERT_EQ(0, pMbuf->_flags);
            ASSERT_EQ(0, pMbuf->_protocol);
            ASSERT_EQ(0, pMbuf->_owner);
            ASSERT_EQ(DefaultType, pMbuf->_type);
            ASSERT_EQ(DefaultType, pMbuf->_pool);
            mbufs[i] = pMbuf;
        }

        // プールに空きがないため割り当てに失敗します。
        nn::mbuf::Mbuf* pMbuf = nn::mbuf::MbufGetm(
            nullptr, 1, nn::mbuf::MbufAllocationMode_DontWait, DefaultType);
        ASSERT_EQ(nullptr, pMbuf);

        // mbuf を全て解放します。
        for (int i = 0; i < PoolCapacity; ++i)
        {
            nn::mbuf::MbufFreem(mbufs[i]);
        }
    }
}

TEST_F(MbufGetm, WaitMinimalMbuf)
{
    // 正しく mbuf を解放できていることを検証するため 2 回テストを実行します。
    for (int k = 0; k < 2; ++k)
    {
        // 最大数の mbuf を割り当てます。
        nn::mbuf::Mbuf* mbufs[PoolCapacity];
        for (int i = 0; i < PoolCapacity; ++i)
        {
            nn::mbuf::Mbuf* pMbuf = nn::mbuf::MbufGetm(
                nullptr, 1, nn::mbuf::MbufAllocationMode_Wait, DefaultType);
            ASSERT_NE(nullptr, pMbuf);
            ASSERT_EQ(nullptr, pMbuf->_pPrev);
            ASSERT_EQ(nullptr, pMbuf->_pNext);
            ASSERT_EQ(nullptr, pMbuf->_pGeneral);
            ASSERT_EQ(0, pMbuf->_len);
            ASSERT_EQ(DataSize, pMbuf->_capacity);
            ASSERT_EQ(0, pMbuf->_top);
            ASSERT_EQ(0, pMbuf->_flags);
            ASSERT_EQ(0, pMbuf->_protocol);
            ASSERT_EQ(0, pMbuf->_owner);
            ASSERT_EQ(DefaultType, pMbuf->_type);
            ASSERT_EQ(DefaultType, pMbuf->_pool);
            mbufs[i] = pMbuf;
        }

        // mbuf を全て解放します。
        for (int i = 0; i < PoolCapacity; ++i)
        {
            nn::mbuf::MbufFreem(mbufs[i]);
        }
    }
}

TEST_F(MbufGetm, DontWaitBiggestMbuf)
{
    // 正しく mbuf を解放できていることを検証するため 2 回テストを実行します。
    for (int k = 0; k < 2; ++k)
    {
        // 最大数の mbuf を割り当てます。
        nn::mbuf::Mbuf* mbufs[PoolCapacity];
        for (int i = 0; i < PoolCapacity; ++i)
        {
            nn::mbuf::Mbuf* pMbuf = nn::mbuf::MbufGetm(
                nullptr, DataSize, nn::mbuf::MbufAllocationMode_DontWait, DefaultType);
            ASSERT_NE(nullptr, pMbuf);
            ASSERT_EQ(nullptr, pMbuf->_pPrev);
            ASSERT_EQ(nullptr, pMbuf->_pNext);
            ASSERT_EQ(nullptr, pMbuf->_pGeneral);
            ASSERT_EQ(0, pMbuf->_len);
            ASSERT_EQ(DataSize, pMbuf->_capacity);
            ASSERT_EQ(0, pMbuf->_top);
            ASSERT_EQ(0, pMbuf->_flags);
            ASSERT_EQ(0, pMbuf->_protocol);
            ASSERT_EQ(0, pMbuf->_owner);
            ASSERT_EQ(DefaultType, pMbuf->_type);
            ASSERT_EQ(DefaultType, pMbuf->_pool);
            mbufs[i] = pMbuf;
        }

        // プールに空きがないため割り当てに失敗します。
        nn::mbuf::Mbuf* pMbuf = nn::mbuf::MbufGetm(
            nullptr, DataSize, nn::mbuf::MbufAllocationMode_DontWait, DefaultType);
        ASSERT_EQ(nullptr, pMbuf);

        // mbuf を全て解放します。
        for (int i = 0; i < PoolCapacity; ++i)
        {
            nn::mbuf::MbufFreem(mbufs[i]);
        }
    }
}

TEST_F(MbufGetm, WaitBiggestMbuf)
{
    // 正しく mbuf を解放できていることを検証するため 2 回テストを実行します。
    for (int k = 0; k < 2; ++k)
    {
        // 最大数の mbuf を割り当てます。
        nn::mbuf::Mbuf* mbufs[PoolCapacity];
        for (int i = 0; i < PoolCapacity; ++i)
        {
            nn::mbuf::Mbuf* pMbuf = nn::mbuf::MbufGetm(
                nullptr, DataSize, nn::mbuf::MbufAllocationMode_Wait, DefaultType);
            ASSERT_NE(nullptr, pMbuf);
            ASSERT_EQ(nullptr, pMbuf->_pPrev);
            ASSERT_EQ(nullptr, pMbuf->_pNext);
            ASSERT_EQ(nullptr, pMbuf->_pGeneral);
            ASSERT_EQ(0, pMbuf->_len);
            ASSERT_EQ(DataSize, pMbuf->_capacity);
            ASSERT_EQ(0, pMbuf->_top);
            ASSERT_EQ(0, pMbuf->_flags);
            ASSERT_EQ(0, pMbuf->_protocol);
            ASSERT_EQ(0, pMbuf->_owner);
            ASSERT_EQ(DefaultType, pMbuf->_type);
            ASSERT_EQ(DefaultType, pMbuf->_pool);
            mbufs[i] = pMbuf;
        }

        // mbuf を全て解放します。
        for (int i = 0; i < PoolCapacity; ++i)
        {
            nn::mbuf::MbufFreem(mbufs[i]);
        }
    }
}

TEST_F(MbufGetm, DontWaitMinimalMbufChain2)
{
    // 正しく mbuf を解放できていることを検証するため 2 回テストを実行します。
    const int capacity = PoolCapacity / 2;
    for (int k = 0; k < 2; ++k)
    {
        // 最大数の mbuf を割り当てます。
        nn::mbuf::Mbuf* mbufs[capacity];
        for (int i = 0; i < capacity; ++i)
        {
            nn::mbuf::Mbuf* pMbuf = nn::mbuf::MbufGetm(
                nullptr, DataSize + 1, nn::mbuf::MbufAllocationMode_DontWait, DefaultType);

            // mbuf chain の先頭の mbuf クラスタです。
            ASSERT_NE(nullptr, pMbuf);
            ASSERT_EQ(nullptr, pMbuf->_pPrev);
            ASSERT_NE(nullptr, pMbuf->_pNext);
            ASSERT_EQ(nullptr, pMbuf->_pGeneral);
            ASSERT_EQ(0, pMbuf->_len);
            ASSERT_EQ(DataSize, pMbuf->_capacity);
            ASSERT_EQ(0, pMbuf->_top);
            ASSERT_EQ(0, pMbuf->_flags);
            ASSERT_EQ(0, pMbuf->_protocol);
            ASSERT_EQ(0, pMbuf->_owner);
            ASSERT_EQ(DefaultType, pMbuf->_type);
            ASSERT_EQ(DefaultType, pMbuf->_pool);

            // mbuf chain の 2 つ目の mbuf クラスタです。
            nn::mbuf::Mbuf* pNext = pMbuf->_pNext;
            ASSERT_NE(nullptr, pNext);
            ASSERT_EQ(pMbuf, pNext->_pPrev);
            ASSERT_EQ(nullptr, pNext->_pNext);
            ASSERT_EQ(nullptr, pNext->_pGeneral);
            ASSERT_EQ(0, pNext->_len);
            ASSERT_EQ(DataSize, pNext->_capacity);
            ASSERT_EQ(0, pNext->_top);
            ASSERT_EQ(0, pNext->_flags);
            ASSERT_EQ(0, pNext->_protocol);
            ASSERT_EQ(0, pNext->_owner);
            ASSERT_EQ(DefaultType, pNext->_type);
            ASSERT_EQ(DefaultType, pNext->_pool);
            mbufs[i] = pMbuf;
        }

        // プールに空きがないため割り当てに失敗します。
        nn::mbuf::Mbuf* pMbuf = nn::mbuf::MbufGetm(
            nullptr, DataSize + 1, nn::mbuf::MbufAllocationMode_DontWait, DefaultType);
        ASSERT_EQ(nullptr, pMbuf);

        // mbuf を全て解放します。
        for (int i = 0; i < capacity; ++i)
        {
            nn::mbuf::MbufFreem(mbufs[i]);
        }
    }
}

TEST_F(MbufGetm, WaitMinimalMbufChain2)
{
    // 正しく mbuf を解放できていることを検証するため 2 回テストを実行します。
    const int capacity = PoolCapacity / 2;
    for (int k = 0; k < 2; ++k)
    {
        // 最大数の mbuf を割り当てます。
        nn::mbuf::Mbuf* mbufs[capacity];
        for (int i = 0; i < capacity; ++i)
        {
            nn::mbuf::Mbuf* pMbuf = nn::mbuf::MbufGetm(
                nullptr, DataSize + 1, nn::mbuf::MbufAllocationMode_Wait, DefaultType);

            // mbuf chain の先頭の mbuf クラスタです。
            ASSERT_NE(nullptr, pMbuf);
            ASSERT_EQ(nullptr, pMbuf->_pPrev);
            ASSERT_NE(nullptr, pMbuf->_pNext);
            ASSERT_EQ(nullptr, pMbuf->_pGeneral);
            ASSERT_EQ(0, pMbuf->_len);
            ASSERT_EQ(DataSize, pMbuf->_capacity);
            ASSERT_EQ(0, pMbuf->_top);
            ASSERT_EQ(0, pMbuf->_flags);
            ASSERT_EQ(0, pMbuf->_protocol);
            ASSERT_EQ(0, pMbuf->_owner);
            ASSERT_EQ(DefaultType, pMbuf->_type);
            ASSERT_EQ(DefaultType, pMbuf->_pool);

            // mbuf chain の 2 つ目の mbuf クラスタです。
            nn::mbuf::Mbuf* pNext = pMbuf->_pNext;
            ASSERT_NE(nullptr, pNext);
            ASSERT_EQ(pMbuf, pNext->_pPrev);
            ASSERT_EQ(nullptr, pNext->_pNext);
            ASSERT_EQ(nullptr, pNext->_pGeneral);
            ASSERT_EQ(0, pNext->_len);
            ASSERT_EQ(DataSize, pNext->_capacity);
            ASSERT_EQ(0, pNext->_top);
            ASSERT_EQ(0, pNext->_flags);
            ASSERT_EQ(0, pNext->_protocol);
            ASSERT_EQ(0, pNext->_owner);
            ASSERT_EQ(DefaultType, pNext->_type);
            ASSERT_EQ(DefaultType, pNext->_pool);
            mbufs[i] = pMbuf;
        }

        // mbuf を全て解放します。
        for (int i = 0; i < capacity; ++i)
        {
            nn::mbuf::MbufFreem(mbufs[i]);
        }
    }
}

TEST_F(MbufGetm, DontWaitBiggestMbufChain2)
{
    // 正しく mbuf を解放できていることを検証するため 2 回テストを実行します。
    const int capacity = PoolCapacity / 2;
    for (int k = 0; k < 2; ++k)
    {
        // 最大数の mbuf を割り当てます。
        nn::mbuf::Mbuf* mbufs[capacity];
        for (int i = 0; i < capacity; ++i)
        {
            nn::mbuf::Mbuf* pMbuf = nn::mbuf::MbufGetm(
                nullptr, DataSize * 2, nn::mbuf::MbufAllocationMode_DontWait, DefaultType);

            // mbuf chain の先頭の mbuf クラスタです。
            ASSERT_NE(nullptr, pMbuf);
            ASSERT_EQ(nullptr, pMbuf->_pPrev);
            ASSERT_NE(nullptr, pMbuf->_pNext);
            ASSERT_EQ(nullptr, pMbuf->_pGeneral);
            ASSERT_EQ(0, pMbuf->_len);
            ASSERT_EQ(DataSize, pMbuf->_capacity);
            ASSERT_EQ(0, pMbuf->_top);
            ASSERT_EQ(0, pMbuf->_flags);
            ASSERT_EQ(0, pMbuf->_protocol);
            ASSERT_EQ(0, pMbuf->_owner);
            ASSERT_EQ(DefaultType, pMbuf->_type);
            ASSERT_EQ(DefaultType, pMbuf->_pool);

            // mbuf chain の 2 つ目の mbuf クラスタです。
            nn::mbuf::Mbuf* pNext = pMbuf->_pNext;
            ASSERT_NE(nullptr, pNext);
            ASSERT_EQ(pMbuf, pNext->_pPrev);
            ASSERT_EQ(nullptr, pNext->_pNext);
            ASSERT_EQ(nullptr, pNext->_pGeneral);
            ASSERT_EQ(0, pNext->_len);
            ASSERT_EQ(DataSize, pNext->_capacity);
            ASSERT_EQ(0, pNext->_top);
            ASSERT_EQ(0, pNext->_flags);
            ASSERT_EQ(0, pNext->_protocol);
            ASSERT_EQ(0, pNext->_owner);
            ASSERT_EQ(DefaultType, pNext->_type);
            ASSERT_EQ(DefaultType, pNext->_pool);
            mbufs[i] = pMbuf;
        }

        // プールに空きがないため割り当てに失敗します。
        nn::mbuf::Mbuf* pMbuf = nn::mbuf::MbufGetm(
            nullptr, DataSize * 2, nn::mbuf::MbufAllocationMode_DontWait, DefaultType);
        ASSERT_EQ(nullptr, pMbuf);

        // mbuf を全て解放します。
        for (int i = 0; i < capacity; ++i)
        {
            nn::mbuf::MbufFreem(mbufs[i]);
        }
    }
}

TEST_F(MbufGetm, WaitBiggestMbufChain2)
{
    // 正しく mbuf を解放できていることを検証するため 2 回テストを実行します。
    const int capacity = PoolCapacity / 2;
    for (int k = 0; k < 2; ++k)
    {
        // 最大数の mbuf を割り当てます。
        nn::mbuf::Mbuf* mbufs[capacity];
        for (int i = 0; i < capacity; ++i)
        {
            nn::mbuf::Mbuf* pMbuf = nn::mbuf::MbufGetm(
                nullptr, DataSize * 2, nn::mbuf::MbufAllocationMode_Wait, DefaultType);

            // mbuf chain の先頭の mbuf クラスタです。
            ASSERT_NE(nullptr, pMbuf);
            ASSERT_EQ(nullptr, pMbuf->_pPrev);
            ASSERT_NE(nullptr, pMbuf->_pNext);
            ASSERT_EQ(nullptr, pMbuf->_pGeneral);
            ASSERT_EQ(0, pMbuf->_len);
            ASSERT_EQ(DataSize, pMbuf->_capacity);
            ASSERT_EQ(0, pMbuf->_top);
            ASSERT_EQ(0, pMbuf->_flags);
            ASSERT_EQ(0, pMbuf->_protocol);
            ASSERT_EQ(0, pMbuf->_owner);
            ASSERT_EQ(DefaultType, pMbuf->_type);
            ASSERT_EQ(DefaultType, pMbuf->_pool);

            // mbuf chain の 2 つ目の mbuf クラスタです。
            nn::mbuf::Mbuf* pNext = pMbuf->_pNext;
            ASSERT_NE(nullptr, pNext);
            ASSERT_EQ(pMbuf, pNext->_pPrev);
            ASSERT_EQ(nullptr, pNext->_pNext);
            ASSERT_EQ(nullptr, pNext->_pGeneral);
            ASSERT_EQ(0, pNext->_len);
            ASSERT_EQ(DataSize, pNext->_capacity);
            ASSERT_EQ(0, pNext->_top);
            ASSERT_EQ(0, pNext->_flags);
            ASSERT_EQ(0, pNext->_protocol);
            ASSERT_EQ(0, pNext->_owner);
            ASSERT_EQ(DefaultType, pNext->_type);
            ASSERT_EQ(DefaultType, pNext->_pool);
            mbufs[i] = pMbuf;
        }

        // mbuf を全て解放します。
        for (int i = 0; i < capacity; ++i)
        {
            nn::mbuf::MbufFreem(mbufs[i]);
        }
    }
}

TEST_F(MbufGetm, DontWaitMinimalMbufChain3)
{
    // 正しく mbuf を解放できていることを検証するため 2 回テストを実行します。
    const int capacity = PoolCapacity / 3;
    for (int k = 0; k < 2; ++k)
    {
        // 最大数の mbuf を割り当てます。
        nn::mbuf::Mbuf* mbufs[capacity];
        for (int i = 0; i < capacity; ++i)
        {
            nn::mbuf::Mbuf* pMbuf = nn::mbuf::MbufGetm(
            nullptr, DataSize * 2 + 1, nn::mbuf::MbufAllocationMode_DontWait, DefaultType);

            // mbuf chain の先頭の mbuf クラスタです。
            ASSERT_NE(nullptr, pMbuf);
            ASSERT_EQ(nullptr, pMbuf->_pPrev);
            ASSERT_NE(nullptr, pMbuf->_pNext);
            ASSERT_EQ(nullptr, pMbuf->_pGeneral);
            ASSERT_EQ(0, pMbuf->_len);
            ASSERT_EQ(DataSize, pMbuf->_capacity);
            ASSERT_EQ(0, pMbuf->_top);
            ASSERT_EQ(0, pMbuf->_flags);
            ASSERT_EQ(0, pMbuf->_protocol);
            ASSERT_EQ(0, pMbuf->_owner);
            ASSERT_EQ(DefaultType, pMbuf->_type);
            ASSERT_EQ(DefaultType, pMbuf->_pool);

            // mbuf chain の 2 つ目の mbuf クラスタです。
            nn::mbuf::Mbuf* pNext = pMbuf->_pNext;
            ASSERT_NE(nullptr, pNext);
            ASSERT_EQ(pMbuf, pNext->_pPrev);
            ASSERT_NE(nullptr, pNext->_pNext);
            ASSERT_EQ(nullptr, pNext->_pGeneral);
            ASSERT_EQ(0, pNext->_len);
            ASSERT_EQ(DataSize, pNext->_capacity);
            ASSERT_EQ(0, pNext->_top);
            ASSERT_EQ(0, pNext->_flags);
            ASSERT_EQ(0, pNext->_protocol);
            ASSERT_EQ(0, pNext->_owner);
            ASSERT_EQ(DefaultType, pNext->_type);
            ASSERT_EQ(DefaultType, pNext->_pool);

            // mbuf chain の 3 つ目の mbuf クラスタです。
            pNext = pNext->_pNext;
            ASSERT_NE(nullptr, pNext);
            ASSERT_EQ(pMbuf->_pNext, pNext->_pPrev);
            ASSERT_EQ(nullptr, pNext->_pNext);
            ASSERT_EQ(nullptr, pNext->_pGeneral);
            ASSERT_EQ(0, pNext->_len);
            ASSERT_EQ(DataSize, pNext->_capacity);
            ASSERT_EQ(0, pNext->_top);
            ASSERT_EQ(0, pNext->_flags);
            ASSERT_EQ(0, pNext->_protocol);
            ASSERT_EQ(0, pNext->_owner);
            ASSERT_EQ(DefaultType, pNext->_type);
            ASSERT_EQ(DefaultType, pNext->_pool);
            mbufs[i] = pMbuf;
        }

        // プールに空きがないため割り当てに失敗します。
        nn::mbuf::Mbuf* pMbuf = nn::mbuf::MbufGetm(
            nullptr, DataSize * 2 + 1, nn::mbuf::MbufAllocationMode_DontWait, DefaultType);
        ASSERT_EQ(nullptr, pMbuf);

        // mbuf を全て解放します。
        for (int i = 0; i < capacity; ++i)
        {
            nn::mbuf::MbufFreem(mbufs[i]);
        }
    }
}

TEST_F(MbufGetm, WaitMinimalMbufChain3)
{
    // 正しく mbuf を解放できていることを検証するため 2 回テストを実行します。
    const int capacity = PoolCapacity / 3;
    for (int k = 0; k < 2; ++k)
    {
        // 最大数の mbuf を割り当てます。
        nn::mbuf::Mbuf* mbufs[capacity];
        for (int i = 0; i < capacity; ++i)
        {
            nn::mbuf::Mbuf* pMbuf = nn::mbuf::MbufGetm(
            nullptr, DataSize * 2 + 1, nn::mbuf::MbufAllocationMode_Wait, DefaultType);

            // mbuf chain の先頭の mbuf クラスタです。
            ASSERT_NE(nullptr, pMbuf);
            ASSERT_EQ(nullptr, pMbuf->_pPrev);
            ASSERT_NE(nullptr, pMbuf->_pNext);
            ASSERT_EQ(nullptr, pMbuf->_pGeneral);
            ASSERT_EQ(0, pMbuf->_len);
            ASSERT_EQ(DataSize, pMbuf->_capacity);
            ASSERT_EQ(0, pMbuf->_top);
            ASSERT_EQ(0, pMbuf->_flags);
            ASSERT_EQ(0, pMbuf->_protocol);
            ASSERT_EQ(0, pMbuf->_owner);
            ASSERT_EQ(DefaultType, pMbuf->_type);
            ASSERT_EQ(DefaultType, pMbuf->_pool);

            // mbuf chain の 2 つ目の mbuf クラスタです。
            nn::mbuf::Mbuf* pNext = pMbuf->_pNext;
            ASSERT_NE(nullptr, pNext);
            ASSERT_EQ(pMbuf, pNext->_pPrev);
            ASSERT_NE(nullptr, pNext->_pNext);
            ASSERT_EQ(nullptr, pNext->_pGeneral);
            ASSERT_EQ(0, pNext->_len);
            ASSERT_EQ(DataSize, pNext->_capacity);
            ASSERT_EQ(0, pNext->_top);
            ASSERT_EQ(0, pNext->_flags);
            ASSERT_EQ(0, pNext->_protocol);
            ASSERT_EQ(0, pNext->_owner);
            ASSERT_EQ(DefaultType, pNext->_type);
            ASSERT_EQ(DefaultType, pNext->_pool);

            // mbuf chain の 3 つ目の mbuf クラスタです。
            pNext = pNext->_pNext;
            ASSERT_NE(nullptr, pNext);
            ASSERT_EQ(pMbuf->_pNext, pNext->_pPrev);
            ASSERT_EQ(nullptr, pNext->_pNext);
            ASSERT_EQ(nullptr, pNext->_pGeneral);
            ASSERT_EQ(0, pNext->_len);
            ASSERT_EQ(DataSize, pNext->_capacity);
            ASSERT_EQ(0, pNext->_top);
            ASSERT_EQ(0, pNext->_flags);
            ASSERT_EQ(0, pNext->_protocol);
            ASSERT_EQ(0, pNext->_owner);
            ASSERT_EQ(DefaultType, pNext->_type);
            ASSERT_EQ(DefaultType, pNext->_pool);
            mbufs[i] = pMbuf;
        }

        // mbuf を全て解放します。
        for (int i = 0; i < capacity; ++i)
        {
            nn::mbuf::MbufFreem(mbufs[i]);
        }
    }
}

TEST_F(MbufAppend, AppendSmallDataToEmptyMbuf)
{
    // 1 クラスタから構成される mbuf チェインを確保します。
    nn::mbuf::Mbuf* pMbuf = nn::mbuf::MbufGetm(
        nullptr, DataSize, nn::mbuf::MbufAllocationMode_DontWait, DefaultType);
    ASSERT_NE(nullptr, pMbuf);

    // mbuf クラスタに収まる範囲でデータを追加します。
    for (size_t i = 0; i < DataSize; ++i)
    {
        uint8_t data = static_cast<uint8_t>(i + 1);
        ASSERT_TRUE(nn::mbuf::MbufAppend(pMbuf, sizeof(data), &data));
        ASSERT_EQ(nullptr, pMbuf->_pPrev);
        ASSERT_EQ(nullptr, pMbuf->_pNext);
        ASSERT_EQ(nullptr, pMbuf->_pGeneral);
        ASSERT_EQ(i + 1, pMbuf->_len);
        ASSERT_EQ(DataSize, pMbuf->_capacity);
        ASSERT_EQ(0, pMbuf->_top);
        ASSERT_EQ(0, pMbuf->_flags);
        ASSERT_EQ(0, pMbuf->_protocol);
        ASSERT_EQ(0, pMbuf->_owner);
        ASSERT_EQ(DefaultType, pMbuf->_type);
        ASSERT_EQ(DefaultType, pMbuf->_pool);
    }

    // 意図したデータが格納されていることを検証します。
    for (size_t i = 0; i < DataSize; ++i)
    {
        ASSERT_EQ(i + 1, pMbuf->_data[i]);
    }

    // さらに小さいデータを追加することで mbuf チェインが構成されます。
    uint8_t data = static_cast<uint8_t>(DataSize + 1);
    ASSERT_TRUE(nn::mbuf::MbufAppend(pMbuf, sizeof(data), &data));

    // mbuf チェインの内容を検証します。
    auto* pNext = pMbuf->_pNext;
    ASSERT_EQ(nullptr, pMbuf->_pPrev);
    ASSERT_NE(nullptr, pMbuf->_pNext);
    ASSERT_EQ(nullptr, pMbuf->_pGeneral);
    ASSERT_EQ(DataSize, pMbuf->_len);
    ASSERT_EQ(DataSize, pMbuf->_capacity);
    ASSERT_EQ(0, pMbuf->_top);
    ASSERT_EQ(0, pMbuf->_flags);
    ASSERT_EQ(0, pMbuf->_protocol);
    ASSERT_EQ(0, pMbuf->_owner);
    ASSERT_EQ(DefaultType, pMbuf->_type);
    ASSERT_EQ(DefaultType, pMbuf->_pool);
    ASSERT_EQ(pMbuf, pNext->_pPrev);
    ASSERT_EQ(nullptr, pNext->_pNext);
    ASSERT_EQ(nullptr, pNext->_pGeneral);
    ASSERT_EQ(1, pNext->_len);
    ASSERT_EQ(DataSize, pNext->_capacity);
    ASSERT_EQ(0, pNext->_top);
    ASSERT_EQ(0, pNext->_flags);
    ASSERT_EQ(0, pNext->_protocol);
    ASSERT_EQ(0, pNext->_owner);
    ASSERT_EQ(DefaultType, pNext->_type);
    ASSERT_EQ(DefaultType, pNext->_pool);

    // mbuf チェインを解放します。
    nn::mbuf::MbufFreem(pMbuf);
}

TEST_F(MbufAppend, AppendLargeDataToEmptyMbuf)
{
    // 1 クラスタから構成される mbuf チェインを確保します。
    nn::mbuf::Mbuf* pMbuf = nn::mbuf::MbufGetm(
        nullptr, DataSize, nn::mbuf::MbufAllocationMode_DontWait, DefaultType);
    ASSERT_NE(nullptr, pMbuf);

    // mbuf クラスタに収まらないデータを生成します。
    uint8_t data[DataSize + 1];
    for (size_t i = 0; i < sizeof(data); ++i)
    {
        data[i] = static_cast<uint8_t>(i + 1);
    }

    // mbuf クラスタに収まらないデータを追加します。
    ASSERT_TRUE(nn::mbuf::MbufAppend(pMbuf, sizeof(data), &data));

    // mbuf チェインの内容を検証します。
    auto* pNext = pMbuf->_pNext;
    ASSERT_EQ(nullptr, pMbuf->_pPrev);
    ASSERT_NE(nullptr, pMbuf->_pNext);
    ASSERT_EQ(nullptr, pMbuf->_pGeneral);
    ASSERT_EQ(DataSize, pMbuf->_len);
    ASSERT_EQ(DataSize, pMbuf->_capacity);
    ASSERT_EQ(0, pMbuf->_top);
    ASSERT_EQ(0, pMbuf->_flags);
    ASSERT_EQ(0, pMbuf->_protocol);
    ASSERT_EQ(0, pMbuf->_owner);
    ASSERT_EQ(DefaultType, pMbuf->_type);
    ASSERT_EQ(DefaultType, pMbuf->_pool);
    ASSERT_EQ(pMbuf, pNext->_pPrev);
    ASSERT_EQ(nullptr, pNext->_pNext);
    ASSERT_EQ(nullptr, pNext->_pGeneral);
    ASSERT_EQ(1, pNext->_len);
    ASSERT_EQ(DataSize, pNext->_capacity);
    ASSERT_EQ(0, pNext->_top);
    ASSERT_EQ(0, pNext->_flags);
    ASSERT_EQ(0, pNext->_protocol);
    ASSERT_EQ(0, pNext->_owner);
    ASSERT_EQ(DefaultType, pNext->_type);
    ASSERT_EQ(DefaultType, pNext->_pool);

    // 意図したデータが格納されていることを検証します。
    for (size_t i = 0; i < DataSize; ++i)
    {
        ASSERT_EQ(i + 1, pMbuf->_data[i]);
    }
    ASSERT_EQ(DataSize + 1, pNext->_data[0]);

    // mbuf チェインを解放します。
    nn::mbuf::MbufFreem(pMbuf);
}

TEST_F(MbufAdj, TrimFromEmptyMbuf)
{
    // 1 クラスタから構成される mbuf チェインを確保します。
    nn::mbuf::Mbuf* pMbuf = nn::mbuf::MbufGetm(
        nullptr, DataSize, nn::mbuf::MbufAllocationMode_DontWait, DefaultType);
    ASSERT_NE(nullptr, pMbuf);

    // データサイズよりも大きなサイズのトリムはできません。
    ASSERT_FALSE(nn::mbuf::MbufAdj(pMbuf, -1));
    ASSERT_FALSE(nn::mbuf::MbufAdj(pMbuf, 1));

    // mbuf チェインを解放します。
    nn::mbuf::MbufFreem(pMbuf);
}

TEST_F(MbufAdj, TrimFrontFromMbuf)
{
    // 1 クラスタから構成される mbuf チェインを確保します。
    nn::mbuf::Mbuf* pMbuf = nn::mbuf::MbufGetm(
        nullptr, DataSize, nn::mbuf::MbufAllocationMode_DontWait, DefaultType);
    ASSERT_NE(nullptr, pMbuf);

    // 適当なデータを設定します。
    char data[4] = { 0x01, 0x02, 0x03, 0x04 };
    ASSERT_TRUE(nn::mbuf::MbufAppend(pMbuf, sizeof(data), &data));
    ASSERT_EQ(nullptr, pMbuf->_pNext);
    ASSERT_EQ(nullptr, pMbuf->_pPrev);
    ASSERT_EQ(0, pMbuf->_top);
    ASSERT_EQ(sizeof(data), pMbuf->_len);

    // データサイズよりも大きなサイズのトリムはできません。
    ASSERT_FALSE(nn::mbuf::MbufAdj(pMbuf, 5));

    // 先頭から最小サイズのトリムを実行します。
    ASSERT_TRUE(nn::mbuf::MbufAdj(pMbuf, 1));
    ASSERT_EQ(1, pMbuf->_top);
    ASSERT_EQ(3, pMbuf->_len);
    ASSERT_EQ(DataSize, pMbuf->_capacity);
    ASSERT_EQ(0x02, pMbuf->_data[pMbuf->_top]);
    ASSERT_EQ(0x03, pMbuf->_data[pMbuf->_top + 1]);
    ASSERT_EQ(0x04, pMbuf->_data[pMbuf->_top + 2]);

    // 0 サイズのトリムは何もしません。
    ASSERT_EQ(1, pMbuf->_top);
    ASSERT_EQ(3, pMbuf->_len);
    ASSERT_EQ(DataSize, pMbuf->_capacity);
    ASSERT_EQ(0x02, pMbuf->_data[pMbuf->_top]);
    ASSERT_EQ(0x03, pMbuf->_data[pMbuf->_top + 1]);
    ASSERT_EQ(0x04, pMbuf->_data[pMbuf->_top + 2]);

    // 先頭から、空になるようなトリムを実行します。
    ASSERT_TRUE(nn::mbuf::MbufAdj(pMbuf, 3));
    ASSERT_EQ(4, pMbuf->_top);
    ASSERT_EQ(0, pMbuf->_len);
    ASSERT_EQ(DataSize, pMbuf->_capacity);

    // mbuf チェインを解放します。
    nn::mbuf::MbufFreem(pMbuf);
}

TEST_F(MbufAdj, TrimBackFromMbuf)
{
    // 1 クラスタから構成される mbuf チェインを確保します。
    nn::mbuf::Mbuf* pMbuf = nn::mbuf::MbufGetm(
        nullptr, DataSize, nn::mbuf::MbufAllocationMode_DontWait, DefaultType);
    ASSERT_NE(nullptr, pMbuf);

    // 適当なデータを設定します。
    char data[4] = { 0x01, 0x02, 0x03, 0x04 };
    ASSERT_TRUE(nn::mbuf::MbufAppend(pMbuf, sizeof(data), &data));
    ASSERT_EQ(nullptr, pMbuf->_pNext);
    ASSERT_EQ(nullptr, pMbuf->_pPrev);
    ASSERT_EQ(0, pMbuf->_top);
    ASSERT_EQ(sizeof(data), pMbuf->_len);

    // データサイズよりも大きなサイズのトリムはできません。
    ASSERT_FALSE(nn::mbuf::MbufAdj(pMbuf, -5));

    // 末尾から最小サイズのトリムを実行します。
    ASSERT_TRUE(nn::mbuf::MbufAdj(pMbuf, -1));
    ASSERT_EQ(0, pMbuf->_top);
    ASSERT_EQ(3, pMbuf->_len);
    ASSERT_EQ(DataSize, pMbuf->_capacity);
    ASSERT_EQ(0x01, pMbuf->_data[pMbuf->_top]);
    ASSERT_EQ(0x02, pMbuf->_data[pMbuf->_top + 1]);
    ASSERT_EQ(0x03, pMbuf->_data[pMbuf->_top + 2]);

    // 0 サイズのトリムは何もしません。
    ASSERT_TRUE(nn::mbuf::MbufAdj(pMbuf, 0));
    ASSERT_EQ(0, pMbuf->_top);
    ASSERT_EQ(3, pMbuf->_len);
    ASSERT_EQ(DataSize, pMbuf->_capacity);
    ASSERT_EQ(0x01, pMbuf->_data[pMbuf->_top]);
    ASSERT_EQ(0x02, pMbuf->_data[pMbuf->_top + 1]);
    ASSERT_EQ(0x03, pMbuf->_data[pMbuf->_top + 2]);

    // 末尾から、空になるようなトリムを実行します。
    ASSERT_TRUE(nn::mbuf::MbufAdj(pMbuf, -3));
    ASSERT_EQ(0, pMbuf->_top);
    ASSERT_EQ(0, pMbuf->_len);
    ASSERT_EQ(DataSize, pMbuf->_capacity);
    ASSERT_EQ(DataSize, pMbuf->_capacity);

    // mbuf チェインを解放します。
    nn::mbuf::MbufFreem(pMbuf);
}

TEST_F(MbufAdj, TrimFrontFromMbufChain)
{
    // 3 クラスタから構成される mbuf チェインを確保します。
    nn::mbuf::Mbuf* pMbuf = nn::mbuf::MbufGetm(
        nullptr, DataSize * 3, nn::mbuf::MbufAllocationMode_DontWait, DefaultType);
    ASSERT_NE(nullptr, pMbuf);

    // 適当なデータを設定します。
    uint8_t data[DataSize * 2 + 2];
    for (size_t i = 0; i < sizeof(data); ++i)
    {
        data[i] = static_cast<uint8_t>(i + 1);
    }
    ASSERT_TRUE(nn::mbuf::MbufAppend(pMbuf, sizeof(data), data));

    // データサイズよりも大きなサイズのトリムはできません。
    int trimSize = 2 * DataSize + 3;
    ASSERT_FALSE(nn::mbuf::MbufAdj(pMbuf, trimSize));

    // 先頭から最小サイズのトリムを実行します。
    ASSERT_TRUE(nn::mbuf::MbufAdj(pMbuf, 1));
    ASSERT_EQ(1, pMbuf->_top);
    ASSERT_EQ(DataSize - 1, pMbuf->_len);
    ASSERT_EQ(DataSize, pMbuf->_capacity);
    ASSERT_EQ(0x02, pMbuf->_data[pMbuf->_top]);

    // 先頭から mbuf クラスタをちょうど超えるサイズのトリムを実行します。
    ASSERT_TRUE(nn::mbuf::MbufAdj(pMbuf, DataSize - 1));
    ASSERT_EQ(nullptr, pMbuf->_pPrev);
    ASSERT_EQ(0, pMbuf->_len);
    ASSERT_EQ(DataSize, pMbuf->_capacity);
    nn::mbuf::Mbuf* pNext = pMbuf->_pNext;
    ASSERT_NE(nullptr, pNext);
    ASSERT_EQ(pMbuf, pNext->_pPrev);
    ASSERT_EQ(0, pNext->_top);
    ASSERT_EQ(DataSize, pNext->_len);
    ASSERT_EQ(DataSize, pNext->_capacity);
    ASSERT_EQ(DataSize + 1, pNext->_data[pNext->_top]);
    nn::mbuf::Mbuf* pTail = pNext->_pNext;
    ASSERT_NE(nullptr, pTail);
    ASSERT_EQ(pNext, pTail->_pPrev);
    ASSERT_EQ(nullptr, pTail->_pNext);
    ASSERT_EQ(0, pTail->_top);
    ASSERT_EQ(2, pTail->_len);
    ASSERT_EQ(DataSize, pTail->_capacity);
    ASSERT_EQ(DataSize * 2 + 1, pTail->_data[pTail->_top]);

    // mbuf クラスタを超えるサイズのトリムを実行します。
    ASSERT_TRUE(nn::mbuf::MbufAdj(pMbuf, DataSize + 1));
    ASSERT_EQ(nullptr, pMbuf->_pPrev);
    ASSERT_EQ(0, pMbuf->_len);
    ASSERT_EQ(DataSize, pMbuf->_capacity);
    pNext = pMbuf->_pNext;
    ASSERT_NE(nullptr, pNext);
    ASSERT_EQ(pMbuf, pNext->_pPrev);
    ASSERT_EQ(0, pNext->_len);
    ASSERT_EQ(DataSize, pNext->_capacity);
    pTail = pNext->_pNext;
    ASSERT_NE(nullptr, pTail);
    ASSERT_EQ(pNext, pTail->_pPrev);
    ASSERT_EQ(nullptr, pTail->_pNext);
    ASSERT_EQ(1, pTail->_top);
    ASSERT_EQ(1, pTail->_len);
    ASSERT_EQ(DataSize, pTail->_capacity);
    ASSERT_EQ(DataSize * 2 + 2, pTail->_data[pTail->_top]);

    // データサイズよりも大きなサイズのトリムはできません。
    ASSERT_FALSE(nn::mbuf::MbufAdj(pMbuf, 2));

    // データが空になるトリムを実行します。
    ASSERT_TRUE(nn::mbuf::MbufAdj(pMbuf, 1));
    ASSERT_EQ(nullptr, pMbuf->_pPrev);
    ASSERT_EQ(0, pMbuf->_len);
    ASSERT_EQ(DataSize, pMbuf->_capacity);
    pNext = pMbuf->_pNext;
    ASSERT_NE(nullptr, pNext);
    ASSERT_EQ(pMbuf, pNext->_pPrev);
    ASSERT_EQ(0, pNext->_len);
    ASSERT_EQ(DataSize, pNext->_capacity);
    pTail = pNext->_pNext;
    ASSERT_NE(nullptr, pTail);
    ASSERT_EQ(pNext, pTail->_pPrev);
    ASSERT_EQ(nullptr, pTail->_pNext);
    ASSERT_EQ(2, pTail->_top);
    ASSERT_EQ(0, pTail->_len);
    ASSERT_EQ(DataSize, pTail->_capacity);

    // データサイズよりも大きなサイズのトリムはできません。
    ASSERT_FALSE(nn::mbuf::MbufAdj(pMbuf, 1));

    // mbuf チェインを解放します。
    nn::mbuf::MbufFreem(pMbuf);
}

TEST_F(MbufAdj, TrimBackFromMbufChain)
{
    // 3 クラスタから構成される mbuf チェインを確保します。
    nn::mbuf::Mbuf* pMbuf = nn::mbuf::MbufGetm(
        nullptr, DataSize * 3, nn::mbuf::MbufAllocationMode_DontWait, DefaultType);
    ASSERT_NE(nullptr, pMbuf);

    // 適当なデータを設定します。
    uint8_t data[DataSize * 2 + 2];
    for (size_t i = 0; i < sizeof(data); ++i)
    {
        data[i] = static_cast<uint8_t>(i + 1);
    }
    ASSERT_TRUE(nn::mbuf::MbufAppend(pMbuf, sizeof(data), data));

    // データサイズよりも大きなサイズのトリムはできません。
    int trimSize = 2 * DataSize + 3;
    ASSERT_FALSE(nn::mbuf::MbufAdj(pMbuf, -trimSize));

    // 末尾から最小サイズのトリムを実行します。
    ASSERT_TRUE(nn::mbuf::MbufAdj(pMbuf, -1));
    ASSERT_EQ(nullptr, pMbuf->_pPrev);
    ASSERT_EQ(0, pMbuf->_top);
    ASSERT_EQ(DataSize, pMbuf->_len);
    ASSERT_EQ(DataSize, pMbuf->_capacity);
    ASSERT_EQ(0x01, pMbuf->_data[pMbuf->_top]);
    nn::mbuf::Mbuf* pNext = pMbuf->_pNext;
    ASSERT_NE(nullptr, pNext);
    ASSERT_EQ(pMbuf, pNext->_pPrev);
    ASSERT_EQ(0, pNext->_top);
    ASSERT_EQ(DataSize, pNext->_len);
    ASSERT_EQ(DataSize, pNext->_capacity);
    ASSERT_EQ(DataSize + 1, pNext->_data[pNext->_top]);
    nn::mbuf::Mbuf* pTail = pNext->_pNext;
    ASSERT_NE(nullptr, pTail);
    ASSERT_EQ(pNext, pTail->_pPrev);
    ASSERT_EQ(0, pTail->_top);
    ASSERT_EQ(1, pTail->_len);
    ASSERT_EQ(DataSize, pTail->_capacity);
    ASSERT_EQ(DataSize * 2 + 1, pTail->_data[pTail->_top]);

    // 末尾から mbuf クラスタが除去される最小のサイズのトリムを実行します。
    ASSERT_TRUE(nn::mbuf::MbufAdj(pMbuf, -1));
    ASSERT_EQ(nullptr, pMbuf->_pPrev);
    ASSERT_EQ(0, pMbuf->_top);
    ASSERT_EQ(DataSize, pMbuf->_len);
    ASSERT_EQ(DataSize, pMbuf->_capacity);
    ASSERT_EQ(0x01, pMbuf->_data[pMbuf->_top]);
    pNext = pMbuf->_pNext;
    ASSERT_NE(nullptr, pNext);
    ASSERT_EQ(pMbuf, pNext->_pPrev);
    ASSERT_EQ(nullptr, pNext->_pNext);
    ASSERT_EQ(0, pNext->_top);
    ASSERT_EQ(DataSize, pNext->_len);
    ASSERT_EQ(DataSize, pNext->_capacity);
    ASSERT_EQ(DataSize + 1, pNext->_data[pNext->_top]);

    // mbuf クラスタが除去されるサイズのトリムを実行します。
    trimSize = DataSize + 1;
    ASSERT_TRUE(nn::mbuf::MbufAdj(pMbuf, -trimSize));
    ASSERT_EQ(nullptr, pMbuf->_pPrev);
    ASSERT_EQ(nullptr, pMbuf->_pNext);
    ASSERT_EQ(0, pMbuf->_top);
    ASSERT_EQ(DataSize - 1, pMbuf->_len);
    ASSERT_EQ(DataSize, pMbuf->_capacity);
    ASSERT_EQ(0x01, pMbuf->_data[pMbuf->_top]);

    // データサイズよりも大きなサイズのトリムはできません。
    trimSize = DataSize;
    ASSERT_FALSE(nn::mbuf::MbufAdj(pMbuf, -trimSize));

    // データが空になるトリムを実行します。
    trimSize = DataSize - 1;
    ASSERT_TRUE(nn::mbuf::MbufAdj(pMbuf, -trimSize));
    ASSERT_EQ(nullptr, pMbuf->_pPrev);
    ASSERT_EQ(nullptr, pMbuf->_pNext);
    ASSERT_EQ(0, pMbuf->_top);
    ASSERT_EQ(0, pMbuf->_len);
    ASSERT_EQ(DataSize, pMbuf->_capacity);

    // mbuf チェインを解放します。
    nn::mbuf::MbufFreem(pMbuf);
}

TEST_F(MbufDup, DontWaitEmptyMbuf)
{
    // 空の mbuf を用意します。
    nn::mbuf::Mbuf* pMbuf = nn::mbuf::MbufGetm(
        nullptr, DataSize, nn::mbuf::MbufAllocationMode_DontWait, DefaultType);
    ASSERT_NE(nullptr, pMbuf);

    // 上記の mbuf を除く残りのプール容量を計算します。
    const int capacity = PoolCapacity - 1;

    // DontWait モードで複製します。
    nn::mbuf::Mbuf* mbufs[capacity];
    for (int i = 0; i < capacity; ++i)
    {
        mbufs[i] = nn::mbuf::MbufDup(pMbuf, nn::mbuf::MbufAllocationMode_DontWait);
        ASSERT_NE(nullptr, mbufs[i]);
        ASSERT_EQ(nullptr, mbufs[i]->_pPrev);
        ASSERT_EQ(nullptr, mbufs[i]->_pNext);
        ASSERT_EQ(nullptr, mbufs[i]->_pGeneral);
        ASSERT_EQ(0, mbufs[i]->_top);
        ASSERT_EQ(0, mbufs[i]->_len);
        ASSERT_EQ(DataSize, mbufs[i]->_capacity);
        ASSERT_EQ(0, mbufs[i]->_flags);
        ASSERT_EQ(0, mbufs[i]->_protocol);
        ASSERT_EQ(0, mbufs[i]->_owner);
        ASSERT_EQ(DefaultType, mbufs[i]->_type);
        ASSERT_EQ(DefaultType, mbufs[i]->_pool);
    }

    // プールに空きがないため失敗します。
    nn::mbuf::Mbuf* pDup = nn::mbuf::MbufDup(pMbuf, nn::mbuf::MbufAllocationMode_DontWait);
    ASSERT_EQ(nullptr, pDup);

    // mbuf を解放します。
    for (int i = 0; i < capacity; ++i)
    {
        nn::mbuf::MbufFreem(mbufs[i]);
    }
    nn::mbuf::MbufFreem(pMbuf);
}

TEST_F(MbufDup, WaitEmptyMbuf)
{
    // 空の mbuf を用意します。
    nn::mbuf::Mbuf* pMbuf = nn::mbuf::MbufGetm(
        nullptr, DataSize, nn::mbuf::MbufAllocationMode_Wait, DefaultType);
    ASSERT_NE(nullptr, pMbuf);

    // 上記の mbuf を除く残りのプール容量を計算します。
    const int capacity = PoolCapacity - 1;

    // DontWait モードで複製します。
    nn::mbuf::Mbuf* mbufs[capacity];
    for (int i = 0; i < capacity; ++i)
    {
        mbufs[i] = nn::mbuf::MbufDup(pMbuf, nn::mbuf::MbufAllocationMode_Wait);
        ASSERT_NE(nullptr, mbufs[i]);
        ASSERT_EQ(nullptr, mbufs[i]->_pPrev);
        ASSERT_EQ(nullptr, mbufs[i]->_pNext);
        ASSERT_EQ(nullptr, mbufs[i]->_pGeneral);
        ASSERT_EQ(0, mbufs[i]->_top);
        ASSERT_EQ(0, mbufs[i]->_len);
        ASSERT_EQ(DataSize, mbufs[i]->_capacity);
        ASSERT_EQ(0, mbufs[i]->_flags);
        ASSERT_EQ(0, mbufs[i]->_protocol);
        ASSERT_EQ(0, mbufs[i]->_owner);
        ASSERT_EQ(DefaultType, mbufs[i]->_type);
        ASSERT_EQ(DefaultType, mbufs[i]->_pool);
    }

    // mbuf を解放します。
    for (int i = 0; i < capacity; ++i)
    {
        nn::mbuf::MbufFreem(mbufs[i]);
    }
    nn::mbuf::MbufFreem(pMbuf);
}

TEST_F(MbufDup, DontWaitSingleMbuf)
{
    // 空の mbuf を用意します。
    nn::mbuf::Mbuf* pMbuf = nn::mbuf::MbufGetm(
        nullptr, DataSize, nn::mbuf::MbufAllocationMode_DontWait, DefaultType);
    ASSERT_NE(nullptr, pMbuf);

    // 小さなデータを追加します。
    uint8_t data[4] = { 1, 2, 3, 4};
    ASSERT_TRUE(nn::mbuf::MbufAppend(pMbuf, sizeof(data), &data));
    ASSERT_TRUE(nn::mbuf::MbufAdj(pMbuf, 1));
    ASSERT_TRUE(nn::mbuf::MbufAdj(pMbuf, -1));
    ASSERT_EQ(1, pMbuf->_top);
    ASSERT_EQ(2, pMbuf->_len);
    ASSERT_EQ(2, pMbuf->_data[pMbuf->_top]);
    ASSERT_EQ(3, pMbuf->_data[pMbuf->_top + 1]);

    // 上記の mbuf を除く残りのプール容量を計算します。
    const int capacity = PoolCapacity - 1;

    // DontWait モードで複製します。
    nn::mbuf::Mbuf* mbufs[capacity];
    for (int i = 0; i < capacity; ++i)
    {
        mbufs[i] = nn::mbuf::MbufDup(pMbuf, nn::mbuf::MbufAllocationMode_DontWait);
        ASSERT_NE(nullptr, mbufs[i]);
        ASSERT_EQ(nullptr, mbufs[i]->_pPrev);
        ASSERT_EQ(nullptr, mbufs[i]->_pNext);
        ASSERT_EQ(nullptr, mbufs[i]->_pGeneral);
        ASSERT_EQ(1, mbufs[i]->_top);
        ASSERT_EQ(2, mbufs[i]->_len);
        ASSERT_EQ(DataSize, mbufs[i]->_capacity);
        ASSERT_EQ(0, mbufs[i]->_flags);
        ASSERT_EQ(0, mbufs[i]->_protocol);
        ASSERT_EQ(0, mbufs[i]->_owner);
        ASSERT_EQ(DefaultType, mbufs[i]->_type);
        ASSERT_EQ(DefaultType, mbufs[i]->_pool);
        ASSERT_EQ(2, mbufs[i]->_data[mbufs[i]->_top]);
        ASSERT_EQ(3, mbufs[i]->_data[mbufs[i]->_top + 1]);
    }

    // プールに空きがないため失敗します。
    nn::mbuf::Mbuf* pDup = nn::mbuf::MbufDup(pMbuf, nn::mbuf::MbufAllocationMode_DontWait);
    ASSERT_EQ(nullptr, pDup);

    // mbuf を解放します。
    for (int i = 0; i < capacity; ++i)
    {
        nn::mbuf::MbufFreem(mbufs[i]);
    }
    nn::mbuf::MbufFreem(pMbuf);
}

TEST_F(MbufDup, WaitSingleMbuf)
{
    // 空の mbuf を用意します。
    nn::mbuf::Mbuf* pMbuf = nn::mbuf::MbufGetm(
        nullptr, DataSize, nn::mbuf::MbufAllocationMode_DontWait, DefaultType);
    ASSERT_NE(nullptr, pMbuf);

    // 小さなデータを追加します。
    uint8_t data[4] = { 1, 2, 3, 4};
    ASSERT_TRUE(nn::mbuf::MbufAppend(pMbuf, sizeof(data), &data));
    ASSERT_TRUE(nn::mbuf::MbufAdj(pMbuf, 1));
    ASSERT_TRUE(nn::mbuf::MbufAdj(pMbuf, -1));
    ASSERT_EQ(1, pMbuf->_top);
    ASSERT_EQ(2, pMbuf->_len);
    ASSERT_EQ(2, pMbuf->_data[pMbuf->_top]);
    ASSERT_EQ(3, pMbuf->_data[pMbuf->_top + 1]);

    // 上記の mbuf を除く残りのプール容量を計算します。
    const int capacity = PoolCapacity - 1;

    // DontWait モードで複製します。
    nn::mbuf::Mbuf* mbufs[capacity];
    for (int i = 0; i < capacity; ++i)
    {
        mbufs[i] = nn::mbuf::MbufDup(pMbuf, nn::mbuf::MbufAllocationMode_Wait);
        ASSERT_NE(nullptr, mbufs[i]);
        ASSERT_EQ(nullptr, mbufs[i]->_pPrev);
        ASSERT_EQ(nullptr, mbufs[i]->_pNext);
        ASSERT_EQ(nullptr, mbufs[i]->_pGeneral);
        ASSERT_EQ(1, mbufs[i]->_top);
        ASSERT_EQ(2, mbufs[i]->_len);
        ASSERT_EQ(DataSize, mbufs[i]->_capacity);
        ASSERT_EQ(0, mbufs[i]->_flags);
        ASSERT_EQ(0, mbufs[i]->_protocol);
        ASSERT_EQ(0, mbufs[i]->_owner);
        ASSERT_EQ(DefaultType, mbufs[i]->_type);
        ASSERT_EQ(DefaultType, mbufs[i]->_pool);
        ASSERT_EQ(2, mbufs[i]->_data[mbufs[i]->_top]);
        ASSERT_EQ(3, mbufs[i]->_data[mbufs[i]->_top + 1]);
    }

    // mbuf を解放します。
    for (int i = 0; i < capacity; ++i)
    {
        nn::mbuf::MbufFreem(mbufs[i]);
    }
    nn::mbuf::MbufFreem(pMbuf);
}

TEST_F(MbufDup, DontWaitMbufChain)
{
    // mbuf チェインを用意します。
    nn::mbuf::Mbuf* pMbuf = nn::mbuf::MbufGetm(
        nullptr, DataSize * 2 + 1, nn::mbuf::MbufAllocationMode_DontWait, DefaultType);
    ASSERT_NE(nullptr, pMbuf);

    // データを追加します。
    uint8_t data[DataSize + 1];
    for (size_t i = 0; i < sizeof(data); ++i)
    {
        data[i] = static_cast<uint8_t>(i + 1);
    }
    ASSERT_TRUE(nn::mbuf::MbufAppend(pMbuf, sizeof(data), data));

    // 上記の mbuf を除く残りのプール容量を計算します。
    const int capacity = (PoolCapacity / 3) - 1;

    // DontWait モードで複製します。
    nn::mbuf::Mbuf* mbufs[capacity];
    for (int i = 0; i < capacity; ++i)
    {

        mbufs[i] = nn::mbuf::MbufDup(pMbuf, nn::mbuf::MbufAllocationMode_DontWait);
        ASSERT_NE(nullptr, mbufs[i]);
        ASSERT_EQ(nullptr, mbufs[i]->_pPrev);
        ASSERT_NE(nullptr, mbufs[i]->_pNext);
        ASSERT_EQ(nullptr, mbufs[i]->_pGeneral);
        ASSERT_EQ(0, mbufs[i]->_top);
        ASSERT_EQ(DataSize, mbufs[i]->_len);
        ASSERT_EQ(DataSize, mbufs[i]->_capacity);
        ASSERT_EQ(0, mbufs[i]->_flags);
        ASSERT_EQ(0, mbufs[i]->_protocol);
        ASSERT_EQ(0, mbufs[i]->_owner);
        ASSERT_EQ(DefaultType, mbufs[i]->_type);
        ASSERT_EQ(DefaultType, mbufs[i]->_pool);
        ASSERT_EQ(1, mbufs[i]->_data[mbufs[i]->_top]);
        ASSERT_EQ(DataSize, mbufs[i]->_data[mbufs[i]->_top + mbufs[i]->_len - 1]);
        nn::mbuf::Mbuf* pNext = mbufs[i]->_pNext;
        ASSERT_NE(nullptr, pNext);
        ASSERT_EQ(mbufs[i], pNext->_pPrev);
        ASSERT_NE(nullptr, pNext->_pNext);
        ASSERT_EQ(nullptr, pNext->_pGeneral);
        ASSERT_EQ(0, pNext->_top);
        ASSERT_EQ(1, pNext->_len);
        ASSERT_EQ(DataSize, pNext->_capacity);
        ASSERT_EQ(0, pNext->_flags);
        ASSERT_EQ(0, pNext->_protocol);
        ASSERT_EQ(0, pNext->_owner);
        ASSERT_EQ(DefaultType, pNext->_type);
        ASSERT_EQ(DefaultType, pNext->_pool);
        ASSERT_EQ(DataSize + 1, pNext->_data[pNext->_top]);
        nn::mbuf::Mbuf* pTail = pNext->_pNext;
        ASSERT_NE(nullptr, pTail);
        ASSERT_EQ(pNext, pTail->_pPrev);
        ASSERT_EQ(nullptr, pTail->_pNext);
        ASSERT_EQ(nullptr, pTail->_pGeneral);
        ASSERT_EQ(0, pTail->_top);
        ASSERT_EQ(0, pTail->_len);
        ASSERT_EQ(DataSize, pNext->_capacity);
        ASSERT_EQ(0, pTail->_flags);
        ASSERT_EQ(0, pTail->_protocol);
        ASSERT_EQ(0, pTail->_owner);
        ASSERT_EQ(DefaultType, pTail->_type);
        ASSERT_EQ(DefaultType, pTail->_pool);
    }

    // プールに空きがないため失敗します。
    nn::mbuf::Mbuf* pDup = nn::mbuf::MbufDup(pMbuf, nn::mbuf::MbufAllocationMode_DontWait);
    ASSERT_EQ(nullptr, pDup);

    // mbuf を解放します。
    for (int i = 0; i < capacity; ++i)
    {
        nn::mbuf::MbufFreem(mbufs[i]);
    }
    nn::mbuf::MbufFreem(pMbuf);
}

TEST_F(MbufDup, WaitMbufChain)
{
    // mbuf チェインを用意します。
    nn::mbuf::Mbuf* pMbuf = nn::mbuf::MbufGetm(
        nullptr, DataSize * 2 + 1, nn::mbuf::MbufAllocationMode_Wait, DefaultType);
    ASSERT_NE(nullptr, pMbuf);

    // データを追加します。
    uint8_t data[DataSize + 1];
    for (size_t i = 0; i < sizeof(data); ++i)
    {
        data[i] = static_cast<uint8_t>(i + 1);
    }
    ASSERT_TRUE(nn::mbuf::MbufAppend(pMbuf, sizeof(data), data));

    // 上記の mbuf を除く残りのプール容量を計算します。
    const int capacity = (PoolCapacity / 3) - 1;

    // DontWait モードで複製します。
    nn::mbuf::Mbuf* mbufs[capacity];
    for (int i = 0; i < capacity; ++i)
    {

        mbufs[i] = nn::mbuf::MbufDup(pMbuf, nn::mbuf::MbufAllocationMode_Wait);
        ASSERT_NE(nullptr, mbufs[i]);
        ASSERT_EQ(nullptr, mbufs[i]->_pPrev);
        ASSERT_NE(nullptr, mbufs[i]->_pNext);
        ASSERT_EQ(nullptr, mbufs[i]->_pGeneral);
        ASSERT_EQ(0, mbufs[i]->_top);
        ASSERT_EQ(DataSize, mbufs[i]->_len);
        ASSERT_EQ(DataSize, mbufs[i]->_capacity);
        ASSERT_EQ(0, mbufs[i]->_flags);
        ASSERT_EQ(0, mbufs[i]->_protocol);
        ASSERT_EQ(0, mbufs[i]->_owner);
        ASSERT_EQ(DefaultType, mbufs[i]->_type);
        ASSERT_EQ(DefaultType, mbufs[i]->_pool);
        ASSERT_EQ(1, mbufs[i]->_data[mbufs[i]->_top]);
        ASSERT_EQ(DataSize, mbufs[i]->_data[mbufs[i]->_top + mbufs[i]->_len - 1]);
        nn::mbuf::Mbuf* pNext = mbufs[i]->_pNext;
        ASSERT_NE(nullptr, pNext);
        ASSERT_EQ(mbufs[i], pNext->_pPrev);
        ASSERT_NE(nullptr, pNext->_pNext);
        ASSERT_EQ(nullptr, pNext->_pGeneral);
        ASSERT_EQ(0, pNext->_top);
        ASSERT_EQ(1, pNext->_len);
        ASSERT_EQ(DataSize, pNext->_capacity);
        ASSERT_EQ(0, pNext->_flags);
        ASSERT_EQ(0, pNext->_protocol);
        ASSERT_EQ(0, pNext->_owner);
        ASSERT_EQ(DefaultType, pNext->_type);
        ASSERT_EQ(DefaultType, pNext->_pool);
        ASSERT_EQ(DataSize + 1, pNext->_data[pNext->_top]);
        nn::mbuf::Mbuf* pTail = pNext->_pNext;
        ASSERT_NE(nullptr, pTail);
        ASSERT_EQ(pNext, pTail->_pPrev);
        ASSERT_EQ(nullptr, pTail->_pNext);
        ASSERT_EQ(nullptr, pTail->_pGeneral);
        ASSERT_EQ(0, pTail->_top);
        ASSERT_EQ(0, pTail->_len);
        ASSERT_EQ(DataSize, pTail->_capacity);
        ASSERT_EQ(0, pTail->_flags);
        ASSERT_EQ(0, pTail->_protocol);
        ASSERT_EQ(0, pTail->_owner);
        ASSERT_EQ(DefaultType, pTail->_type);
        ASSERT_EQ(DefaultType, pTail->_pool);
    }

    // mbuf を解放します。
    for (int i = 0; i < capacity; ++i)
    {
        nn::mbuf::MbufFreem(mbufs[i]);
    }
    nn::mbuf::MbufFreem(pMbuf);
}

TEST_F(MbufReserve, Mbuf)
{
    // mbuf を用意します。
    nn::mbuf::Mbuf* pMbuf = nn::mbuf::MbufGetm(
        nullptr, DataSize, nn::mbuf::MbufAllocationMode_DontWait, DefaultType);
    ASSERT_NE(nullptr, pMbuf);

    // 最小サイズの予約です。
    nn::mbuf::MbufReserve(pMbuf, 1);
    ASSERT_EQ(nullptr, pMbuf->_pPrev);
    ASSERT_EQ(nullptr, pMbuf->_pNext);
    ASSERT_EQ(nullptr, pMbuf->_pGeneral);
    ASSERT_EQ(1, pMbuf->_top);
    ASSERT_EQ(0, pMbuf->_len);
    ASSERT_EQ(DataSize, pMbuf->_capacity);
    ASSERT_EQ(0, pMbuf->_flags);
    ASSERT_EQ(0, pMbuf->_protocol);
    ASSERT_EQ(0, pMbuf->_owner);
    ASSERT_EQ(DefaultType, pMbuf->_type);
    ASSERT_EQ(DefaultType, pMbuf->_pool);

    // 最大サイズの予約です。
    nn::mbuf::MbufReserve(pMbuf, DataSize - 1);
    ASSERT_EQ(nullptr, pMbuf->_pPrev);
    ASSERT_EQ(nullptr, pMbuf->_pNext);
    ASSERT_EQ(nullptr, pMbuf->_pGeneral);
    ASSERT_EQ(DataSize, pMbuf->_top);
    ASSERT_EQ(0, pMbuf->_len);
    ASSERT_EQ(DataSize, pMbuf->_capacity);
    ASSERT_EQ(0, pMbuf->_flags);
    ASSERT_EQ(0, pMbuf->_protocol);
    ASSERT_EQ(0, pMbuf->_owner);
    ASSERT_EQ(DefaultType, pMbuf->_type);
    ASSERT_EQ(DefaultType, pMbuf->_pool);

    // mbuf を解放します。
    nn::mbuf::MbufFreem(pMbuf);
}

TEST_F(MbufReserve, MbufChain)
{
    // mbuf チェインを用意します。
    nn::mbuf::Mbuf* pNext;
    nn::mbuf::Mbuf* pMbuf = nn::mbuf::MbufGetm(
        nullptr, DataSize * 2, nn::mbuf::MbufAllocationMode_DontWait, DefaultType);
    ASSERT_NE(nullptr, pMbuf);

    // 最小サイズの予約です。
    nn::mbuf::MbufReserve(pMbuf, 1);
    ASSERT_EQ(nullptr, pMbuf->_pPrev);
    ASSERT_NE(nullptr, pMbuf->_pNext);
    ASSERT_EQ(nullptr, pMbuf->_pGeneral);
    ASSERT_EQ(1, pMbuf->_top);
    ASSERT_EQ(0, pMbuf->_len);
    ASSERT_EQ(DataSize, pMbuf->_capacity);
    ASSERT_EQ(0, pMbuf->_flags);
    ASSERT_EQ(0, pMbuf->_protocol);
    ASSERT_EQ(0, pMbuf->_owner);
    ASSERT_EQ(DefaultType, pMbuf->_type);
    ASSERT_EQ(DefaultType, pMbuf->_pool);
    pNext = pMbuf->_pNext;
    ASSERT_NE(nullptr, pNext);
    ASSERT_EQ(pMbuf, pNext->_pPrev);
    ASSERT_EQ(nullptr, pNext->_pNext);
    ASSERT_EQ(nullptr, pNext->_pGeneral);
    ASSERT_EQ(0, pNext->_top);
    ASSERT_EQ(0, pNext->_len);
    ASSERT_EQ(DataSize, pNext->_capacity);
    ASSERT_EQ(0, pNext->_flags);
    ASSERT_EQ(0, pNext->_protocol);
    ASSERT_EQ(0, pNext->_owner);
    ASSERT_EQ(DefaultType, pNext->_type);
    ASSERT_EQ(DefaultType, pNext->_pool);

    // 最大サイズの予約です。
    nn::mbuf::MbufReserve(pMbuf, DataSize - 1);
    ASSERT_EQ(nullptr, pMbuf->_pPrev);
    ASSERT_NE(nullptr, pMbuf->_pNext);
    ASSERT_EQ(nullptr, pMbuf->_pGeneral);
    ASSERT_EQ(DataSize, pMbuf->_top);
    ASSERT_EQ(0, pMbuf->_len);
    ASSERT_EQ(DataSize, pMbuf->_capacity);
    ASSERT_EQ(0, pMbuf->_flags);
    ASSERT_EQ(0, pMbuf->_protocol);
    ASSERT_EQ(0, pMbuf->_owner);
    ASSERT_EQ(DefaultType, pMbuf->_type);
    ASSERT_EQ(DefaultType, pMbuf->_pool);
    pNext = pMbuf->_pNext;
    ASSERT_NE(nullptr, pNext);
    ASSERT_EQ(pMbuf, pNext->_pPrev);
    ASSERT_EQ(nullptr, pNext->_pNext);
    ASSERT_EQ(nullptr, pNext->_pGeneral);
    ASSERT_EQ(0, pNext->_top);
    ASSERT_EQ(0, pNext->_len);
    ASSERT_EQ(DataSize, pNext->_capacity);
    ASSERT_EQ(0, pNext->_flags);
    ASSERT_EQ(0, pNext->_protocol);
    ASSERT_EQ(0, pNext->_owner);
    ASSERT_EQ(DefaultType, pNext->_type);
    ASSERT_EQ(DefaultType, pNext->_pool);

    // mbuf を解放します。
    nn::mbuf::MbufFreem(pMbuf);
}

TEST_F(MbufPrepend, PrependMbufWithoutAllocation)
{
    // mbuf を用意します。
    nn::mbuf::Mbuf* pMbuf = nn::mbuf::MbufGetm(
        nullptr, DataSize, nn::mbuf::MbufAllocationMode_DontWait, DefaultType);
    ASSERT_NE(nullptr, pMbuf);

    // 予約します。
    nn::mbuf::MbufReserve(pMbuf, DataSize);

    // 予約された領域の一部を利用します。
    nn::mbuf::Mbuf* pNew;
    pNew = nn::mbuf::MbufPrepend(pMbuf, 1, nn::mbuf::MbufAllocationMode_Wait);
    ASSERT_EQ(pMbuf, pNew);
    ASSERT_EQ(nullptr, pMbuf->_pPrev);
    ASSERT_EQ(nullptr, pMbuf->_pNext);
    ASSERT_EQ(nullptr, pMbuf->_pGeneral);
    ASSERT_EQ(DataSize - 1, pMbuf->_top);
    ASSERT_EQ(1, pMbuf->_len);
    ASSERT_EQ(DataSize, pMbuf->_capacity);
    ASSERT_EQ(0, pMbuf->_flags);
    ASSERT_EQ(0, pMbuf->_protocol);
    ASSERT_EQ(0, pMbuf->_owner);
    ASSERT_EQ(DefaultType, pMbuf->_type);
    ASSERT_EQ(DefaultType, pMbuf->_pool);

    // 予約された領域を全て利用します。
    pNew = nn::mbuf::MbufPrepend(pMbuf, DataSize - 1, nn::mbuf::MbufAllocationMode_DontWait);
    ASSERT_EQ(pMbuf, pNew);
    ASSERT_EQ(nullptr, pMbuf->_pPrev);
    ASSERT_EQ(nullptr, pMbuf->_pNext);
    ASSERT_EQ(nullptr, pMbuf->_pGeneral);
    ASSERT_EQ(0, pMbuf->_top);
    ASSERT_EQ(DataSize, pMbuf->_len);
    ASSERT_EQ(DataSize, pMbuf->_capacity);
    ASSERT_EQ(0, pMbuf->_flags);
    ASSERT_EQ(0, pMbuf->_protocol);
    ASSERT_EQ(0, pMbuf->_owner);
    ASSERT_EQ(DefaultType, pMbuf->_type);
    ASSERT_EQ(DefaultType, pMbuf->_pool);

    // mbuf を解放します。
    nn::mbuf::MbufFreem(pMbuf);
}

TEST_F(MbufPrepend, PrependMbufChainWithoutAllocation)
{
    // mbuf チェインを用意します。
    nn::mbuf::Mbuf* pNext;
    nn::mbuf::Mbuf* pMbuf = nn::mbuf::MbufGetm(
        nullptr, DataSize + 1, nn::mbuf::MbufAllocationMode_DontWait, DefaultType);
    ASSERT_NE(nullptr, pMbuf);

    // 予約します。
    nn::mbuf::MbufReserve(pMbuf, DataSize);

    // 予約された領域の一部を利用します。
    nn::mbuf::Mbuf* pNew;
    pNew = nn::mbuf::MbufPrepend(pMbuf, 1, nn::mbuf::MbufAllocationMode_DontWait);
    ASSERT_EQ(pMbuf, pNew);
    ASSERT_EQ(nullptr, pMbuf->_pPrev);
    ASSERT_NE(nullptr, pMbuf->_pNext);
    ASSERT_EQ(nullptr, pMbuf->_pGeneral);
    ASSERT_EQ(DataSize - 1, pMbuf->_top);
    ASSERT_EQ(1, pMbuf->_len);
    ASSERT_EQ(DataSize, pMbuf->_capacity);
    ASSERT_EQ(0, pMbuf->_flags);
    ASSERT_EQ(0, pMbuf->_protocol);
    ASSERT_EQ(0, pMbuf->_owner);
    ASSERT_EQ(DefaultType, pMbuf->_type);
    ASSERT_EQ(DefaultType, pMbuf->_pool);
    pNext = pMbuf->_pNext;
    ASSERT_NE(nullptr, pNext);
    ASSERT_EQ(pMbuf, pNext->_pPrev);
    ASSERT_EQ(nullptr, pNext->_pNext);
    ASSERT_EQ(nullptr, pNext->_pGeneral);
    ASSERT_EQ(0, pNext->_top);
    ASSERT_EQ(0, pNext->_len);
    ASSERT_EQ(DataSize, pNext->_capacity);
    ASSERT_EQ(0, pNext->_flags);
    ASSERT_EQ(0, pNext->_protocol);
    ASSERT_EQ(0, pNext->_owner);
    ASSERT_EQ(DefaultType, pNext->_type);
    ASSERT_EQ(DefaultType, pNext->_pool);

    // 予約された領域を全て利用します。
    pNew = nn::mbuf::MbufPrepend(pMbuf, DataSize - 1, nn::mbuf::MbufAllocationMode_Wait);
    ASSERT_EQ(pMbuf, pNew);
    ASSERT_EQ(nullptr, pMbuf->_pPrev);
    ASSERT_NE(nullptr, pMbuf->_pNext);
    ASSERT_EQ(nullptr, pMbuf->_pGeneral);
    ASSERT_EQ(0, pMbuf->_top);
    ASSERT_EQ(DataSize, pMbuf->_len);
    ASSERT_EQ(DataSize, pMbuf->_capacity);
    ASSERT_EQ(0, pMbuf->_flags);
    ASSERT_EQ(0, pMbuf->_protocol);
    ASSERT_EQ(0, pMbuf->_owner);
    ASSERT_EQ(DefaultType, pMbuf->_type);
    ASSERT_EQ(DefaultType, pMbuf->_pool);
    pNext = pMbuf->_pNext;
    ASSERT_NE(nullptr, pNext);
    ASSERT_EQ(pMbuf, pNext->_pPrev);
    ASSERT_EQ(nullptr, pNext->_pNext);
    ASSERT_EQ(nullptr, pNext->_pGeneral);
    ASSERT_EQ(0, pNext->_top);
    ASSERT_EQ(0, pNext->_len);
    ASSERT_EQ(DataSize, pNext->_capacity);
    ASSERT_EQ(0, pNext->_flags);
    ASSERT_EQ(0, pNext->_protocol);
    ASSERT_EQ(0, pNext->_owner);
    ASSERT_EQ(DefaultType, pNext->_type);
    ASSERT_EQ(DefaultType, pNext->_pool);

    // mbuf を解放します。
    nn::mbuf::MbufFreem(pMbuf);
}

TEST_F(MbufPrepend, PrependMbufWithAllocation)
{
    // mbuf を用意します。
    nn::mbuf::Mbuf* pMbuf = nn::mbuf::MbufGetm(
        nullptr, DataSize, nn::mbuf::MbufAllocationMode_DontWait, DefaultType);
    ASSERT_NE(nullptr, pMbuf);

    // 予約します。
    nn::mbuf::MbufReserve(pMbuf, 1);

    // 新しい mbuf を先頭に追加します。
    nn::mbuf::Mbuf* pNew;
    pNew = nn::mbuf::MbufPrepend(pMbuf, 2, nn::mbuf::MbufAllocationMode_DontWait);
    ASSERT_NE(nullptr, pNew);
    ASSERT_NE(pMbuf, pNew);
    ASSERT_EQ(nullptr, pNew->_pPrev);
    ASSERT_EQ(pMbuf, pNew->_pNext);
    ASSERT_EQ(nullptr, pNew->_pGeneral);
    ASSERT_EQ(0, pNew->_top);
    ASSERT_EQ(2, pNew->_len);
    ASSERT_EQ(DataSize, pNew->_capacity);
    ASSERT_EQ(0, pNew->_flags);
    ASSERT_EQ(0, pNew->_protocol);
    ASSERT_EQ(0, pNew->_owner);
    ASSERT_EQ(DefaultType, pNew->_type);
    ASSERT_EQ(DefaultType, pNew->_pool);
    ASSERT_EQ(pNew, pMbuf->_pPrev);
    ASSERT_EQ(nullptr, pMbuf->_pNext);
    ASSERT_EQ(nullptr, pMbuf->_pGeneral);
    ASSERT_EQ(1, pMbuf->_top);
    ASSERT_EQ(0, pMbuf->_len);
    ASSERT_EQ(DataSize, pMbuf->_capacity);
    ASSERT_EQ(0, pMbuf->_flags);
    ASSERT_EQ(0, pMbuf->_protocol);
    ASSERT_EQ(0, pMbuf->_owner);
    ASSERT_EQ(DefaultType, pMbuf->_type);
    ASSERT_EQ(DefaultType, pMbuf->_pool);

    // 容量が無くなるまで Prepend します。
    for (int i = 0; i < PoolCapacity - 2; ++i)
    {
        nn::mbuf::Mbuf* pNext = pNew;
        pNew = nn::mbuf::MbufPrepend(pNew, 2, nn::mbuf::MbufAllocationMode_Wait);
        ASSERT_NE(nullptr, pNew);
        ASSERT_EQ(nullptr, pNew->_pPrev);
        ASSERT_EQ(pNext, pNew->_pNext);
        ASSERT_EQ(nullptr, pNew->_pGeneral);   ASSERT_EQ(0, pNew->_top);
        ASSERT_EQ(2, pNew->_len);
        ASSERT_EQ(DataSize, pNew->_capacity);
        ASSERT_EQ(0, pNew->_flags);
        ASSERT_EQ(0, pNew->_protocol);
        ASSERT_EQ(0, pNew->_owner);
        ASSERT_EQ(DefaultType, pNew->_type);
        ASSERT_EQ(DefaultType, pNew->_pool);
        ASSERT_NE(nullptr, pNext);
        ASSERT_EQ(pNew, pNext->_pPrev);
        ASSERT_NE(nullptr, pNext->_pNext);
        ASSERT_EQ(nullptr, pNext->_pGeneral);
        ASSERT_EQ(0, pNext->_top);
        ASSERT_EQ(2, pNext->_len);
        ASSERT_EQ(DataSize, pNext->_capacity);
        ASSERT_EQ(0, pNext->_flags);
        ASSERT_EQ(0, pNext->_protocol);
        ASSERT_EQ(0, pNext->_owner);
        ASSERT_EQ(DefaultType, pNext->_type);
        ASSERT_EQ(DefaultType, pNext->_pool);
    }

    // プールに空きがない場合は失敗します。
    pMbuf = nn::mbuf::MbufPrepend(pNew, 2, nn::mbuf::MbufAllocationMode_DontWait);
    ASSERT_EQ(nullptr, pMbuf);

    // mbuf を解放します。
    nn::mbuf::MbufFreem(pNew);
}
