﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/hid.h>
#include <nn/ldn.h>
#include <nn/os.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/socket/socket_Api.h>
#include <nn/socket/socket_Types.h>
#include <nnt/nnt_Argument.h>
#include "../Common/Definition.h"
#include "../Common/HidController.h"
#include "../../Common/Util.h"

nn::ldn::NetworkInfo g_ScanResult[nn::ldn::ScanResultCountMax];

namespace
{
    NN_ALIGNAS(4096) uint8_t g_SocketMemoryPoolBuffer[nn::socket::DefaultSocketMemoryPoolSize];
}

NN_ALIGNAS(4096) char g_LdnThreadStack[8192];
NN_ALIGNAS(4096) char g_MonitorThreadStack[8192];

int LdnThreadPriority       = nn::os::DefaultThreadPriority + 2;
int MonitorThreadPriority   = nn::os::DefaultThreadPriority + 1;

const int CursorAppedex         = 3;

enum ApplicationState
{
    ApplicationState_Initialized,
    ApplicationState_Connecting,
    ApplicationState_Connected,
    ApplicationState_Disconnecting,
    ApplicationState_Disconnected
};

enum Type
{
    Type_Iperf,
    Type_BroadcastServer,
    Type_BroadcastClient,
};

struct ApplicationResource
{
    nn::os::ThreadType          monitorThread;
    nn::os::SystemEventType     stateChangeEvent;
    nn::os::EventType           cancelEvent;
    nn::os::MutexType           mutex;
    nn::ldn::NetworkInfo        networkInfo;
    nn::ldn::Ipv4Address        ipv4Address;
    nn::ldn::SubnetMask         subnetMask;
    uint32_t                    myCounter;
    uint32_t                    counter[nn::ldn::NodeCountMax];
    int                         socket;
    ApplicationState            state;
    int                         argc;
    char**                      argv;
    int                         cursor;
};

extern "C"
{
int        testMain(int argc, char **argv);

const int  MemoryPoolSize = 10 * 1024 * 1024;
uint8_t    mempool[MemoryPoolSize];
size_t     mempoolSize = MemoryPoolSize;
}

void MonitorThread(void* arg) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(arg);

    nn::Result result;
    nn::ldn::NetworkInfo networkInfo;

    // 接続状態の変化と監視のキャンセルイベントを同時に待ち受けます。
    auto& app = *static_cast<ApplicationResource*>(arg);
    nn::os::MultiWaitType       multiWait;
    nn::os::MultiWaitHolderType stateChangeHolder;
    nn::os::MultiWaitHolderType cancelHolder;
    nn::os::InitializeMultiWait(&multiWait);
    nn::os::InitializeMultiWaitHolder(&stateChangeHolder, &app.stateChangeEvent);
    nn::os::InitializeMultiWaitHolder(&cancelHolder, &app.cancelEvent);
    nn::os::LinkMultiWaitHolder(&multiWait, &stateChangeHolder);
    nn::os::LinkMultiWaitHolder(&multiWait, &cancelHolder);
    while (nn::os::WaitAny(&multiWait) != &cancelHolder)
    {
        // WaitAny では必ず手動でシグナルをクリアしなければなりません。
        nn::os::TryWaitSystemEvent(&app.stateChangeEvent);

        // ネットワークが破棄された場合には監視を終了します。
        nn::ldn::State ldnState = nn::ldn::GetState();
        if (ldnState != nn::ldn::State_StationConnected)
        {
            nn::os::SignalEvent(&app.cancelEvent);
            break;
        }

        // 現在のネットワークの状態を取得します。
        result = nn::ldn::GetNetworkInfo(&networkInfo);
        if (result <= nn::ldn::ResultInvalidState())
        {
            nn::os::SignalEvent(&app.cancelEvent);
            break;
        }
        nn::os::LockMutex(&app.mutex);
        std::memcpy(&app.networkInfo, &networkInfo, sizeof(nn::ldn::NetworkInfo));
        nn::os::UnlockMutex(&app.mutex);
    }

    nn::os::UnlinkMultiWaitHolder(&stateChangeHolder);
    nn::os::UnlinkMultiWaitHolder(&cancelHolder);
    nn::os::FinalizeMultiWaitHolder(&stateChangeHolder);
    nn::os::FinalizeMultiWaitHolder(&cancelHolder);
    nn::os::FinalizeMultiWait(&multiWait);
}

void IperfFunction(void* arg) NN_NOEXCEPT
{
    auto& app = *static_cast<ApplicationResource*>(arg);

    int    argC;
    char** argV;

    char argvName[]             = "iperf";
    char argvServer[]           = "-s";
    char argvClient[]           = "-c";
    char argvBroadcastServer[]  = "-U";
    char argvBroadcastAddress[] = "255.255.255.255";
    char *argvAddress;

    nn::socket::InAddr addr;
    char** arguments;
    int    count = 0;

    if(app.networkInfo.ldn.nodes[app.cursor].ipv4Address == app.ipv4Address) //サーバー
    {
        count = 1;
        arguments = (char **)std::malloc(sizeof(char *) * (app.argc + count));
        arguments[0] = argvName;
        arguments[1] = argvServer;
    }
    else
    {
        count = 2;
        arguments = (char **)std::malloc(sizeof(char *) * (app.argc + count));
        arguments[0] = argvName;

        if(app.cursor == app.networkInfo.ldn.nodeCount) //ブロードキャストサーバー
        {
            arguments[1] = argvServer;
            arguments[2] = argvBroadcastServer;
        }
        else if(app.cursor == app.networkInfo.ldn.nodeCount + 1) //ブロードキャストクライアント接続
        {
            arguments[1] = argvClient;
            if( nn::ldn::BroadcastIpv4Address.raw == 0xFFFFFFFF )
            {
                arguments[2] = argvBroadcastAddress;
            }
            else
            {
                addr.S_addr = nn::socket::InetHtonl(nn::ldn::BroadcastIpv4Address.raw);
                argvAddress = nn::socket::InetNtoa(addr);
                arguments[2] = argvAddress;
            }
        }
        else // クライアント接続
        {
            arguments[1] = argvClient;
            addr.S_addr = nn::socket::InetHtonl(app.networkInfo.ldn.nodes[app.cursor].ipv4Address.raw);
            argvAddress = nn::socket::InetNtoa(addr);
            arguments[2] = argvAddress;
        }
    }

    for (int i = 1; i < app.argc; ++i)
    {
        arguments[i + count] = app.argv[i];
    }

    argV = arguments;
    argC = app.argc + count;

    NN_LOG("start(argc:%d) ", argC);
    for(int i = 0; i < argC; i++)
    {
        NN_LOG("%s ", argV[i], argV[i]);
    }
    NN_LOG("\n");

    testMain(argC, argV);

    std::free(arguments);
}

nn::Result Scan(ApplicationResource* pApp, int* pOutScanResultCount) NN_NOEXCEPT
{
    // IntentId が一致する LDN ネットワークのみを探索対象とします。
    nn::ldn::ScanFilter filter = { };
    filter.networkType = nn::ldn::NetworkType_Ldn;
    filter.networkId.intentId = BasicSampleIntentId;
    filter.flag = nn::ldn::ScanFilterFlag_IntentId | nn::ldn::ScanFilterFlag_NetworkType;

    // LDN ネットワークを探索します。
    NNS_LDN_RETURN_IF_FAILED(nn::ldn::Scan(
        g_ScanResult, pOutScanResultCount, nn::ldn::ScanResultCountMax,
        filter, nn::ldn::AutoChannel));

    return nn::ResultSuccess();
}

nn::Result Connect(ApplicationResource* pApp, const nn::ldn::NetworkInfo& network) NN_NOEXCEPT
{
    // 接続に必要なセキュリティパラメータを設定します。
    nn::ldn::SecurityConfig security = { };
    nn::ldn::UserConfig user = { };
    security.securityMode   = static_cast<nn::Bit16>(nn::ldn::SecurityMode_Any);
    security.passphraseSize = sizeof(Passphrase);
    std::memcpy(security.passphrase, Passphrase, sizeof(Passphrase));
    std::strncpy(user.userName, "perfSta", nn::ldn::UserNameBytesMax);

    // LDN ネットワークに接続します。
    NNS_LDN_RETURN_IF_FAILED(nn::ldn::Connect(
        network, security, user, 0, nn::ldn::ConnectOption_None));

    // ネットワークの情報を取得します。
    NNS_LDN_RETURN_IF_FAILED(nn::ldn::GetNetworkInfo(&pApp->networkInfo));

    // 自身の IPv4 アドレスとサブネットマスクを取得します。
    NNS_LDN_RETURN_IF_FAILED(nn::ldn::GetIpv4Address(&pApp->ipv4Address, &pApp->subnetMask));

    // ネットワークの状態を監視するスレッドを生成します。
    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
        &pApp->monitorThread, MonitorThread, pApp,
        g_MonitorThreadStack, sizeof(g_MonitorThreadStack),
        MonitorThreadPriority));
    nn::os::StartThread(&pApp->monitorThread);

    return nn::ResultSuccess();
}

void Disconnect(ApplicationResource* pApp) NN_NOEXCEPT
{
    // ネットワークの状態を監視するスレッドの終了まで待機します。
    nn::os::WaitThread(&pApp->monitorThread);
    nn::os::DestroyThread(&pApp->monitorThread);

    // ネットワークから切断します。
    nn::ldn::Disconnect();
}

void IperfPrint(void* arg) NN_NOEXCEPT
{
    auto& app = *static_cast<ApplicationResource*>(arg);

    NN_LOG("----------------------------------------------------------\n");
    for (int i = 0; i < app.networkInfo.ldn.nodeCount; i++)
    {
        if(app.cursor == i)
        {
            NN_LOG(">");
        }
        else
        {
            NN_LOG(" ");
        }

        if(app.networkInfo.ldn.nodes[i].ipv4Address.raw ==  app.ipv4Address.raw)
        {
            NN_LOG("Start iperf server\n");
        }
        else
        {
            NN_LOG("Connect to %s as a client\n", ConvertToString(app.networkInfo.ldn.nodes[i].ipv4Address));
        }
    }

    if(app.cursor == app.networkInfo.ldn.nodeCount)
    {
        NN_LOG(">");
    }
    else
    {
        NN_LOG(" ");
    }
    NN_LOG("Start iperf server(using Broadcast)\n");

    if(app.cursor == app.networkInfo.ldn.nodeCount + 1)
    {
        NN_LOG(">");
    }
    else
    {
        NN_LOG(" ");
    }
    NN_LOG("Connect to %s as a client\n");

}

void LdnStation(void* arg) NN_NOEXCEPT
{
    nn::Result result;
    auto& app = *static_cast<ApplicationResource*>(arg);

    // ステーションとして起動します。
    result = nn::ldn::OpenStation();
    if (result <= nn::ldn::ResultInvalidState())
    {
        // 無線スイッチオフなど通信機能を利用できない状態です。
        // 通信機能を利用できる状態になってから再度 LDN ライブラリを初期化してください。
        return;
    }
    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // LDN ネットワークを探索します。
    NN_LOG("Scaning...\n\n");
    int scanResultCount;
    result = Scan(&app, &scanResultCount);
    if (result <= nn::ldn::ResultInvalidState())
    {
        // 無線スイッチオフなど通信機能を利用できない状態です。
        // 通信機能を利用できる状態になってから再度 LDN ライブラリを初期化してください。
        return;
    }
    else if (scanResultCount == 0)
    {
        NN_LOG("Not found LDN network\n\n");
        nn::ldn::CloseStation();
        app.state = ApplicationState_Disconnected;
        return;
    }
    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // LDN ネットワークへの接続を試行します。
    result = Connect(&app, g_ScanResult[0]);
    if (result <= nn::ldn::ResultInvalidState())
    {
        // 無線スイッチオフなど通信機能を利用できない状態です。
        // 通信機能を利用できる状態になってから再度 LDN ライブラリを初期化してください。
        return;
    }
    else if (result <= nn::ldn::ResultConnectionFailed())
    {
        NN_LOG("Failed to connect to LDN network\n\n");
        nn::ldn::CloseStation();
        app.state = ApplicationState_Disconnected;
        return;
    }
    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(result);
    NN_LOG("Connect is success\n");
    NN_LOG("IP Address (AP)  : %s/%s\n",
        ConvertToString(app.networkInfo.ldn.nodes[0].ipv4Address),
        ConvertToString(app.subnetMask));
    NN_LOG("IP Address (STA) : %s/%s\n\n",
        ConvertToString(app.ipv4Address), ConvertToString(app.subnetMask));
    app.state = ApplicationState_Connected;

    // Iperf での起動用のリストを表示します。
    IperfPrint(&app);

    // メインスレッドから停止を指示されるか、ネットワークが破棄されるまで待機します。
    nn::os::WaitEvent(&app.cancelEvent);

    // ソケットを破棄します。
    nn::socket::Shutdown(app.socket, nn::socket::ShutdownMethod::Shut_RdWr);
    nn::socket::Close(app.socket);

    // LDN ネットワークから切断し、データの送受信を終了します。
    Disconnect(&app);
    nn::ldn::CloseStation();
    app.state = ApplicationState_Disconnected;
    NN_LOG("Disconnect is success\n\n");
}

extern "C" void nnMain()
{
    nn::Result result;

    // LDN ライブラリを初期化します。
    result = nn::ldn::Initialize();
    if (result <= nn::ldn::ResultDeviceOccupied())
    {
        // 他の通信機能を使用中のためローカル通信機能を利用できません。
        // 通信機能を終了してから再度 LDN ライブラリを初期化してください。
    }
    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // socket ライブラリを初期化します。
    result = nn::socket::Initialize(
        g_SocketMemoryPoolBuffer,
        nn::socket::DefaultSocketMemoryPoolSize,
        nn::socket::MinSocketAllocatorSize,
        nn::socket::DefaultConcurrencyLimit);

    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // サンプルの操作方法を出力します。
    NN_LOG("Press A to scan and connect to LDN network\n");
    NN_LOG("Press B to disconnect\n\n");

    // アプリケーションが使用するリソースを初期化します。
    ApplicationResource app;
    std::memset(&app, 0, sizeof(app));

    app.argc = ::nnt::GetHostArgc();
    app.argv = ::nnt::GetHostArgv();

    // LDN ネットワークの接続状態の変更を通知するイベントを取得します。
    nn::ldn::AttachStateChangeEvent(&app.stateChangeEvent);

    InitializeHidController();

    nn::os::ThreadType ldnThread;

    for (;;)
    {
        UpdateHidController();

        // 各状態に応じた処理です。
        if (app.state == ApplicationState_Initialized)
        {
            if (HasHidControllerAnyButtons(nn::hid::NpadButton::A::Mask))
            {
                app.state = ApplicationState_Connecting;
                NN_LOG("Start to scan\n\n");
                nn::os::InitializeEvent(
                    &app.cancelEvent, false, nn::os::EventClearMode_ManualClear);
                nn::os::InitializeMutex(&app.mutex, false, 0);
                NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
                    &ldnThread, LdnStation, &app,
                    g_LdnThreadStack, sizeof(g_LdnThreadStack), LdnThreadPriority));
                nn::os::StartThread(&ldnThread);
            }
        }
        else if (app.state == ApplicationState_Connected)
        {
            if (HasHidControllerAnyButtons(nn::hid::NpadButton::B::Mask))
            {
                app.state = ApplicationState_Disconnecting;
                NN_LOG("Disconnecting\n\n");
                nn::os::SignalEvent(&app.cancelEvent);
            }
            else if (HasHidControllerAnyButtons(nn::hid::NpadButton::A::Mask))
            {
                NN_LOG("Start Iperf\n\n");
                IperfFunction(&app);
            }
            else if (HasHidControllerAnyButtons(nn::hid::NpadButton::Up::Mask))
            {
                app.cursor--;
                if(app.cursor < 0)
                {
                    app.cursor = app.networkInfo.ldn.nodeCount - 2 + CursorAppedex;
                }
                IperfPrint(&app);
            }
            else if (HasHidControllerAnyButtons(nn::hid::NpadButton::Down::Mask))
            {
                app.cursor++;
                if(app.cursor > app.networkInfo.ldn.nodeCount - 2 + CursorAppedex)
                {
                    app.cursor = 0;
                }
                IperfPrint(&app);
            }
        }
        else if (app.state == ApplicationState_Disconnected)
        {
            app.state = ApplicationState_Initialized;
            nn::os::WaitThread(&ldnThread);
            nn::os::DestroyThread(&ldnThread);
            nn::os::FinalizeMutex(&app.mutex);
            nn::os::FinalizeEvent(&app.cancelEvent);
        }

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
    }

    FinalizeHidController();

    // socket ライブラリの使用を終了します。
    result = nn::socket::Finalize();
    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // LDN ライブラリの使用を終了します。
    nn::ldn::Finalize();
}
