﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <cstdio>
#include <cstring>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/hid.h>
#include <nn/ldn.h>
#include <nn/os.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/socket/socket_Api.h>
#include <nn/socket/socket_Types.h>
#include "Definition.h"
#include "DisplayInfo.h"
#include "Util.h"
#include "Graphics.h"
#include "HidController.h"
#include "LdnAccessPoint.h"
#include "LdnStation.h"
#include "UpdateState.h"
#include "PacketTransfer.h"

NN_ALIGNAS(4096) char g_LdnThreadStack[8192];

const int       CountThresholdLow = 30;
const int       CountThresholdHigh = 90;

const int LdnThreadPriority = nn::os::DefaultThreadPriority;
int g_Count = 0;
int g_Diff = 0;

void ChangeSettingsValue(ApplicationResource* pApp, bool direct)
{
    switch (pApp->settingCursor)
    {
    case SettingCursor_Group:
        {
            if (direct)
            {
                if (pApp->ldnSetting.group < 4)
                {
                    ++pApp->ldnSetting.group;
                }
            }
            else
            {
                if (pApp->ldnSetting.group > 0)
                {
                    --pApp->ldnSetting.group;
                }
            }
        }
        break;
    case SettingCursor_Mode:
        {
            if (pApp->ldnSetting.mode == LdnMode_AccessPoint)
            {
                pApp->ldnSetting.mode = LdnMode_Station;
            }
            else
            {
                pApp->ldnSetting.mode = LdnMode_AccessPoint;
            }
        }
        break;
    case SettingCursor_FrequencyBand:
        {
            if (pApp->ldnSetting.band == LdnBand_2GHz)
            {
                pApp->ldnSetting.band = LdnBand_5GHz;
            }
            else
            {
                pApp->ldnSetting.band = LdnBand_2GHz;
            }
        }
        break;
    case SettingCursor_Channel:
        {
            if (pApp->ldnSetting.band == LdnBand_5GHz)
            {
                if (direct)
                {
                    ++pApp->ldnSetting.channel5GHz;
                    if (pApp->ldnSetting.channel5GHz >= LdnChannel5G_CountMax)
                    {
                        pApp->ldnSetting.channel5GHz = LdnChannel5G_36;
                    }
                }
                else
                {
                    --pApp->ldnSetting.channel5GHz;
                    if (pApp->ldnSetting.channel5GHz <= LdnChannel5G_None)
                    {
                        pApp->ldnSetting.channel5GHz = LdnChannel5G_48;
                    }
                }
            }
            else
            {
                if (direct)
                {
                    ++pApp->ldnSetting.channel2GHz;
                    if (pApp->ldnSetting.channel2GHz >= LdnChannel2G_CountMax)
                    {
                        pApp->ldnSetting.channel2GHz = LdnChannel2G_1;
                    }
                }
                else
                {
                    --pApp->ldnSetting.channel2GHz;
                    if (pApp->ldnSetting.channel2GHz <= LdnChannel2G_None)
                    {
                        pApp->ldnSetting.channel2GHz = LdnChannel2G_11;
                    }
                }
            }
        }
        break;
    case SettingCursor_Security:
        {
            if (pApp->ldnSetting.secrity == LdnSecurity_StaticAES)
            {
                pApp->ldnSetting.secrity = LdnSecurity_None;
            }
            else
            {
                pApp->ldnSetting.secrity = LdnSecurity_StaticAES;
            }
        }
        break;
    case SettingCursor_NodeCountMax:
        {
            if (direct)
            {
                if (pApp->ldnSetting.nodeCountMax < nn::ldn::NodeCountMax)
                {
                    ++pApp->ldnSetting.nodeCountMax;
                }
            }
            else
            {
                if (pApp->ldnSetting.nodeCountMax > 1)
                {
                    --pApp->ldnSetting.nodeCountMax;
                }
            }
        }
        break;
    case SettingCursor_CastTo:
        {
            if (!direct)
            {
                ++pApp->translateSetting.castTo;
                if (pApp->translateSetting.castTo >= PacketCastTo_CountMax)
                {
                    pApp->translateSetting.castTo = PacketCastTo_Broadcast;
                }
            }
            else
            {
                --pApp->translateSetting.castTo;
                if (pApp->translateSetting.castTo <= PacketCastTo_None)
                {
                    pApp->translateSetting.castTo = PacketCastTo_Ping;
                }
            }
        }
        break;
    case SettingCursor_Rate:
        {
            if (direct)
            {
                if (pApp->translateSetting.rate < RateMax)
                {
                    if (g_Count < CountThresholdHigh)
                    {
                        ++pApp->translateSetting.rate;
                    }
                    else
                    {
                        pApp->translateSetting.rate += 10;
                    }
                }
                else
                {
                    pApp->translateSetting.rate = RateMax;
                }
            }
            else
            {
                if (pApp->translateSetting.rate > RateMin)
                {
                    if (g_Count < CountThresholdHigh)
                    {
                        --pApp->translateSetting.rate;
                    }
                    else
                    {
                        pApp->translateSetting.rate -= 10;
                    }
                }
                else
                {
                    pApp->translateSetting.rate = RateMin;
                }
            }
        }
        break;
    case SettingCursor_PacketSize:
        {
            if (direct)
            {
                if (pApp->translateSetting.packetSize < PacketSizeMax)
                {
                    pApp->translateSetting.packetSize += 100;
                }
            }
            else
            {
                if (pApp->translateSetting.packetSize >= PacketSizeMin)
                {
                    pApp->translateSetting.packetSize -= 100;
                }
            }
        }
        break;
    default:
        break;
    }
}// NOLINT(readability/fn_size)


void UpdateState(void* arg) NN_NOEXCEPT
{
    auto& app = *static_cast<ApplicationResource*>(arg);

    for (;;)
    {
        UpdateHidController();

        if (app.appState == ApplicationState_Initialized)
        {
            if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::Down::Mask) || app.touch == Touch_Down)
            {
                ++app.initCursor;
                if (app.initCursor >= InitCursor_CountMax)
                {
                    app.initCursor = InitCursor_Exit;
                }
            }
            else if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::Up::Mask) || app.touch == Touch_Up)
            {
                --app.initCursor;
                if (app.initCursor <= InitCursor_None)
                {
                    app.initCursor = InitCursor_Start;
                }
            }
            else if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::A::Mask) || app.touch == Touch_A)
            {
                if (app.initCursor == InitCursor_Start)
                {
                    app.appState = ApplicationState_Ldn;
                    app.time = 0;
                    ::std::memset(app.counter, 0, sizeof(app.counter[0]) * nn::ldn::NodeCountMax);
                    ::std::memset(app.receiveInfo, 0, sizeof(ReceiveInfo) * nn::ldn::NodeCountMax);
                    nn::os::InitializeEvent(&app.cancelEvent, false, nn::os::EventClearMode_ManualClear);

                    if (app.ldnSetting.mode == LdnMode_AccessPoint)
                    {
                        app.netState = NetworkState_AP_Creating;
                        NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
                            &app.ldnThread, LdnAccessPoint, &app,
                            g_LdnThreadStack, sizeof(g_LdnThreadStack), LdnThreadPriority));
                    }
                    else if (app.ldnSetting.mode == LdnMode_Station)
                    {
                        app.netState = NetworkState_STA_Connecting;
                        NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
                            &app.ldnThread, LdnStation, &app,
                            g_LdnThreadStack, sizeof(g_LdnThreadStack), LdnThreadPriority));
                    }
                    nn::os::StartThread(&app.ldnThread);
                }
                else if (app.initCursor == InitCursor_Setting)
                {
                    app.appState = ApplicationState_Setting;
                }
                else if (app.initCursor == InitCursor_Exit)
                {
                    app.appState = ApplicationState_Exit;
                    break;
                }
            }
        }

        else if (app.appState == ApplicationState_Setting)
        {
            if (HasHidControllerAnyButtons(nn::hid::NpadButton::Down::Mask))
            {
                if (g_Count == 0 || g_Count > CountThresholdLow)
                {
                    if (!app.focus)
                    {
                        ++app.settingCursor;
                        if (app.ldnSetting.mode == LdnMode_Station && app.settingCursor == SettingCursor_FrequencyBand)
                        {
                            app.settingCursor = SettingCursor_CastTo;
                        }
                        if (app.translateSetting.castTo == PacketCastTo_Ping && app.settingCursor == SettingCursor_Rate)
                        {
                            app.settingCursor = SettingCursor_CastTo;
                        }
                        if (app.settingCursor >= SettingCursor_CountMax)
                        {
                            app.settingCursor = SettingCursor_PacketSize;
                        }
                    }
                    else
                    {
                        ChangeSettingsValue(&app, false);
                    }
                }
                ++g_Count;
            }
            else if (HasHidControllerAnyButtons(nn::hid::NpadButton::Up::Mask))
            {
                if (g_Count == 0 || g_Count > CountThresholdLow)
                {
                    if (!app.focus)
                    {
                        --app.settingCursor;
                        if (app.ldnSetting.mode == LdnMode_Station && app.settingCursor == SettingCursor_NodeCountMax)
                        {
                            app.settingCursor = SettingCursor_Mode;
                        }
                        if (app.settingCursor <= SettingCursor_None)
                        {
                            app.settingCursor = SettingCursor_Group;
                        }
                    }
                    else
                    {
                        ChangeSettingsValue(&app, true);
                    }
                }
                ++g_Count;
            }
            else if (HasHidControllerAnyButtonsUp(nn::hid::NpadButton::Up::Mask))
            {
                g_Count = 0;
            }
            else if (HasHidControllerAnyButtonsUp(nn::hid::NpadButton::Down::Mask))
            {
                g_Count = 0;
            }
            else if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::A::Mask) || app.touch == Touch_A)
            {
                app.focus = !app.focus;
            }
            else if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::B::Mask) || app.touch == Touch_B)
            {
                if (!app.focus)
                {
                    app.appState = ApplicationState_Initialized;
                }
                else
                {
                    app.focus = false;
                }
            }
            else if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::Left::Mask) || app.touch == Touch_Left)
            {
                app.focus = false;
            }
            else if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::Right::Mask) || app.touch == Touch_Right)
            {
                app.focus = true;
            }

            // タッチ判定
            if (app.touch == Touch_Down)
            {
                if (!app.focus)
                {
                    ++app.settingCursor;
                    if (app.ldnSetting.mode == LdnMode_Station && app.settingCursor == SettingCursor_FrequencyBand)
                    {
                        app.settingCursor = SettingCursor_CastTo;
                    }
                    if (app.settingCursor >= SettingCursor_CountMax)
                    {
                        app.settingCursor = SettingCursor_PacketSize;
                    }
                }
                else
                {
                    ChangeSettingsValue(&app, false);
                }
            }
            else if (app.touch == Touch_Up)
            {
                if (!app.focus)
                {
                    --app.settingCursor;
                    if (app.ldnSetting.mode == LdnMode_Station && app.settingCursor == SettingCursor_NodeCountMax)
                    {
                        app.settingCursor = SettingCursor_Mode;
                    }
                    if (app.settingCursor <= SettingCursor_None)
                    {
                        app.settingCursor = SettingCursor_Mode;
                    }
                }
                else
                {
                    ChangeSettingsValue(&app, true);
                }
            }

            //Rateの設定だけ特別処理
            if (app.settingCursor == SettingCursor_Rate && app.focus)
            {
                if (app.diffY != 0)
                {
                    app.translateSetting.rate -= g_Diff - (app.diffY / 10);
                }

                if (app.translateSetting.rate < RateMin)
                {
                    app.translateSetting.rate = RateMin;
                }
                else if (app.translateSetting.rate > RateMax)
                {
                    app.translateSetting.rate = RateMax;
                }

                g_Diff = (app.diffY / 10);
            }

        }

        else if (app.appState == ApplicationState_Ldn)
        {
            if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::B::Mask) || app.touch == Touch_B)
            {
                if (app.ldnSetting.mode == LdnMode_AccessPoint && app.netState == NetworkState_AP_Created)
                {
                    app.netState = NetworkState_AP_Destroying;
                }
                else if (app.ldnSetting.mode == LdnMode_Station && app.netState == NetworkState_STA_Connected)
                {
                    app.netState = NetworkState_STA_Disconnecting;
                }
                nn::os::SignalEvent(&app.cancelEvent);
            }
            else if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::X::Mask) || app.touch == Touch_X)
            {
                if (app.ldnSetting.mode == LdnMode_AccessPoint)
                {
                    nn::os::LockMutex(&app.mutex);
                    if (0 < app.networkInfo.ldn.nodeCount)
                    {
                        for (int i = 1; i < nn::ldn::NodeCountMax; ++i)
                        {
                            const auto& node = app.networkInfo.ldn.nodes[i];
                            if (node.isConnected)
                            {
                                nn::ldn::Reject(node.ipv4Address);
                            }
                        }
                        nn::ldn::GetNetworkInfo(&app.networkInfo);
                    }
                    nn::os::UnlockMutex(&app.mutex);
                }
            }
            else if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::Y::Mask) || app.touch == Touch_Y)
            {
                nn::os::InitializeEvent(&app.stopCommunicateEvent, false, nn::os::EventClearMode_ManualClear);
                nn::os::LockMutex(&app.mutex);
                app.time = 0;
                ::std::memset(app.counter, 0, sizeof(app.counter[0]) * nn::ldn::NodeCountMax);
                ::std::memset(app.receiveInfo, 0, sizeof(ReceiveInfo) * nn::ldn::NodeCountMax);
                nn::os::UnlockMutex(&app.mutex);
                app.appState = ApplicationState_Communicating;
                // データの送受信を開始します.
                CreateSocket(&app);
            }

            if (app.netState == NetworkState_AP_Destroyed || app.netState == NetworkState_STA_Disconnected)
            {
                app.appState = ApplicationState_Initialized;
                nn::os::WaitThread(&app.ldnThread);
                nn::os::DestroyThread(&app.ldnThread);
                nn::os::FinalizeEvent(&app.cancelEvent);
            }
        }

        else if (app.appState == ApplicationState_Communicating)
        {
            if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::Y::Mask) || app.touch == Touch_Y)
            {
                nn::os::SignalEvent(&app.stopCommunicateEvent);
                // データの送受信を終了します。
                DestroySocket(&app);
                app.appState = ApplicationState_Ldn;
                nn::os::FinalizeEvent(&app.stopCommunicateEvent);
            }
        }

        app.touch = Touch_None;
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
    }
}// NOLINT(readability/fn_size)
