﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstring>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/nn_SdkLog.h>
#include <nn/ldn.h>
#include <nn/os.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/socket.h>
#include <nn/socket/socket_TypesPrivate.h>
#include "PacketTransfer.h"
#include "IcmpTransfer.h"

namespace
{
    NN_ALIGNAS(4096) uint8_t g_SocketMemoryPoolBuffer[nn::socket::DefaultSocketMemoryPoolSize];
}

NN_ALIGNAS(4096) char g_SendThreadStack[8192];
NN_ALIGNAS(4096) char g_ReceiveThreadStack[8192];
NN_ALIGNAS(4096) char g_TimeThreadThreadStack[8192];

const int SendThreadPriority = nn::os::DefaultThreadPriority - 1;
const int ReceiveThreadPriority = nn::os::DefaultThreadPriority;
const int TimeThreadPriority = nn::os::DefaultThreadPriority;

void InitializeSocket() NN_NOEXCEPT
{
    // socket ライブラリを初期化します。
    nn::Result result = nn::socket::Initialize(
        g_SocketMemoryPoolBuffer,
        nn::socket::DefaultSocketMemoryPoolSize,
        nn::socket::MinSocketAllocatorSize,
        nn::socket::DefaultConcurrencyLimit);
    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(result);
}

void FinalizeSocket() NN_NOEXCEPT
{
    // socket ライブラリの使用を終了します。
    nn::Result result = nn::socket::Finalize();
    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(result);
}

void SendThread(void* arg) NN_NOEXCEPT
{
    auto& app = *static_cast<ApplicationResource*>(arg);
    PacketData data;
    IcmpPacketData icmpData;

    // ブロードキャスト送信用のアドレスを生成します。
    nn::socket::SockAddrIn addr;
    addr.sin_family = nn::socket::Family::Af_Inet;
    addr.sin_port = nn::socket::InetHtons(UdpPort);

    if (app.translateSetting.castTo == PacketCastTo_Broadcast || app.translateSetting.castTo == PacketCastTo_Ping)
    {
        addr.sin_addr.S_addr = nn::socket::InetHtonl(app.networkInfo.ldn.nodes[0].ipv4Address.raw | 0x000000FF);
    }
    else if (app.translateSetting.castTo == PacketCastTo_UnicastToAP)
    {
        addr.sin_addr.S_addr = nn::socket::InetHtonl(app.networkInfo.ldn.nodes[0].ipv4Address.raw);
    }

    ::std::memset(data.data, 0xFF, sizeof(data.data));

    nn::os::TimerEventType  timerEvent;

    nn::os::InitializeTimerEvent(&timerEvent, nn::os::EventClearMode_AutoClear);
    nn::TimeSpan interval;
    if (app.translateSetting.castTo == PacketCastTo_Ping)
    {
        interval = nn::TimeSpan::FromSeconds(1);
    }
    else
    {
        interval = nn::TimeSpan::FromMicroSeconds(1000000 / app.translateSetting.rate);
    }
    nn::os::StartPeriodicTimerEvent(&timerEvent, interval, interval);

    int index = 0;
    app.sendWait = false;

    // キャンセルされるまでパケットを送信します。
    while (!nn::os::TryWaitEvent(&app.stopCommunicateEvent))
    {
        nn::os::WaitTimerEvent(&timerEvent);

        if (app.networkInfo.ldn.nodeCount > 1)
        {
            if (app.translateSetting.castTo == PacketCastTo_Ping)
            {
                CreateIcmpPacket(&icmpData);

                nn::socket::SendTo(
                    app.icmp_sock,
                    &icmpData,
                    sizeof(IcmpPacketData),
                    nn::socket::MsgFlag::Msg_None,
                    reinterpret_cast<nn::socket::SockAddr*>(&addr),
                    sizeof(addr));
            }
            else
            {
                if (app.translateSetting.castTo == PacketCastTo_UnicastToAll)
                {
                    for (;;)
                    {
                        ++index;
                        if (index == app.networkInfo.ldn.nodeCount)
                        {
                            index = 0;
                        }

                        if (app.networkInfo.ldn.nodes[index].ipv4Address.raw != app.ipv4Address.raw &&
                            app.networkInfo.ldn.nodes[index].ipv4Address.raw != 0)
                        {
                            break;
                        }
                    }
                    addr.sin_addr.S_addr = nn::socket::InetHtonl(app.networkInfo.ldn.nodes[index].ipv4Address.raw);
                    app.counter[index] += 1;
                    data.counter = nn::socket::InetHtonl(app.counter[index]);
                }
                else
                {
                    app.counter[0] += 1;
                    data.counter = nn::socket::InetHtonl(app.counter[0]);
                }

                data.dataSize = nn::socket::InetHtons(app.translateSetting.packetSize);

                app.sendWait = true;
                nn::socket::SendTo(
                    app.socket,
                    &data,
                    app.translateSetting.packetSize,
                    nn::socket::MsgFlag::Msg_None,
                    reinterpret_cast<nn::socket::SockAddr*>(&addr),
                    sizeof(addr));
                app.sendWait = false;
            }


            //NN_LOG("Send : %x\n", nn::socket::InetNtohl(addr.sin_addr.S_addr));
        }
    }
    nn::os::FinalizeTimerEvent(&timerEvent);
}// NOLINT(readability/fn_size)

void ReceiveThread(void* arg) NN_NOEXCEPT
{
    auto& app = *static_cast<ApplicationResource*>(arg);
    nn::os::LockMutex(&app.mutex);
    app.receiveError = false;
    nn::os::UnlockMutex(&app.mutex);

    // キャンセルされるまでパケットを受信し続けます。
    while (!nn::os::TryWaitEvent(&app.stopCommunicateEvent))
    {
        // 任意のステーションからパケットを受信するためのアドレスを生成します。
        nn::socket::SockAddrIn addr;
        addr.sin_family = nn::socket::Family::Af_Inet;
        addr.sin_port = nn::socket::InetHtons(UdpPort);
        addr.sin_addr.S_addr = nn::socket::InAddr_Any;
        nn::socket::SockLenT length = sizeof(addr);

        if (app.translateSetting.castTo == PacketCastTo_Ping)
        {
            // 受信用のバッファです。
            uint8_t buf[1024];

            size_t size = nn::socket::RecvFrom(
                app.icmp_sock,
                buf,
                sizeof(buf),
                nn::socket::MsgFlag::Msg_None,
                reinterpret_cast<nn::socket::SockAddr*>(&addr),
                &length);
            if (size == 0 || size == -1)
            {
                app.receiveError = true;
            }

            nn::ldn::Ipv4Address from = { nn::socket::InetNtohl(addr.sin_addr.S_addr) };

            nn::socket::Icmp *icmpHeader;
            icmpHeader = reinterpret_cast<nn::socket::Icmp*>(buf + sizeof(ip));

            int64_t packetTick64 = (int64_t)buf[48] | (int64_t)buf[49] << 8 | (int64_t)buf[50] << 16 | (int64_t)buf[51] << 24 |
                (int64_t)buf[52] << 32 | (int64_t)buf[53] << 40 | (int64_t)buf[54] << 48 | (int64_t)buf[55] << 56;

            if (icmpHeader->icmp_type == nn::socket::IcmpType::Icmp_EchoReply)
            {
                // 受信した時刻から RTT を計算し、保存します。
                nn::os::LockMutex(&app.mutex);
                for (int i = 0; i < app.networkInfo.ldn.nodeCountMax; ++i)
                {
                    if (app.networkInfo.ldn.nodes[i].ipv4Address == from && app.ipv4Address != from)
                    {
                        nn::os::Tick packetTick(packetTick64);
                        nn::TimeSpan rtt = nn::os::ConvertToTimeSpan(nn::os::GetSystemTick() - packetTick);
                        app.receiveInfo[i].rtt = rtt.GetMilliSeconds();

                        NN_LOG("form: %d.%d.%d.%d[%d] rtt: %lld[ms]\n",
                            from.raw >> 24 & 0xFF, from.raw >> 16 & 0xFF, from.raw >> 8 & 0xFF, from.raw & 0xFF, i, rtt.GetMilliSeconds());
                        break;
                    }
                }
                nn::os::UnlockMutex(&app.mutex);
            }
            else if (icmpHeader->icmp_type == nn::socket::IcmpType::Icmp_Echo)
            {
                if (app.ipv4Address != from)
                {
                    IcmpPacketData icmpData;

                    EchoIcmpPacket(&icmpData, packetTick64);

                    nn::socket::SendTo(
                        app.icmp_sock,
                        &icmpData,
                        sizeof(IcmpPacketData),
                        nn::socket::MsgFlag::Msg_None,
                        reinterpret_cast<nn::socket::SockAddr*>(&addr),
                        sizeof(addr));
                }
            }
        }
        else
        {
            // 受信用のバッファです。
            PacketData data;

            size_t size = nn::socket::RecvFrom(
                app.socket,
                &data,
                sizeof(data),
                nn::socket::MsgFlag::Msg_None,
                reinterpret_cast<nn::socket::SockAddr*>(&addr),
                &length);
            if (size == 0 || size == -1)
            {
                app.receiveError = true;
                continue;
            }
            nn::ldn::Ipv4Address from = { nn::socket::InetNtohl(addr.sin_addr.S_addr) };
            int packetCount = nn::socket::InetNtohl(data.counter);

            // 受信したカウンタを保存します。
            for (int i = 0; i < app.networkInfo.ldn.nodeCountMax; ++i)
            {
                if (app.networkInfo.ldn.nodes[i].ipv4Address == from && app.ipv4Address != from)
                {
                    if (app.receiveInfo[i].startTime == 0)
                    {
                        //NN_LOG("Start  ");
                        app.receiveInfo[i].startTime = app.time;
                        app.receiveInfo[i].startCount = packetCount;
                        app.receiveInfo[i].packetCount = packetCount;
                        //NN_LOG("Frame : %d / Count : %d\n", app.receiveInfo[i].startTime, app.receiveInfo[i].startCount);
                    }
                    app.receiveInfo[i].receivePacketSize += size;
                    //NN_LOG("Size : %d\n", app.receiveInfo[i].receivePacketSize);
                    if (packetCount - app.receiveInfo[i].packetCount >= 1)
                    {
                        if (packetCount - app.receiveInfo[i].packetCount > 1)
                        {
                            //NN_LOG("--Packet Loss!--\n");
                            app.receiveInfo[i].packetLoss += (packetCount - app.receiveInfo[i].packetCount - 1);
                        }
                        app.receiveInfo[i].packetCount = packetCount;
                    }
                    else
                    {
                        app.packetSwitched = true;
                    }
                    //NN_LOG("Count : %d\n", app.receiveInfo[i].packetCount);
                    break;
                }
            }
        }
    }
}// NOLINT(readability/fn_size)

void SetWriteInfo(ApplicationResource* pApp)
{
    int index = 0;
    for (int i = 0; i < nn::ldn::NodeCountMax; ++i)
    {
        if (pApp->networkInfo.ldn.nodes[i].ipv4Address == pApp->ipv4Address)
        {
            index = i;
            break;
        }
    }
    for (int i = 0; i < nn::ldn::NodeCountMax; ++i)
    {
        if (pApp->networkInfo.ldn.nodes[i].ipv4Address.raw != 0 && i != index)
        {
            pApp->receiveInfo[i].packetDataRate =
                static_cast<double>(pApp->receiveInfo[i].receivePacketSize - pApp->receiveInfo[i].previousReceivePacketSize) / 1000;
            pApp->receiveInfo[i].packetLossRate =
                static_cast<double>(pApp->receiveInfo[i].packetLoss * 100) / static_cast<double>(pApp->receiveInfo[i].packetCount - pApp->receiveInfo[i].startCount);
            /*
            NN_LOG("Log[%d], ReceiveFrom[%d], %d sec, PS:%lld, pPS:%lld, PC:%d, sPC:%d, PR:%d, pPR:%d\n",
                i, index,
                (pApp->time),
                pApp->receiveInfo[i].receivePacketSize,
                pApp->receiveInfo[i].previousReceivePacketSize,
                pApp->receiveInfo[i].packetCount,
                pApp->receiveInfo[i].startCount,
                pApp->receiveInfo[i].packetLoss,
                pApp->receiveInfo[i].previousPacketLoss);
*/
            pApp->receiveInfo[i].previousPacketLoss = pApp->receiveInfo[i].packetLoss;
            pApp->receiveInfo[i].previousReceivePacketSize = pApp->receiveInfo[i].receivePacketSize;
        }
    }
}

void TimeCount(void* arg) NN_NOEXCEPT
{
    auto& app = *static_cast<ApplicationResource*>(arg);
    nn::os::TimerEventType  timerEvent;

    nn::os::InitializeTimerEvent(&timerEvent, nn::os::EventClearMode_AutoClear);
    const nn::TimeSpan interval = nn::TimeSpan::FromMilliSeconds(1000);
    nn::os::StartPeriodicTimerEvent(&timerEvent, interval, interval);

    while (!nn::os::TryWaitEvent(&app.stopCommunicateEvent))
    {
        nn::os::WaitTimerEvent(&timerEvent);

        if (app.appState == ApplicationState_Communicating)
        {
            app.time++;
            SetWriteInfo(&app);
        }
        app.receiveError = false;
        app.packetSwitched = false;
    }
    nn::os::FinalizeTimerEvent(&timerEvent);
}

void CreateSocket(ApplicationResource* pApp) NN_NOEXCEPT
{
    // データの送受信に使用する socket を生成します。
    pApp->socket = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp);
    NN_ABORT_UNLESS_NOT_EQUAL(pApp->socket, -1);
    nn::socket::SockAddrIn addr;
    addr.sin_family = nn::socket::Family::Af_Inet;
    addr.sin_port = nn::socket::InetHtons(UdpPort);
    addr.sin_addr.S_addr = nn::socket::InAddr_Any;
    NN_ABORT_UNLESS_EQUAL(nn::socket::Bind(
        pApp->socket, reinterpret_cast<nn::socket::SockAddr*>(&addr), sizeof(addr)), 0);

    // Ping に使用する socket を生成します。
    pApp->icmp_sock = nn::socket::Socket(nn::socket::Family::Pf_Inet, nn::socket::Type::Sock_Raw, nn::socket::Protocol::IpProto_Icmp);

    // ブロードキャスト通信を有効化します。
    int isEnabled = 1;
    nn::socket::SetSockOpt(
        pApp->socket, nn::socket::Level::Sol_Socket, nn::socket::Option::So_Broadcast, &isEnabled, sizeof(isEnabled));

    // データ送信用のスレッドを生成します。
    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
        &pApp->sendThread, SendThread, pApp,
        g_SendThreadStack, sizeof(g_SendThreadStack),
        SendThreadPriority));
    nn::os::StartThread(&pApp->sendThread);

    // データ受信用のスレッドを生成します。
    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
        &pApp->receiveThread, ReceiveThread, pApp,
        g_ReceiveThreadStack, sizeof(g_ReceiveThreadStack),
        ReceiveThreadPriority));
    nn::os::StartThread(&pApp->receiveThread);

    //時間計測用のスレッドを生成します。
    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
        &pApp->timeThread, TimeCount, pApp,
        g_TimeThreadThreadStack, sizeof(g_TimeThreadThreadStack),
        TimeThreadPriority));
    nn::os::StartThread(&pApp->timeThread);
}

void DestroySocket(ApplicationResource* pApp) NN_NOEXCEPT
{
    nn::socket::Shutdown(pApp->socket, nn::socket::ShutdownMethod::Shut_RdWr);
    nn::socket::Shutdown(pApp->icmp_sock, nn::socket::ShutdownMethod::Shut_RdWr);

    // スレッドの終了まで待機します。
    nn::os::WaitThread(&pApp->sendThread);
    nn::os::WaitThread(&pApp->receiveThread);
    nn::os::WaitThread(&pApp->timeThread);
    nn::os::DestroyThread(&pApp->sendThread);
    nn::os::DestroyThread(&pApp->receiveThread);
    nn::os::DestroyThread(&pApp->timeThread);

    // ソケットを破棄します。
    nn::socket::Close(pApp->socket);
    nn::socket::Close(pApp->icmp_sock);
}
