﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/hid.h>
#include <nn/ldn.h>
#include <nn/os.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/socket/socket_Api.h>
#include <nn/socket/socket_Types.h>
#include "Definition.h"
#include "Util.h"
#include "PacketTransfer.h"
#include "LdnStation.h"

NN_ALIGNAS(4096) char g_StationMonitorThreadStack[8192];
const int StationMonitorThreadPriority = nn::os::DefaultThreadPriority + 1;

nn::ldn::NetworkInfo g_ScanResult[nn::ldn::ScanResultCountMax];

const int ChannelCount = 5;

void StationMonitorThread(void* arg) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(arg);

    nn::Result result;
    nn::ldn::NetworkInfo networkInfo;

    // 接続状態の変化と監視のキャンセルイベントを同時に待ち受けます。
    auto& app = *static_cast<ApplicationResource*>(arg);
    nn::os::MultiWaitType       multiWait;
    nn::os::MultiWaitHolderType stateChangeHolder;
    nn::os::MultiWaitHolderType cancelHolder;
    nn::os::InitializeMultiWait(&multiWait);
    nn::os::InitializeMultiWaitHolder(&stateChangeHolder, &app.stateChangeEvent);
    nn::os::InitializeMultiWaitHolder(&cancelHolder, &app.cancelEvent);
    nn::os::LinkMultiWaitHolder(&multiWait, &stateChangeHolder);
    nn::os::LinkMultiWaitHolder(&multiWait, &cancelHolder);
    while (nn::os::WaitAny(&multiWait) != &cancelHolder)
    {
        // WaitAny では必ず手動でシグナルをクリアしなければなりません。
        nn::os::TryWaitSystemEvent(&app.stateChangeEvent);

        // ネットワークが破棄された場合には監視を終了します。
        nn::ldn::State ldnState = nn::ldn::GetState();
        if (ldnState != nn::ldn::State_StationConnected)
        {
            nn::os::SignalEvent(&app.cancelEvent);
            break;
        }

        // 現在のネットワークの状態を取得します。
        result = nn::ldn::GetNetworkInfo(&networkInfo);
        if (result <= nn::ldn::ResultInvalidState())
        {
            nn::os::SignalEvent(&app.cancelEvent);
            break;
        }
        nn::os::LockMutex(&app.mutex);
        ::std::memcpy(&app.networkInfo, &networkInfo, sizeof(nn::ldn::NetworkInfo));
        nn::os::UnlockMutex(&app.mutex);
    }

    nn::os::UnlinkMultiWaitHolder(&stateChangeHolder);
    nn::os::UnlinkMultiWaitHolder(&cancelHolder);
    nn::os::FinalizeMultiWaitHolder(&stateChangeHolder);
    nn::os::FinalizeMultiWaitHolder(&cancelHolder);
    nn::os::FinalizeMultiWait(&multiWait);

    FinalizeSocket();
}


nn::Result Scan(ApplicationResource* pApp, int* pOutScanResultCount) NN_NOEXCEPT
{
    nn::ldn::IntentId BasicSampleIntentId =
        nn::ldn::MakeIntentId(LocalCommunicationId, BasicSampleSceneId[pApp->ldnSetting.group]);

    int scanChannel[ChannelCount] = { nn::ldn::AutoChannel, 36, 40, 44, 48 };

    // IntentId が一致する LDN ネットワークのみを探索対象とします。
    nn::ldn::ScanFilter filter = {};
    filter.networkType = nn::ldn::NetworkType_Ldn;
    filter.networkId.intentId = BasicSampleIntentId;
    filter.flag = nn::ldn::ScanFilterFlag_IntentId | nn::ldn::ScanFilterFlag_NetworkType;

    // LDN ネットワークを探索します。
    for (int i = 0; i < ChannelCount; ++i)
    {
        NNS_LDN_RETURN_IF_FAILED(nn::ldn::Scan(
            g_ScanResult, pOutScanResultCount, nn::ldn::ScanResultCountMax,
            filter, scanChannel[i]));

        if (*pOutScanResultCount != 0)
        {
            break;
        }
    }
    return nn::ResultSuccess();

}

nn::Result Connect(ApplicationResource* pApp, const nn::ldn::NetworkInfo& network) NN_NOEXCEPT
{
    // 接続に必要なセキュリティパラメータを設定します。
    nn::ldn::SecurityConfig security = {};
    nn::ldn::UserConfig     user = {};
    security.securityMode = static_cast<nn::Bit16>(nn::ldn::SecurityMode_Any);
    security.passphraseSize = sizeof(Passphrase);
    ::std::memcpy(security.passphrase, Passphrase, sizeof(Passphrase));
    ::std::strncpy(user.userName, "Station", nn::ldn::UserNameBytesMax);

    // LDN ネットワークに接続します。
    NNS_LDN_RETURN_IF_FAILED(nn::ldn::Connect(
        network, security, user, 0, nn::ldn::ConnectOption_None));

    // ネットワークの情報を取得します。
    NNS_LDN_RETURN_IF_FAILED(nn::ldn::GetNetworkInfo(&pApp->networkInfo));

    // 自身の IPv4 アドレスとサブネットマスクを取得します。
    NNS_LDN_RETURN_IF_FAILED(nn::ldn::GetIpv4Address(&pApp->ipv4Address, &pApp->subnetMask));

    // ネットワークの状態を監視するスレッドを生成します。
    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
        &pApp->monitorThread, StationMonitorThread, pApp,
        g_StationMonitorThreadStack, sizeof(g_StationMonitorThreadStack),
        StationMonitorThreadPriority));
    nn::os::StartThread(&pApp->monitorThread);

    return nn::ResultSuccess();
}

void Disconnect(ApplicationResource* pApp) NN_NOEXCEPT
{
    // ネットワークの状態を監視するスレッドの終了まで待機します。
    nn::os::WaitThread(&pApp->monitorThread);
    nn::os::DestroyThread(&pApp->monitorThread);

    // ネットワークから切断します。
    nn::ldn::Disconnect();
}

void LdnStation(void* arg) NN_NOEXCEPT
{
    nn::Result result;
    auto& app = *static_cast<ApplicationResource*>(arg);

    // ステーションとして起動します。
    result = nn::ldn::OpenStation();
    if (result <= nn::ldn::ResultInvalidState())
    {
        // 無線スイッチオフなど通信機能を利用できない状態です。
        // 通信機能を利用できる状態になってから再度 LDN ライブラリを初期化してください。
        return;
    }
    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // LDN ネットワークを探索します。
    int scanResultCount;
    result = Scan(&app, &scanResultCount);

    if (result <= nn::ldn::ResultInvalidState())
    {
        // 無線スイッチオフなど通信機能を利用できない状態です。
        // 通信機能を利用できる状態になってから再度 LDN ライブラリを初期化してください。
        return;
    }
    else if (scanResultCount == 0)
    {
        nn::ldn::CloseStation();
        app.netState = NetworkState_STA_Disconnected;
        return;
    }
    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // LDN ネットワークへの接続を試行します。
    result = Connect(&app, g_ScanResult[0]);
    if (result <= nn::ldn::ResultInvalidState() ||
        result <= nn::ldn::ResultConnectionFailed())
    {
        nn::ldn::CloseStation();
        app.netState = NetworkState_STA_Disconnected;
        return;
    }
    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(result);
    app.netState = NetworkState_STA_Connected;

    InitializeSocket();

    // メインスレッドから停止を指示されるか、ネットワークが破棄されるまで待機します。
    nn::os::WaitEvent(&app.cancelEvent);

    // LDN ネットワークから切断し、データの送受信を終了します。
    //DestroySocket(&app);
    Disconnect(&app);
    nn::ldn::CloseStation();
    app.netState = NetworkState_STA_Disconnected;
}
