﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/ldn/detail/TcpIp/ldn_LdnAutoIp.h>
#include <nn/ldn/detail/Utility/ldn_Random.h>
#include <nn/ldn/detail/Utility/ldn_ReverseByteOrder.h>
#include <nnt.h>

namespace
{
    // 初期化に使用するバッファです。
    char g_ServerBuffer[4 * 1024];

    // IPv4 アドレスの範囲は 169.254.x.y/24 (1 <= x <= 127 && 1 <= y <= 127) とします。
    static const int NetworkAddressCountMax = 127;
    static const int HostAddressCountMax    = 127;

    /**
     * @brief       ランダムな MAC アドレスを生成します。
     * @return      ランダムな MAC アドレスです。
     */
    nn::ldn::MacAddress CreateRandomMacAddress() NN_NOEXCEPT
    {
        return nn::ldn::MakeMacAddress(
            nn::ldn::detail::Random<uint8_t>(),
            nn::ldn::detail::Random<uint8_t>(),
            nn::ldn::detail::Random<uint8_t>(),
            nn::ldn::detail::Random<uint8_t>(),
            nn::ldn::detail::Random<uint8_t>(),
            nn::ldn::detail::Random<uint8_t>());
    }

    /**
     * @brief       IP アドレスの正当性を検証します。
     * @param[in]   ipv4Address     対象の IP アドレスです。
     * @return      LdnAutoIp で生成されうる IP アドレスであれば true です。
     */
    bool IsValid(nn::ldn::Ipv4Address ipv4Address) NN_NOEXCEPT
    {
        auto  a = nn::ldn::detail::ConvertToNetworkByteOrder(ipv4Address);
        auto* p = reinterpret_cast<const uint8_t*>(&a);
        return p[0] == 169 &&
               p[1] == 254 &&
               1 <= p[2] && p[2] <= NetworkAddressCountMax &&
               1 <= p[3] && p[3] <= HostAddressCountMax;
    }

    /**
     * @brief       サブネットマスクの正当性を検証します。
     * @param[in]   subnetMask      対象のサブネットマスクです。
     * @return      LdnAutoIp で生成されうるサブネットマスクであれば true です。
     */
    bool IsValid(nn::ldn::SubnetMask subnetMask) NN_NOEXCEPT
    {
        return subnetMask.raw == nn::ldn::MakeSubnetMask(24).raw;
    }

    /**
     * @brief       IP アドレスに重複が無いことを検証します。
     * @param[in]   addresses       対象の IP アドレスのリストです。
     * @param[in]   addressCount    addresses に含まれる IP アドレスの総数です。
     * @return      IP アドレスに重複が無ければ true です。
     */
    bool IsUnique(const nn::ldn::Ipv4Address* addresses, int addressCount) NN_NOEXCEPT
    {
        bool tables[HostAddressCountMax + 1] = { };
        for (int i = 0; i < addressCount; ++i)
        {
            auto a = nn::ldn::detail::ConvertToNetworkByteOrder(addresses[i]);
            auto* p = reinterpret_cast<const uint8_t*>(&a);
            auto host = p[3];
            if (tables[host])
            {
                return false;
            }
            tables[host] = true;
        }
        return true;
    }


} // namespace <unnamed>

class LdnAutoIpServer : public ::testing::Test
{
protected:

    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
    {
        m_pServer = new nn::ldn::detail::LdnAutoIpServer(
            g_ServerBuffer, nn::ldn::detail::LdnAutoIpServer::GetRequiredBufferSize());
    }

    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
    {
        delete m_pServer;
        m_pServer = nullptr;
    }

protected:

    nn::ldn::detail::IAddressingServer* m_pServer;
};

class LdnAutoIpClient : public ::testing::Test
{
protected:

    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
    {
        m_pClient = new nn::ldn::detail::LdnAutoIpClient();
    }

    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
    {
        delete m_pClient;
        m_pClient = nullptr;
    }

protected:

    nn::ldn::detail::IAddressingClient* m_pClient;
};

TEST_F(LdnAutoIpServer, StartServerMultipleTimes)
{
    auto& server = *this->m_pServer;

    // サーバーを初期化します。
    auto macAddress = CreateRandomMacAddress();
    server.StartServer(macAddress);
    auto ipv4Address = server.GetIpv4Address();
    ASSERT_TRUE(IsValid(ipv4Address));
    auto subnetMask = server.GetSubnetMask();
    ASSERT_TRUE(IsValid(subnetMask));
    server.StopServer();

    // ClearEntries() を挟まずに初期化します。
    // 毎回同じ IP アドレスが生成されるはずです。
    const int retryCountMax = 256;
    for (int i = 0; i < retryCountMax; ++i)
    {
        server.StartServer(macAddress);
        ASSERT_EQ(ipv4Address, server.GetIpv4Address());
        ASSERT_EQ(subnetMask.raw, server.GetSubnetMask().raw);
        server.StopServer();
    }

    // ClearEntries() を挟んで初期化します。
    // 毎回同じ IP アドレスが生成されるとは限りません。
    for (int i = 0; i < retryCountMax; ++i)
    {
        server.ClearEntries();
        server.StartServer(macAddress);
        ASSERT_TRUE(IsValid(server.GetIpv4Address()));
        ASSERT_TRUE(IsValid(server.GetSubnetMask()));
        server.StopServer();
    }
}

TEST_F(LdnAutoIpServer, StopServerInDestructor)
{
    auto& server = *this->m_pServer;
    auto macAddress = CreateRandomMacAddress();
    server.StartServer(macAddress);
    auto ipv4Address = server.GetIpv4Address();
    ASSERT_TRUE(IsValid(ipv4Address));
    auto subnetMask = server.GetSubnetMask();
    ASSERT_TRUE(IsValid(subnetMask));
}

TEST_F(LdnAutoIpServer, UniqueAssignment)
{
    auto& server = *this->m_pServer;

    // サーバーを初期化します。
    auto serverMacAddress = CreateRandomMacAddress();
    server.StartServer(serverMacAddress);
    auto serverIpv4Address = server.GetIpv4Address();
    ASSERT_TRUE(IsValid(serverIpv4Address));
    auto subnetMask = server.GetSubnetMask();
    ASSERT_TRUE(IsValid(subnetMask));

    // 最大数の IPv4 アドレスを割り当てます。
    nn::ldn::Ipv4Address addresses[HostAddressCountMax];
    addresses[0] = serverIpv4Address;
    for (int i = 1; i < HostAddressCountMax; ++i)
    {
        auto macAddress = CreateRandomMacAddress();
        addresses[i] = server.Assign(macAddress);
        ASSERT_TRUE(IsValid(addresses[i]));
        ASSERT_TRUE(IsValid(server.GetSubnetMask()));
    }

    // 重複が無いことを確認します。
    ASSERT_TRUE(IsUnique(addresses, HostAddressCountMax));

    // 自分以外の全ての IPv4 アドレスを解放します。
    for (int i = 0; i < HostAddressCountMax - 1; ++i)
    {
        server.Free(addresses[i]);
    }

    // サーバーを終了します。
    server.StopServer();
}

TEST_F(LdnAutoIpServer, LeastRecentlyUsed)
{
    auto& server = *this->m_pServer;

    // サーバーを初期化します。
    auto serverMacAddress = CreateRandomMacAddress();
    server.StartServer(serverMacAddress);
    auto serverIpv4Address = server.GetIpv4Address();
    ASSERT_TRUE(IsValid(serverIpv4Address));
    auto subnetMask = server.GetSubnetMask();
    ASSERT_TRUE(IsValid(subnetMask));

    // 最大数の IPv4 アドレスを割り当てます。
    nn::ldn::Ipv4Address addresses[HostAddressCountMax];
    addresses[0] = serverIpv4Address;
    for (int i = 1; i < HostAddressCountMax; ++i)
    {
        auto macAddress = CreateRandomMacAddress();
        addresses[i] = server.Assign(macAddress);
        ASSERT_TRUE(IsValid(addresses[i]));
        ASSERT_TRUE(IsValid(server.GetSubnetMask()));
    }

    // 適当にいくつかの IPv4 アドレスを解放します。
    int freeIndexes[] = { 32, 8, 16, 64, 4, 2 };
    int freeCount = static_cast<int>(sizeof(freeIndexes) / sizeof(freeIndexes[0]));
    for (int i = 0; i < freeCount; ++i)
    {
        server.Free(addresses[freeIndexes[i]]);
    }

    // MAC アドレスが異なるクライアントが接続しにきたとき、
    // Free した順に IPv4 アドレスが割り当てられることを確認します。
    for (int i = 0; i < freeCount; ++i)
    {
        auto macAddress = CreateRandomMacAddress();
        auto ipv4Address = server.Assign(macAddress);
        ASSERT_EQ(addresses[freeIndexes[i]], ipv4Address);
    }

    // 自分以外の全ての IPv4 アドレスを解放します。
    for (int i = 1; i < HostAddressCountMax - 1; ++i)
    {
        server.Free(addresses[i]);
    }

    // サーバーを終了します。
    server.StopServer();
}

TEST_F(LdnAutoIpServer, Reuse)
{
    auto& server = *this->m_pServer;

    // サーバーを初期化します。
    auto serverMacAddress = CreateRandomMacAddress();
    server.StartServer(serverMacAddress);
    auto serverIpv4Address = server.GetIpv4Address();
    ASSERT_TRUE(IsValid(serverIpv4Address));
    auto subnetMask = server.GetSubnetMask();
    ASSERT_TRUE(IsValid(subnetMask));

    // 適当な数の IPv4 アドレスを割り当てます。
    nn::ldn::MacAddress macAddresses[HostAddressCountMax];
    nn::ldn::Ipv4Address ipv4Addresses[HostAddressCountMax];
    macAddresses[0] = serverMacAddress;
    ipv4Addresses[0] = serverIpv4Address;
    for (int i = 1; i < 8; ++i)
    {
        macAddresses[i] = CreateRandomMacAddress();
        ipv4Addresses[i] = server.Assign(macAddresses[i]);
        ASSERT_TRUE(IsValid(ipv4Addresses[i]));
        ASSERT_TRUE(IsValid(server.GetSubnetMask()));
    }

    // 適当な順に解放します。
    int freeIndexes[] = { 7, 4, 3, 5, 2, 6, 1 };
    int freeCount = static_cast<int>(sizeof(freeIndexes) / sizeof(freeIndexes[0]));
    for (int i = 0; i < freeCount; ++i)
    {
        server.Free(ipv4Addresses[freeIndexes[i]]);
    }

    // MAC アドレスが一致するクライアントが接続しにきたとき、
    // 前回と同じ IPv4 アドレスが割り当てられることを確認します。
    for (int i = 1; i < 8; ++i)
    {
        auto ipv4Address = server.Assign(macAddresses[i]);
        ASSERT_EQ(ipv4Addresses[i], ipv4Address);
        ASSERT_TRUE(IsValid(server.GetSubnetMask()));
    }

    // 自分以外の全ての IPv4 アドレスを解放します。
    for (int i = 0; i < freeCount; ++i)
    {
        server.Free(ipv4Addresses[freeIndexes[i]]);
    }

    // サーバーを終了します。
    server.StopServer();

    // すぐにサーバーを起動します。
    server.StartServer(serverMacAddress);

    // 前回と同じ IPv4 アドレスが割り当てられることを確認します。
    for (int i = 7; i >= 1; --i)
    {
        auto ipv4Address = server.Assign(macAddresses[i]);
        ASSERT_EQ(ipv4Addresses[i], ipv4Address);
        ASSERT_TRUE(IsValid(server.GetSubnetMask()));
    }

    // サーバーを終了します。
    server.StopServer();
}

TEST_F(LdnAutoIpServer, ClearEntries)
{
    auto& server = *this->m_pServer;

    // サーバーを初期化します。
    auto serverMacAddress = CreateRandomMacAddress();
    server.StartServer(serverMacAddress);
    auto serverIpv4Address = server.GetIpv4Address();
    ASSERT_TRUE(IsValid(serverIpv4Address));
    auto subnetMask = server.GetSubnetMask();
    ASSERT_TRUE(IsValid(subnetMask));

    // 適当な数の IPv4 アドレスを割り当てます。
    nn::ldn::MacAddress macAddresses[HostAddressCountMax];
    nn::ldn::Ipv4Address ipv4Addresses[HostAddressCountMax];
    macAddresses[0] = serverMacAddress;
    ipv4Addresses[0] = serverIpv4Address;
    for (int i = 1; i < 8; ++i)
    {
        macAddresses[i] = CreateRandomMacAddress();
        ipv4Addresses[i] = server.Assign(macAddresses[i]);
        ASSERT_TRUE(IsValid(ipv4Addresses[i]));
        ASSERT_TRUE(IsValid(server.GetSubnetMask()));
    }

    // 自分以外の全ての IPv4 アドレスを解放します。
    for (int i = 1; i < 8; ++i)
    {
        server.Free(ipv4Addresses[i]);
    }

    // サーバーを終了します。
    server.StopServer();

    // エントリを消去します。
    server.ClearEntries();

    // すぐにサーバーを起動します。
    server.StartServer(serverMacAddress);
    serverIpv4Address = server.GetIpv4Address();
    ASSERT_TRUE(IsValid(serverIpv4Address));
    subnetMask = server.GetSubnetMask();
    ASSERT_TRUE(IsValid(subnetMask));

    // 追加の接続を挟みます。
    server.Assign(CreateRandomMacAddress());

    // 前回と同じ IPv4 アドレスが割り当てられないことを確認します。
    for (int i = 1; i < 8; ++i)
    {
        auto ipv4Address = server.Assign(macAddresses[i]);
        ASSERT_NE(ipv4Addresses[i], ipv4Address);
        ASSERT_TRUE(IsValid(server.GetSubnetMask()));
    }

    // サーバーを終了します。
    server.StopServer();
}

TEST_F(LdnAutoIpServer, AddEntry)
{
    auto& server = *this->m_pServer;

    // MAC アドレスと IP アドレスのペアを登録します。
    nn::ldn::MacAddress macAddresses[HostAddressCountMax];
    nn::ldn::Ipv4Address ipv4Addresses[HostAddressCountMax];
    for (int i = 0; i < HostAddressCountMax; ++i)
    {
        auto host = ((i + 32) % HostAddressCountMax);
        macAddresses[host] =
            CreateRandomMacAddress();
        ipv4Addresses[host] =
            nn::ldn::MakeIpv4Address(169, 254, 32, static_cast<uint8_t>(host + 1));
        server.AddEntry(ipv4Addresses[host] , macAddresses[host]);
    }

    // サーバーを初期化します。
    server.StartServer(macAddresses[0]);
    auto serverIpv4Address = server.GetIpv4Address();
    ASSERT_EQ(ipv4Addresses[0], serverIpv4Address);
    auto subnetMask = server.GetSubnetMask();
    ASSERT_TRUE(IsValid(subnetMask));

    // AddEntry で登録した IP アドレスが割り当てられることを確認します。
    for (int i = HostAddressCountMax - 1; i >= 1; --i)
    {
        auto ipv4Address = server.Assign(macAddresses[i]);
        ASSERT_EQ(ipv4Addresses[i], ipv4Address);
    }

    // サーバーを終了します。
    server.StopServer();
}

TEST_F(LdnAutoIpClient, StartClientMultipleTimes)
{
    auto& client = *m_pClient;
    const int retryCountMax = 256;
    for (int i = 0; i < retryCountMax; ++i)
    {
        auto host = static_cast<uint8_t>(((i + 1) % HostAddressCountMax) + 1);
        auto gateway = static_cast<uint8_t>((i % HostAddressCountMax) + 1);
        auto hostAddress = nn::ldn::MakeIpv4Address(169, 254, 0, host);
        auto gatewayAddress = nn::ldn::MakeIpv4Address(169, 254, 0, gateway);
        client.StartClient(gatewayAddress, hostAddress);
        ASSERT_EQ(hostAddress, client.GetIpv4Address());
        ASSERT_EQ(gatewayAddress, client.GetGatewayAddress());
        ASSERT_TRUE(IsValid(client.GetSubnetMask()));
        client.StopClient();
    }
}

TEST_F(LdnAutoIpClient, StopClientInDestructor)
{
    auto& client = *m_pClient;
    auto address = nn::ldn::MakeIpv4Address(169, 254, 0, 2);
    auto gateway = nn::ldn::MakeIpv4Address(169, 254, 0, 1);
    client.StartClient(gateway, address);
    ASSERT_EQ(address, client.GetIpv4Address());
    ASSERT_EQ(gateway, client.GetGatewayAddress());
    ASSERT_TRUE(IsValid(client.GetSubnetMask()));
}
