﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <memory>
#include <nn/ldn/ldn_Types.h>
#include <nn/ldn/detail/NetworkInterface/ldn_Wlan.h>
#include <nnt.h>

namespace
{
    // このテストでは Wlan を使用します。
    typedef nn::ldn::detail::Wlan NetworkInterface;
    const size_t NetworkInterfaceBufferSize = sizeof(nn::ldn::detail::impl::WlanBuffer);

    // ネットワーク・インタフェース用のバッファです。
    NN_ALIGNAS(4096) char g_NetworkInterfaceBuffer[NetworkInterfaceBufferSize];

    // スキャン用のバッファです。
    nn::ldn::detail::L2ScanResult g_ScanBuffer[nn::ldn::ScanResultCountMax];

} // namespace <unnamed>

//
// OpenStation/CloseStation を繰り返し実行します。
//
TEST(OpenStation, MultipleTimes)
{
    std::unique_ptr<nn::ldn::detail::INetworkInterface> pNetworkInterface(
        new NetworkInterface(g_NetworkInterfaceBuffer, NetworkInterfaceBufferSize));
    ASSERT_EQ(nn::ldn::detail::L2State_None, pNetworkInterface->GetState());

    const int repeatCountMax = 64;
    for (int i = 0; i < repeatCountMax; ++i)
    {
        NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->OpenStation());
        ASSERT_EQ(nn::ldn::detail::L2State_Station, pNetworkInterface->GetState());
        NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->CloseStation());
        ASSERT_EQ(nn::ldn::detail::L2State_None, pNetworkInterface->GetState());
    }
}

//
// OpenStation や CloseStation でイベントがシグナルされないことを検証します。
//
TEST(OpenStation, NotSignaled)
{
    std::unique_ptr<nn::ldn::detail::INetworkInterface> pNetworkInterface(
        new NetworkInterface(g_NetworkInterfaceBuffer, NetworkInterfaceBufferSize));
    ASSERT_EQ(nn::ldn::detail::L2State_None, pNetworkInterface->GetState());

    // 状態変化を取得するイベントを取得します。
    auto& stateChangeEvent = pNetworkInterface->GetStateChangeEvent();
    ASSERT_FALSE(stateChangeEvent.TryWait());

    // OpenStation 関数でイベントがシグナルされないことを確認します。
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->OpenStation());
    ASSERT_EQ(nn::ldn::detail::L2State_Station, pNetworkInterface->GetState());
    ASSERT_FALSE(stateChangeEvent.TryWait());

    // CloseStation 関数でイベントがシグナルされないことを確認します。
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->CloseStation());
    ASSERT_EQ(nn::ldn::detail::L2State_None, pNetworkInterface->GetState());
    ASSERT_FALSE(stateChangeEvent.TryWait());
}

//
// OpenStation の後、 CloseStation をとばして Finalize します。
//
TEST(OpenStation, CloseStationByFinalize)
{
    // ネットワークインタフェースを初期化します。
    auto* pNetworkInterface = new NetworkInterface(
        g_NetworkInterfaceBuffer, NetworkInterfaceBufferSize);
    ASSERT_EQ(nn::ldn::detail::L2State_None, pNetworkInterface->GetState());

    // OpenStation 関数でステーションとしての動作を開始します。
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->OpenStation());
    ASSERT_EQ(nn::ldn::detail::L2State_Station, pNetworkInterface->GetState());

    // CloseStation を実行せず、 ネットワーク・インタフェースを解放して STA としての動作を停止します。
    delete pNetworkInterface;
}

//
// 自動チャンネルでスキャンを実行します。
//
TEST(Scan, AutoChannel)
{
    std::unique_ptr<nn::ldn::detail::INetworkInterface> pNetworkInterface(
        new NetworkInterface(g_NetworkInterfaceBuffer, NetworkInterfaceBufferSize));
    ASSERT_EQ(nn::ldn::detail::L2State_None, pNetworkInterface->GetState());

    // OpenStation 関数でステーションとしての動作を開始します。
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->OpenStation());
    ASSERT_EQ(nn::ldn::detail::L2State_Station, pNetworkInterface->GetState());

    // ネットワーク・インタフェースの詳細を取得します。
    nn::ldn::detail::NetworkInterfaceProfile profile;
    pNetworkInterface->GetNetworkInterfaceProfile(&profile);

    // 自動チャンネルスキャンを繰り返します。
    const int repeatCountMax = 16;
    for (int i = 0; i < repeatCountMax; ++i)
    {
        int count;
        NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->Scan(
            g_ScanBuffer, &count, nn::ldn::ScanResultCountMax, nn::ldn::AutoChannel));
        ASSERT_GE(count, 0);
        ASSERT_LE(count, nn::ldn::ScanResultCountMax);
        for (int i = 0; i < count; ++i)
        {
            ASSERT_TRUE(pNetworkInterface->IsAutoSelectableChannel(g_ScanBuffer[i].channel));
            ASSERT_LT(g_ScanBuffer[i].rssi, 0);
            ASSERT_LE(g_ScanBuffer[i].dataSize, profile.beaconDataSizeMax);
        }
    }

    // CloseStation 関数でステーションとしての動作を終了します。
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->CloseStation());
    ASSERT_EQ(nn::ldn::detail::L2State_None, pNetworkInterface->GetState());
}

//
// チャンネルを指定してスキャンを実行します。
//
TEST(Scan, ManualChannel)
{
    std::unique_ptr<nn::ldn::detail::INetworkInterface> pNetworkInterface(
        new NetworkInterface(g_NetworkInterfaceBuffer, NetworkInterfaceBufferSize));
    ASSERT_EQ(nn::ldn::detail::L2State_None, pNetworkInterface->GetState());

    // OpenStation 関数でステーションとしての動作を開始します。
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->OpenStation());
    ASSERT_EQ(nn::ldn::detail::L2State_Station, pNetworkInterface->GetState());

    // ネットワーク・インタフェースの詳細を取得します。
    nn::ldn::detail::NetworkInterfaceProfile profile;
    pNetworkInterface->GetNetworkInterfaceProfile(&profile);

    // 自動チャンネルスキャンを繰り返します。
    for (int i = 0; i < profile.supportedChannelCount; ++i)
    {
        int channel = profile.supportedChannels[i];
        int count;
        NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->Scan(
            g_ScanBuffer, &count, nn::ldn::ScanResultCountMax, channel));
        ASSERT_GE(count, 0);
        ASSERT_LE(count, nn::ldn::ScanResultCountMax);
        for (int i = 0; i < count; ++i)
        {
            ASSERT_EQ(channel, g_ScanBuffer[i].channel);
            ASSERT_LT(g_ScanBuffer[i].rssi, 0);
            ASSERT_LE(g_ScanBuffer[i].dataSize, profile.beaconDataSizeMax);
        }
    }

    // CloseStation 関数でステーションとしての動作を終了します。
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->CloseStation());
    ASSERT_EQ(nn::ldn::detail::L2State_None, pNetworkInterface->GetState());
}

//
// 接続前に GetConnectionStatus() を実行します。
//
TEST(GetConnectionStatus, BeforeConnect)
{
    std::unique_ptr<nn::ldn::detail::INetworkInterface> pNetworkInterface(
        new NetworkInterface(g_NetworkInterfaceBuffer, NetworkInterfaceBufferSize));
    ASSERT_EQ(nn::ldn::detail::L2State_None, pNetworkInterface->GetState());

    // OpenStation 関数でステーションとしての動作を開始します。
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->OpenStation());
    ASSERT_EQ(nn::ldn::detail::L2State_Station, pNetworkInterface->GetState());

    // ネットワーク・インタフェースの詳細を取得します。
    nn::ldn::detail::NetworkInterfaceProfile profile;
    pNetworkInterface->GetNetworkInterfaceProfile(&profile);

    // GetConnectionStatus 関数で現在の状態を取得します。
    nn::ldn::detail::L2StationInfo station;
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->GetConnectionStatus(&station));
    ASSERT_EQ(nn::ldn::detail::L2StationState_Disconnected, station.status);
    ASSERT_EQ(profile.physicalAddress, station.macAddress);
    ASSERT_EQ(nn::ldn::detail::L2DisconnectReason_Unknown, station.disconnectReason);
    ASSERT_EQ(nn::ldn::detail::RssiMin, station.rssi);

    // CloseStation 関数でステーションとしての動作を終了します。
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->CloseStation());
    ASSERT_EQ(nn::ldn::detail::L2State_None, pNetworkInterface->GetState());
}
