﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <nn/ldn.h>
#include <nn/ldn/ldn_PrivateApi.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nnt.h>
#include <nnt/ldn/testLdn_Assert.h>
#include <nnt/ldn/testLdn_CommandLineParser.h>
#include <nnt/ldn/testLdn_HtcsSynchronization.h>
#include <nnt/ldn/testLdn_Log.h>
#include <nnt/ldn/testLdn_NifmUtility.h>
#include <nnt/ldn/testLdn_Utility.h>
#include <nnt/result/testResult_Assert.h>
#include "../Common/testLdn_Definitions.h"

namespace
{
    // テストの設定値です。
    nnt::ldn::TestConfig g_TestConfig;

    // 同期に使用するオブジェクトとバッファです。
    nnt::ldn::ISynchronizationServer* g_pServer;
    char g_SynchronizationBuffer[16 * 1024];

    // 同期用のマクロです。
    #define NNT_LDN_SYNC(keyword)\
        ASSERT_EQ(::nnt::ldn::SynchronizationResult_Success, g_pServer->Synchronize(keyword))

    void SetLdnSettings() NN_NOEXCEPT
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::SetOperationMode(
            static_cast<nn::ldn::OperationMode>(g_TestConfig.operationMode)));
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::SetWirelessControllerRestriction(
            static_cast<nn::ldn::WirelessControllerRestriction>(
                g_TestConfig.wirelessControllerRestriction)));
    }

} // namespace <unnamed>

//
// パスフレーズが異なるステーションが接続できないことを確認します。
//
TEST(Restrict, DifferentKey)
{
    // テストケース開始前の同期です。
    NNT_LDN_SYNC("Restrict.DifferentKey.Start");

    // このテストは 2 台以上のノードが存在しなければ実行できません。
    int nodeCountMax = std::min<int>(nn::ldn::NodeCountMax, g_TestConfig.nodeCount);
    if (nodeCountMax <= 1)
    {
        return;
    }

    // AP としての動作を開始します。
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    nnt::ldn::AccessPointStarter starter;
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());

    // ネットワークのパラメータを適当に設定します。
    nn::ldn::NetworkConfig network = { };
    network.intentId = nn::ldn::MakeIntentId(LocalId, g_TestConfig.sceneId);
    network.nodeCountMax = static_cast<int8_t>(nodeCountMax);
    network.channel = g_TestConfig.channel;
    nn::ldn::SecurityConfig security = { };
    security.securityMode = static_cast<nn::Bit16>(nn::ldn::SecurityMode_Product);
    security.passphraseSize = sizeof(Passphrase);
    std::memcpy(security.passphrase, Passphrase, sizeof(Passphrase));
    nn::ldn::UserConfig user = { };
    std::strncpy(user.userName, ApUserName, nn::ldn::UserNameBytesMax);

    // ネットワークを構築します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetwork(network, security, user));
    ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetState());
    NNT_LDN_SYNC("Restrict.DifferentKey.NetworkCreated");

    // 一定時間、ステーションの接続を受け入れます。
    NNT_LDN_SYNC("Restrict.DifferentKey.Connected");

    // ステーションの接続数を制限できていることを確認します。
    nn::ldn::NetworkInfo networkInfo;
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(&networkInfo));
    ASSERT_EQ(nodeCountMax, networkInfo.ldn.nodeCountMax);
    ASSERT_EQ(1, networkInfo.ldn.nodeCount);
    int nodeCount = static_cast<int>(std::count_if(
        std::begin(networkInfo.ldn.nodes), std::end(networkInfo.ldn.nodes),
        [](const nn::ldn::NodeInfo& node) { return node.isConnected; }));
    ASSERT_EQ(1, nodeCount);

    // ネットワークを破棄します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::DestroyNetwork());
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());
    NNT_LDN_SYNC("Restrict.DifferentKey.NetworkDestroyed");
}

//
// 接続台数制限の確認です。ノードの最大接続数を 1 ～ 7 台に制限します。
//
TEST(Restrict, NodeCount)
{
    // テストケース開始前の同期です。
    NNT_LDN_SYNC("Restrict.NodeCount.Start");

    // 少なくとも 1 台のステーションが接続できない nodeCountMax を計算します。
    int nodeCountMax = std::min<int>(nn::ldn::NodeCountMax, g_TestConfig.nodeCount) - 1;

    // AP としての動作を開始します。
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    nnt::ldn::AccessPointStarter starter;
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());

    // ネットワークのパラメータを適当に設定します。
    nn::ldn::NetworkConfig network = { };
    network.intentId = nn::ldn::MakeIntentId(LocalId, g_TestConfig.sceneId);
    network.nodeCountMax = static_cast<int8_t>(nodeCountMax);
    network.channel = g_TestConfig.channel;
    nn::ldn::SecurityConfig security = { };
    security.securityMode = static_cast<nn::Bit16>(nn::ldn::SecurityMode_Product);
    security.passphraseSize = sizeof(Passphrase);
    std::memcpy(security.passphrase, Passphrase, sizeof(Passphrase));
    nn::ldn::UserConfig user = { };
    std::strncpy(user.userName, ApUserName, nn::ldn::UserNameBytesMax);

    // ネットワークを構築します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetwork(network, security, user));
    ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetState());
    NNT_LDN_SYNC("Restrict.NodeCount.NetworkCreated");

    // 一定時間、ステーションの接続を受け入れます。
    NNT_LDN_SYNC("Restrict.NodeCount.Connected");

    // ステーションの接続数を制限できていることを確認します。
    nn::ldn::NetworkInfo networkInfo;
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(&networkInfo));
    ASSERT_EQ(nodeCountMax, networkInfo.ldn.nodeCountMax);
    ASSERT_EQ(nodeCountMax, networkInfo.ldn.nodeCount);
    int nodeCount = static_cast<int>(std::count_if(
        std::begin(networkInfo.ldn.nodes), std::end(networkInfo.ldn.nodes),
        [](const nn::ldn::NodeInfo& node) { return node.isConnected; }));
    ASSERT_EQ(nodeCountMax, nodeCount);

    // ネットワークを破棄します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::DestroyNetwork());
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());
}

//
// 接続台数制限の確認です。ノードの最大接続数は 8 台です。
//
TEST(Restrict, NodeCountMax)
{
    // テストケース開始前の同期です。
    NNT_LDN_SYNC("Restrict.NodeCountMax.Start");

    // このテストは 9 台以上のノードが存在しなければ実行できません。
    int nodeCountMax = nn::ldn::NodeCountMax;
    if (g_TestConfig.nodeCount <= nodeCountMax)
    {
        return;
    }

    // AP としての動作を開始します。
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    nnt::ldn::AccessPointStarter starter;
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());

    // ネットワークのパラメータを適当に設定します。
    nn::ldn::NetworkConfig network = { };
    network.intentId = nn::ldn::MakeIntentId(LocalId, g_TestConfig.sceneId);
    network.nodeCountMax = static_cast<int8_t>(nodeCountMax);
    network.channel = g_TestConfig.channel;
    nn::ldn::SecurityConfig security = { };
    security.securityMode = static_cast<nn::Bit16>(nn::ldn::SecurityMode_Product);
    security.passphraseSize = sizeof(Passphrase);
    std::memcpy(security.passphrase, Passphrase, sizeof(Passphrase));
    nn::ldn::UserConfig user = { };
    std::strncpy(user.userName, ApUserName, nn::ldn::UserNameBytesMax);

    // ネットワークを構築します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetwork(network, security, user));
    ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetState());
    NNT_LDN_SYNC("Restrict.NodeCountMax.NetworkCreated");

    // 一定時間、ステーションの接続を受け入れます。
    NNT_LDN_SYNC("Restrict.NodeCountMax.Connected");

    // ステーションの接続数を制限できていることを確認します。
    nn::ldn::NetworkInfo networkInfo;
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(&networkInfo));
    ASSERT_EQ(nodeCountMax, networkInfo.ldn.nodeCountMax);
    ASSERT_EQ(nodeCountMax, networkInfo.ldn.nodeCount);
    int nodeCount = static_cast<int>(std::count_if(
        std::begin(networkInfo.ldn.nodes), std::end(networkInfo.ldn.nodes),
        [](const nn::ldn::NodeInfo& node) { return node.isConnected; }));
    ASSERT_EQ(nodeCountMax, nodeCount);

    // ネットワークを破棄します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::DestroyNetwork());
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());
}

//
// AcceptPolicy による接続制限です。
//
TEST(Restrict, AcceptPolicyAlwaysReject)
{
    // テストケース開始前の同期です。
    NNT_LDN_SYNC("Restrict.AcceptPolicyAlwaysReject.Start");

    // このテストは 2 台以上のノードが存在しなければ実行できません。
    int nodeCountMax = std::min<int>(nn::ldn::NodeCountMax, g_TestConfig.nodeCount);
    if (g_TestConfig.nodeCount <= 1)
    {
        return;
    }

    // AP としての動作を開始します。
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    nnt::ldn::AccessPointStarter starter;
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());

    // ネットワークのパラメータを適当に設定します。
    nn::ldn::NetworkConfig network = { };
    network.intentId = nn::ldn::MakeIntentId(LocalId, g_TestConfig.sceneId);
    network.nodeCountMax = static_cast<int8_t>(nodeCountMax);
    network.channel = g_TestConfig.channel;
    nn::ldn::SecurityConfig security = { };
    security.securityMode = static_cast<nn::Bit16>(nn::ldn::SecurityMode_Product);
    security.passphraseSize = sizeof(Passphrase);
    std::memcpy(security.passphrase, Passphrase, sizeof(Passphrase));
    nn::ldn::UserConfig user = { };
    std::strncpy(user.userName, ApUserName, nn::ldn::UserNameBytesMax);

    // AcceptPolicy を設定します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::SetStationAcceptPolicy(nn::ldn::AcceptPolicy_AlwaysAccept));

    // ネットワークを構築します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetwork(network, security, user));
    ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetState());
    NNT_LDN_SYNC("Restrict.AcceptPolicyAlwaysReject.NetworkCreated");

    // ステーションの接続まで待機します。
    NNT_LDN_SYNC("Restrict.AcceptPolicyAlwaysReject.Connected");
    nn::ldn::NetworkInfo networkInfo;
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(&networkInfo));
    ASSERT_EQ(nodeCountMax, networkInfo.ldn.nodeCountMax);
    ASSERT_EQ(nodeCountMax, networkInfo.ldn.nodeCount);
    int nodeCount = static_cast<int>(std::count_if(
        std::begin(networkInfo.ldn.nodes), std::end(networkInfo.ldn.nodes),
        [](const nn::ldn::NodeInfo& node) { return node.isConnected; }));
    ASSERT_EQ(nodeCountMax, nodeCount);

    // ステーションが一切接続できない AcceptPolicy_AlwaysReject に切り替えて、
    // この変更が Advertise でステーションに通知されるまで待機します。
    NNT_ASSERT_RESULT_SUCCESS(
        nn::ldn::SetStationAcceptPolicy(nn::ldn::AcceptPolicy_AlwaysReject));
    nn::os::SleepThread(AdvertisingTime);
    NNT_LDN_SYNC("Restrict.AcceptPolicyAlwaysReject.AcceptPolicyUpdated");

    // すぐにステーションが全て切断されることはありません。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(&networkInfo));
    ASSERT_EQ(nodeCountMax, networkInfo.ldn.nodeCountMax);
    ASSERT_EQ(nodeCountMax, networkInfo.ldn.nodeCount);
    nodeCount = static_cast<int>(std::count_if(
        std::begin(networkInfo.ldn.nodes), std::end(networkInfo.ldn.nodes),
        [](const nn::ldn::NodeInfo& node) { return node.isConnected; }));
    ASSERT_EQ(nodeCountMax, nodeCount);

    // 再度ステーションを募集して、AcceptPolicy が有効なことを確認します。
    for (int i = 1; i < nn::ldn::NodeCountMax; ++i)
    {
        const auto& node = networkInfo.ldn.nodes[i];
        if (node.isConnected)
        {
            NNT_ASSERT_RESULT_SUCCESS(nn::ldn::Reject(node.ipv4Address));
        }
    }
    NNT_LDN_SYNC("Restrict.AcceptPolicyAlwaysReject.Rejected");
    NNT_LDN_SYNC("Restrict.AcceptPolicyAlwaysReject.Reconnected");
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(&networkInfo));
    ASSERT_EQ(nodeCountMax, networkInfo.ldn.nodeCountMax);
    ASSERT_EQ(1, networkInfo.ldn.nodeCount);
    nodeCount = static_cast<int>(std::count_if(
        std::begin(networkInfo.ldn.nodes), std::end(networkInfo.ldn.nodes),
        [](const nn::ldn::NodeInfo& node) { return node.isConnected; }));
    ASSERT_EQ(1, nodeCount);

    // ネットワークを破棄します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::DestroyNetwork());
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());
    NNT_LDN_SYNC("Restrict.AcceptPolicyAlwaysReject.NetworkDestroyed");
}

//
// AcceptPolicy による接続制限です。
//
TEST(Restrict, AcceptPolicyBlackList)
{
    // テストケース開始前の同期です。
    NNT_LDN_SYNC("Restrict.AcceptPolicyBlackList.Start");

    // このテストは 2 台以上のノードが存在しなければ実行できません。
    int nodeCountMax = std::min<int>(nn::ldn::NodeCountMax, g_TestConfig.nodeCount);
    if (g_TestConfig.nodeCount <= 1)
    {
        return;
    }

    // AP としての動作を開始します。
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    nnt::ldn::AccessPointStarter starter;
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());

    // 状態変化を監視するイベントを取得します。
    nn::os::SystemEvent stateChangeEvent;
    nn::ldn::AttachStateChangeEvent(stateChangeEvent.GetBase());

    // ネットワークのパラメータを適当に設定します。
    nn::ldn::NetworkConfig network = { };
    network.intentId = nn::ldn::MakeIntentId(LocalId, g_TestConfig.sceneId);
    network.nodeCountMax = static_cast<int8_t>(nodeCountMax);
    network.channel = g_TestConfig.channel;
    nn::ldn::SecurityConfig security = { };
    security.securityMode = static_cast<nn::Bit16>(nn::ldn::SecurityMode_Product);
    security.passphraseSize = sizeof(Passphrase);
    std::memcpy(security.passphrase, Passphrase, sizeof(Passphrase));
    nn::ldn::UserConfig user = { };
    std::strncpy(user.userName, ApUserName, nn::ldn::UserNameBytesMax);

    // AcceptPolicy を設定します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::SetStationAcceptPolicy(nn::ldn::AcceptPolicy_AlwaysAccept));

    // ネットワークを構築します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetwork(network, security, user));
    ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetState());
    NNT_LDN_SYNC("Restrict.AcceptPolicyBlackList.NetworkCreated");

    // 最大数のステーションが接続できるはずです。
    NNT_LDN_SYNC("Restrict.AcceptPolicyBlackList.Connected");
    nn::ldn::NetworkInfo networkInfo;
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(&networkInfo));
    ASSERT_EQ(nodeCountMax, networkInfo.ldn.nodeCountMax);
    ASSERT_EQ(nodeCountMax, networkInfo.ldn.nodeCount);
    int nodeCount = static_cast<int>(std::count_if(
        std::begin(networkInfo.ldn.nodes), std::end(networkInfo.ldn.nodes),
        [](const nn::ldn::NodeInfo& node) { return node.isConnected; }));
    ASSERT_EQ(nodeCountMax, nodeCount);

    // 指定されたステーションを拒絶する AcceptPolicy_BlackList に切り替えて、
    // この変更が Advertise でステーションに通知されるまで待機します。
    nn::ldn::NodeInfo filteredNode = networkInfo.ldn.nodes[1];
    NNT_ASSERT_RESULT_SUCCESS(
        nn::ldn::SetStationAcceptPolicy(nn::ldn::AcceptPolicy_BlackList));
    nn::ldn::AddAcceptFilterEntry(filteredNode);
    nn::os::SleepThread(AdvertisingTime);
    NNT_LDN_SYNC("Restrict.AcceptPolicyBlackList.AcceptPolicyUpdated");

    // すぐにステーションが全て切断されることはありません。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(&networkInfo));
    ASSERT_EQ(nodeCountMax, networkInfo.ldn.nodeCountMax);
    ASSERT_EQ(nodeCountMax, networkInfo.ldn.nodeCount);
    nodeCount = static_cast<int>(std::count_if(
        std::begin(networkInfo.ldn.nodes), std::end(networkInfo.ldn.nodes),
        [](const nn::ldn::NodeInfo& node) { return node.isConnected; }));
    ASSERT_EQ(nodeCountMax, nodeCount);

    // 再度ステーションを募集して、 AcceptPolicy が有効なことを確認します。
    for (int i = 1; i < nn::ldn::NodeCountMax; ++i)
    {
        const auto& node = networkInfo.ldn.nodes[i];
        if (node.isConnected)
        {
            NNT_ASSERT_RESULT_SUCCESS(nn::ldn::Reject(node.ipv4Address));
        }
    }
    NNT_LDN_SYNC("Restrict.AcceptPolicyBlackList.Disconnected");
    NNT_LDN_SYNC("Restrict.AcceptPolicyBlackList.Reconnected");
    do
    {
        ASSERT_TRUE(stateChangeEvent.TimedWait(AdvertisingTime));
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(&networkInfo));
    } while (networkInfo.ldn.nodeCount < nodeCountMax - 1);
    ASSERT_EQ(nodeCountMax, networkInfo.ldn.nodeCountMax);
    ASSERT_EQ(nodeCountMax - 1, networkInfo.ldn.nodeCount);
    ASSERT_FALSE(std::any_of(
        std::begin(networkInfo.ldn.nodes), std::end(networkInfo.ldn.nodes),
        [&filteredNode](const nn::ldn::NodeInfo& node)
        {
            return node.macAddress == filteredNode.macAddress;
        }));

    // ネットワークを破棄します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::DestroyNetwork());
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());
}

//
// AcceptPolicy による接続制限です。
//
TEST(Restrict, AcceptPolicyWhiteList)
{
    // テストケース開始前の同期です。
    NNT_LDN_SYNC("Restrict.AcceptPolicyWhiteList.Start");

    // このテストは 2 台以上のノードが存在しなければ実行できません。
    int nodeCountMax = std::min<int>(nn::ldn::NodeCountMax, g_TestConfig.nodeCount);
    if (g_TestConfig.nodeCount <= 1)
    {
        return;
    }

    // AP としての動作を開始します。
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    nnt::ldn::AccessPointStarter starter;
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());

    // 状態変化を監視するイベントを取得します。
    nn::os::SystemEvent stateChangeEvent;
    nn::ldn::AttachStateChangeEvent(stateChangeEvent.GetBase());

    // ネットワークのパラメータを適当に設定します。
    nn::ldn::NetworkConfig network = { };
    network.intentId = nn::ldn::MakeIntentId(LocalId, g_TestConfig.sceneId);
    network.nodeCountMax = static_cast<int8_t>(nodeCountMax);
    network.channel = g_TestConfig.channel;
    nn::ldn::SecurityConfig security = { };
    security.securityMode = static_cast<nn::Bit16>(nn::ldn::SecurityMode_Product);
    security.passphraseSize = sizeof(Passphrase);
    std::memcpy(security.passphrase, Passphrase, sizeof(Passphrase));
    nn::ldn::UserConfig user = { };
    std::strncpy(user.userName, ApUserName, nn::ldn::UserNameBytesMax);

    // AcceptPolicy を設定します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::SetStationAcceptPolicy(nn::ldn::AcceptPolicy_AlwaysAccept));

    // ネットワークを構築します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetwork(network, security, user));
    ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetState());
    NNT_LDN_SYNC("Restrict.AcceptPolicyWhiteList.NetworkCreated");

    // 最大数のステーションが接続できるはずです。
    NNT_LDN_SYNC("Restrict.AcceptPolicyWhiteList.Connected");
    nn::ldn::NetworkInfo networkInfo;
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(&networkInfo));
    ASSERT_EQ(nodeCountMax, networkInfo.ldn.nodeCountMax);
    ASSERT_EQ(nodeCountMax, networkInfo.ldn.nodeCount);
    int nodeCount = static_cast<int>(std::count_if(
        std::begin(networkInfo.ldn.nodes), std::end(networkInfo.ldn.nodes),
        [](const nn::ldn::NodeInfo& node) { return node.isConnected; }));
    ASSERT_EQ(nodeCountMax, nodeCount);

    // 指定されたステーションを受け入れる AcceptPolicy_WhiteList に切り替えて、
    // この変更が Advertise でステーションに通知されるまで待機します。
    nn::ldn::NodeInfo filteredNode = networkInfo.ldn.nodes[1];
    NNT_ASSERT_RESULT_SUCCESS(
        nn::ldn::SetStationAcceptPolicy(nn::ldn::AcceptPolicy_WhiteList));
    nn::ldn::AddAcceptFilterEntry(filteredNode);
    nn::os::SleepThread(AdvertisingTime);
    NNT_LDN_SYNC("Restrict.AcceptPolicyWhiteList.AcceptPolicyUpdated");

    // すぐにステーションが全て切断されることはありません。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(&networkInfo));
    ASSERT_EQ(nodeCountMax, networkInfo.ldn.nodeCountMax);
    ASSERT_EQ(nodeCountMax, networkInfo.ldn.nodeCount);
    nodeCount = static_cast<int>(std::count_if(
        std::begin(networkInfo.ldn.nodes), std::end(networkInfo.ldn.nodes),
        [](const nn::ldn::NodeInfo& node) { return node.isConnected; }));
    ASSERT_EQ(nodeCountMax, nodeCount);

    // 再度ステーションを募集して、 AcceptPolicy が有効なことを確認します。
    for (int i = 1; i < nn::ldn::NodeCountMax; ++i)
    {
        const auto& node = networkInfo.ldn.nodes[i];
        if (node.isConnected)
        {
            NNT_ASSERT_RESULT_SUCCESS(nn::ldn::Reject(node.ipv4Address));
        }
    }
    NNT_LDN_SYNC("Restrict.AcceptPolicyWhiteList.Disconnected");
    NNT_LDN_SYNC("Restrict.AcceptPolicyWhiteList.Reconnected");
    do
    {
        ASSERT_TRUE(stateChangeEvent.TimedWait(AdvertisingTime));
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(&networkInfo));
    } while (networkInfo.ldn.nodeCount <= 1);
    ASSERT_EQ(nodeCountMax, networkInfo.ldn.nodeCountMax);
    ASSERT_EQ(2, networkInfo.ldn.nodeCount);
    ASSERT_TRUE(std::any_of(
        std::begin(networkInfo.ldn.nodes), std::end(networkInfo.ldn.nodes),
        [&filteredNode](const nn::ldn::NodeInfo& node)
        {
            return node.macAddress == filteredNode.macAddress;
        }));

    // ネットワークを破棄します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::DestroyNetwork());
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());
}

//
// ネットワークの破棄でイベントがシグナルされることを検証します。
//
TEST(StateChangeEvent, DestroyNetwork)
{
    // テストケース開始前の同期です。
    NNT_LDN_SYNC("StateChangeEvent.DestroyNetwork.Start");

    // 最大台数のステーションを接続させます。
    int nodeCountMax = std::min<int>(nn::ldn::NodeCountMax, g_TestConfig.nodeCount);

    // AP としての動作を開始します。
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    nnt::ldn::AccessPointStarter starter;
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());

    // 状態変化を監視するイベントを取得します。
    nn::os::SystemEvent stateChangeEvent;
    nn::ldn::AttachStateChangeEvent(stateChangeEvent.GetBase());

    // ネットワークのパラメータを適当に設定します。
    nn::ldn::NetworkConfig network = { };
    network.intentId = nn::ldn::MakeIntentId(LocalId, g_TestConfig.sceneId);
    network.nodeCountMax = static_cast<int8_t>(nodeCountMax);
    network.channel = g_TestConfig.channel;
    nn::ldn::SecurityConfig security = { };
    security.securityMode = static_cast<nn::Bit16>(nn::ldn::SecurityMode_Product);
    security.passphraseSize = sizeof(Passphrase);
    std::memcpy(security.passphrase, Passphrase, sizeof(Passphrase));
    nn::ldn::UserConfig user = { };
    std::strncpy(user.userName, ApUserName, nn::ldn::UserNameBytesMax);

    // 最初はステーションが接続できない状態にします。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::SetStationAcceptPolicy(nn::ldn::AcceptPolicy_AlwaysReject));

    // ネットワークを構築し、イベントがシグナルされていることを確認します。
    ASSERT_FALSE(stateChangeEvent.TryWait());
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetwork(network, security, user));
    ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetState());
    ASSERT_TRUE(stateChangeEvent.TryWait());
    NNT_LDN_SYNC("StateChangeEvent.DestroyNetwork.NetworkCreated");

    // イベントは AutoClear です。
    ASSERT_FALSE(stateChangeEvent.TryWait());

    // この状態でステーションの接続を示すイベントがシグナルされないことを確認します。
    ASSERT_FALSE(stateChangeEvent.TimedWait(AdvertisingTime));

    // ステーションの接続を許可します。
    nn::ldn::NetworkInfo networkInfo;
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(&networkInfo));
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::SetStationAcceptPolicy(nn::ldn::AcceptPolicy_AlwaysAccept));
    NNT_LDN_SYNC("StateChangeEvent.DestroyNetwork.AcceptPolicyUpdated");

    // ステーションの接続時にイベントが発生することを確認します。
    // ステーションが最大数接続するまで繰り返します。
    while (networkInfo.ldn.nodeCount < nodeCountMax)
    {
        NNT_LDN_ASSERT_SIGNALED_WEAK(stateChangeEvent, ConnectionTimeout);
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(&networkInfo));
    }
    NNT_LDN_SYNC("StateChangeEvent.DestroyNetwork.AdvertiseReceived");

    // ネットワークの破棄でイベントが発生することを検証します。
    stateChangeEvent.TryWait();
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::DestroyNetwork());
    ASSERT_TRUE(stateChangeEvent.TryWait());
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());
}

//
// ステーションが自発的に切断した際にイベントがシグナルされることを検証します。
//
TEST(StateChangeEvent, Disconect)
{
    // テストケース開始前の同期です。
    NNT_LDN_SYNC("StateChangeEvent.Disconect.Start");

    // 最大台数のステーションを接続させます。
    int nodeCountMax = std::min<int>(nn::ldn::NodeCountMax, g_TestConfig.nodeCount);

    // AP としての動作を開始します。
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    nnt::ldn::AccessPointStarter starter;
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());

    // 状態変化を監視するイベントを取得します。
    nn::os::SystemEvent stateChangeEvent;
    nn::ldn::AttachStateChangeEvent(stateChangeEvent.GetBase());

    // ネットワークのパラメータを適当に設定します。
    nn::ldn::NetworkConfig network = { };
    network.intentId = nn::ldn::MakeIntentId(LocalId, g_TestConfig.sceneId);
    network.nodeCountMax = static_cast<int8_t>(nodeCountMax);
    network.channel = g_TestConfig.channel;
    nn::ldn::SecurityConfig security = { };
    security.securityMode = static_cast<nn::Bit16>(nn::ldn::SecurityMode_Product);
    security.passphraseSize = sizeof(Passphrase);
    std::memcpy(security.passphrase, Passphrase, sizeof(Passphrase));
    nn::ldn::UserConfig user = { };
    std::strncpy(user.userName, ApUserName, nn::ldn::UserNameBytesMax);

    // 最初はステーションが接続できない状態にします。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::SetStationAcceptPolicy(nn::ldn::AcceptPolicy_AlwaysReject));

    // ネットワークを構築し、イベントがシグナルされていることを確認します。
    ASSERT_FALSE(stateChangeEvent.TryWait());
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetwork(network, security, user));
    ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetState());
    ASSERT_TRUE(stateChangeEvent.TryWait());
    NNT_LDN_SYNC("StateChangeEvent.Disconect.NetworkCreated");

    // イベントは AutoClear です。
    ASSERT_FALSE(stateChangeEvent.TryWait());

    // この状態でステーションの接続を示すイベントがシグナルされないことを確認します。
    ASSERT_FALSE(stateChangeEvent.TimedWait(AdvertisingTime));

    // ステーションの接続を許可します。
    nn::ldn::NetworkInfo networkInfo;
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::SetStationAcceptPolicy(nn::ldn::AcceptPolicy_AlwaysAccept));
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(&networkInfo));
    NNT_LDN_SYNC("StateChangeEvent.Disconect.AcceptPolicyUpdated");

    // ステーションの接続時にイベントが発生することを確認します。
    // ステーションが最大数接続するまで繰り返します。
    while (networkInfo.ldn.nodeCount < nodeCountMax)
    {
        NNT_LDN_ASSERT_SIGNALED_WEAK(stateChangeEvent, ConnectionTimeout);
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(&networkInfo));
    }
    stateChangeEvent.TryWait();
    NNT_LDN_SYNC("StateChangeEvent.Disconect.AdvertiseReceived");

    // ステーションが自発的に切断するまで待機し、イベントが発生することを確認します。
    while (1 < networkInfo.ldn.nodeCount)
    {
        NNT_LDN_ASSERT_SIGNALED_WEAK(stateChangeEvent, DisconnectTimeout);
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(&networkInfo));
    }
    ASSERT_EQ(1, networkInfo.ldn.nodeCount);

    // ネットワークの破棄でイベントが発生することを検証します。
    NNT_LDN_SYNC("StateChangeEvent.Disconect.DestroyReady");
    ASSERT_FALSE(stateChangeEvent.TryWait());
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::DestroyNetwork());
    ASSERT_TRUE(stateChangeEvent.TryWait());
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());
}

//
// アクセスポイントがステーションを明示的に切断した際にイベントがシグナルされることを検証します。
//
TEST(StateChangeEvent, Reject)
{
    // テストケース開始前の同期です。
    NNT_LDN_SYNC("StateChangeEvent.Reject.Start");

    // 最大台数のステーションを接続させます。
    int nodeCountMax = std::min<int>(nn::ldn::NodeCountMax, g_TestConfig.nodeCount);

    // AP としての動作を開始します。
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    nnt::ldn::AccessPointStarter starter;
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());

    // 状態変化を監視するイベントを取得します。
    nn::os::SystemEvent stateChangeEvent;
    nn::ldn::AttachStateChangeEvent(stateChangeEvent.GetBase());

    // ネットワークのパラメータを適当に設定します。
    nn::ldn::NetworkConfig network = { };
    network.intentId = nn::ldn::MakeIntentId(LocalId, g_TestConfig.sceneId);
    network.nodeCountMax = static_cast<int8_t>(nodeCountMax);
    network.channel = g_TestConfig.channel;
    nn::ldn::SecurityConfig security = { };
    security.securityMode = static_cast<nn::Bit16>(nn::ldn::SecurityMode_Product);
    security.passphraseSize = sizeof(Passphrase);
    std::memcpy(security.passphrase, Passphrase, sizeof(Passphrase));
    nn::ldn::UserConfig user = { };
    std::strncpy(user.userName, ApUserName, nn::ldn::UserNameBytesMax);

    // 最初はステーションが接続できない状態にします。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::SetStationAcceptPolicy(nn::ldn::AcceptPolicy_AlwaysReject));

    // ネットワークを構築し、イベントがシグナルされていることを確認します。
    ASSERT_FALSE(stateChangeEvent.TryWait());
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetwork(network, security, user));
    ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetState());
    ASSERT_TRUE(stateChangeEvent.TryWait());
    NNT_LDN_SYNC("StateChangeEvent.Reject.NetworkCreated");

    // イベントは AutoClear です。
    ASSERT_FALSE(stateChangeEvent.TryWait());

    // この状態でステーションの接続を示すイベントがシグナルされないことを確認します。
    ASSERT_FALSE(stateChangeEvent.TimedWait(AdvertisingTime));

    // ステーションの接続を許可します。
    nn::ldn::NetworkInfo networkInfo;
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::SetStationAcceptPolicy(nn::ldn::AcceptPolicy_AlwaysAccept));
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(&networkInfo));
    NNT_LDN_SYNC("StateChangeEvent.Reject.AcceptPolicyUpdated");

    // ステーションの接続時にイベントが発生することを確認します。
    // ステーションが最大数接続するまで繰り返します。
    while (networkInfo.ldn.nodeCount < nodeCountMax)
    {
        NNT_LDN_ASSERT_SIGNALED_WEAK(stateChangeEvent, ConnectionTimeout);
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(&networkInfo));
    }
    NNT_LDN_SYNC("StateChangeEvent.DestroyNetwork.AdvertiseReceived");
    stateChangeEvent.TryWait();

    // ステーションを明示的に切断し、イベントが発生することを確認します。
    for (int i = 1; i < nodeCountMax; ++i)
    {
        ASSERT_FALSE(stateChangeEvent.TryWait());
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::Reject(networkInfo.ldn.nodes[i].ipv4Address));
        ASSERT_TRUE(stateChangeEvent.TryWait());
    }
    ASSERT_FALSE(stateChangeEvent.TryWait());

    // ネットワークの破棄でイベントが発生することを検証します。
    ASSERT_FALSE(stateChangeEvent.TryWait());
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::DestroyNetwork());
    ASSERT_TRUE(stateChangeEvent.TryWait());
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());
}

//
// 初期状態では Advertise に空のユーザデータが登録されていることを検証します。
//
TEST(Advertise, UserDataEmpty)
{
    // テストケース開始前の同期です。
    NNT_LDN_SYNC("Advertise.UserDataEmpty.Start");

    // 最大台数のステーションを接続させます。
    int nodeCountMax = std::min<int>(nn::ldn::NodeCountMax, g_TestConfig.nodeCount);

    // LDN ライブラリを初期化します。
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();

    // 状態変化を監視するイベントを取得します。
    nn::os::SystemEvent stateChangeEvent;
    nn::ldn::AttachStateChangeEvent(stateChangeEvent.GetBase());

    // 各セキュリティ・モードについてテストを実施します。
    nn::ldn::SecurityMode securities[] = { nn::ldn::SecurityMode_Debug,
        nn::ldn::SecurityMode_Product };
    for (auto securityMode : securities)
    {
        // AP としての動作を開始します。
        nnt::ldn::AccessPointStarter starter;
        ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());

        // ネットワークのパラメータを適当に設定します。
        nn::ldn::NetworkConfig network = {};
        network.intentId = nn::ldn::MakeIntentId(LocalId, g_TestConfig.sceneId);
        network.nodeCountMax = static_cast<int8_t>(nodeCountMax);
        network.channel = g_TestConfig.channel;
        nn::ldn::SecurityConfig security = {};
        security.securityMode = static_cast<nn::Bit16>(securityMode);
        security.passphraseSize = sizeof(Passphrase);
        std::memcpy(security.passphrase, Passphrase, sizeof(Passphrase));
        nn::ldn::UserConfig user = {};
        std::strncpy(user.userName, ApUserName, nn::ldn::UserNameBytesMax);

        // ネットワークを構築します。
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetwork(network, security, user));
        ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetState());
        NNT_LDN_SYNC("Advertise.UserDataEmpty.NetworkCreated");

        // ステーションの接続を待機します。
        NNT_LDN_SYNC("Advertise.UserDataEmpty.Connected");
        nn::ldn::NetworkInfo networkInfo;
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(&networkInfo));
        ASSERT_EQ(networkInfo.ldn.nodeCountMax, networkInfo.ldn.nodeCount);
        ASSERT_EQ(0U, networkInfo.ldn.advertiseDataSize);

        // ステーションが Advertise を受信するまで待機します。
        NNT_LDN_SYNC("Advertise.UserDataEmpty.AdvertiseReceived");

        // ネットワークを破棄します。
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::DestroyNetwork());
    }
}

//
// Advertise を正しく更新できることを検証します。
//
TEST(Advertise, UserDataUpdate)
{
    // テストケース開始前の同期です。
    NNT_LDN_SYNC("Advertise.UserDataUpdate.Start");

    // 最大台数のステーションを接続させます。
    int nodeCountMax = std::min<int>(nn::ldn::NodeCountMax, g_TestConfig.nodeCount);

    // LDN ライブラリを初期化します。
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();

    // 状態変化を監視するイベントを取得します。
    nn::os::SystemEvent stateChangeEvent;
    nn::ldn::AttachStateChangeEvent(stateChangeEvent.GetBase());

    // 各セキュリティ・モードについてテストを実施します。
    nn::ldn::SecurityMode securities[] = { nn::ldn::SecurityMode_Debug,
        nn::ldn::SecurityMode_Product };
    for (auto securityMode : securities)
    {
        // AP としての動作を開始します。
        nnt::ldn::AccessPointStarter starter;
        ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());

        // ネットワークのパラメータを適当に設定します。
        nn::ldn::NetworkConfig network = {};
        network.intentId = nn::ldn::MakeIntentId(LocalId, g_TestConfig.sceneId);
        network.nodeCountMax = static_cast<int8_t>(nodeCountMax);
        network.channel = g_TestConfig.channel;
        nn::ldn::SecurityConfig security = {};
        security.securityMode = static_cast<nn::Bit16>(securityMode);
        security.passphraseSize = sizeof(Passphrase);
        std::memcpy(security.passphrase, Passphrase, sizeof(Passphrase));
        nn::ldn::UserConfig user = {};
        std::strncpy(user.userName, ApUserName, nn::ldn::UserNameBytesMax);

        // 最小サイズのユーザデータを設定します。
        const nn::Bit8 smallUserData = 0xA5;
        nn::ldn::SetAdvertiseData(&smallUserData, sizeof(smallUserData));

        // ネットワークを構築します。
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetwork(network, security, user));
        ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetState());
        NNT_LDN_SYNC("Advertise.UserDataUpdate.NetworkCreated");

        // ステーションの接続を待機します。
        NNT_LDN_SYNC("Advertise.UserDataUpdate.Connected");
        nn::ldn::NetworkInfo networkInfo;
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(&networkInfo));
        ASSERT_EQ(networkInfo.ldn.nodeCountMax, networkInfo.ldn.nodeCount);
        ASSERT_EQ(1U, networkInfo.ldn.advertiseDataSize);
        ASSERT_EQ(smallUserData, networkInfo.ldn.advertiseData[0]);

        // 最大サイズのユーザデータを設定します。
        nn::Bit8 largeUserData[nn::ldn::AdvertiseDataSizeMax];
        for (size_t i = 0; i < nn::ldn::AdvertiseDataSizeMax; ++i)
        {
            largeUserData[i] = static_cast<nn::Bit8>((i + 1) & 0xFF);
        }
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::SetAdvertiseData(
            &largeUserData, sizeof(largeUserData)));
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(&networkInfo));
        ASSERT_EQ(sizeof(largeUserData), networkInfo.ldn.advertiseDataSize);
        ASSERT_EQ(0, std::memcmp(
            networkInfo.ldn.advertiseData, largeUserData, sizeof(largeUserData)));
        nn::os::SleepThread(AdvertisingTime);
        NNT_LDN_SYNC("Advertise.UserDataUpdate.AdvertiseUpdated");

        // Advertise を STA に確実に伝えるために待機します。
        NNT_LDN_SYNC("Advertise.UserDataUpdate.AdvertiseReceived");

        // ネットワークを破棄します。
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::DestroyNetwork());
    }
}

//
// Advertise で前回のユーザデータを引き継ぐことを確認します。
//
TEST(Advertise, UserDataKeep)
{
    // テストケース開始前の同期です。
    NNT_LDN_SYNC("Advertise.UserDataKeep.Start");

    // 最大台数のステーションを接続させます。
    int nodeCountMax = std::min<int>(nn::ldn::NodeCountMax, g_TestConfig.nodeCount);

    // LDN ライブラリを初期化します。
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();

    // 状態変化を監視するイベントを取得します。
    nn::os::SystemEvent stateChangeEvent;
    nn::ldn::AttachStateChangeEvent(stateChangeEvent.GetBase());

    // 各セキュリティ・モードについてテストを実施します。
    nn::ldn::SecurityMode securities[] = { nn::ldn::SecurityMode_Debug,
                                           nn::ldn::SecurityMode_Product };
    for (auto securityMode : securities)
    {
        // AP としての動作を開始します。
        nnt::ldn::AccessPointStarter starter;
        ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());

        // ネットワークのパラメータを適当に設定します。
        nn::ldn::NetworkConfig network = { };
        network.intentId = nn::ldn::MakeIntentId(LocalId, g_TestConfig.sceneId);
        network.nodeCountMax = static_cast<int8_t>(nodeCountMax);
        network.channel = g_TestConfig.channel;
        nn::ldn::SecurityConfig security = { };
        security.securityMode = static_cast<nn::Bit16>(securityMode);
        security.passphraseSize = sizeof(Passphrase);
        std::memcpy(security.passphrase, Passphrase, sizeof(Passphrase));
        nn::ldn::UserConfig user = { };
        std::strncpy(user.userName, ApUserName, nn::ldn::UserNameBytesMax);

        // 最大サイズのユーザデータを生成します。
        nn::Bit8 largeUserData[nn::ldn::AdvertiseDataSizeMax];
        for (size_t i = 0; i < nn::ldn::AdvertiseDataSizeMax; ++i)
        {
            largeUserData[i] = static_cast<nn::Bit8>((i + 1) & 0xFF);
        }

        // 最大サイズのユーザーデータを設定します。
        {
            // ネットワークを構築します。
            NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetwork(network, security, user));
            ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetState());

            // 最大サイズのユーザデータを設定します。
            nn::ldn::NetworkInfo networkInfo;
            NNT_ASSERT_RESULT_SUCCESS(nn::ldn::SetAdvertiseData(
                &largeUserData, sizeof(largeUserData)));
            NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(&networkInfo));
            ASSERT_EQ(sizeof(largeUserData), networkInfo.ldn.advertiseDataSize);
            ASSERT_EQ(0, std::memcmp(
                networkInfo.ldn.advertiseData, largeUserData, sizeof(largeUserData)));

            // ネットワークを破棄します。
            NNT_ASSERT_RESULT_SUCCESS(nn::ldn::DestroyNetwork());
        }

        // 前回のユーザデータを引き継ぐことを確認します。
        {
            // ネットワークを構築します。
            NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetwork(network, security, user));
            ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetState());
            NNT_LDN_SYNC("Advertise.UserDataKeep.NetworkCreated");

            // ステーションの接続を待機します。
            NNT_LDN_SYNC("Advertise.UserDataKeep.Connected");
            nn::ldn::NetworkInfo networkInfo;
            NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(&networkInfo));
            ASSERT_EQ(networkInfo.ldn.nodeCountMax, networkInfo.ldn.nodeCount);
            ASSERT_EQ(nn::ldn::AdvertiseDataSizeMax, networkInfo.ldn.advertiseDataSize);
            ASSERT_EQ(0, std::memcmp(
                networkInfo.ldn.advertiseData, largeUserData, sizeof(largeUserData)));

            // ネットワークを破棄します。
            NNT_ASSERT_RESULT_SUCCESS(nn::ldn::DestroyNetwork());
        }
    }
}

//
// Advertise でノード情報が正しく配信されることを検証します。
//
TEST(Advertise, NodeInfo)
{
    // テストケース開始前の同期です。
    NNT_LDN_SYNC("Advertise.NodeInfo.Start");

    // 最大台数のステーションを接続させます。
    int nodeCountMax = std::min<int>(nn::ldn::NodeCountMax, g_TestConfig.nodeCount);

    // LDN ライブラリを初期化します。
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();

    // 状態変化を監視するイベントを取得します。
    nn::os::SystemEvent stateChangeEvent;
    nn::ldn::AttachStateChangeEvent(stateChangeEvent.GetBase());

    // 各セキュリティ・モードについてテストを実施します。
    nn::ldn::SecurityMode securities[] = { nn::ldn::SecurityMode_Debug,
                                           nn::ldn::SecurityMode_Product };
    for (auto securityMode : securities)
    {
        // AP としての動作を開始します。
        nnt::ldn::AccessPointStarter starter;
        ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());

        // ネットワークのパラメータを適当に設定します。
        nn::ldn::NetworkConfig network = { };
        network.intentId = nn::ldn::MakeIntentId(LocalId, g_TestConfig.sceneId);
        network.nodeCountMax = static_cast<int8_t>(nodeCountMax);
        network.channel = g_TestConfig.channel;
        nn::ldn::SecurityConfig security = { };
        security.securityMode = static_cast<nn::Bit16>(securityMode);
        security.passphraseSize = sizeof(Passphrase);
        std::memcpy(security.passphrase, Passphrase, sizeof(Passphrase));

        // 適当な名前でネットワークを構築します。
        {
            // ネットワークを構築します。
            nn::ldn::UserConfig user = { };
            std::strncpy(user.userName, ApUserName, nn::ldn::UserNameBytesMax);
            NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetwork(network, security, user));
            ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetState());
            NNT_LDN_SYNC("Advertise.NodeInfo.NetworkCreated");

            // 自身の IPv4 アドレスを取得します。
            nn::ldn::Ipv4Address address;
            nn::ldn::SubnetMask  mask;
            NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetIpv4Address(&address, &mask));
            nn::ldn::Ipv4Address netAddress = nn::ldn::MakeNetworkAddress(address, mask);

            // ステーションの接続を待機します。
            NNT_LDN_SYNC("Advertise.NodeInfo.Connected");
            nn::ldn::NetworkInfo networkInfo;
            NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(&networkInfo));
            ASSERT_EQ(network.nodeCountMax, networkInfo.ldn.nodeCount);

            // 全員のユーザ名を確認します。
            const auto& ap = networkInfo.ldn.nodes[0];
            ASSERT_TRUE(ap.isConnected);
            ASSERT_EQ(0, ap.nodeId);
            ASSERT_STREQ(ApUserName, networkInfo.ldn.nodes[0].userName);
            ASSERT_EQ(address, ap.ipv4Address);
            ASSERT_NE(nn::ldn::ZeroMacAddress, ap.macAddress);
            ASSERT_NE(nn::ldn::BroadcastMacAddress, ap.macAddress);
            for (int i = 1; i < nodeCountMax; ++i)
            {
                const auto& node = networkInfo.ldn.nodes[i];
                ASSERT_TRUE(node.isConnected);
                ASSERT_EQ(i, node.nodeId);
                ASSERT_STREQ(StaUserName, networkInfo.ldn.nodes[i].userName);
                ASSERT_NE(address, node.ipv4Address);
                ASSERT_EQ(netAddress, nn::ldn::MakeNetworkAddress(node.ipv4Address, mask));
                ASSERT_NE(nn::ldn::ZeroMacAddress, node.macAddress);
                ASSERT_NE(nn::ldn::BroadcastMacAddress, node.macAddress);
            }
            for (int i = nodeCountMax; i < nn::ldn::NodeCountMax; ++i)
            {
                ASSERT_FALSE(networkInfo.ldn.nodes[i].isConnected);
                ASSERT_EQ(i, networkInfo.ldn.nodes[i].nodeId);
            }

            // ネットワークを破棄します。
            NNT_LDN_SYNC("Advertise.NodeInfo.AdvertiseReceived");
            NNT_ASSERT_RESULT_SUCCESS(nn::ldn::DestroyNetwork());
        }
    }
}

//
// Advertise で配信される情報を検証します。
//
TEST(Advertise, Others)
{
    // テストケース開始前の同期です。
    NNT_LDN_SYNC("Advertise.Others.Start");

    // 最大台数のステーションを接続させます。
    int nodeCountMax = std::min<int>(nn::ldn::NodeCountMax, g_TestConfig.nodeCount);

    // LDN ライブラリを初期化します。
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();

    // 状態変化を監視するイベントを取得します。
    nn::os::SystemEvent stateChangeEvent;
    nn::ldn::AttachStateChangeEvent(stateChangeEvent.GetBase());

    // 各セキュリティ・モードについてテストを実施します。
    nn::ldn::SecurityMode securities[] = { nn::ldn::SecurityMode_Debug,
                                           nn::ldn::SecurityMode_Product };
    for (auto securityMode : securities)
    {
        // AP としての動作を開始します。
        nnt::ldn::AccessPointStarter starter;
        ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());

        // ネットワークのパラメータを適当に設定します。
        nn::ldn::NetworkConfig network = { };
        network.intentId = nn::ldn::MakeIntentId(LocalId, g_TestConfig.sceneId);
        network.nodeCountMax = static_cast<int8_t>(nodeCountMax);
        network.channel = g_TestConfig.channel;
        nn::ldn::SecurityConfig security = { };
        security.securityMode = static_cast<nn::Bit16>(securityMode);
        security.passphraseSize = sizeof(Passphrase);
        std::memcpy(security.passphrase, Passphrase, sizeof(Passphrase));
        nn::ldn::UserConfig user = { };
        std::strncpy(user.userName, ApUserName, nn::ldn::UserNameBytesMax);

        // ネットワークを構築します。
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetwork(network, security, user));
        ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetState());

        // ネットワークの情報を検証します。
        nn::ldn::NetworkInfo networkInfo;
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(&networkInfo));
        ASSERT_EQ(securityMode, networkInfo.ldn.securityMode);
        ASSERT_EQ(nn::ldn::AcceptPolicy_AlwaysAccept, networkInfo.ldn.stationAcceptPolicy);
        ASSERT_EQ(nodeCountMax, networkInfo.ldn.nodeCountMax);
        ASSERT_EQ(1, networkInfo.ldn.nodeCount);
        ASSERT_EQ(network.intentId, networkInfo.networkId.intentId);
        ASSERT_EQ(networkInfo.ldn.nodes[0].macAddress, networkInfo.common.bssid);
        ASSERT_EQ(nn::ldn::SsidLengthMax, networkInfo.common.ssid.length);
        ASSERT_NE(0, networkInfo.common.channel);
        ASSERT_EQ(nn::ldn::LinkLevelMax, networkInfo.common.linkLevel);
        ASSERT_EQ(nn::ldn::NetworkType_Ldn, networkInfo.common.networkType);
        NNT_LDN_SYNC("Advertise.Others.NetworkCreated");

        // ステーションの接続を待機します。
        NNT_LDN_SYNC("Advertise.Others.Connected");

        // ネットワークの情報を検証します。
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(&networkInfo));
        ASSERT_EQ(securityMode, networkInfo.ldn.securityMode);
        ASSERT_EQ(nn::ldn::AcceptPolicy_AlwaysAccept, networkInfo.ldn.stationAcceptPolicy);
        ASSERT_EQ(nodeCountMax, networkInfo.ldn.nodeCountMax);
        ASSERT_EQ(nodeCountMax, networkInfo.ldn.nodeCount);
        ASSERT_EQ(network.intentId, networkInfo.networkId.intentId);
        ASSERT_EQ(networkInfo.ldn.nodes[0].macAddress, networkInfo.common.bssid);
        ASSERT_EQ(nn::ldn::SsidLengthMax, networkInfo.common.ssid.length);
        ASSERT_NE(0, networkInfo.common.channel);
        ASSERT_TRUE(nn::ldn::LinkLevelMin <= networkInfo.common.linkLevel &&
                    networkInfo.common.linkLevel <= nn::ldn::LinkLevelMax);
        ASSERT_EQ(nn::ldn::NetworkType_Ldn, networkInfo.common.networkType);

        // ネットワークを破棄します。
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::DestroyNetwork());
    }
}

//
// ノード接続時の NodeLatestUpdate の検証です。
//
TEST(NodeLatestUpdate, Connect)
{
    // テストケース開始前の同期です。
    NNT_LDN_SYNC("NodeLatestUpdate.Connect.Start");

    // 最大台数のステーションを接続させます。
    int nodeCountMax = std::min<int>(nn::ldn::NodeCountMax, g_TestConfig.nodeCount);

    // AP としての動作を開始します。
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    nnt::ldn::AccessPointStarter starter;
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());

    // 状態変化を監視するイベントを取得します。
    nn::os::SystemEvent stateChangeEvent;
    nn::ldn::AttachStateChangeEvent(stateChangeEvent.GetBase());

    // ネットワークのパラメータを適当に設定します。
    nn::ldn::NetworkConfig network = { };
    network.intentId = nn::ldn::MakeIntentId(LocalId, g_TestConfig.sceneId);
    network.nodeCountMax = static_cast<int8_t>(nodeCountMax);
    network.channel = g_TestConfig.channel;
    nn::ldn::SecurityConfig security = { };
    security.securityMode = static_cast<nn::Bit16>(nn::ldn::SecurityMode_Product);
    security.passphraseSize = sizeof(Passphrase);
    std::memcpy(security.passphrase, Passphrase, sizeof(Passphrase));
    nn::ldn::UserConfig user = { };
    std::strncpy(user.userName, ApUserName, nn::ldn::UserNameBytesMax);

    // ネットワークを構築します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetwork(network, security, user));
    ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetState());
    NNT_LDN_SYNC("NodeLatestUpdate.Connect.NetworkCreated");

    // ステーションの接続を待ち受けます。
    int count = 0;
    nn::ldn::NetworkInfo networkInfo;
    nn::ldn::NodeLatestUpdate updates[nn::ldn::NodeCountMax];
    while (count < nodeCountMax)
    {
        NNT_LDN_ASSERT_SIGNALED_WEAK(stateChangeEvent, ConnectionTimeout);
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(
            &networkInfo, updates, nn::ldn::NodeCountMax));
        for (int i = 0; i < nn::ldn::NodeCountMax; ++i)
        {
            const auto stateChange = updates[i].stateChange;
            if (stateChange == nn::ldn::NodeStateChange_Connect)
            {
                ++count;
            }
            else if (stateChange == nn::ldn::NodeStateChange_Disconnect)
            {
                --count;
            }
        }
    }
    ASSERT_EQ(nodeCountMax, count);
    ASSERT_EQ(nodeCountMax, networkInfo.ldn.nodeCount);
    NNT_LDN_SYNC("NodeLatestUpdate.Connect.Connected");

    // ネットワークを破棄します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::DestroyNetwork());
}

//
// ノード切断時の NodeLatestUpdate の検証です。
//
TEST(NodeLatestUpdate, Disconnect)
{
    // テストケース開始前の同期です。
    NNT_LDN_SYNC("NodeLatestUpdate.Disconnect.Start");

    // 最大台数のステーションを接続させます。
    int nodeCountMax = std::min<int>(nn::ldn::NodeCountMax, g_TestConfig.nodeCount);

    // AP としての動作を開始します。
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    nnt::ldn::AccessPointStarter starter;
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());

    // 状態変化を監視するイベントを取得します。
    nn::os::SystemEvent stateChangeEvent;
    nn::ldn::AttachStateChangeEvent(stateChangeEvent.GetBase());

    // ネットワークのパラメータを適当に設定します。
    nn::ldn::NetworkConfig network = { };
    network.intentId = nn::ldn::MakeIntentId(LocalId, g_TestConfig.sceneId);
    network.nodeCountMax = static_cast<int8_t>(nodeCountMax);
    network.channel = g_TestConfig.channel;
    nn::ldn::SecurityConfig security = { };
    security.securityMode = static_cast<nn::Bit16>(nn::ldn::SecurityMode_Product);
    security.passphraseSize = sizeof(Passphrase);
    std::memcpy(security.passphrase, Passphrase, sizeof(Passphrase));
    nn::ldn::UserConfig user = { };
    std::strncpy(user.userName, ApUserName, nn::ldn::UserNameBytesMax);

    // ネットワークを構築します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetwork(network, security, user));
    ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetState());
    NNT_LDN_SYNC("NodeLatestUpdate.Disconnect.NetworkCreated");

    // ステーションの接続を待ち受けます。
    nn::ldn::NetworkInfo networkInfo;
    nn::ldn::NodeLatestUpdate updates[nn::ldn::NodeCountMax];
    do
    {
        NNT_LDN_ASSERT_SIGNALED_WEAK(stateChangeEvent, ConnectionTimeout);
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(
            &networkInfo, updates, nn::ldn::NodeCountMax));
    } while (networkInfo.ldn.nodeCount < nodeCountMax);

    // ステーションに接続状態を伝えるため、一定時間待機します。
    NNT_LDN_SYNC("NodeLatestUpdate.Disconnect.Connected");

    // ステーションの切断を待ち受けます。
    int count = nodeCountMax;
    while (1 < count)
    {
        NNT_LDN_ASSERT_SIGNALED_WEAK(stateChangeEvent, ConnectionTimeout);
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(
            &networkInfo, updates, nn::ldn::NodeCountMax));
        count -= static_cast<int>(std::count_if(std::begin(updates), std::end(updates),
        [](const nn::ldn::NodeLatestUpdate& update)
        {
            return update.stateChange == nn::ldn::NodeStateChange_Disconnect;
        }));
    }
    ASSERT_EQ(1, count);

    // これ以上、接続状態に変化はありません。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(
        &networkInfo, updates, nn::ldn::NodeCountMax));
    ASSERT_TRUE(std::all_of(std::begin(updates), std::end(updates),
    [](const nn::ldn::NodeLatestUpdate& update)
    {
        return update.stateChange == nn::ldn::NodeStateChange_None;
    }));

    // ネットワークを破棄します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::DestroyNetwork());
}

//
// ノード切断時の NodeLatestUpdate の検証です。
//
TEST(NodeLatestUpdate, DisconnectAndConnect)
{
    // テストケース開始前の同期です。
    NNT_LDN_SYNC("NodeLatestUpdate.DisconnectAndConnect.Start");

    // 最大台数のステーションを接続させます。
    int nodeCountMax = std::min<int>(nn::ldn::NodeCountMax, g_TestConfig.nodeCount);

    // AP としての動作を開始します。
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    nnt::ldn::AccessPointStarter starter;
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());

    // 状態変化を監視するイベントを取得します。
    nn::os::SystemEvent stateChangeEvent;
    nn::ldn::AttachStateChangeEvent(stateChangeEvent.GetBase());

    // ネットワークのパラメータを適当に設定します。
    nn::ldn::NetworkConfig network = { };
    network.intentId = nn::ldn::MakeIntentId(LocalId, g_TestConfig.sceneId);
    network.nodeCountMax = static_cast<int8_t>(nodeCountMax);
    network.channel = g_TestConfig.channel;
    nn::ldn::SecurityConfig security = { };
    security.securityMode = static_cast<nn::Bit16>(nn::ldn::SecurityMode_Product);
    security.passphraseSize = sizeof(Passphrase);
    std::memcpy(security.passphrase, Passphrase, sizeof(Passphrase));
    nn::ldn::UserConfig user = { };
    std::strncpy(user.userName, ApUserName, nn::ldn::UserNameBytesMax);

    // ネットワークを構築します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetwork(network, security, user));
    ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetState());
    NNT_LDN_SYNC("NodeLatestUpdate.DisconnectAndConnect.NetworkCreated");

    // ステーションの接続を待ち受けます。
    nn::ldn::NetworkInfo networkInfo;
    nn::ldn::NodeLatestUpdate updates[nn::ldn::NodeCountMax];
    do
    {
        NNT_LDN_ASSERT_SIGNALED_WEAK(stateChangeEvent, ConnectionTimeout);
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(
            &networkInfo, updates, nn::ldn::NodeCountMax));
    } while (networkInfo.ldn.nodeCount < nodeCountMax);

    // ステーションに接続状態を伝えるため、一定時間待機します。
    NNT_LDN_SYNC("NodeLatestUpdate.DisconnectAndConnect.Connected");

    // ステーションの切断を待ち受けます。
    while (1 < networkInfo.ldn.nodeCount)
    {
        NNT_LDN_ASSERT_SIGNALED_WEAK(stateChangeEvent, ConnectionTimeout);
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(&networkInfo));
    }
    ASSERT_EQ(1, networkInfo.ldn.nodeCount);
    NNT_LDN_SYNC("NodeLatestUpdate.DisconnectAndConnect.Disconnected");

    // ステーションが再度接続してくるまで待機します。
    while (networkInfo.ldn.nodeCount < nodeCountMax)
    {
        NNT_LDN_ASSERT_SIGNALED_WEAK(stateChangeEvent, ConnectionTimeout);
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(&networkInfo));
    }
    ASSERT_EQ(nodeCountMax, networkInfo.ldn.nodeCount);
    NNT_LDN_SYNC("NodeLatestUpdate.DisconnectAndConnect.Reconnected");

    // NodeLatestUpdate を確認します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(
        &networkInfo, updates, nn::ldn::NodeCountMax));
    ASSERT_EQ(nn::ldn::NodeStateChange_None, updates[0].stateChange);
    ASSERT_TRUE(std::all_of(&updates[1], &updates[nodeCountMax],
                            [](const nn::ldn::NodeLatestUpdate& update)
    {
        return update.stateChange == nn::ldn::NodeStateChange_DisconnectAndConnect;
    }));

    // これ以上、接続状態に変化はありません。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(
        &networkInfo, updates, nn::ldn::NodeCountMax));
    ASSERT_TRUE(std::all_of(std::begin(updates), std::end(updates),
                             [](const nn::ldn::NodeLatestUpdate& update)
    {
        return update.stateChange == nn::ldn::NodeStateChange_None;
    }));

    // ネットワークを破棄します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::DestroyNetwork());
}

//
// ネットワーク再構築時に IPv4 アドレスを持ち越せていることの検証です。
//
TEST(Reconnection, ReuseIpv4Address)
{
    // テストケース開始前の同期です。
    NNT_LDN_SYNC("Reconnection.ReuseIpv4Address.Start");

    // 最大台数のステーションを接続させます。
    int nodeCountMax = std::min<int>(nn::ldn::NodeCountMax, g_TestConfig.nodeCount);

    // AP としての動作を開始します。
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    nnt::ldn::AccessPointStarter starter;
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());

    // 状態変化を監視するイベントを取得します。
    nn::os::SystemEvent stateChangeEvent;
    nn::ldn::AttachStateChangeEvent(stateChangeEvent.GetBase());

    // ネットワークのパラメータを適当に設定します。
    nn::ldn::NetworkConfig network = { };
    network.intentId = nn::ldn::MakeIntentId(LocalId, g_TestConfig.sceneId);
    network.nodeCountMax = static_cast<int8_t>(nodeCountMax);
    network.channel = g_TestConfig.channel;
    nn::ldn::SecurityConfig security = { };
    security.securityMode = static_cast<nn::Bit16>(nn::ldn::SecurityMode_Product);
    security.passphraseSize = sizeof(Passphrase);
    std::memcpy(security.passphrase, Passphrase, sizeof(Passphrase));
    nn::ldn::UserConfig user = { };
    std::strncpy(user.userName, ApUserName, nn::ldn::UserNameBytesMax);

    // ネットワークを構築します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetwork(network, security, user));
    ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetState());
    NNT_LDN_SYNC("Reconnection.ReuseIpv4Address.NetworkCreated");

    // ステーションの接続を待ち受けます。
    NNT_LDN_SYNC("Reconnection.ReuseIpv4Address.Connected");
    nn::ldn::NetworkInfo networkInfo;
    nn::ldn::SecurityParameter securityParam;
    nn::ldn::NetworkConfig networkConfig;
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(&networkInfo));
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetSecurityParameter(&securityParam));
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkConfig(&networkConfig));
    NNT_LDN_SYNC("Reconnection.ReuseIpv4Address.GetNetworkInfo");

    // ネットワークを破棄します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::DestroyNetwork());

    // IPv4 アドレスを再利用するためのエントリを作成します。
    nn::ldn::AddressEntry entries[nn::ldn::NodeCountMax];
    int nodeCount = 0;
    for (int i = 0; i < nn::ldn::NodeCountMax; ++i)
    {
        const auto& node = networkInfo.ldn.nodes[i];
        if (node.isConnected)
        {
            auto& entry = entries[nodeCount];
            entry.ipv4Address = node.ipv4Address;
            entry.macAddress = node.macAddress;
            ++nodeCount;
        }
    }
    ASSERT_EQ(nodeCountMax, nodeCount);

    // ネットワークを再構築します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetworkPrivate(
        networkConfig, security, securityParam, user, nodeCount, entries));
    ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetState());
    NNT_LDN_SYNC("Reconnection.ReuseIpv4Address.NetworkReconstructed");

    // ステーションの接続を待ち受けます。
    NNT_LDN_SYNC("Reconnection.ReuseIpv4Address.Reconnected");
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(&networkInfo));
    ASSERT_EQ(nodeCountMax, nodeCount);

    // IPv4 アドレスが再利用されていることを確認します。
    for (int i = 0; i < nn::ldn::NodeCountMax; ++i)
    {
        const auto& node = networkInfo.ldn.nodes[i];
        for (int j = 0; j < nodeCount; ++j)
        {
            const auto& entry = entries[j];
            if (node.macAddress == entry.macAddress)
            {
                ASSERT_EQ(entry.ipv4Address, node.ipv4Address);
            }
        }
    }

    // ネットワークを破棄します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::DestroyNetwork());

    // ステーションの処理完了まで待機します。
    NNT_LDN_SYNC("Reconnection.ReuseIpv4Address.Finished");
}

//
// アプリ起因の切断時に切断理由を正しく取得できることを確認します。
//
TEST(DisconnectReason, DisconnectedByUser)
{
    // テストケース開始前の同期です。
    NNT_LDN_SYNC("DisconnectReason.DisconnectedByUser.Start");

    // 最大台数のステーションを接続させます。
    int nodeCountMax = std::min<int>(nn::ldn::NodeCountMax, g_TestConfig.nodeCount);

    // AP としての動作を開始します。
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    nnt::ldn::AccessPointStarter starter;
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());

    // 状態変化を監視するイベントを取得します。
    nn::os::SystemEvent stateChangeEvent;
    nn::ldn::AttachStateChangeEvent(stateChangeEvent.GetBase());

    // ネットワークのパラメータを適当に設定します。
    nn::ldn::NetworkConfig network = { };
    network.intentId = nn::ldn::MakeIntentId(LocalId, g_TestConfig.sceneId);
    network.nodeCountMax = static_cast<int8_t>(nodeCountMax);
    network.channel = g_TestConfig.channel;
    nn::ldn::SecurityConfig security = { };
    security.securityMode = static_cast<nn::Bit16>(nn::ldn::SecurityMode_Product);
    security.passphraseSize = sizeof(Passphrase);
    std::memcpy(security.passphrase, Passphrase, sizeof(Passphrase));
    nn::ldn::UserConfig user = { };
    std::strncpy(user.userName, ApUserName, nn::ldn::UserNameBytesMax);

    // ネットワークを構築します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetwork(network, security, user));
    ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetState());
    NNT_LDN_SYNC("DisconnectReason.DisconnectedByUser.NetworkCreated");

    // ステーションの接続、切断を待機します。
    NNT_LDN_SYNC("DisconnectReason.DisconnectedByUser.Connected");
    NNT_LDN_SYNC("DisconnectReason.DisconnectedByUser.Disconnected");
    NNT_LDN_SYNC("DisconnectReason.DisconnectedByUser.Reconnected");
    NNT_LDN_SYNC("DisconnectReason.DisconnectedByUser.Disconnected");

    // ネットワークを破棄します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::DestroyNetwork());
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());
}

//
// システム起因の切断時に切断理由を正しく取得できることを確認します。
//
TEST(DisconnectReason, DisconnectedByWifiOff)
{
    // テストケース開始前の同期です。
    NNT_LDN_SYNC("DisconnectReason.DisconnectedByWifiOff.Start");

    // 最大台数のステーションを接続させます。
    int nodeCountMax = std::min<int>(nn::ldn::NodeCountMax, g_TestConfig.nodeCount);

    // AP としての動作を開始します。
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    nnt::ldn::AccessPointStarter starter;
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());

    // 状態変化を監視するイベントを取得します。
    nn::os::SystemEvent stateChangeEvent;
    nn::ldn::AttachStateChangeEvent(stateChangeEvent.GetBase());

    // ネットワークのパラメータを適当に設定します。
    nn::ldn::NetworkConfig network = { };
    network.intentId = nn::ldn::MakeIntentId(LocalId, g_TestConfig.sceneId);
    network.nodeCountMax = static_cast<int8_t>(nodeCountMax);
    network.channel = g_TestConfig.channel;
    nn::ldn::SecurityConfig security = { };
    security.securityMode = static_cast<nn::Bit16>(nn::ldn::SecurityMode_Product);
    security.passphraseSize = sizeof(Passphrase);
    std::memcpy(security.passphrase, Passphrase, sizeof(Passphrase));
    nn::ldn::UserConfig user = { };
    std::strncpy(user.userName, ApUserName, nn::ldn::UserNameBytesMax);

    // ネットワークを構築します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetwork(network, security, user));
    ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetState());
    NNT_LDN_SYNC("DisconnectReason.DisconnectedByWifiOff.NetworkCreated");

    // ステーションの接続、切断を待機します。
    NNT_LDN_SYNC("DisconnectReason.DisconnectedByWifiOff.Connected");
    NNT_LDN_SYNC("DisconnectReason.DisconnectedByWifiOff.Disconnected");

    // ネットワークを破棄します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::DestroyNetwork());
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());
}

//
// アプリ起因のネットワーク破棄時に切断理由を正しく取得できることを確認します。
//
TEST(DisconnectReason, DestroyedByUser)
{
    // テストケース開始前の同期です。
    NNT_LDN_SYNC("DisconnectReason.DestroyedByUser.Start");

    // 最大台数のステーションを接続させます。
    int nodeCountMax = std::min<int>(nn::ldn::NodeCountMax, g_TestConfig.nodeCount);

    // AP としての動作を開始します。
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::OpenAccessPoint());
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());

    // 状態変化を監視するイベントを取得します。
    nn::os::SystemEvent stateChangeEvent;
    nn::ldn::AttachStateChangeEvent(stateChangeEvent.GetBase());

    // ネットワークのパラメータを適当に設定します。
    nn::ldn::NetworkConfig network = { };
    network.intentId = nn::ldn::MakeIntentId(LocalId, g_TestConfig.sceneId);
    network.nodeCountMax = static_cast<int8_t>(nodeCountMax);
    network.channel = g_TestConfig.channel;
    nn::ldn::SecurityConfig security = { };
    security.securityMode = static_cast<nn::Bit16>(nn::ldn::SecurityMode_Product);
    security.passphraseSize = sizeof(Passphrase);
    std::memcpy(security.passphrase, Passphrase, sizeof(Passphrase));
    nn::ldn::UserConfig user = { };
    std::strncpy(user.userName, ApUserName, nn::ldn::UserNameBytesMax);

    // ネットワークを構築します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetwork(network, security, user));
    ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetState());
    NNT_LDN_SYNC("DisconnectReason.DestroyedByUser.NetworkCreated");

    // ステーションが最大数接続するまで待機します。
    NNT_LDN_SYNC("DisconnectReason.DestroyedByUser.Connected");

    // ネットワークを破棄します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::DestroyNetwork());
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());

    // もう一度ネットワークを構築します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetwork(network, security, user));
    ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetState());
    NNT_LDN_SYNC("DisconnectReason.DestroyedByUser.NetworkReconstructed");

    // ステーションが最大数接続するまで待機します。
    NNT_LDN_SYNC("DisconnectReason.DestroyedByUser.Reconnected");

    // ネットワークを破棄します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CloseAccessPoint());
    ASSERT_EQ(nn::ldn::State_Initialized, nn::ldn::GetState());
}

//
// システム起因のネットワーク破棄時に切断理由を正しく取得できることを確認します。
//
TEST(DisconnectReason, DestroyedByWifiOff)
{
    // テストケース開始前の同期です。
    NNT_LDN_SYNC("DisconnectReason.DestroyedByWifiOff.Start");

    // 最大台数のステーションを接続させます。
    int nodeCountMax = std::min<int>(nn::ldn::NodeCountMax, g_TestConfig.nodeCount);

    // AP としての動作を開始します。
    nnt::ldn::NifmInitializer nifmInitializer;
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::OpenAccessPoint());
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());

    // 状態変化を監視するイベントを取得します。
    nn::os::SystemEvent stateChangeEvent;
    nn::ldn::AttachStateChangeEvent(stateChangeEvent.GetBase());

    // ネットワークのパラメータを適当に設定します。
    nn::ldn::NetworkConfig network = { };
    network.intentId = nn::ldn::MakeIntentId(LocalId, g_TestConfig.sceneId);
    network.nodeCountMax = static_cast<int8_t>(nodeCountMax);
    network.channel = g_TestConfig.channel;
    nn::ldn::SecurityConfig security = { };
    security.securityMode = static_cast<nn::Bit16>(nn::ldn::SecurityMode_Product);
    security.passphraseSize = sizeof(Passphrase);
    std::memcpy(security.passphrase, Passphrase, sizeof(Passphrase));
    nn::ldn::UserConfig user = { };
    std::strncpy(user.userName, ApUserName, nn::ldn::UserNameBytesMax);

    // ネットワークを構築します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetwork(network, security, user));
    ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetState());
    NNT_LDN_SYNC("DisconnectReason.DestroyedByWifiOff.NetworkCreated");

    // ステーションが最大数接続するまで待機します。
    NNT_LDN_SYNC("DisconnectReason.DestroyedByWifiOff.Connected");

    // ネットワークを破棄します。
    nnt::ldn::ScopedWirelessCommunicationSwitchSetter setter(false);
    ASSERT_EQ(nn::ldn::State_Error, nn::ldn::GetState());
    NNT_ASSERT_RESULT_FAILURE(nn::ldn::ResultWifiOff, nn::ldn::OpenStation());
}

//
// AP 側のアプリから STA を切断する場合に切断理由を正しく取得できることを確認します。
//
TEST(DisconnectReason, Rejected)
{
    // テストケース開始前の同期です。
    NNT_LDN_SYNC("DisconnectReason.Rejected.Start");

    // 最大台数のステーションを接続させます。
    int nodeCountMax = std::min<int>(nn::ldn::NodeCountMax, g_TestConfig.nodeCount);

    // AP としての動作を開始します。
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::OpenAccessPoint());
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());

    // 状態変化を監視するイベントを取得します。
    nn::os::SystemEvent stateChangeEvent;
    nn::ldn::AttachStateChangeEvent(stateChangeEvent.GetBase());

    // ネットワークのパラメータを適当に設定します。
    nn::ldn::NetworkConfig network = { };
    network.intentId = nn::ldn::MakeIntentId(LocalId, g_TestConfig.sceneId);
    network.nodeCountMax = static_cast<int8_t>(nodeCountMax);
    network.channel = g_TestConfig.channel;
    nn::ldn::SecurityConfig security = { };
    security.securityMode = static_cast<nn::Bit16>(nn::ldn::SecurityMode_Product);
    security.passphraseSize = sizeof(Passphrase);
    std::memcpy(security.passphrase, Passphrase, sizeof(Passphrase));
    nn::ldn::UserConfig user = { };
    std::strncpy(user.userName, ApUserName, nn::ldn::UserNameBytesMax);

    // ネットワークを構築します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetwork(network, security, user));
    ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetState());
    NNT_LDN_SYNC("DisconnectReason.Rejected.NetworkCreated");

    // ステーションが最大数接続するまで待機します。
    NNT_LDN_SYNC("DisconnectReason.Rejected.Connected");
    nn::ldn::NetworkInfo networkInfo;
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(&networkInfo));

    // ステーションを全て切断します。
    for (int i = 1; i < nn::ldn::NodeCountMax; ++i)
    {
        const auto& node = networkInfo.ldn.nodes[i];
        if (node.isConnected)
        {
            NNT_ASSERT_RESULT_SUCCESS(nn::ldn::Reject(node.ipv4Address));
        }
    }

    // ネットワークを破棄します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::DestroyNetwork());
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());
}

//
// ローカル通信バージョンの比較テストです。
//
TEST(LocalCommunicationVersion, Compare)
{
    // テストケース開始前の同期です。
    NNT_LDN_SYNC("LocalCommunicationVersion.Compare.Start");

    // 最大台数のステーションを接続させます。
    int nodeCountMax = std::min<int>(nn::ldn::NodeCountMax, g_TestConfig.nodeCount);

    // AP としての動作を開始します。
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::OpenAccessPoint());
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());

    // 状態変化を監視するイベントを取得します。
    nn::os::SystemEvent stateChangeEvent;
    nn::ldn::AttachStateChangeEvent(stateChangeEvent.GetBase());

    // ネットワークのパラメータを適当に設定します。
    nn::ldn::NetworkConfig network = { };
    network.intentId = nn::ldn::MakeIntentId(LocalId, g_TestConfig.sceneId);
    network.nodeCountMax = static_cast<int8_t>(nodeCountMax);
    network.channel = g_TestConfig.channel;
    network.localCommunicationVersion = 1024;
    nn::ldn::SecurityConfig security = { };
    security.securityMode = static_cast<nn::Bit16>(nn::ldn::SecurityMode_Product);
    security.passphraseSize = sizeof(Passphrase);
    std::memcpy(security.passphrase, Passphrase, sizeof(Passphrase));
    nn::ldn::UserConfig user = { };
    std::strncpy(user.userName, ApUserName, nn::ldn::UserNameBytesMax);

    // ネットワークを構築します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetwork(network, security, user));
    ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetState());
    NNT_LDN_SYNC("LocalCommunicationVersion.Compare.NetworkCreated");

    // ステーションが最大数接続するまで待機します。
    NNT_LDN_SYNC("LocalCommunicationVersion.Compare.Connected");

    // ネットワークを破棄します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::DestroyNetwork());
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());
}

//
// デフォルトのローカル通信識別子に関するテストです。
//
TEST(LocalCommunicationId, Default)
{
    // テストケース開始前の同期です。
    NNT_LDN_SYNC("LocalCommunicationId.Default.Start");

    // 最大台数のステーションを接続させます。
    int nodeCountMax = std::min<int>(nn::ldn::NodeCountMax, g_TestConfig.nodeCount);

    // AP としての動作を開始します。
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::OpenAccessPoint());
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());

    // 状態変化を監視するイベントを取得します。
    nn::os::SystemEvent stateChangeEvent;
    nn::ldn::AttachStateChangeEvent(stateChangeEvent.GetBase());

    // ネットワークのパラメータを適当に設定します。
    nn::ldn::NetworkConfig network = {};
    network.intentId = nn::ldn::MakeIntentId(
        nn::ldn::DefaultLocalCommunicationId, g_TestConfig.sceneId);
    network.nodeCountMax = static_cast<int8_t>(nodeCountMax);
    network.channel = g_TestConfig.channel;
    nn::ldn::SecurityConfig security = {};
    security.securityMode = static_cast<nn::Bit16>(nn::ldn::SecurityMode_Product);
    security.passphraseSize = sizeof(Passphrase);
    std::memcpy(security.passphrase, Passphrase, sizeof(Passphrase));
    nn::ldn::UserConfig user = {};
    std::strncpy(user.userName, ApUserName, nn::ldn::UserNameBytesMax);

    // ネットワークを構築します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetwork(network, security, user));
    ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetState());
    NNT_LDN_SYNC("LocalCommunicationId.Default.NetworkCreated");

    // デフォルトのローカル通信識別子が設定されていることを確認します。
    nn::ldn::NetworkInfo networkInfo;
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(&networkInfo));
    ASSERT_EQ(LocalId, networkInfo.networkId.intentId.localCommunicationId);

    // ステーションが最大数接続するまで待機します。
    NNT_LDN_SYNC("LocalCommunicationId.Default.Connected");

    // ネットワークを破棄します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::DestroyNetwork());
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());
}

//
// テストのエントリポイントです。
//
extern "C" void nnMain()
{
    // コマンドライン引数に関する設定です。
    nnt::ldn::CommandLineParserSetting setting = { };
    setting.flag = static_cast<nn::Bit32>(
        nnt::ldn::CommandLineOptionFlag_NodeCount |
        nnt::ldn::CommandLineOptionFlag_NodeIndex |
        nnt::ldn::CommandLineOptionFlag_SceneId |
        nnt::ldn::CommandLineOptionFlag_Channel |
        nnt::ldn::CommandLineOptionFlag_OperationMode |
        nnt::ldn::CommandLineOptionFlag_WirelessControllerRestriction);
    setting.nodeCountMin = 2;
    setting.nodeCountMax = nnt::ldn::SynchronizationClientCountMax;
    setting.defaultSceneId = 0x1101;

    // コマンドライン引数を解析します。
    int argc = nn::os::GetHostArgc();
    char **argv = nn::os::GetHostArgv();
    ::testing::InitGoogleTest(&argc, argv);
    nnt::ldn::Parse(&g_TestConfig, setting, argc, argv);

    // 無線スイッチをオンにします。
    {
        nnt::ldn::NifmInitializer nifmInitializer;
        nnt::ldn::SetWirelessCommunicationSwitch(true);
    }

    // 他のインスタンスと同期するためにテストサーバを生成します。
    nnt::ldn::HtcsSynchronizationServer server(
        g_SynchronizationBuffer, sizeof(g_SynchronizationBuffer));
    int result = server.CreateServer("nn::ldn::IntegrationNetwork", g_TestConfig.nodeCount - 1);

    // テストを実行します。
    int exitCode;
    if (result == nnt::ldn::SynchronizationResult_Success)
    {
        g_pServer = &server;
        exitCode = RUN_ALL_TESTS();
        server.DestroyServer();
    }
    else
    {
        NNT_LDN_LOG_ERROR("Failed to sync: %d\n", result);
        exitCode = result;
    }
    nnt::Exit(exitCode);
}
