﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <nn/ldn.h>
#include <nn/ldn/ldn_Settings.h>
#include <nn/ldn/ldn_SystemApi.h>
#include <nn/os.h>
#include <nn/socket.h>
#include <nnt.h>
#include <nnt/ldn/testLdn_CommandLineParser.h>
#include <nnt/ldn/testLdn_HtcsSynchronization.h>
#include <nnt/ldn/testLdn_Log.h>
#include <nnt/ldn/testLdn_NifmUtility.h>
#include "Config.h"
#include "Random.h"

namespace
{
    // 同期に使用するオブジェクトとバッファです。
    nnt::ldn::ISynchronization* g_pSync;
    nnt::ldn::ISynchronizationClient* g_pClient;
    nnt::ldn::ISynchronizationServer* g_pServer;
    char g_SynchronizationBuffer[16 * 1024];

    // 同期用のマクロです。
    #define NNT_LDN_SYNC(keyword)\
        ASSERT_EQ(::nnt::ldn::SynchronizationResult_Success, g_pSync->Synchronize(keyword))

    //! 接続に関するテスト結果の定義です。
    struct TestResult
    {
        //! 接続の試行回数です。
        int32_t trial;

        //! 接続成功回数です。
        int32_t success;

        //! 接続失敗の回数です。
        int32_t failure;

        //! 連続して失敗した回数です。
        int32_t burst;

        //! 連続して失敗した回数の最大値です。
        int32_t burstMax;

        //! 意図せず切断されてしまった回数です。
        int32_t disconnected;

        //! 失敗原因の内訳です。
        struct
        {
            int32_t notFound;
            int32_t timeout;
            int32_t rejected;
        } reason;
    };

    //! 現在のシーンです。
    enum TestScene
    {
        TestScene_None,
        TestScene_Finalizing,
        TestScene_Setup,
        TestScene_Timeout,

        // アクセスポイント特有のシーンです。
        TestScene_SetupAccessPoint,
        TestScene_CreatingNetwork,
        TestScene_NetworkCreated,
        TestScene_RunningAccessPointTest,
        TestScene_CleanupAccessPoint,

        // ステーション特有のシーンです。
        TestScene_SetupStation,
        TestScene_Scanning,
        TestScene_Connecting,
        TestScene_Connected,
        TestScene_RunningStationTest,
        TestScene_CleanupStation,
    };

    //! テストケースの一覧です。
    enum TestCase
    {
        TestCase_DestroyNetwork,
        TestCase_Reject,
        TestCase_Disconnect,
        TestCase_WififOffAp,
        TestCase_WififOffSta,
        TestCase_Count,
    };

    //! テストの状態です。
    struct TestStatus
    {
        int32_t testCase;
        int32_t scene;
        int32_t testCount;
        int32_t elapsed;
        int32_t wait;
        TestResult result;
        int32_t socket;
        int32_t sendCounter;
        int32_t receivedCounter;
        nn::ldn::Ipv4Address address;
        nn::ldn::SubnetMask mask;
    };

    //! AP-STA 間の情報共有のために配信される Advertise データです。
    struct AdvertiseData
    {
        int32_t testCase;
        int32_t testCount;
    };

    //! データ通信で送受信するデータです。
    struct Data
    {
        int32_t counter;
        int32_t hash;
    };

    // テストの設定値です。
    nnt::ldn::TestConfig g_TestConfig;

    // 状態変化を監視するイベントです。
    nn::os::SystemEvent g_StateChangeEvent;

    // socket のバッファです。
    nn::socket::ConfigDefaultWithMemory g_SocketConfig;

    // スキャンバッファです。
    nn::ldn::NetworkInfo g_ScanBuffer[nn::ldn::ScanResultCountMax];
    nn::ldn::NetworkInfo g_Network;

    // 送信スレッドです。
    nn::os::ThreadType g_SendThread;
    NN_ALIGNAS(4096) char g_SendThreadStack[16 * 1024];
    const int SendThreadPriority = nn::os::DefaultThreadPriority;

    // 受信スレッドです。
    nn::os::ThreadType g_ReceiveThread;
    NN_ALIGNAS(4096) char g_ReceiveThreadStack[16 * 1024];
    const int ReceiveThreadPriority = nn::os::DefaultThreadPriority;

    void SetLdnSettings() NN_NOEXCEPT
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::SetOperationMode(
            static_cast<nn::ldn::OperationMode>(g_TestConfig.operationMode)));
    }

    void PrintTestCase(TestCase testCase) NN_NOEXCEPT
    {
        switch (testCase)
        {
        case TestCase_DestroyNetwork:
            NNT_LDN_LOG_INFO("Test Case    : DestroyNetwork\n");
            break;
        case TestCase_Reject:
            NNT_LDN_LOG_INFO("Test Case    : Reject\n");
            break;
        case TestCase_Disconnect:
            NNT_LDN_LOG_INFO("Test Case    : Disconnect\n");
            break;
        case TestCase_WififOffAp:
            NNT_LDN_LOG_INFO("Test Case    : WifiOff (AP)\n");
            break;
        case TestCase_WififOffSta:
            NNT_LDN_LOG_INFO("Test Case    : WifiOff (STA)\n");
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    void PrintDisconnectReason(nn::ldn::DisconnectReason reason) NN_NOEXCEPT
    {
        switch (reason)
        {
        case nn::ldn::DisconnectReason_Unknown:
            NNT_LDN_LOG_INFO("Disconnected : Unknown\n");
            break;
        case nn::ldn::DisconnectReason_None:
            NNT_LDN_LOG_INFO("Disconnected : None\n");
            break;
        case nn::ldn::DisconnectReason_DisconnectedByUser:
            NNT_LDN_LOG_INFO("Disconnected : Disconnected by User\n");
            break;
        case nn::ldn::DisconnectReason_DisconnectedBySystem:
            NNT_LDN_LOG_INFO("Disconnected : Disconnected by System\n");
            break;
        case nn::ldn::DisconnectReason_DestroyedByUser:
            NNT_LDN_LOG_INFO("Disconnected : Destroyed by User\n");
            break;
        case nn::ldn::DisconnectReason_DestroyedBySystem:
            NNT_LDN_LOG_INFO("Disconnected : Destroyed by System\n");
            break;
        case nn::ldn::DisconnectReason_Rejected:
            NNT_LDN_LOG_INFO("Disconnected : Rejected\n");
            break;
        case nn::ldn::DisconnectReason_SignalLost:
            NNT_LDN_LOG_INFO("Disconnected : Signal Lost\n");
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    void PrintTestResult(const TestResult& result) NN_NOEXCEPT
    {
        NNT_LDN_LOG_INFO("Conn.Success : %d/%d (%.2f%%)\n",
            result.success, result.trial, result.success * 100.0f / result.trial);
        NNT_LDN_LOG_INFO("               Not Found : %d\n", result.reason.notFound);
        NNT_LDN_LOG_INFO("               Timeout   : %d\n", result.reason.timeout);
        NNT_LDN_LOG_INFO("               Rejected  : %d\n", result.reason.rejected);
        NNT_LDN_LOG_INFO("Conn.Burst   : %d\n", result.burstMax);
    }

    void SendThread(void* arg) NN_NOEXCEPT
    {
        auto& status = *static_cast<TestStatus*>(arg);

        // ブロードキャスト送信用のアドレスを生成します。
        NNT_LDN_LOG_DEBUG("send thread started\n");
        nn::socket::SockAddrIn addr;
        addr.sin_family = nn::socket::Family::Af_Inet;
        addr.sin_port = nn::socket::InetHtons(PortNumber);
        addr.sin_addr.S_addr = nn::socket::InetHtonl(
            nn::ldn::MakeBroadcastAddress(status.address, status.mask).raw);

        // 一定時間毎にパケットの送信を繰り返します。
        auto interval = nn::TimeSpan::FromMicroSeconds(1000000 / g_TestConfig.packetRate);
        for (;;)
        {
            // 開始時刻を取得します。
            nn::os::Tick startedAt = nn::os::GetSystemTick();

            // 送信データを生成します。
            Data data;
            data.counter = nn::socket::InetHtonl(status.sendCounter);
            data.hash = data.counter ^ INT32_C(0x5C5C5C5C);
            ++status.sendCounter;

            // カウンタの値を送信します。
            auto ret = nn::socket::SendTo(
                status.socket, &data, sizeof(data), nn::socket::MsgFlag::Msg_None,
                reinterpret_cast<nn::socket::SockAddr*>(&addr), sizeof(addr));
            if (ret < 0)
            {
                break;
            }

            // パケットレート調整のためスリープします。
            auto elapsed = (nn::os::GetSystemTick() - startedAt).ToTimeSpan();
            auto diff = interval - elapsed;
            if (0 < diff)
            {
                nn::os::SleepThread(diff);
            }
        }
        NNT_LDN_LOG_DEBUG("send thread finished\n");
    }

    void ReceiveThread(void* arg) NN_NOEXCEPT
    {
        auto& status = *static_cast<TestStatus*>(arg);

        NNT_LDN_LOG_DEBUG("receive thread started\n");

        // 失敗するまでパケットを受信し続けます。
        for (;;)
        {
            // 任意のステーションからパケットを受信するためのアドレスを生成します。
            nn::socket::SockAddrIn addr;
            addr.sin_family = nn::socket::Family::Af_Inet;
            addr.sin_port = nn::socket::InetHtons(PortNumber);
            addr.sin_addr.S_addr = nn::socket::InAddr_Any;

            // パケットの受信まで待機します。
            Data data;
            nn::socket::SockLenT length = sizeof(addr);
            auto size = nn::socket::RecvFrom(
                status.socket, &data, sizeof(data), nn::socket::MsgFlag::Msg_None,
                reinterpret_cast<nn::socket::SockAddr*>(&addr), &length);
            if (size == 0 || size < 0)
            {
                break;
            }

            // フレームの内容が正しくない場合には無視します。
            if (size != sizeof(data) || data.hash != (data.counter ^ INT32_C(0x5C5C5C5C)))
            {
                continue;
            }

            // 受信カウンタをインクリメントします。
            ++status.receivedCounter;
        }
        NNT_LDN_LOG_DEBUG("receive thread finished\n");
    }

    void StartCommunication(TestStatus* pStatus) NN_NOEXCEPT
    {
        // UDP 通信に使用するソケットを生成します。
        pStatus->socket = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp);
        ASSERT_NE(nn::socket::InvalidSocket, pStatus->socket);

        // ソケットを Bind します。
        nn::socket::SockAddrIn addr;
        addr.sin_family = nn::socket::Family::Af_Inet;
        addr.sin_port = nn::socket::InetHtons(PortNumber);
        addr.sin_addr.S_addr = nn::socket::InAddr_Any;
        ASSERT_EQ(0, nn::socket::Bind(
            pStatus->socket, reinterpret_cast<nn::socket::SockAddr*>(&addr), sizeof(addr)));

        // ブロードキャスト通信を有効化します。
        int isEnabled = 1;
        nn::socket::SetSockOpt(
            pStatus->socket, nn::socket::Level::Sol_Socket, nn::socket::Option::So_Broadcast, &isEnabled, sizeof(isEnabled));

        // データ送信を開始します。
        NNT_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(
            &g_SendThread, SendThread, pStatus,
            g_SendThreadStack, sizeof(g_SendThreadStack), SendThreadPriority));
        nn::os::StartThread(&g_SendThread);

        // データ受信を開始します。
        NNT_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(
            &g_ReceiveThread, ReceiveThread, pStatus,
            g_ReceiveThreadStack, sizeof(g_ReceiveThreadStack), ReceiveThreadPriority));
        nn::os::StartThread(&g_ReceiveThread);
    }

    void StopCommunication(TestStatus* pStatus) NN_NOEXCEPT
    {
        // データの送受信を終了します。
        nn::socket::Shutdown(pStatus->socket, nn::socket::ShutdownMethod::Shut_RdWr);
        nn::socket::Close(pStatus->socket);
        pStatus->socket = static_cast<int32_t>(nn::socket::InvalidSocket);

        // スレッドを停止します。
        nn::os::WaitThread(&g_SendThread);
        nn::os::WaitThread(&g_ReceiveThread);
        nn::os::DestroyThread(&g_SendThread);
        nn::os::DestroyThread(&g_ReceiveThread);
    }

    void TestCaseDestroyNetworkForAccessPoint(
        TestScene* pOutNextScene, TestStatus* pStatus) NN_NOEXCEPT
    {
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(pStatus->wait));
        *pOutNextScene = TestScene_CleanupAccessPoint;
    }

    void TestCaseRejectForAccessPoint(
        TestScene* pOutNextScene, TestStatus* pStatus) NN_NOEXCEPT
    {
        // 一定時間待機します。
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(pStatus->wait));

        // ネットワーク情報を取得します。
        nn::ldn::NetworkInfo networkInfo;
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(&networkInfo));

        // ステーションを切断します。
        int counter = 0;
        for (int i = 1; i < nn::ldn::NodeCountMax; ++i)
        {
            const auto& node = networkInfo.ldn.nodes[i];
            if (node.isConnected)
            {
                if (nn::ldn::Reject(node.ipv4Address).IsSuccess())
                {
                    ++counter;
                    NNT_LDN_LOG_DEBUG("rejected node %d\n", node.nodeId);
                }
                else
                {
                    NNT_LDN_LOG_WARN("failed to reject node %d\n", node.nodeId);
                }
            }
        }

        // Reject 以外で切断したステーションがいる場合には警告を出します。
        int diff = g_TestConfig.nodeCount - counter - 1;
        if (0 < diff)
        {
            NNT_LDN_LOG_WARN("%d stations disconnected unintentionally\n", diff);
        }
        *pOutNextScene = TestScene_CleanupAccessPoint;
    }

    void TestCaseDisconnectForAccessPoint(
        TestScene* pOutNextScene, TestStatus* pStatus) NN_NOEXCEPT
    {
        NN_UNUSED(pStatus);

        // ステーションの切断まで待機します。
        if (g_StateChangeEvent.TryWait())
        {
            nn::ldn::NetworkInfo networkInfo;
            NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(&networkInfo));
            if (networkInfo.ldn.nodeCount == 1)
            {
                *pOutNextScene = TestScene_CleanupAccessPoint;
            }
        }
    }

    void TestCaseWifiOffApForAccessPoint(
        TestScene* pOutNextScene, TestStatus* pStatus) NN_NOEXCEPT
    {
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(pStatus->wait));
        nnt::ldn::NifmInitializer initializer;
        {
            nnt::ldn::ScopedWirelessCommunicationSwitchSetter setter(false);
            ASSERT_EQ(nn::ldn::State_Error, nn::ldn::GetState());
        }
        *pOutNextScene = TestScene_Finalizing;
    }

    void TestCaseWifiOffStaForAccessPoint(
        TestScene* pOutNextScene, TestStatus* pStatus) NN_NOEXCEPT
    {
        NN_UNUSED(pStatus);

        // ステーションの切断まで待機します。
        if (g_StateChangeEvent.TryWait())
        {
            nn::ldn::NetworkInfo networkInfo;
            NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(&networkInfo));
            if (networkInfo.ldn.nodeCount == 1)
            {
                *pOutNextScene = TestScene_CleanupAccessPoint;
            }
        }
    }

    void TestCaseDestroyNetworkForStation(
        TestScene* pOutNextScene, TestStatus* pStatus) NN_NOEXCEPT
    {
        if (g_StateChangeEvent.TryWait() &&
            nn::ldn::GetState() != nn::ldn::State_StationConnected)
        {
            *pOutNextScene = TestScene_CleanupStation;
            auto disconnectReason = nn::ldn::GetDisconnectReason();
            if (disconnectReason != nn::ldn::DisconnectReason_DestroyedByUser)
            {
                ++pStatus->result.disconnected;
                PrintDisconnectReason(disconnectReason);
            }
        }
    }

    void TestCaseRejectForStation(
        TestScene* pOutNextScene, TestStatus* pStatus) NN_NOEXCEPT
    {
        if (g_StateChangeEvent.TryWait() &&
            nn::ldn::GetState() != nn::ldn::State_StationConnected)
        {
            *pOutNextScene = TestScene_CleanupStation;
            auto disconnectReason = nn::ldn::GetDisconnectReason();
            if (disconnectReason != nn::ldn::DisconnectReason_Rejected)
            {
                ++pStatus->result.disconnected;
                PrintDisconnectReason(disconnectReason);
            }
        }
    }

    void TestCaseDisconnectForStation(
        TestScene* pOutNextScene, TestStatus* pStatus) NN_NOEXCEPT
    {
        // 一定時間待機します。
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(pStatus->wait));

        // 切断します。
        nn::Result result = nn::ldn::Disconnect();
        if (result.IsFailure())
        {
            ++pStatus->result.disconnected;
            PrintDisconnectReason(nn::ldn::GetDisconnectReason());
        }
        *pOutNextScene = TestScene_CleanupStation;
    }

    void TestCaseWifiOffApForStation(
        TestScene* pOutNextScene, TestStatus* pStatus) NN_NOEXCEPT
    {
        if (g_StateChangeEvent.TryWait() &&
            nn::ldn::GetState() != nn::ldn::State_StationConnected)
        {
            *pOutNextScene = TestScene_CleanupStation;
            auto disconnectReason = nn::ldn::GetDisconnectReason();
            if (disconnectReason != nn::ldn::DisconnectReason_DestroyedBySystem)
            {
                ++pStatus->result.disconnected;
                PrintDisconnectReason(disconnectReason);
            }
        }
    }

    void TestCaseWifiOffStaForStation(
        TestScene* pOutNextScene, TestStatus* pStatus) NN_NOEXCEPT
    {
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(pStatus->wait));
        nnt::ldn::NifmInitializer initializer;
        {
            nnt::ldn::ScopedWirelessCommunicationSwitchSetter setter(false);
            ASSERT_EQ(nn::ldn::State_Error, nn::ldn::GetState());
        }
        *pOutNextScene = TestScene_Finalizing;
    }

    void UpdateNone(TestStatus* pStatus) NN_NOEXCEPT
    {
        // LDN ライブラリを初期化します。
        NNT_LDN_LOG_DEBUG("initializing the ldn library...\n");
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::InitializeSystem());
        SetLdnSettings();

        // socket ライブラリを初期化します。
        NNT_LDN_LOG_DEBUG("initializing the socket library...\n");
        NNT_ASSERT_RESULT_SUCCESS(nn::socket::Initialize(g_SocketConfig));

        // 状態変化を通知するイベントを取得します。
        nn::ldn::AttachStateChangeEvent(g_StateChangeEvent.GetBase());

        // 次のシーンへの遷移です。
        pStatus->scene = TestScene_Setup;
    }

    void UpdateFinalizing(TestStatus* pStatus) NN_NOEXCEPT
    {
        // データ通信を終了します。
        if (pStatus->socket != nn::socket::InvalidSocket)
        {
            StopCommunication(pStatus);
        }

        // 状態変化を通知するイベントを破棄します。
        nn::os::DestroySystemEvent(g_StateChangeEvent.GetBase());

        // socket ライブラリを初期化します。
        NNT_LDN_LOG_DEBUG("finalizing the socket library...\n");
        NNT_ASSERT_RESULT_SUCCESS(nn::socket::Finalize());

        // LDN ライブラリを終了します。
        NNT_LDN_LOG_DEBUG("finalizing the ldn library...\n");
        nn::ldn::FinalizeSystem();

        // 次のシーンへの遷移です。
        pStatus->scene = TestScene_None;
    }

    void UpdateSetup(TestStatus* pStatus) NN_NOEXCEPT
    {
        // テストカウンタをインクリメントします。
        auto hour   = pStatus->elapsed / 3600;
        auto min    = (pStatus->elapsed / 60) % 60;
        auto second = pStatus->elapsed % 60;
        NNT_LDN_LOG_INFO_WITHOUT_PREFIX("\n");
        NNT_LDN_LOG_INFO("Time         : %02d:%02d:%02d\n", hour, min, second);
        NNT_LDN_LOG_INFO("Received     : %d frames\n", pStatus->receivedCounter);
        NN_UNUSED(second);

        // 待ち時間を決定します。
        pStatus->wait = Random(0, 1000);
        NNT_LDN_LOG_INFO("Wait         : %d ms\n", pStatus->wait);

        // アクセスポイントとステーションの分岐です。
        if (g_TestConfig.nodeIndex == 0)
        {
            pStatus->scene = TestScene_SetupAccessPoint;
        }
        else
        {
            pStatus->scene = TestScene_SetupStation;
        }
    }

    void UpdateSetupAccessPoint(TestStatus* pStatus) NN_NOEXCEPT
    {
        // 終了時刻を過ぎているかどうかを判定し、他の開発機と同期します。
        bool isTimedOut = g_TestConfig.duration <= pStatus->elapsed;
        g_pServer->SetData(&isTimedOut, sizeof(isTimedOut));
        NNT_LDN_SYNC("Aging.Setup");

        // 終了時刻が過ぎているかどうかに寄って処理を分岐します。
        if (isTimedOut)
        {
            pStatus->scene = TestScene_Timeout;
        }
        else
        {
            NNT_LDN_LOG_DEBUG("starting accesspoint mode...\n");
            NNT_ASSERT_RESULT_SUCCESS(nn::ldn::OpenAccessPoint());
            pStatus->scene = TestScene_CreatingNetwork;
        }
    }

    void UpdateCreatingNetwork(TestStatus* pStatus) NN_NOEXCEPT
    {
        // テストケースをランダムに決定します。
        pStatus->testCase = static_cast<uint16_t>(Random(0, TestCase_Count - 1));
        PrintTestCase(static_cast<TestCase>(pStatus->testCase));

        // ネットワークのパラメータを適当に設定します。
        nn::ldn::NetworkConfig network = { };
        network.intentId = nn::ldn::MakeIntentId(LocalCommunicationId, g_TestConfig.sceneId);
        network.nodeCountMax = static_cast<int8_t>(g_TestConfig.nodeCount);
        network.channel = g_TestConfig.channel;
        nn::ldn::SecurityConfig security = { };
        security.securityMode = static_cast<nn::Bit16>(g_TestConfig.isEncrypted ?
            nn::ldn::SecurityMode_Product : nn::ldn::SecurityMode_Debug);
        security.passphraseSize = sizeof(Passphrase);
        std::memcpy(security.passphrase, Passphrase, sizeof(Passphrase));
        nn::ldn::UserConfig user = { };
        std::strncpy(user.userName, ApUserName, nn::ldn::UserNameBytesMax);

        // Advertise でテストの情報を配信します。
        AdvertiseData advertise;
        advertise.testCase = nn::socket::InetHtonl(pStatus->testCase);
        advertise.testCount = nn::socket::InetHtonl(pStatus->testCount);
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::SetAdvertiseData(&advertise, sizeof(advertise)));

        // 全てのステーションの接続を受け入れます。
        NNT_ASSERT_RESULT_SUCCESS(
            nn::ldn::SetStationAcceptPolicy(nn::ldn::AcceptPolicy_AlwaysAccept));

        // ネットワークを構築します。
        NNT_LDN_LOG_DEBUG("creating a network...\n");
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetwork(network, security, user));
        ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetState());

        // ネットワークの情報を取得します。
        ++pStatus->testCount;
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(&g_Network));
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetIpv4Address(&pStatus->address, &pStatus->mask));
        NNT_LDN_LOG_INFO("Test Counter : %d\n", pStatus->testCount);
        NNT_LDN_LOG_INFO("Channel      : %d\n", g_Network.common.channel);
        NNT_LDN_LOG_INFO("SSID         : %s\n", g_Network.common.ssid.raw);

        // データフレームの送受信を開始します。
        StartCommunication(pStatus);
        NNT_LDN_SYNC("Aging.NetworkCreated");

        // 次のシーンへの遷移です。
        pStatus->scene = TestScene_NetworkCreated;
    }

    void UpdateNetworkCreated(TestStatus* pStatus) NN_NOEXCEPT
    {
        // 状態変化が無い場合、何もしません。
        if (!g_StateChangeEvent.TryWait())
        {
            return;
        }

        // ステーションの接続状態を確認します。
        nn::ldn::NetworkInfo networkInfo;
        nn::ldn::NodeLatestUpdate updates[nn::ldn::NodeCountMax];
        NNT_ASSERT_RESULT_SUCCESS(
            nn::ldn::GetNetworkInfo(&networkInfo, updates, nn::ldn::NodeCountMax));
        for (int i = 0; i < nn::ldn::NodeCountMax; ++i)
        {
            const auto& node = networkInfo.ldn.nodes[i];
            switch (updates[i].stateChange)
            {
            case nn::ldn::NodeStateChange_None:
                break;
            case nn::ldn::NodeStateChange_Connect:
                NNT_LDN_LOG_DEBUG("node %d connected\n", node.nodeId);
                break;
            case nn::ldn::NodeStateChange_Disconnect:
                NNT_LDN_LOG_WARN("node %d disconnected\n", node.nodeId);
                break;
            case nn::ldn::NodeStateChange_DisconnectAndConnect:
                NNT_LDN_LOG_WARN("node %d disconnected and connected\n", node.nodeId);
                break;
            default:
                NN_UNEXPECTED_DEFAULT;
            }
            NN_UNUSED(node);
        }
        NNT_LDN_LOG_DEBUG("%d nodes are available\n", networkInfo.ldn.nodeCount);

        // テストの実行に必要なステーションが集まった場合、次のシーンに遷移します。
        if (networkInfo.ldn.nodeCount == g_TestConfig.nodeCount)
        {
            NNT_LDN_SYNC("Aging.Connected");
            NNT_LDN_LOG_DEBUG("start\n");
            NNT_ASSERT_RESULT_SUCCESS(
                nn::ldn::SetStationAcceptPolicy(nn::ldn::AcceptPolicy_AlwaysReject));
            pStatus->scene = TestScene_RunningAccessPointTest;
        }
    }

    void UpdateRunningAccessPointTest(TestStatus* pStatus) NN_NOEXCEPT
    {
        // テストケースに応じた処理です。
        TestScene nextScene = TestScene_RunningAccessPointTest;
        switch (pStatus->testCase)
        {
        case TestCase_DestroyNetwork:
            TestCaseDestroyNetworkForAccessPoint(&nextScene, pStatus);
            break;
        case TestCase_Reject:
            TestCaseRejectForAccessPoint(&nextScene, pStatus);
            break;
        case TestCase_Disconnect:
            TestCaseDisconnectForAccessPoint(&nextScene, pStatus);
            break;
        case TestCase_WififOffAp:
            TestCaseWifiOffApForAccessPoint(&nextScene, pStatus);
            break;
        case TestCase_WififOffSta:
            TestCaseWifiOffStaForAccessPoint(&nextScene, pStatus);
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }

        // テストが完了した場合は次のシーンに遷移します。
        if (nextScene != TestScene_RunningAccessPointTest)
        {
            pStatus->scene = nextScene;
        }
    }

    void UpdateCleanupAccessPoint(TestStatus* pStatus) NN_NOEXCEPT
    {
        // アクセスポイントを終了します。
        NNT_LDN_LOG_DEBUG("clean up access point...\n");
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CloseAccessPoint());

        // データ通信を終了します。
        if (pStatus->socket != nn::socket::InvalidSocket)
        {
            StopCommunication(pStatus);
        }

        // 次のテストに移ります。
        pStatus->scene = TestScene_Setup;
    }

    void UpdateSetupStation(TestStatus* pStatus) NN_NOEXCEPT
    {
        // 他の開発機と同期します。
        NNT_LDN_SYNC("Aging.Setup");
        uint8_t isTimedOut;
        size_t dataSize;
        int result = g_pClient->GetData(&isTimedOut, &dataSize, sizeof(isTimedOut));
        ASSERT_EQ(nnt::ldn::SynchronizationResult_Success, result);
        ASSERT_EQ(1U, dataSize);

        // 終了時刻が過ぎているかどうかに寄って処理を分岐します。
        if (isTimedOut != 0U)
        {
            pStatus->scene = TestScene_Timeout;
        }
        else
        {
            NNT_LDN_LOG_DEBUG("starting station mode...\n");
            NNT_ASSERT_RESULT_SUCCESS(nn::ldn::OpenStation());
            pStatus->scene = TestScene_Scanning;
            NNT_LDN_SYNC("Aging.NetworkCreated");
        }
    }

    void UpdateScanning(TestStatus* pStatus) NN_NOEXCEPT
    {
        // アクセスポイントを終了します。
        NNT_LDN_LOG_DEBUG("scanning...\n");

        // IntentID が一致するネットワークをスキャン対象とします。
        int count;
        nn::ldn::ScanFilter filter = { };
        filter.networkId.intentId = nn::ldn::MakeIntentId(LocalCommunicationId, g_TestConfig.sceneId);
        filter.flag = static_cast<nn::Bit32>(nn::ldn::ScanFilterFlag_IntentId);

        // ネットワークをスキャンします。
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::Scan(
            g_ScanBuffer, &count, nn::ldn::ScanResultCountMax,
            filter, g_TestConfig.channel));

        // ネットワークを発見したら次のテストに移ります。
        for (int i = 0; i < count; ++i)
        {
            auto& network = g_ScanBuffer[i];
            auto advertise = *reinterpret_cast<const AdvertiseData*>(network.ldn.advertiseData);
            advertise.testCase = nn::socket::InetNtohl(advertise.testCase);
            advertise.testCount = nn::socket::InetNtohl(advertise.testCount);
            if (network.ldn.advertiseDataSize == sizeof(AdvertiseData) &&
                pStatus->testCount <= advertise.testCount)
            {
                g_Network = network;
                pStatus->scene = TestScene_Connecting;
                pStatus->testCase = advertise.testCase;
                pStatus->testCount = advertise.testCount;
                PrintTestCase(static_cast<TestCase>(pStatus->testCase));
                NNT_LDN_LOG_INFO("Test Counter : %d\n", pStatus->testCount);
                NNT_LDN_LOG_INFO("Channel      : %d\n", network.common.channel);
                NNT_LDN_LOG_INFO("SSID         : %s\n", network.common.ssid.raw);
                return;
            }
        }
    }

    void UpdateConnecting(TestStatus* pStatus) NN_NOEXCEPT
    {
        NNT_LDN_LOG_DEBUG("connecting...\n");

        // ネットワークへの接続に必要なパラメータを生成します。
        nn::ldn::SecurityConfig security = { };
        security.securityMode = static_cast<nn::Bit16>(g_TestConfig.isEncrypted ?
            nn::ldn::SecurityMode_Product : nn::ldn::SecurityMode_Debug);
        security.passphraseSize = sizeof(Passphrase);
        std::memcpy(security.passphrase, Passphrase, sizeof(Passphrase));
        nn::ldn::UserConfig user = { };
        std::strncpy(user.userName, StaUserName, nn::ldn::UserNameBytesMax);

        // ネットワークへの接続を試行します。
        ++pStatus->result.trial;
        nn::Result result = nn::ldn::Connect(
            g_Network, security, user, 0, nn::ldn::ConnectOption_None);
        if (result.IsSuccess())
        {
            ++pStatus->result.success;
            pStatus->result.burst = 0;
            if (nn::ldn::GetNetworkInfo(&g_Network).IsSuccess() &&
                nn::ldn::GetIpv4Address(&pStatus->address, &pStatus->mask).IsSuccess())
            {
                StartCommunication(pStatus);
                pStatus->scene = TestScene_Connected;
                NNT_LDN_LOG_DEBUG("waiting for other stations...\n");
            }
            else
            {
                ++pStatus->result.disconnected;
                PrintDisconnectReason(nn::ldn::GetDisconnectReason());
            }
        }
        else
        {
            ++pStatus->result.failure;
            ++pStatus->result.burst;
            pStatus->result.burstMax = std::max(pStatus->result.burst, pStatus->result.burstMax);
            if (nn::ldn::ResultNetworkNotFound::Includes(result))
            {
                ++pStatus->result.reason.notFound;
                NNT_LDN_LOG_WARN("network not found\n");
            }
            else if (nn::ldn::ResultConnectionTimeout::Includes(result))
            {
                ++pStatus->result.reason.timeout;
                NNT_LDN_LOG_WARN("connection timeout\n");
            }
            else if (nn::ldn::ResultConnectionRejected::Includes(result))
            {
                ++pStatus->result.reason.rejected;
                NNT_LDN_LOG_WARN("connection rejected\n");
            }
            else
            {
                NNT_ASSERT_RESULT_SUCCESS(result);
            }

            // スキャンからやり直します。
            pStatus->scene = TestScene_Scanning;
        }
    }

    void UpdateConnected(TestStatus* pStatus) NN_NOEXCEPT
    {
        // 状態変化が無い場合、何もしません。
        if (!g_StateChangeEvent.TryWait())
        {
            return;
        }

        // 切断されてしまった場合には最初からやり直します。
        if (nn::ldn::GetState() != nn::ldn::State_StationConnected)
        {
            ++pStatus->result.disconnected;
            pStatus->scene = TestScene_CleanupStation;
            PrintDisconnectReason(nn::ldn::GetDisconnectReason());
            return;
        }

        // ステーションの接続状態を確認します。
        nn::ldn::NetworkInfo networkInfo;
        nn::ldn::NodeLatestUpdate updates[nn::ldn::NodeCountMax];
        NNT_ASSERT_RESULT_SUCCESS(
            nn::ldn::GetNetworkInfo(&networkInfo, updates, nn::ldn::NodeCountMax));
        for (int i = 0; i < nn::ldn::NodeCountMax; ++i)
        {
            const auto& node = networkInfo.ldn.nodes[i];
            switch (updates[i].stateChange)
            {
            case nn::ldn::NodeStateChange_None:
                break;
            case nn::ldn::NodeStateChange_Connect:
                NNT_LDN_LOG_DEBUG("node %d connected\n", node.nodeId);
                break;
            case nn::ldn::NodeStateChange_Disconnect:
                NNT_LDN_LOG_WARN("node %d disconnected\n", node.nodeId);
                break;
            case nn::ldn::NodeStateChange_DisconnectAndConnect:
                NNT_LDN_LOG_WARN("node %d disconnected and connected\n", node.nodeId);
                break;
            default:
                NN_UNEXPECTED_DEFAULT;
            }
            NN_UNUSED(node);
        }
        NNT_LDN_LOG_DEBUG("%d nodes are available\n", networkInfo.ldn.nodeCount);

        // テストの実行に必要なステーションが集まった場合、次のシーンに遷移します。
        if (networkInfo.ldn.nodeCount == g_TestConfig.nodeCount)
        {
            NNT_LDN_SYNC("Aging.Connected");
            NNT_LDN_LOG_DEBUG("start\n");
            pStatus->scene = TestScene_RunningStationTest;
        }
    }

    void UpdateRunningStationTest(TestStatus* pStatus) NN_NOEXCEPT
    {
        // テストケースに応じた処理です。
        TestScene nextScene = TestScene_RunningStationTest;
        switch (pStatus->testCase)
        {
        case TestCase_DestroyNetwork:
            TestCaseDestroyNetworkForStation(&nextScene, pStatus);
            break;
        case TestCase_Reject:
            TestCaseRejectForStation(&nextScene, pStatus);
            break;
        case TestCase_Disconnect:
            TestCaseDisconnectForStation(&nextScene, pStatus);
            break;
        case TestCase_WififOffAp:
            TestCaseWifiOffApForStation(&nextScene, pStatus);
            break;
        case TestCase_WififOffSta:
            TestCaseWifiOffStaForStation(&nextScene, pStatus);
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }

        // テストが完了した場合は次のシーンに遷移します。
        if (nextScene != TestScene_RunningStationTest)
        {
            pStatus->scene = nextScene;
        }
    }

    void UpdateCleanupStation(TestStatus* pStatus) NN_NOEXCEPT
    {
        // ステーションを終了します。
        NNT_LDN_LOG_DEBUG("closing the station mode...\n");
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CloseStation());

        // データ通信を終了します。
        if (pStatus->socket != nn::socket::InvalidSocket)
        {
            StopCommunication(pStatus);
        }

        // 次のテストに移ります。
        pStatus->scene = TestScene_Setup;

        // 現時点のテスト結果を出力します。
        PrintTestResult(pStatus->result);
    }

} // namespace <unnamed>

//
// 接続・切断関係のエイジングです。
//
TEST(Aging, Connection)
{
    // テストの設定を表示します。
    NNT_LDN_LOG_INFO("Seed         : %d\n", g_TestConfig.seed);
    NNT_LDN_LOG_INFO("Scene Id     : %d\n", g_TestConfig.sceneId);
    NNT_LDN_LOG_INFO("Node Type    : %s\n", g_TestConfig.nodeIndex == 0 ? "AP" : "STA");
    NNT_LDN_LOG_INFO("Node Index   : %d/%d\n", g_TestConfig.nodeIndex, g_TestConfig.nodeCount);
    NNT_LDN_LOG_INFO("Encryption   : %s\n", g_TestConfig.isEncrypted ? "enabled" : "disabled");
    NNT_LDN_LOG_INFO("Packet Rate  : %d packets/sec\n", g_TestConfig.packetRate);
    NNT_LDN_LOG_INFO("Channel      : %d\n", g_TestConfig.channel);
    NNT_LDN_LOG_INFO_WITHOUT_PREFIX("\n");

    // テストの状態を初期化しておきます。
    TestStatus status;
    std::memset(&status, 0, sizeof(status));
    status.socket = static_cast<int32_t>(nn::socket::InvalidSocket);

    // 他の開発機と同期します。
    NNT_LDN_LOG_DEBUG("synchronizing...\n");
    NNT_LDN_SYNC("Aging.Start");

    // テスト開始時点の時刻を取得しておきます。
    nn::os::Tick startedAt = nn::os::GetSystemTick();

    // ネットワークの構築と破棄を繰り返します。
    bool isRunning = true;
    while (isRunning)
    {
        // テストが始まってからの経過時間を取得します。
        auto now = nn::os::GetSystemTick();
        status.elapsed = static_cast<int>((now - startedAt).ToTimeSpan().GetSeconds());

        // シーンに応じた処理です。
        switch (status.scene)
        {
        case TestScene_None:
            UpdateNone(&status);
            break;
        case TestScene_Finalizing:
            UpdateFinalizing(&status);
            break;
        case TestScene_Setup:
            UpdateSetup(&status);
            break;
        case TestScene_Timeout:
            UpdateFinalizing(&status);
            isRunning = false;
            break;
        case TestScene_SetupAccessPoint:
            UpdateSetupAccessPoint(&status);
            break;
        case TestScene_CreatingNetwork:
            UpdateCreatingNetwork(&status);
            break;
        case TestScene_NetworkCreated:
            UpdateNetworkCreated(&status);
            break;
        case TestScene_RunningAccessPointTest:
            UpdateRunningAccessPointTest(&status);
            break;
        case TestScene_CleanupAccessPoint:
            UpdateCleanupAccessPoint(&status);
            break;
        case TestScene_SetupStation:
            UpdateSetupStation(&status);
            break;
        case TestScene_Scanning:
            UpdateScanning(&status);
            break;
        case TestScene_Connecting:
            UpdateConnecting(&status);
            break;
        case TestScene_Connected:
            UpdateConnected(&status);
            break;
        case TestScene_RunningStationTest:
            UpdateRunningStationTest(&status);
            break;
        case TestScene_CleanupStation:
            UpdateCleanupStation(&status);
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }

        // 60FPS 程度になるようにスリープします。
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
    }

    // 接続成功率が閾値を上回っていれば成功とみなします。
    if (0 < status.result.trial)
    {
        float successRate = status.result.success * 100.0f / status.result.trial;
        ASSERT_GE(successRate, g_TestConfig.threshold);
    }
}

//
// テストのエントリポイントです。
//
extern "C" void nnMain()
{
    // コマンドライン引数に関する設定です。
    nnt::ldn::CommandLineParserSetting setting = { };
    setting.flag = static_cast<nn::Bit32>(
        nnt::ldn::CommandLineOptionFlag_NodeCount |
        nnt::ldn::CommandLineOptionFlag_NodeIndex |
        nnt::ldn::CommandLineOptionFlag_Channel |
        nnt::ldn::CommandLineOptionFlag_Encryption |
        nnt::ldn::CommandLineOptionFlag_Duration |
        nnt::ldn::CommandLineOptionFlag_PacketRate |
        nnt::ldn::CommandLineOptionFlag_Threshold |
        nnt::ldn::CommandLineOptionFlag_SceneId |
        nnt::ldn::CommandLineOptionFlag_OperationMode);
    setting.nodeCountMin = 2;
    setting.nodeCountMax = 8;
    setting.defaultSceneId = SceneId;

    // コマンドライン引数を解析します。
    int argc = nn::os::GetHostArgc();
    char **argv = nn::os::GetHostArgv();
    ::testing::InitGoogleTest(&argc, argv);
    nnt::ldn::Parse(&g_TestConfig, setting, argc, argv);

    // 乱数のシード値を設定しておきます。
    SetSeed(g_TestConfig.seed);

    // 他の開発機と同期する準備です。
    int result;
    if (g_TestConfig.nodeIndex == 0)
    {
        g_pServer = new nnt::ldn::HtcsSynchronizationServer(
            g_SynchronizationBuffer, sizeof(g_SynchronizationBuffer));
        result = g_pServer->CreateServer("nn::ldn::AgingConnection", g_TestConfig.nodeCount - 1);
        g_pSync = g_pServer;
    }
    else
    {
        g_pClient = new nnt::ldn::HtcsSynchronizationClient(
            g_SynchronizationBuffer, sizeof(g_SynchronizationBuffer));
        result = g_pClient->Connect("nn::ldn::AgingConnection");
        g_pSync = g_pClient;
    }

    // テストを実行します。
    int exitCode;
    if (result == nnt::ldn::SynchronizationResult_Success)
    {
        exitCode = RUN_ALL_TESTS();
    }
    else
    {
        NNT_LDN_LOG_ERROR("failed to sync: %d\n", result);
        exitCode = result;
    }
    delete g_pSync;
    nnt::Exit(exitCode);
}
