﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <tuple>
#include <nn/nn_Log.h>
#include <nn/lcs.h>
#include <nn/lcs/lcs_DebugApi.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/socket.h>
#include <nn/util/util_Optional.h>
#include <nnt.h>
#include <nnt/lcs/testLcs_Utility.h>
#include <nnt/ldn/testLdn_HtcsSynchronization.h>
#include <nnt/ldn/testLdn_NifmUtility.h>
#include <nnt/ldn/testLdn_Utility.h>
#include <nnt/result/testResult_Assert.h>
#include "Params.h"
#include "Precondition.h"
#include "Utils.h"

namespace
{
    // テストの設定値です。
    nnt::lcs::TestConfig g_TestConfig;

    // LCS/Socket の初期化用バッファです。
    NN_ALIGNAS(nn::lcs::RequiredBufferAlignment) char g_Buffer[4 * 1024 * 1024];
    NN_STATIC_ASSERT(nn::lcs::RequiredBufferSize < sizeof(g_Buffer));
    nn::socket::ConfigDefaultWithMemory g_SocketConfig;

    // EULA 取得用のバッファです。
    // char g_EulaBuffer[1 * 1024 * 1024];

    // 同期に使用するオブジェクトとバッファです。
    nnt::ldn::ISynchronizationServer* g_pServer;
    nnt::ldn::ISynchronizationClient* g_pClient;
    nnt::ldn::ISynchronization* g_pSync;
    char g_SynchronizationBuffer[16 * 1024];

    // 同期用のマクロです。
#define NNT_LDN_SYNC(keyword)\
        ASSERT_EQ(::nnt::ldn::SynchronizationResult_Success, g_pSync->Synchronize(keyword))

    // テストを実施する条件を指定するためのマクロです。
#define NNT_LDN_REQUIRES_NODE_COUNT(count) \
        if (g_TestConfig.nodeCount < (count))\
        {\
            return;\
        }

    // NNT_ASSERT_RESEULT_SUCEESS と NNT_ASSERT_RESEULT_FAILURE の分岐と *_FAILURE に Result の型ではなく値を渡すためのマクロです。
#define NNT_ASSERT_UNLESS_RESULT(expectedResult, actualResult) \
        do \
        { \
            if (expectedResult) \
            { \
                ASSERT_PRED_FORMAT3([](const char* , const char* expectedExpression, const char* resultExpression, bool (*isExpected)(nn::Result), nn::Result expectedResult, nn::Result result){ return nnt::result::detail::IncludeResultImpl(expectedExpression, resultExpression, isExpected, expectedResult, result); }, std::get<1>(expectedResult.value()), std::get<0>(expectedResult.value()), actualResult); \
            } \
            else \
            { \
                NNT_ASSERT_RESULT_SUCCESS(actualResult); \
            } \
        } while (NN_STATIC_CONDITION(false))

#define DISABLE_WIRELESS_COMMUNICATION() \
        do \
        { \
            NN_LOG("Disable wireless communication.\n"); \
            nnt::ldn::SetWirelessCommunicationSwitch(false); \
        } while (NN_STATIC_CONDITION(false))

#define ENABLE_WIRELESS_COMMUNICATION() \
        do \
        { \
            NN_LOG("Enable wireless communication.\n"); \
            nnt::ldn::SetWirelessCommunicationSwitch(true); \
        } while (NN_STATIC_CONDITION(false))

    enum TestState
    {
        TestState_End,

        TestState_Begin,

        TestState_BranchByNodeIndex,

        TestState_Host,                                   // ホストのフローです。
        TestState_HostCreateSessionSettings = TestState_Host,
        TestState_HostOpenSession,
        TestState_HostSyncSessionOpened,
        TestState_HostSyncJoined,
        TestState_HostSyncReadyToStart,
        TestState_HostStartContentShare,
        TestState_HostSyncStarted,

        TestState_Client,                                 // クライアントのフローです。
        TestState_ClientSyncSessionOpened = TestState_Client,
        TestState_ClientInitializeJoinWaitLoop,
        TestState_ClientDecrementJoinWaitLoop,
        TestState_ClientScanSession,
        TestState_ClientBranchJoin,
        TestState_ClientCheckStateBeforeJoin,
        TestState_ClientJoin,
        TestState_ClientSyncJoined,
        TestState_ClientWaitChangeStateToJoined,
        TestState_ClientClearEvent,
        TestState_ClientSyncReadyToStart,
        TestState_ClientSyncStarted,
        TestState_ClientWaitChangeState,

        TestState_BranchByShareRole,

        TestState_Sender,                                // 送信者のフローです。
        TestState_SenderSkipWaitComplete = TestState_Sender,
        TestState_SenderCheckTransferringStateBeforeCompleted,
        TestState_SenderWaitComplete,
        TestState_SenderSyncCompleted,
        TestState_SenderCheckDownloadedContents,

        TestState_Receiver,                              // 受信者のフローです。
        TestState_ReceiverInitializeWaitSuspendedState = TestState_Receiver,
        TestState_ReceiverSkipWaitSuspendedState,
        TestState_ReceiverCheckStateBeforeSuspended,
        TestState_ReceiverWaitSuspendedState,
        TestState_ReceiverCheckAfterWaitSuspendedState,
        TestState_ReceiverCheckSuspendedReason,
        TestState_ReceiverResumeContentsShare,
        TestState_ReceiverInitializeWaitComplete,
        TestState_ReceiverSkipWaitComplete,
        TestState_ReceiverCheckStateBeforeCompleted,
        TestState_ReceiverWaitComplete,
        TestState_ReceiverSyncCompleted,
        TestState_ReceiverCheckDownloadedContents,

        TestState_Leave,                                 // セッション終了のフローです。
        TestState_CheckCompletedState = TestState_Leave,
        TestState_LeaveSession,
        TestState_SyncDestroyed,
    };

    // パッチ配信の正常系テストを実行するステートマシンです。
    class ContentsShareStateMachine
    {
    private:
        nnt::lcs::NsInitializer     m_NsInitializer;
        nnt::ldn::SystemInitializer m_LdnInitializer;
        nnt::lcs::Initializer       m_LcsInitializer;

        nnt::lcs::Precondition m_Cond;

        nn::os::SystemEventType  m_StateChangeEvent;
        bool                     m_IsInitializedStateChangeEvent;
        nn::lcs::SessionSettings m_Settings;
        int                      m_RetryCount;
        nn::lcs::SessionInfo     m_Sessions[nn::lcs::ScanResultCountMax];
        int                      m_SessionIndex;
        nn::lcs::SessionInfo     m_Session;
        nn::lcs::State           m_State;
        nn::os::Tick             m_BeginTime;

        TestState m_TestState;

        nn::util::optional<std::tuple<nn::Result, bool(*)(nn::Result)>>         m_ExpectedResult;
        nn::util::optional<nn::util::optional<bool>>                            m_IsExpectedStateTransition;
        nn::util::optional<nn::util::optional<bool(*)(const nn::lcs::State&)>>  m_ExpectedState;
        nn::util::optional<bool(*)(const nn::lcs::ContentsShareFailureReason&)> m_ExpectedContentsShareFailureReason;

#define TESTSTATE(state, next_state, isExpectedPostTransition) \
            break; \
        case state: \
            NN_LOG("State : %d, Changed : %5s, TestState : %d (%s)\n", nn::lcs::GetState(), nn::os::TryWaitSystemEvent(&m_StateChangeEvent) ? "true" : "false", state, #state); \
            m_TestState = (next_state); \
            isExpectedStateTransition = (isExpectedPostTransition); \
            expectedState = nn::util::nullopt

#define TESTSTATE_WITH_POSTSTATE(state, next_state, isExpectedPostTransition, ...) \
            break; \
        case state: \
            NN_LOG("State : %d, Changed : %5s, TestState : %d (%s)\n", nn::lcs::GetState(), nn::os::TryWaitSystemEvent(&m_StateChangeEvent) ? "true" : "false", state, #state); \
            m_TestState = (next_state); \
            isExpectedStateTransition = (isExpectedPostTransition); \
            expectedState = SetComparator<nn::lcs::State, __VA_ARGS__>::Compare

#define TESTSTATE_BRANCH(state, isExpectedPostTransition, condition, t_state, f_state) \
            TESTSTATE(state, (condition) ? (t_state) : (f_state), isExpectedPostTransition)

    private:
        template <typename... InterceptStates>
        bool IsStateEquals(TestState state, InterceptStates... states)
        {
            return (m_TestState == state) || IsStateEquals(states...);
        }

        bool IsStateEquals()
        {
            return false;
        }

    public:
        explicit ContentsShareStateMachine(const nnt::lcs::Precondition& cond) NN_NOEXCEPT :
                m_LcsInitializer(g_Buffer, sizeof(g_Buffer), nnt::lcs::CreateDefaultHostConfig()),
                m_Cond(cond),
                m_IsInitializedStateChangeEvent(false),
                m_TestState(TestState_Begin)
        {
        }

        ~ContentsShareStateMachine() NN_NOEXCEPT
        {
        }

        // 指定されたテストステートの直前までテストを実行します。
        // テストステートを指定しなかった場合はステートを一つだけ実行して終わります。
        template <typename... InterceptStates>
        void Run(InterceptStates... interceptStates) NN_NOEXCEPT
        {
            // 上書き期待値は目的ステート到達後にリセットします。
            NN_UTIL_SCOPE_EXIT
            {
                m_ExpectedResult                     = nn::util::nullopt;
                m_IsExpectedStateTransition          = nn::util::nullopt;
                m_ExpectedState                      = nn::util::nullopt;
                m_ExpectedContentsShareFailureReason = nn::util::nullopt;
            };

            // 既に目的テストステート若しくは終了ステートなら何もしません。
            if (IsStateEquals(interceptStates...) || m_TestState == TestState_End)
            {
                return;
            }

            if (!m_IsInitializedStateChangeEvent)
            {
                // イベントを取得します。
                nn::lcs::AttachStateChangeEvent(&m_StateChangeEvent);
                m_IsInitializedStateChangeEvent = true;
            }

            do
            {
                // テストステート正常終了条件です。
                nn::util::optional<bool>                           isExpectedStateTransition; // true : LCS ステート遷移があることを期待, false : LCS ステート遷移がないことを期待, 無効 : LCS ステート遷移には関心がない
                nn::util::optional<bool(*)(const nn::lcs::State&)> expectedState;             // 期待される LCS ステートの遷移先, 無効 : 遷移先には関心がない

                switch (m_TestState)
                {
                case TestState_End:
                    break;

                TESTSTATE(TestState_Begin, TestState_BranchByNodeIndex, nn::util::nullopt);

                //
                // ホストとクライアントの分岐をします。
                //
                TESTSTATE_BRANCH(TestState_BranchByNodeIndex, nn::util::nullopt,
                    g_TestConfig.nodeIndex == m_Cond.masterIndex,
                    TestState_Host,
                    TestState_Client);

                //
                // ホストのフローです。
                //
                TESTSTATE(TestState_HostCreateSessionSettings, TestState_HostOpenSession, false);
                    // セッションを構築します。
                    m_Settings = nnt::lcs::CreateSessionSettings(nn::lcs::NodeCountMax, m_Cond.application.id);

                TESTSTATE_WITH_POSTSTATE(TestState_HostOpenSession, TestState_HostSyncSessionOpened, true, nn::lcs::State_Opened);
                    // セッションをオープンします。
                    NNT_ASSERT_UNLESS_RESULT(m_ExpectedResult, nn::lcs::OpenSession(m_Settings));

                TESTSTATE(TestState_HostSyncSessionOpened, TestState_HostSyncJoined, nn::util::nullopt);
                    // セッションオープン完了の同期をします。
                    NNT_LDN_SYNC("PatchShareFailure.RunTest.SessionOpened");

                TESTSTATE(TestState_HostSyncJoined, TestState_HostSyncReadyToStart, true);
                    // クライアントの接続完了の同期をします。。
                    NNT_LDN_SYNC("PatchShareFailure.RunTest.Joined");

                TESTSTATE(TestState_HostSyncReadyToStart, TestState_HostStartContentShare, false);
                    // 配信準備完了の同期をします。
                    NNT_LDN_SYNC("PatchShareFailure.RunTest.ReadyToStart");

                TESTSTATE(TestState_HostStartContentShare, TestState_HostSyncStarted, true);
                    // コンテンツの配信を開始します。
                    NNT_ASSERT_UNLESS_RESULT(m_ExpectedResult, nn::lcs::StartContentsShare());

                TESTSTATE(TestState_HostSyncStarted, TestState_BranchByShareRole, nn::util::nullopt);
                    // 配信開始の同期をします。
                    NNT_LDN_SYNC("PatchShareFailure.RunTest.Started");

                //
                // クライアントのフローです。
                //
                TESTSTATE(TestState_ClientSyncSessionOpened, TestState_ClientInitializeJoinWaitLoop, nn::util::nullopt);
                    // セッションの構築を待機します。
                    NNT_LDN_SYNC("PatchShareFailure.RunTest.SessionOpened");

                TESTSTATE(TestState_ClientInitializeJoinWaitLoop, TestState_ClientDecrementJoinWaitLoop, nn::util::nullopt);
                    // セッションスキャンリトライ回数を指定します。
                    m_RetryCount = g_TestConfig.nodeCount * 10;

                TESTSTATE(TestState_ClientDecrementJoinWaitLoop, TestState_ClientScanSession, nn::util::nullopt);
                    // スキャンリトライ回数が規定に達したらテスト失敗です。
                    ASSERT_NE(m_RetryCount--, 0);

                TESTSTATE(TestState_ClientScanSession, TestState_ClientBranchJoin, nn::util::nullopt);
                {
                    // セッションをスキャンします。
                    int count;
                    NNT_ASSERT_UNLESS_RESULT(m_ExpectedResult, nnt::lcs::TimedScan(
                        m_Sessions, &count, &m_SessionIndex, nn::lcs::ScanResultCountMax,
                        nnt::lcs::DefaultHostUserName, ScanTimeout));
                }

                TESTSTATE_BRANCH(TestState_ClientBranchJoin, nn::util::nullopt,
                    (m_Session = m_Sessions[m_SessionIndex]).nodeCount == g_TestConfig.nodeIndex,
                    TestState_ClientCheckStateBeforeJoin,
                    TestState_ClientDecrementJoinWaitLoop);

                TESTSTATE_WITH_POSTSTATE(TestState_ClientCheckStateBeforeJoin, TestState_ClientJoin, false, nn::lcs::State_Joined);
                    //

                TESTSTATE_WITH_POSTSTATE(TestState_ClientJoin, TestState_ClientSyncJoined, true, nn::lcs::State_Joined);
                    // セッションに参加します。
                    NNT_ASSERT_UNLESS_RESULT(m_ExpectedResult, nnt::lcs::TimedJoin(m_Session, JoinTimeout));

                TESTSTATE(TestState_ClientSyncJoined, TestState_ClientWaitChangeStateToJoined, nn::util::nullopt);
                    // 全クライアントがセッションに参加し終わるまで待ちます。
                    NNT_LDN_SYNC("PatchShareFailure.RunTest.Joined");

                TESTSTATE(TestState_ClientWaitChangeStateToJoined, TestState_ClientClearEvent, nn::util::nullopt);
                    // 他のクライアントの接続がすぐに反映されない可能性があるため、一定時間待機します。
                    nn::os::SleepThread(StateChangeTimeout);

                TESTSTATE_WITH_POSTSTATE(TestState_ClientClearEvent, TestState_ClientSyncReadyToStart, nn::util::nullopt, nn::lcs::State_Joined);
                    //
                    nn::os::ClearSystemEvent(&m_StateChangeEvent);

                TESTSTATE(TestState_ClientSyncReadyToStart, TestState_ClientSyncStarted, nn::util::nullopt);
                    // ホストが配信準備完了するのを待ちます。
                    NNT_LDN_SYNC("PatchShareFailure.RunTest.ReadyToStart");

                TESTSTATE(TestState_ClientSyncStarted, TestState_ClientWaitChangeState, nn::util::nullopt);
                    // コンテンツ配信の開始まで待機します。
                    NNT_LDN_SYNC("PatchShareFailure.RunTest.Started");

                TESTSTATE(TestState_ClientWaitChangeState, TestState_BranchByShareRole, true);
                    // 状態遷移を待ちます。LCS ステート遷移期待値チェックは下で行うのでここではイベントシグナルクリアはしません。
                    nn::os::TimedWaitSystemEvent(&m_StateChangeEvent, StateChangeTimeout);

                //
                // 配信者と受信者の分岐をします。
                //
                TESTSTATE_BRANCH(TestState_BranchByShareRole, nn::util::nullopt,
                    m_Cond.nodes[g_TestConfig.nodeIndex].application.version < m_Cond.application.latestVersion,
                    TestState_Receiver,
                    TestState_Sender);

                //
                // 配信者のフローです。
                //
                TESTSTATE_BRANCH(TestState_SenderSkipWaitComplete, nn::util::nullopt,
                    (m_State = nn::lcs::GetState()) != nn::lcs::State_Completed,
                    TestState_SenderCheckTransferringStateBeforeCompleted,
                    TestState_SenderSyncCompleted);

                TESTSTATE_WITH_POSTSTATE(TestState_SenderCheckTransferringStateBeforeCompleted, TestState_SenderWaitComplete, nn::util::nullopt, nn::lcs::State_Transferring);
                    // Transferring 状態であることを確認します。

                TESTSTATE(TestState_SenderWaitComplete, TestState_SenderSkipWaitComplete, nn::util::nullopt);
                    // コンテンツ配信の完了まで待機します。
                    ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&m_StateChangeEvent, ShareTimeout));
                    nn::os::ClearSystemEvent(&m_StateChangeEvent);

                TESTSTATE(TestState_SenderSyncCompleted, TestState_SenderCheckDownloadedContents, nn::util::nullopt);
                    // 全台の配信完了を待ちます。
                    NNT_LDN_SYNC("PatchShareFailure.RunTest.Completed");

                TESTSTATE(TestState_SenderCheckDownloadedContents, TestState_Leave, nn::util::nullopt);
                {
                    // この端末は送信者なので何も受信していません。
                    nn::lcs::ContentsInfo contents[nn::lcs::DownloadableContentsCountMax];
                    int contentCount;
                    NNT_ASSERT_UNLESS_RESULT(m_ExpectedResult, nn::lcs::GetDownloadedContents(
                        contents, &contentCount, nn::lcs::DownloadableContentsCountMax));
                    ASSERT_EQ(0, contentCount);
                }

                //
                // 受信者のフローです。
                //
                TESTSTATE(TestState_ReceiverInitializeWaitSuspendedState, TestState_ReceiverSkipWaitSuspendedState, nn::util::nullopt);
                    m_BeginTime = nn::os::GetSystemTick();

                TESTSTATE_BRANCH(TestState_ReceiverSkipWaitSuspendedState, nn::util::nullopt,
                    ((m_State = nn::lcs::GetState()) != nn::lcs::State_Suspended) && ((nn::os::GetSystemTick() - m_BeginTime).ToTimeSpan() < ShareTimeout),
                    TestState_ReceiverCheckStateBeforeSuspended,
                    TestState_ReceiverCheckAfterWaitSuspendedState
                    );

                TESTSTATE_WITH_POSTSTATE(TestState_ReceiverCheckStateBeforeSuspended, TestState_ReceiverWaitSuspendedState, nn::util::nullopt, nn::lcs::State_Transferring);
                    // Suspended に遷移する前は Transferring であることを確認します。

                TESTSTATE(TestState_ReceiverWaitSuspendedState, TestState_ReceiverSkipWaitSuspendedState, nn::util::nullopt);
                    // アプリケーション終了のための Suspended 状態になるまで待機します。
                    ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&m_StateChangeEvent, ShareTimeout - (nn::os::GetSystemTick() - m_BeginTime).ToTimeSpan()));
                    nn::os::ClearSystemEvent(&m_StateChangeEvent);

                TESTSTATE_WITH_POSTSTATE(TestState_ReceiverCheckAfterWaitSuspendedState, TestState_ReceiverCheckSuspendedReason, nn::util::nullopt, nn::lcs::State_Suspended);
                    // LCS ステートを確認します。
                    nn::os::ClearSystemEvent(&m_StateChangeEvent);

                TESTSTATE(TestState_ReceiverCheckSuspendedReason, TestState_ReceiverResumeContentsShare, nn::util::nullopt);
                    // アプリケーション終了のための Suspended 状態であることを確認します。
                    ASSERT_EQ(nn::lcs::SuspendedReason_NeedTerminateApplication, nn::lcs::GetSuspendedReason());

                TESTSTATE(TestState_ReceiverResumeContentsShare, TestState_ReceiverInitializeWaitComplete, nn::util::nullopt);
                    // コンテンツ配信を再開します。
                    NNT_ASSERT_UNLESS_RESULT(m_ExpectedResult, nn::lcs::ResumeContentsShare());

                TESTSTATE(TestState_ReceiverInitializeWaitComplete, TestState_ReceiverSkipWaitComplete, nn::util::nullopt);
                    m_BeginTime = nn::os::GetSystemTick();

                TESTSTATE_BRANCH(TestState_ReceiverSkipWaitComplete, nn::util::nullopt,
                    ((m_State = nn::lcs::GetState()) != nn::lcs::State_Completed) && ((nn::os::GetSystemTick() - m_BeginTime).ToTimeSpan() < ShareTimeout),
                    TestState_ReceiverCheckStateBeforeCompleted,
                    TestState_ReceiverSyncCompleted);

                TESTSTATE_WITH_POSTSTATE(TestState_ReceiverCheckStateBeforeCompleted, TestState_ReceiverWaitComplete, nn::util::nullopt, nn::lcs::State_Transferring);
                // Completed に遷移する前は Transferring であることを確認します。

                TESTSTATE(TestState_ReceiverWaitComplete, TestState_ReceiverSkipWaitComplete, nn::util::nullopt);
                    // コンテンツの受信完了まで待機します。Transferring 以外の状態にはなりません。
                    ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&m_StateChangeEvent, ShareTimeout - (nn::os::GetSystemTick() - m_BeginTime).ToTimeSpan()));
                    nn::os::ClearSystemEvent(&m_StateChangeEvent);

                TESTSTATE(TestState_ReceiverSyncCompleted, TestState_ReceiverCheckDownloadedContents, nn::util::nullopt);
                    // コンテンツ配信完了の同期します。
                    nn::os::ClearSystemEvent(&m_StateChangeEvent);
                    NNT_LDN_SYNC("PatchShareFailure.RunTest.Completed");

                TESTSTATE(TestState_ReceiverCheckDownloadedContents, TestState_Leave, nn::util::nullopt);
                {
                    // 要求した通り、コンテンツを受信できていることを確認します。
                    nn::lcs::ContentsInfo contents[nn::lcs::DownloadableContentsCountMax];
                    int contentCount;
                    NNT_ASSERT_UNLESS_RESULT(m_ExpectedResult, nn::lcs::GetDownloadedContents(
                        contents, &contentCount, nn::lcs::DownloadableContentsCountMax));
                    ASSERT_EQ(1, contentCount);

                    // 受信したパッチの情報が正しいことを確認します。
                    const auto& content = contents[0];
                    ASSERT_FALSE(content.contentsFlag.Test<nn::lcs::ContentsType::Application>());
                    ASSERT_FALSE(content.contentsFlag.Test<nn::lcs::ContentsType::System>());
                    ASSERT_TRUE(content.contentsFlag.Test<nn::lcs::ContentsType::Patch>());
                    ASSERT_EQ(0U, content.attributeFlag);
                    ASSERT_STREQ(m_Cond.application.latestDisplayVersion, content.displayVersion);
                    ASSERT_EQ(m_Cond.application.id, content.applicationId);
                }

                //
                // セッション破棄のフローです。
                //
                TESTSTATE_WITH_POSTSTATE(TestState_CheckCompletedState, TestState_LeaveSession, false, nn::lcs::State_Completed);
                    // セッションの破棄前に状態チェックを行います。

                TESTSTATE_WITH_POSTSTATE(TestState_LeaveSession, TestState_SyncDestroyed, true, nn::lcs::State_Initialized);
                    // セッションを破棄します。
                    NNT_ASSERT_UNLESS_RESULT(m_ExpectedResult, nn::lcs::LeaveSession());

                TESTSTATE(TestState_SyncDestroyed, TestState_End, nn::util::nullopt);
                    NNT_LDN_SYNC("PatchShareFailure.RunTest.Destroyed");

                    break;
                default:
                    // ここに来たときはライブラリではなくテストの不具合なのでアボートしておきます。
                    NN_ABORT("Unexpected TestState : %d", m_TestState);
                }

                //NN_LOG("State : %d, Changed : %5s ", nn::lcs::GetTestState(), nn::os::TryWaitSystemEvent(&m_StateChangeEvent) ? "true" : "false");

                // テストステート正常終了条件を上書きします。
                isExpectedStateTransition = m_IsExpectedStateTransition.value_or(isExpectedStateTransition);
                expectedState = m_ExpectedState.value_or(expectedState);
                auto expectedContentsShareFailureReason = m_ExpectedContentsShareFailureReason;

                // 特定の LCS ステート遷移状態が期待されているなら、LCS ステート遷移の有無が期待されているものかを確認します。
                if (isExpectedStateTransition)
                {
                    ASSERT_EQ(isExpectedStateTransition.value(), nn::os::TryWaitSystemEvent(&m_StateChangeEvent));

                    if (isExpectedStateTransition.value())
                    {
                        nn::os::ClearSystemEvent(&m_StateChangeEvent);
                    }
                }

                // LCS ステート遷移がないことが期待されておらず特定の LCS ステートが期待されているなら、現在の LCS ステートが期待されているものかを確認します。
                if (isExpectedStateTransition.value_or(true) && expectedState)
                {
                    if (nn::lcs::GetState() == nn::lcs::State_ContentsShareFailed)
                    {
                        NN_LOG("ContentShareFailedReason : %d\n", nn::lcs::GetContentsShareFailureReason());
                    }

                    ASSERT_PRED1(expectedState.value(), nn::lcs::GetState());

                    // 特定の配信失敗が期待されているなら、配信失敗理由が期待されているものかを確認します。
                    if (expectedState.value()(nn::lcs::State_ContentsShareFailed))
                    {
                        if (expectedContentsShareFailureReason)
                        {
                            ASSERT_PRED1(expectedContentsShareFailureReason.value(), nn::lcs::GetContentsShareFailureReason());
                        }
                    }
                }
            } while (sizeof...(interceptStates) != 0 && !IsStateEquals(interceptStates...) && m_TestState != TestState_End);
        } // NOLINT(impl/function_size)

        // 現在のテストステートを取得します。
        TestState GetTestState() const NN_NOEXCEPT
        {
            return m_TestState;
        }

        // 現在のテストステートを設定します。
        void SetTestState(TestState state) NN_NOEXCEPT
        {
            m_TestState = state;
        }

        void GetSession(nn::lcs::SessionInfo* pOutSession) NN_NOEXCEPT
        {
            *pOutSession = m_Sessions[m_SessionIndex];
        }

        // 次のテストステートで特定の Result 値が得られることを期待します。
        template <typename ResultType>
        void ExpectSpecificResult(ResultType expectedResult) NN_NOEXCEPT
        {
            m_ExpectedResult = std::make_tuple(static_cast<nn::Result>(expectedResult), ResultType::Includes);
        }

        // 次のテストステート実行後の LCS ステート遷移の有無の期待値を指定します。
        void ExpectStateTransition(const nn::util::optional<bool>& expectation) NN_NOEXCEPT
        {
            m_IsExpectedStateTransition = expectation;
        }

        // 次のテストステート実行後に特定の LCS ステートへであることを期待します。
        template <nn::lcs::State... States>
        void ExpectSpecificState() NN_NOEXCEPT
        {
            m_ExpectedState = nn::util::optional<bool(*)(const nn::lcs::State&)>(SetComparator<nn::lcs::State, States...>::Compare);
        }

        void ExpectSpecificState() NN_NOEXCEPT
        {
            m_ExpectedState = nn::util::nullopt;
        }

        // 次のテストステート実行後に特定の失敗理由であることを期待します。
        template <nn::lcs::ContentsShareFailureReason... Reasons>
        void ExpectSpecificContentsShareFailureReason() NN_NOEXCEPT
        {
            m_ExpectedContentsShareFailureReason = SetComparator<nn::lcs::ContentsShareFailureReason, Reasons...>::Compare;
        }
    };

    class ContentShareFailureTestHelper
    {
    public:
        nnt::lcs::NsInitializer     nsInitializer;
        nnt::ldn::SystemInitializer ldnInitializer;
        nnt::lcs::Initializer       lcsInitializer;

        nnt::lcs::Precondition cond;

        nn::os::SystemEventType  stateChangeEvent;
        nn::lcs::SessionInfo     sessions[nn::lcs::ScanResultCountMax];
        int                      sessionIndex;
        nn::lcs::SessionInfo     session;
        nn::lcs::State           state;

        explicit ContentShareFailureTestHelper(const nnt::lcs::Precondition& cond) NN_NOEXCEPT :
                lcsInitializer(g_Buffer, sizeof(g_Buffer), nnt::lcs::CreateDefaultHostConfig()),
                cond(cond)
        {
            // イベントを取得します。
            nn::lcs::AttachStateChangeEvent(&stateChangeEvent);
        }

        bool IsHost() const NN_NOEXCEPT
        {
            return g_TestConfig.nodeIndex == 0;
        }

        bool IsMaster() const NN_NOEXCEPT
        {
            return g_TestConfig.nodeIndex == cond.masterIndex;
        }

        bool IsSender() const NN_NOEXCEPT
        {
            return cond.nodes[g_TestConfig.nodeIndex].application.version == cond.application.latestVersion;
        }

        void OpenSession() NN_NOEXCEPT
        {
            if (g_TestConfig.nodeIndex == cond.masterIndex)
            {
                ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

                // セッションをオープンします。
                auto settings = nnt::lcs::CreateSessionSettings(nn::lcs::NodeCountMax, cond.application.id);
                NNT_ASSERT_RESULT_SUCCESS(nn::lcs::OpenSession(settings));

                ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
                nn::os::ClearSystemEvent(&stateChangeEvent);
                ASSERT_EQ(nn::lcs::State_Opened, nn::lcs::GetState());
            }
            else
            {
            }
        }

        void SyncSessionOpened() NN_NOEXCEPT
        {
            // セッションオープン完了の同期をします。
            NNT_LDN_SYNC("PatchShareFailure.RunTest.SessionOpened");
        }

        void WaitReadyToJoinSession() NN_NOEXCEPT
        {
            // 自分が接続を始める番になるまで同期を繰り返して待ちます。
            for (int syncCount = 0; syncCount < g_TestConfig.nodeIndex; ++syncCount)
            {
                NNT_LDN_SYNC("PatchShareFailure.RunTest.ReadyToJoin");
            }
        }

        void ScanSession() NN_NOEXCEPT
        {
            if (IsHost())
            {
            }
            else
            {
                // セッションをスキャンします。
                int count;
                NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::TimedScan(
                    sessions, &count, &sessionIndex, nn::lcs::ScanResultCountMax,
                    nnt::lcs::DefaultHostUserName, ScanTimeout));

                session = sessions[sessionIndex];
                ASSERT_EQ(session.nodeCount, g_TestConfig.nodeIndex);
            }
        }

        void JoinSession() NN_NOEXCEPT
        {
            if (IsHost())
            {
            }
            else
            {
                // セッションに参加します。
                NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::TimedJoin(session, JoinTimeout));

                ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
                nn::os::ClearSystemEvent(&stateChangeEvent);
                ASSERT_EQ(nn::lcs::State_Joined, nn::lcs::GetState());
            }
        }

        void WaitJoinSessionOtherClient() NN_NOEXCEPT
        {
            // 他の端末が接続を完了するまで待ちます。
            for (int syncCount = g_TestConfig.nodeIndex; syncCount < g_TestConfig.nodeCount; ++syncCount)
            {
                NNT_LDN_SYNC("PatchShareFailure.RunTest.ReadyToJoin");
            }
        }

        void SyncJoined() NN_NOEXCEPT
        {
            // 全クライアントがセッションに参加し終わるまで待ちます。
            NNT_LDN_SYNC("PatchShareFailure.RunTest.Joined");

            if (IsHost())
            {
                ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
            }
            else
            {
                // 他のクライアントの接続がすぐに反映されない可能性があるため、一定時間待機します。
                nn::os::SleepThread(StateChangeTimeout);
            }

            nn::os::ClearSystemEvent(&stateChangeEvent);
        }

        void SyncReadyToStart() NN_NOEXCEPT
        {
            // 配信準備完了の同期をします。
            NNT_LDN_SYNC("PatchShareFailure.RunTest.ReadyToStart");

            if (IsHost())
            {
                ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
            }
            else
            {
            }
        }

        void StartContentShare() NN_NOEXCEPT
        {
            if (IsHost())
            {
                // コンテンツの配信を開始します。
                NNT_ASSERT_RESULT_SUCCESS(nn::lcs::StartContentsShare());

                ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
                nn::os::ClearSystemEvent(&stateChangeEvent);
            }
            else
            {
            }
        }

        void SyncStarted() NN_NOEXCEPT
        {
            // 配信開始の同期をします。
            NNT_LDN_SYNC("PatchShareFailure.RunTest.Started");
        }

        void WaitStart() NN_NOEXCEPT
        {
            if (IsSender())
            {
            }
            else
            {
                // 状態遷移を待ちます。
                ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&stateChangeEvent, StateChangeTimeout));
                nn::os::ClearSystemEvent(&stateChangeEvent);
            }
        }

        void WaitSuspendedForNeedTerminateApplication() NN_NOEXCEPT
        {
            if (IsSender())
            {
            }
            else
            {
                auto beginTime = nn::os::GetSystemTick();

                while (((state = nn::lcs::GetState()) != nn::lcs::State_Suspended) && ((nn::os::GetSystemTick() - beginTime).ToTimeSpan() < ShareTimeout))
                {
                    // Suspended に遷移する前は Transferring であることを確認します。
                    ASSERT_EQ(nn::lcs::State_Transferring, nn::lcs::GetState());

                    // アプリケーション終了のための Suspended 状態になるまで待機します。
                    ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&stateChangeEvent, ShareTimeout - (nn::os::GetSystemTick() - beginTime).ToTimeSpan()));
                    nn::os::ClearSystemEvent(&stateChangeEvent);
                }

                // アプリケーション終了のための Suspended 状態であることを確認します。
                ASSERT_EQ(nn::lcs::State_Suspended, nn::lcs::GetState());
                ASSERT_EQ(nn::lcs::SuspendedReason_NeedTerminateApplication, nn::lcs::GetSuspendedReason());
                nn::os::ClearSystemEvent(&stateChangeEvent);
            }
        }

        void ResumeContentShare() NN_NOEXCEPT
        {
            if (IsSender())
            {
            }
            else
            {
                // コンテンツ配信を再開します。
                NNT_ASSERT_RESULT_SUCCESS(nn::lcs::ResumeContentsShare());
            }
        }

        void WaitCompleted() NN_NOEXCEPT
        {
            if (IsSender())
            {
                while ((state = nn::lcs::GetState()) != nn::lcs::State_Completed)
                {
                    // Transferring 状態であることを確認します。
                    ASSERT_EQ(nn::lcs::State_Transferring, nn::lcs::GetState());

                    // コンテンツ配信の完了まで待機します。
                    ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&stateChangeEvent, ShareTimeout));
                    nn::os::ClearSystemEvent(&stateChangeEvent);
                }
            }
            else
            {
                auto beginTime = nn::os::GetSystemTick();

                while (((state = nn::lcs::GetState()) != nn::lcs::State_Completed) && ((nn::os::GetSystemTick() - beginTime).ToTimeSpan() < ShareTimeout))
                {
                    // Completed に遷移する前は Transferring であることを確認します。
                    ASSERT_EQ(nn::lcs::State_Transferring, nn::lcs::GetState());

                    // コンテンツの受信完了まで待機します。Transferring 以外の状態にはなりません。
                    ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&stateChangeEvent, ShareTimeout - (nn::os::GetSystemTick() - beginTime).ToTimeSpan()));
                    nn::os::ClearSystemEvent(&stateChangeEvent);
                }

                nn::os::ClearSystemEvent(&stateChangeEvent);
            }
        }

        void SyncCompleted() NN_NOEXCEPT
        {
            // 全台の配信完了を待ちます。
            NNT_LDN_SYNC("PatchShareFailure.RunTest.Completed");
        }

        void VerifyDownloadedContents() NN_NOEXCEPT
        {
            if (IsSender())
            {
                // この端末は送信者なので何も受信していません。
                nn::lcs::ContentsInfo contents[nn::lcs::DownloadableContentsCountMax];
                int contentCount;
                NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetDownloadedContents(
                    contents, &contentCount, nn::lcs::DownloadableContentsCountMax));
                ASSERT_EQ(0, contentCount);
            }
            else
            {
                // 要求した通り、コンテンツを受信できていることを確認します。
                nn::lcs::ContentsInfo contents[nn::lcs::DownloadableContentsCountMax];
                int contentCount;
                NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetDownloadedContents(
                    contents, &contentCount, nn::lcs::DownloadableContentsCountMax));
                ASSERT_EQ(1, contentCount);

                // 受信したパッチの情報が正しいことを確認します。
                const auto& content = contents[0];
                ASSERT_FALSE(content.contentsFlag.Test<nn::lcs::ContentsType::Application>());
                ASSERT_FALSE(content.contentsFlag.Test<nn::lcs::ContentsType::System>());
                ASSERT_TRUE(content.contentsFlag.Test<nn::lcs::ContentsType::Patch>());
                ASSERT_EQ(0U, content.attributeFlag);
                ASSERT_STREQ(cond.application.latestDisplayVersion, content.displayVersion);
                ASSERT_EQ(cond.application.id, content.applicationId);
            }
        }

        void AssertCompleted() NN_NOEXCEPT
        {
            // セッションの破棄前に状態チェックを行います。
            ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
            ASSERT_EQ(nn::lcs::State_Completed, nn::lcs::GetState());
        }

        void LeaveSession() NN_NOEXCEPT
        {
            // セッションを破棄します。
            NNT_ASSERT_RESULT_SUCCESS(nn::lcs::LeaveSession());
            ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
            nn::os::ClearSystemEvent(&stateChangeEvent);
            ASSERT_EQ(nn::lcs::State_Initialized, nn::lcs::GetState());
        }

        void SyncDestroyed() NN_NOEXCEPT
        {
            NNT_LDN_SYNC("PatchShareFailure.RunTest.Destroyed");
        }
    };
} // namespace <unnamed>

//
// スキャンから参加までの間にホストがセッションを破棄した時の挙動を確認します。
// クライアント : ResultSessionNotFound
//
TEST(JoinDestroyedSession, JoinDestroyedSession)
{
    // テストケース開始前の同期です。
    NNT_LDN_REQUIRES_NODE_COUNT(2);
    NNT_LDN_SYNC("JoinDestroyedSession.JoinDestroyedSession.Start");

    const auto cond = (GetPrecondition("JoinDestroyedSession", "JoinDestroyedSession", nnt::lcs::Application::PatchShareFailure01));

    {
        ContentShareFailureTestHelper helper(cond);

        helper.OpenSession();
        helper.SyncSessionOpened();

        if (helper.IsHost())
        {
            // 同期します
            NNT_LDN_SYNC("JoinDestroyedSession.JoinDestroyedSession.Scan");

            // セッションを破棄します
            NNT_ASSERT_RESULT_SUCCESS(nn::lcs::LeaveSession());

            // 同期します
            NNT_LDN_SYNC("JoinDestroyedSession.JoinDestroyedSession.Destroy");
        }
        else
        {
            int count;
            NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::TimedScan(
                helper.sessions, &count, &helper.sessionIndex, nn::lcs::ScanResultCountMax,
                nnt::lcs::DefaultHostUserName, ScanTimeout));

            helper.session = helper.sessions[helper.sessionIndex];

            // 同期します
            NNT_LDN_SYNC("JoinDestroyedSession.JoinDestroyedSession.Scan");

            // 同期します
            NNT_LDN_SYNC("JoinDestroyedSession.JoinDestroyedSession.Destroy");

            // 破棄されたセッションに参加しようとして失敗します。
            NNT_ASSERT_RESULT_FAILURE(nn::lcs::ResultSessionNotFound, nn::lcs::JoinSession(helper.session));
        }
    }

    NNT_LDN_SYNC("JoinDestroyedSession.JoinDestroyedSession.Stop");
}

//
// フライトモードのままコンテンツ配信を開始しようとするとコンテンツ配信に失敗することを確認します。
// 全端末 : ResultWifiOff
//
TEST(FlightMode, BeforeStartSession)
{
    // テストケース開始前の同期です。
    NNT_LDN_REQUIRES_NODE_COUNT(2);
    NNT_LDN_SYNC("FlightMode.BeforeStartSession.Start");

    const auto cond = (GetPrecondition("FlightMode", "BeforeStartSession", nnt::lcs::Application::PatchShareFailure01));

    // Initalize 前にフライトモードにして ResultWifiOff が発生することを確認します。
    {
        nnt::lcs::NsInitializer nsInitializer;
        nnt::ldn::SystemInitializer ldnInitializer;

        nnt::ldn::ScopedWirelessCommunicationSwitchSetter wifiSwitch(false);

        NNT_ASSERT_RESULT_FAILURE(nn::lcs::ResultWifiOff, nn::lcs::Initialize(g_Buffer, sizeof(g_Buffer), nnt::lcs::CreateDefaultHostConfig()));
    }

    {
        ContentsShareStateMachine stateMachine(cond);

        if (g_TestConfig.nodeIndex == 0)
        {
            // セッションを開始するときに ResultWifiOff が発生することを確認します。
            stateMachine.Run(TestState_HostOpenSession);

            nnt::ldn::ScopedWirelessCommunicationSwitchSetter wifiSwitch(false);

            stateMachine.ExpectSpecificResult(nn::lcs::ResultWifiOff());
            stateMachine.ExpectStateTransition(nn::util::nullopt);
            stateMachine.ExpectSpecificState<nn::lcs::State_Initialized>();
            stateMachine.Run();

            // 他の端末のテストを進めるために同期だけは行います。
            NNT_LDN_SYNC("PatchShareFailure.RunTest.SessionOpened");
        }
        else
        {
            // セッションをスキャンするときに ResultWifiOff が発生することを確認します。
            stateMachine.Run(TestState_ClientScanSession);

            nnt::ldn::ScopedWirelessCommunicationSwitchSetter wifiSwitch(false);

            stateMachine.ExpectSpecificResult(nn::lcs::ResultWifiOff());
            stateMachine.ExpectStateTransition(nn::util::nullopt);
            stateMachine.ExpectSpecificState<nn::lcs::State_Initialized>();
            stateMachine.Run();
        }
    }

    {
        ContentsShareStateMachine stateMachine(cond);

        if (g_TestConfig.nodeIndex == 0)
        {
            stateMachine.Run(TestState_HostSyncJoined);
            stateMachine.ExpectStateTransition(false);
            stateMachine.Run();
        }
        else
        {
            // セッションに参加するときに ResultWifiOff が発生することを確認します。

            nn::lcs::SessionInfo session = {};
            while (session.nodeCount == 0)
            {
                stateMachine.Run(TestState_ClientBranchJoin);
                stateMachine.Run();
                stateMachine.GetSession(&session);
            }

            nnt::ldn::ScopedWirelessCommunicationSwitchSetter wifiSwitch(false);

            NNT_ASSERT_RESULT_FAILURE(nn::lcs::ResultWifiOff, nnt::lcs::TimedJoin(session, JoinTimeout));

            stateMachine.SetTestState(TestState_ClientSyncJoined);
            stateMachine.Run();
        }
    }
}

//
// セッションを構築してからホストをフライトモードにすると配信に失敗することを確認します。
// ホスト       : ContentsShareFailureReason_FlightMode
// クライアント : ContentsShareFailureReason_NodeLeaved
//
TEST(FlightMode, HostBeforeStartShare)
{
    // テストケース開始前の同期です。
    NNT_LDN_REQUIRES_NODE_COUNT(3);
    NNT_LDN_SYNC("FlightMode.HostBeforeStartShare.Start");

    const auto cond = (GetPrecondition("FlightMode", "HostBeforeStartShare", nnt::lcs::Application::PatchShareFailure01));

    {
        ContentsShareStateMachine stateMachine(cond);

        NN_UTIL_SCOPE_EXIT
        {
            ENABLE_WIRELESS_COMMUNICATION();
        };

        // ホストではフライトモードを理由に配信開始が失敗することを確認します。
        if (g_TestConfig.nodeIndex == 0)
        {
            stateMachine.Run(TestState_HostStartContentShare);

            DISABLE_WIRELESS_COMMUNICATION();
            nn::os::SleepThread(StateChangeTimeout);

            stateMachine.ExpectSpecificResult(nn::lcs::ResultInvalidState());
            stateMachine.ExpectStateTransition(true);
            stateMachine.ExpectSpecificState<nn::lcs::State_ContentsShareFailed>();
            stateMachine.ExpectSpecificContentsShareFailureReason<nn::lcs::ContentsShareFailureReason_FlightMode>();
            stateMachine.Run();

            stateMachine.Run();
        }
        // クライアントではステート遷移待ち中にセッション破棄を理由としたコンテンツ配信失敗が発生することを確認します。
        else
        {
            stateMachine.Run(TestState_ClientSyncStarted);

            stateMachine.ExpectSpecificState<nn::lcs::State_ContentsShareFailed>();
            stateMachine.ExpectSpecificContentsShareFailureReason<nn::lcs::ContentsShareFailureReason_NodeLeaved>();

            stateMachine.Run();
        }
    }
}

//
// コンテンツ配信を開始してからしてからホストをフライトモードにし、配信に失敗することを確認します。
// ホスト       : ContentsShareFailureReason_FlightMode
// クライアント : ContentsShareFailureReason_NodeLeaved
//
TEST(FlightMode, HostAfterStartShare)
{
    // テストケース開始前の同期です。
    NNT_LDN_REQUIRES_NODE_COUNT(3);
    NNT_LDN_SYNC("FlightMode.HostAfterStartShare.Start");

    const auto cond = (GetPrecondition("FlightMode", "HostAfterStartShare", nnt::lcs::Application::PatchShareFailure01));

    {
        ContentShareFailureTestHelper helper(cond);

        NN_UTIL_SCOPE_EXIT
        {
            ENABLE_WIRELESS_COMMUNICATION();
        };

        helper.OpenSession();
        helper.SyncSessionOpened();
        helper.WaitReadyToJoinSession();
        helper.ScanSession();
        helper.JoinSession();
        helper.WaitJoinSessionOtherClient();
        helper.SyncJoined();
        helper.SyncReadyToStart();
        helper.StartContentShare();
        helper.SyncStarted();
        NN_LOG("WaitStart\n");
        helper.WaitStart();

        if (helper.IsSender())
        {
            NN_LOG("Sender\n");
            // クライアントが受信を開始してからフライトモードに入ります。
            ASSERT_EQ(nn::lcs::State_Transferring, nn::lcs::GetState());

            NNT_LDN_SYNC("FlightMode.HostAfterStartShare.WifiOff");

            NN_LOG("Sleep!!!!\n");
            DISABLE_WIRELESS_COMMUNICATION();

            // ContentsShareFailureReason_FlightMode を理由に配信に失敗します。
            ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&helper.stateChangeEvent, ShareTimeout));
            ASSERT_EQ(nn::lcs::State_ContentsShareFailed, nn::lcs::GetState());
            ASSERT_EQ(nn::lcs::ContentsShareFailureReason_FlightMode, nn::lcs::GetContentsShareFailureReason());
        }
        else
        {
            NN_LOG("receiver\n");
            if (g_TestConfig.nodeIndex == 1)
            {
                // 1 番目の受信者は受信を開始します。
                helper.WaitSuspendedForNeedTerminateApplication();
                helper.ResumeContentShare();
            }
            else
            {
                // 他の受信者は待ち状態です。
                ASSERT_EQ(nn::lcs::State_Transferring, nn::lcs::GetState());
            }

            // 1 秒ほど受信します。
            nn::os::SleepThread(StateChangeTimeout);

            NNT_LDN_SYNC("FlightMode.HostAfterStartShare.WifiOff");

            // 配信に失敗するまで待ちます。
            nn::lcs::State state;
            while ((state = nn::lcs::GetState()) != nn::lcs::State_ContentsShareFailed)
            {
                // 配信に失敗するまでは Transferring 状態であることを確認します。
                ASSERT_EQ(nn::lcs::State_Transferring, state);

                ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&helper.stateChangeEvent, ShareTimeout));
            }
            // 失敗の理由が ContentsShareFailureReason_NodeLeaved であることを確認します。
            ASSERT_EQ(nn::lcs::ContentsShareFailureReason_NodeLeaved, nn::lcs::GetContentsShareFailureReason());
        }
    }
}

//
// セッションを構築してから特定のクライアントをフライトモードにすると該当クライアントのみ受信に失敗することを確認します。
// ホスト             : 成功
// クライアント 1     : ContentsShareFailureReason_FlightMode
// クライアントその他 : 成功
//
TEST(FlightMode, ClientBeforeStartShare)
{
    // テストケース開始前の同期です。
    NNT_LDN_REQUIRES_NODE_COUNT(3);
    NNT_LDN_SYNC("FlightMode.ClientBeforeStartShare.Start");

    const auto cond = (GetPrecondition("FlightMode", "ClientBeforeStartShare", nnt::lcs::Application::PatchShareFailure02));

    {
        ContentsShareStateMachine stateMachine(cond);

        NN_UTIL_SCOPE_EXIT
        {
            ENABLE_WIRELESS_COMMUNICATION();
        };

        // ホストではセッションを維持してパッチ配信が成功することを確認します。
        if (g_TestConfig.nodeIndex == 0)
        {
            stateMachine.Run(TestState_HostSyncReadyToStart);

            // 1 番目のクライアントの離脱によりシグナルされるイベントは無視します。
            stateMachine.ExpectStateTransition(nn::util::nullopt);

            stateMachine.Run(TestState_End);
        }
        // 1 番目のクライアントではフライトモードを理由にセッションから離脱することを確認します。
        else if (g_TestConfig.nodeIndex == 1)
        {
            stateMachine.Run(TestState_ClientSyncReadyToStart);

            DISABLE_WIRELESS_COMMUNICATION();

            stateMachine.ExpectStateTransition(true);
            stateMachine.ExpectSpecificState<nn::lcs::State_ContentsShareFailed>();
            stateMachine.ExpectSpecificContentsShareFailureReason<nn::lcs::ContentsShareFailureReason_FlightMode>();

            nn::os::SleepThread(StateChangeTimeout);

            stateMachine.Run();

            // 他の端末のテストを進めるために同期だけは行います。
            NNT_LDN_SYNC("PatchShareFailure.RunTest.Started");
            g_pSync->SetTimeout(ShareTimeout);
            NNT_LDN_SYNC("PatchShareFailure.RunTest.Completed");
            g_pSync->SetTimeout(nn::TimeSpan::FromSeconds(60));
            NNT_LDN_SYNC("PatchShareFailure.RunTest.Destroyed");
        }
        // 他のクライアントではセッションを維持してパッチ配信が成功することを確認します。
        else
        {
            stateMachine.Run(TestState_ClientSyncReadyToStart);

            // 1 番目のクライアントが抜けるためイベントがシグナルされることを確認します。
            stateMachine.ExpectStateTransition(true);
            stateMachine.ExpectSpecificState<nn::lcs::State_Joined>();

            nn::os::SleepThread(StateChangeTimeout);

            stateMachine.Run();

            stateMachine.Run(TestState_End);
        }
    }
}

//
// 最新のパッチを持つスレーブを順番にフライトモードにしていきます。
// 残ったスレーブが配信を継続しようとすること、最後の1台が抜けるとエラーになることを確認します。
// ホスト       : ContentsShareFailureReason_NodeLeaved
// クライアント : ContentsShareFailureReason_FlightMode
//
TEST(FlightMode, SenderAfterStartShare)
{
    // テストケース開始前の同期です。
    NNT_LDN_REQUIRES_NODE_COUNT(3);
    NNT_LDN_SYNC("FlightMode.SenderAfterStartShare.Start");

    const auto cond = (GetPrecondition("FlightMode", "SenderAfterStartShare", nnt::lcs::Application::PatchShareFailure03));

    {
        ContentsShareStateMachine stateMachine(cond);

        NN_UTIL_SCOPE_EXIT
        {
            ENABLE_WIRELESS_COMMUNICATION();
        };

        // ホストが受信者になります。
        if (g_TestConfig.nodeIndex == 0)
        {
            // 送信者の数 - 1 だけ Suspended に戻ります。
            for (int suspendedCount = 0; suspendedCount < g_TestConfig.nodeCount - 2; )
            {
                stateMachine.Run(TestState_ReceiverCheckStateBeforeCompleted);

                stateMachine.ExpectSpecificState<nn::lcs::State_Transferring, nn::lcs::State_Suspended>();
                stateMachine.Run();

                if (nn::lcs::GetState() == nn::lcs::State_Suspended)
                {
                    stateMachine.SetTestState(TestState_Receiver);

                    ++suspendedCount;
                }
            }

            // 最後の送信者が抜けると、コンテンツ配信に失敗します。
            do
            {
                stateMachine.Run(TestState_ReceiverCheckStateBeforeCompleted);

                stateMachine.ExpectSpecificState<nn::lcs::State_Transferring, nn::lcs::State_ContentsShareFailed>();
                stateMachine.Run();
            } while (nn::lcs::GetState() != nn::lcs::State_ContentsShareFailed);
            ASSERT_EQ(nn::lcs::ContentsShareFailureReason_NodeLeaved, nn::lcs::GetContentsShareFailureReason());
        }
        // クライアントが送信者になります。
        else
        {
            for (;;)
            {
                stateMachine.Run(TestState_SenderWaitComplete);

                nn::os::SleepThread(StateChangeTimeout);

                // 送信者になったらフライトモードに入ります。
                nn::lcs::Progress progress;
                nn::lcs::GetProgress(&progress);
                if (progress.transferRole == nn::lcs::TransferRole_Sender)
                {
                    DISABLE_WIRELESS_COMMUNICATION();

                    stateMachine.Run();
                    stateMachine.Run(TestState_SenderCheckTransferringStateBeforeCompleted);

                    stateMachine.ExpectSpecificState<nn::lcs::State_ContentsShareFailed>();
                    stateMachine.ExpectSpecificContentsShareFailureReason<nn::lcs::ContentsShareFailureReason_FlightMode>();

                    stateMachine.Run();

                    break;
                }

                stateMachine.Run();
            }
        }
    }
}

//
// 1 台だけが、配信可能な RequiredSystemVersion のパッチを持っており、その端末をスレーブにすると
// 残った ExFat ドライバを持たない送信者は ExFat ドライバを持つパッチ受信者に配信できないため
// 配信を継続しようとせずコンテンツ配信に失敗します。
// このとき、一度は受信ができそうだった端末は NodeLeaved で失敗します。
// ホスト             : 受信者がいなくなり配信完了
// クライアント 1     : ContentsShareFailureReason_NodeLeaved
// クライアント 2     : ContentsShareFailureReason_FlightMode
// クライアントその他 : 受信者がいなくなり配信完了
//
TEST(FlightMode, SenderAfterStartShareContainsIncompatibleRequiredSystemVersion)
{
    // テストケース開始前の同期です。
    NNT_LDN_REQUIRES_NODE_COUNT(4);
    NNT_LDN_SYNC("FlightMode.SenderAfterStartShareContainsIncompatibleRequiredSystemVersion.Start");

    const auto cond = (GetPrecondition("FlightMode", "SenderAfterStartShareContainsIncompatibleRequiredSystemVersion", nnt::lcs::Application::PatchShareFailure04));

    {
        ContentsShareStateMachine stateMachine(cond);

        NN_UTIL_SCOPE_EXIT
        {
            ENABLE_WIRELESS_COMMUNICATION();
        };

        // 2 番目の端末が受信者になります。
        if (g_TestConfig.nodeIndex == 1)
        {
            for (;;)
            {
                stateMachine.Run(TestState_ReceiverCheckStateBeforeCompleted);

                stateMachine.ExpectSpecificState<nn::lcs::State_Transferring, nn::lcs::State_ContentsShareFailed>();

                stateMachine.Run();

                if (nn::lcs::GetState() == nn::lcs::State_ContentsShareFailed)
                {
                    ASSERT_EQ(nn::lcs::ContentsShareFailureReason_NodeLeaved, nn::lcs::GetContentsShareFailureReason());
                    break;
                }
            }

            NNT_LDN_SYNC("PatchShareFailure.RunTest.Completed");
            NNT_LDN_SYNC("PatchShareFailure.RunTest.Destroyed");
        }
        // 3番目の端末が送信者になります。
        else if (g_TestConfig.nodeIndex == 2)
        {
            for (;;)
            {
                stateMachine.Run(TestState_SenderWaitComplete);

                nn::os::SleepThread(StateChangeTimeout);

                // 送信者になったらフライトモードに入ります。
                nn::lcs::Progress progress;
                nn::lcs::GetProgress(&progress);
                if (progress.transferRole == nn::lcs::TransferRole_Sender)
                {
                    DISABLE_WIRELESS_COMMUNICATION();

                    stateMachine.Run();
                    stateMachine.Run(TestState_SenderCheckTransferringStateBeforeCompleted);

                    stateMachine.ExpectSpecificState<nn::lcs::State_ContentsShareFailed>();
                    stateMachine.ExpectSpecificContentsShareFailureReason<nn::lcs::ContentsShareFailureReason_FlightMode>();

                    stateMachine.Run();

                    break;
                }
            }

            stateMachine.SetTestState(TestState_SenderSyncCompleted);
            stateMachine.Run(TestState_End);
        }
        // その他の端末は配信できないパッチを持っている送信者です。
        else
        {
            stateMachine.Run(TestState_End);
        }
    }
}

//
// 最新のパッチを持つスレーブを順番にフライトモードにしていきます。
// 全ての送信者が、配信可能な RequiredSystemVersion のパッチを持っているので
// 残ったスレーブは配信を継続しようすることを確認します。
// ホスト       : ContentsShareFailureReason_NodeLeaved
// クライアント : ContentsShareFailureReason_FlightMode
//
TEST(FlightMode, SenderAfterStartShareContainsCompatibleRequiredSystemVersion)
{
    // テストケース開始前の同期です。
    NNT_LDN_REQUIRES_NODE_COUNT(3);
    NNT_LDN_SYNC("FlightMode.SenderAfterStartShareContainsCompatibleRequiredSystemVersion.Start");

    const auto cond = (GetPrecondition("FlightMode", "SenderAfterStartShareContainsCompatibleRequiredSystemVersion", nnt::lcs::Application::PatchShareFailure05));

    {
        ContentsShareStateMachine stateMachine(cond);

        NN_UTIL_SCOPE_EXIT
        {
            ENABLE_WIRELESS_COMMUNICATION();
        };

        // ホストが受信者になります。
        if (g_TestConfig.nodeIndex == 0)
        {
            // 送信者の数 - 1 だけ Suspended に戻ります。
            for (int suspendedCount = 0; suspendedCount < g_TestConfig.nodeCount - 2; )
            {
                stateMachine.Run(TestState_ReceiverCheckStateBeforeCompleted);

                stateMachine.ExpectSpecificState<nn::lcs::State_Transferring, nn::lcs::State_Suspended>();
                stateMachine.Run();

                if (nn::lcs::GetState() == nn::lcs::State_Suspended)
                {
                    stateMachine.SetTestState(TestState_Receiver);

                    ++suspendedCount;
                }
            }

            // 最後の送信者が抜けると、コンテンツ配信に失敗します。
            do
            {
                stateMachine.Run(TestState_ReceiverCheckStateBeforeCompleted);

                stateMachine.ExpectSpecificState<nn::lcs::State_Transferring, nn::lcs::State_ContentsShareFailed>();
                stateMachine.Run();
            } while (nn::lcs::GetState() != nn::lcs::State_ContentsShareFailed);
            ASSERT_EQ(nn::lcs::ContentsShareFailureReason_NodeLeaved, nn::lcs::GetContentsShareFailureReason());
        }
        // クライアントが送信者になります。
        else
        {
            for (;;)
            {
                stateMachine.Run(TestState_SenderWaitComplete);

                nn::os::SleepThread(StateChangeTimeout);

                // 送信者になったらフライトモードに入ります。
                nn::lcs::Progress progress;
                nn::lcs::GetProgress(&progress);
                if (progress.transferRole == nn::lcs::TransferRole_Sender)
                {
                    DISABLE_WIRELESS_COMMUNICATION();

                    stateMachine.Run();
                    stateMachine.Run(TestState_SenderCheckTransferringStateBeforeCompleted);

                    stateMachine.ExpectSpecificState<nn::lcs::State_ContentsShareFailed>();
                    stateMachine.ExpectSpecificContentsShareFailureReason<nn::lcs::ContentsShareFailureReason_FlightMode>();

                    stateMachine.Run();

                    break;
                }

                stateMachine.Run();
            }
        }
    }
}

//
// アプリ終了要求による中断中に受信者が離脱した時の挙動を確認します。
//
TEST(Leave, SuspendedNeedTerminateApplication)
{
    // テストケース開始前の同期です。
    NNT_LDN_REQUIRES_NODE_COUNT(2);
    NNT_LDN_SYNC("Leave.SuspendedNeedTerminateApplication.Start");

    const auto cond = (GetPrecondition("Leave", "SuspendedNeedTerminateApplication", nnt::lcs::Application::PatchShareFailure01));

    {
        ContentShareFailureTestHelper helper(cond);

        helper.OpenSession();
        helper.SyncSessionOpened();
        helper.WaitReadyToJoinSession();
        helper.ScanSession();
        helper.JoinSession();
        helper.WaitJoinSessionOtherClient();
        helper.SyncJoined();
        helper.SyncReadyToStart();
        helper.StartContentShare();
        helper.SyncStarted();
        helper.WaitStart();
        helper.WaitSuspendedForNeedTerminateApplication();

        if (helper.IsSender())
        {
            // 全クライアント (パッチ受信者) が抜けると State_Completed になります。
            auto beginTick = nn::os::GetSystemTick();
            auto elapsed = nn::TimeSpan();
            nn::lcs::State state;
            while ((state = nn::lcs::GetState()) != nn::lcs::State_Completed)
            {
                ASSERT_EQ(nn::lcs::State_Transferring, state);

                auto timeout = ShareTimeout - elapsed;
                ASSERT_LT(nn::TimeSpan(), timeout);
                ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&helper.stateChangeEvent, timeout));
                nn::os::ClearSystemEvent(&helper.stateChangeEvent);

                elapsed = (nn::os::GetSystemTick() - beginTick).ToTimeSpan();
            }

            // 完了したのでセッションを破棄します。
            NNT_ASSERT_RESULT_SUCCESS(nn::lcs::LeaveSession());
        }
        else
        {
            // 中断中に離脱します。
            NNT_ASSERT_RESULT_SUCCESS(nn::lcs::LeaveSession());
            ASSERT_EQ(nn::lcs::State_Initialized, nn::lcs::GetState());

            // 離脱して State_Initialized になったので受信済みコンテンツの取得は失敗します。
            nn::lcs::ContentsInfo contents[nn::lcs::DownloadableContentsCountMax];
            int contentCount;
            NNT_ASSERT_RESULT_FAILURE(nn::lcs::ResultInvalidState, nn::lcs::GetDownloadedContents(
                contents, &contentCount, nn::lcs::DownloadableContentsCountMax));
        }

        NNT_LDN_SYNC("Leave.SuspendedNeedTerminateApplication.Leaved");

        helper.SyncDestroyed();
    }

    NNT_LDN_SYNC("Leave.SuspendedNeedTerminateApplication.Stop");
}

//
// ストレージ空き領域不足による中断中に受信者 (ホスト) が離脱した時の挙動を確認します。
//
TEST(Leave, SuspendedStorageSpaceNotEnoughOnRequiredNode)
{
    // テストケース開始前の同期です。
    NNT_LDN_REQUIRES_NODE_COUNT(2);
    NNT_LDN_SYNC("Leave.SuspendedStorageSpaceNotEnoughOnRequiredNode.Start");

    const auto cond = (GetPrecondition("Leave", "SuspendedStorageSpaceNotEnoughOnRequiredNode", nnt::lcs::Application::PatchShareFailure06));

    {
        ContentShareFailureTestHelper helper(cond);

        helper.OpenSession();
        helper.SyncSessionOpened();
        helper.WaitReadyToJoinSession();
        helper.ScanSession();
        helper.JoinSession();
        helper.WaitJoinSessionOtherClient();
        helper.SyncJoined();
        helper.SyncReadyToStart();
        helper.StartContentShare();
        helper.SyncStarted();
        helper.WaitStart();

        if (helper.IsSender())
        {
            // ホストの受信者が離脱すると State_PatchShareFailure になります。
            auto beginTick = nn::os::GetSystemTick();
            auto elapsed = nn::TimeSpan();
            while (nn::lcs::GetState() != nn::lcs::State_ContentsShareFailed)
            {
                auto timeout = ShareTimeout - elapsed;
                ASSERT_LT(nn::TimeSpan(), timeout);
                ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&helper.stateChangeEvent, timeout));
                nn::os::ClearSystemEvent(&helper.stateChangeEvent);

                elapsed = (nn::os::GetSystemTick() - beginTick).ToTimeSpan();
            }
            ASSERT_EQ(nn::lcs::ContentsShareFailureReason_NodeLeaved, nn::lcs::GetContentsShareFailureReason());

            // 解散したのでセッションを破棄します。
            NNT_ASSERT_RESULT_SUCCESS(nn::lcs::LeaveSession());
        }
        else
        {
            auto beginTick = nn::os::GetSystemTick();
            auto elapsed = nn::TimeSpan();

            // 中断が起きるまで待ちます。
            nn::lcs::State state;
            while ((state = nn::lcs::GetState()) != nn::lcs::State_Suspended)
            {
                // 中断するまでは State_Transferring であることを確認します。
                ASSERT_EQ(nn::lcs::State_Transferring, state);

                auto timeout = ShareTimeout - elapsed;
                ASSERT_LT(nn::TimeSpan(), timeout);
                ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&helper.stateChangeEvent, timeout));
                nn::os::ClearSystemEvent(&helper.stateChangeEvent);

                elapsed = (nn::os::GetSystemTick() - beginTick).ToTimeSpan();
            }

            // ストレージ空き領域不足による中断であることを確認します。
            ASSERT_EQ(nn::lcs::SuspendedReason_StorageSpaceNotEnoughOnRequredNode, nn::lcs::GetSuspendedReason());

            // 中断中に離脱します。
            NNT_ASSERT_RESULT_SUCCESS(nn::lcs::LeaveSession());
            ASSERT_EQ(nn::lcs::State_Initialized, nn::lcs::GetState());

            // 離脱して State_Initialized になったので受信済みコンテンツの取得は失敗します。
            nn::lcs::ContentsInfo contents[nn::lcs::DownloadableContentsCountMax];
            int contentCount;
            NNT_ASSERT_RESULT_FAILURE(nn::lcs::ResultInvalidState, nn::lcs::GetDownloadedContents(
                contents, &contentCount, nn::lcs::DownloadableContentsCountMax));
        }

        NNT_LDN_SYNC("Leave.SuspendedStorageSpaceNotEnoughOnRequiredNode.Leaved");

        helper.SyncDestroyed();
    }

    NNT_LDN_SYNC("Leave.SuspendedStorageSpaceNotEnoughOnRequiredNode.Stop");
}

//
// ストレージ空き領域不足による中断中に受信者 (クライアント) が離脱した時の挙動を確認します。
//
TEST(Leave, SuspendedStorageSpaceNotEnough)
{
    // テストケース開始前の同期です。
    NNT_LDN_REQUIRES_NODE_COUNT(2);
    NNT_LDN_SYNC("Leave.SuspendedStorageSpaceNotEnough.Start");

    const auto cond = (GetPrecondition("Leave", "SuspendedStorageSpaceNotEnough", nnt::lcs::Application::PatchShareFailure01));

    {
        ContentShareFailureTestHelper helper(cond);

        helper.OpenSession();
        helper.SyncSessionOpened();
        helper.WaitReadyToJoinSession();
        helper.ScanSession();
        helper.JoinSession();
        helper.WaitJoinSessionOtherClient();
        helper.SyncJoined();
        helper.SyncReadyToStart();
        helper.StartContentShare();
        helper.SyncStarted();
        helper.WaitStart();

        if (helper.IsSender())
        {
            // 全クライアント (パッチ受信者) が抜けると State_Completed になります。
            auto beginTick = nn::os::GetSystemTick();
            auto elapsed = nn::TimeSpan();
            while (nn::lcs::GetState() != nn::lcs::State_Completed)
            {
                auto timeout = ShareTimeout - elapsed;
                ASSERT_LT(nn::TimeSpan(), timeout);
                ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&helper.stateChangeEvent, timeout));
                nn::os::ClearSystemEvent(&helper.stateChangeEvent);

                elapsed = (nn::os::GetSystemTick() - beginTick).ToTimeSpan();
            }

            // 完了したのでセッションを破棄します。
            NNT_ASSERT_RESULT_SUCCESS(nn::lcs::LeaveSession());
        }
        else
        {
            auto beginTick = nn::os::GetSystemTick();
            auto elapsed = nn::TimeSpan();

            // 中断が起きるまで待ちます。
            nn::lcs::State state;
            while ((state = nn::lcs::GetState()) != nn::lcs::State_Suspended)
            {
                // 中断するまでは State_Transferring であることを確認します。
                ASSERT_EQ(nn::lcs::State_Transferring, state);

                auto timeout = ShareTimeout - elapsed;
                ASSERT_LT(nn::TimeSpan(), timeout);
                ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&helper.stateChangeEvent, timeout));
                nn::os::ClearSystemEvent(&helper.stateChangeEvent);

                elapsed = (nn::os::GetSystemTick() - beginTick).ToTimeSpan();
            }

            // ストレージ空き領域不足による中断であることを確認します。
            ASSERT_EQ(nn::lcs::SuspendedReason_StorageSpaceNotEnough, nn::lcs::GetSuspendedReason());

            // 中断中に離脱します。
            NNT_ASSERT_RESULT_SUCCESS(nn::lcs::LeaveSession());
            ASSERT_EQ(nn::lcs::State_Initialized, nn::lcs::GetState());

            // 離脱して State_Initialized になったので受信済みコンテンツの取得は失敗します。
            nn::lcs::ContentsInfo contents[nn::lcs::DownloadableContentsCountMax];
            int contentCount;
            NNT_ASSERT_RESULT_FAILURE(nn::lcs::ResultInvalidState, nn::lcs::GetDownloadedContents(
                contents, &contentCount, nn::lcs::DownloadableContentsCountMax));
        }

        NNT_LDN_SYNC("Leave.SuspendedStorageSpaceNotEnough.Leaved");

        helper.SyncDestroyed();
    }

    NNT_LDN_SYNC("Leave.SuspendedStorageSpaceNotEnough.Stop");
}

//
// ApplicationDeliveryProtocol の互換性がない場合の中断中に離脱した時の挙動を確認します。
//
TEST(Leave, SuspendedIncompatibleContentsInfo)
{
    // テストケース開始前の同期です。
    NNT_LDN_REQUIRES_NODE_COUNT(2);
    NNT_LDN_SYNC("Leave.SuspendedIncompatibleContentsInfo.Start");

    const auto cond = (GetPrecondition("Leave", "SuspendedIncompatibleContentsInfo", nnt::lcs::Application::PatchShareFailure04));

    {
        ContentShareFailureTestHelper helper(cond);

        helper.OpenSession();
        helper.SyncSessionOpened();
        helper.WaitReadyToJoinSession();
        helper.ScanSession();
        helper.JoinSession();
        helper.WaitJoinSessionOtherClient();
        helper.SyncJoined();
        helper.SyncReadyToStart();
        helper.StartContentShare();
        helper.SyncStarted();
        helper.WaitStart();

        if (helper.IsSender())
        {
            // 全クライアント (パッチ受信者) が抜けると State_Completed になります。
            auto beginTick = nn::os::GetSystemTick();
            auto elapsed = nn::TimeSpan();
            while (nn::lcs::GetState() != nn::lcs::State_Completed)
            {
                auto timeout = ShareTimeout - elapsed;
                ASSERT_LT(nn::TimeSpan(), timeout);
                ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&helper.stateChangeEvent, timeout));
                nn::os::ClearSystemEvent(&helper.stateChangeEvent);

                elapsed = (nn::os::GetSystemTick() - beginTick).ToTimeSpan();
            }

            // 完了したのでセッションを破棄します。
            NNT_ASSERT_RESULT_SUCCESS(nn::lcs::LeaveSession());
        }
        else
        {
            auto beginTick = nn::os::GetSystemTick();
            auto elapsed = nn::TimeSpan();

            // 中断が起きるまで待ちます。
            nn::lcs::State state;
            while ((state = nn::lcs::GetState()) != nn::lcs::State_Suspended)
            {
                // 中断するまでは State_Transferring であることを確認します。
                ASSERT_EQ(nn::lcs::State_Transferring, state);

                auto timeout = ShareTimeout - elapsed;
                ASSERT_LT(nn::TimeSpan(), timeout);
                ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&helper.stateChangeEvent, timeout));
                nn::os::ClearSystemEvent(&helper.stateChangeEvent);

                elapsed = (nn::os::GetSystemTick() - beginTick).ToTimeSpan();
            }

            // ストレージ空き領域不足による中断であることを確認します。
            ASSERT_EQ(nn::lcs::SuspendedReason_IncompatibleContentsInfo, nn::lcs::GetSuspendedReason());

            // 中断中に離脱します。
            NNT_ASSERT_RESULT_SUCCESS(nn::lcs::LeaveSession());
            ASSERT_EQ(nn::lcs::State_Initialized, nn::lcs::GetState());

            // 離脱して State_Initialized になったので受信済みコンテンツの取得は失敗します。
            nn::lcs::ContentsInfo contents[nn::lcs::DownloadableContentsCountMax];
            int contentCount;
            NNT_ASSERT_RESULT_FAILURE(nn::lcs::ResultInvalidState, nn::lcs::GetDownloadedContents(
                contents, &contentCount, nn::lcs::DownloadableContentsCountMax));
        }

        NNT_LDN_SYNC("Leave.SuspendedIncompatibleContentsInfo.Leaved");

        helper.SyncDestroyed();
    }

    NNT_LDN_SYNC("Leave.SuspendedIncompatibleContentsInfo.Stop");
}

#if false // PutSleep/Wakeup をしたあとの nn::ldn::InitializeSystem() が返ってきません。
//
// セッションを構築してからスリープをして、ResultSleep が発生することを確認します。
//
TEST(PatchShareFailure, Sleep)
{
    // テストケース開始前の同期です。
    NNT_LDN_REQUIRES_NODE_COUNT(2);
    NNT_LDN_SYNC("PatchShareFailure.Sleep.Start");

    const auto cond = (GetPrecondition("PatchShareFailure", "Sleep", nnt::lcs::Application::PatchShareFailure01));

    NN_LOG("Open\n");
    {
        ContentShareFailureTestHelper helper(cond);

        nnt::ldn::PutSleep();
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));
        nnt::ldn::WakeUp();

        if (helper.IsMaster())
        {
            auto settings = nnt::lcs::CreateSessionSettings(nn::lcs::NodeCountMax, cond.application.id);
            NNT_ASSERT_RESULT_FAILURE(nn::lcs::ResultSleep, nn::lcs::OpenSession(settings));
        }
        NNT_LDN_SYNC("PatchShareFailure.Sleep.Wait");
    }
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));

    NN_LOG("Scan\n");
    {
        ContentShareFailureTestHelper helper(cond);

        helper.OpenSession();
        helper.SyncSessionOpened();

        nnt::ldn::PutSleep();
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));
        nnt::ldn::WakeUp();

        if (!helper.IsMaster())
        {
            // セッションをスキャンします。
            int count;
            NNT_ASSERT_RESULT_FAILURE(nn::lcs::ResultSleep, nnt::lcs::TimedScan(
                helper.sessions, &count, &helper.sessionIndex, nn::lcs::ScanResultCountMax,
                nnt::lcs::DefaultHostUserName, ScanTimeout));
        }
        NNT_LDN_SYNC("PatchShareFailure.Sleep.Wait");
    }
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));

    NN_LOG("Join\n");
    {
        ContentShareFailureTestHelper helper(cond);

        helper.OpenSession();
        helper.SyncSessionOpened();
        helper.WaitReadyToJoinSession();
        helper.ScanSession();

        nnt::ldn::PutSleep();
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));
        nnt::ldn::WakeUp();

        if (!helper.IsMaster())
        {
            // セッションに参加します。
            NNT_ASSERT_RESULT_FAILURE(nn::lcs::ResultSleep, nnt::lcs::TimedJoin(helper.session, JoinTimeout));
        }
        NNT_LDN_SYNC("PatchShareFailure.Sleep.Wait");
    }
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));

    NN_LOG("Start\n");
    {
        ContentShareFailureTestHelper helper(cond);

        helper.OpenSession();
        helper.SyncSessionOpened();
        helper.WaitReadyToJoinSession();
        helper.ScanSession();
        helper.JoinSession();
        helper.WaitJoinSessionOtherClient();
        helper.SyncJoined();
        helper.SyncReadyToStart();

        nnt::ldn::PutSleep();
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));
        nnt::ldn::WakeUp();

        if (!helper.IsMaster())
        {
            // コンテンツの配信を開始します。
            NNT_ASSERT_RESULT_FAILURE(nn::lcs::ResultSleep, nn::lcs::StartContentsShare());
        }
        NNT_LDN_SYNC("PatchShareFailure.Sleep.Wait");
    }
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));

    NN_LOG("Resume\n");
    {
        ContentShareFailureTestHelper helper(cond);

        helper.OpenSession();
        helper.SyncSessionOpened();
        helper.WaitReadyToJoinSession();
        helper.ScanSession();
        helper.JoinSession();
        helper.WaitJoinSessionOtherClient();
        helper.SyncJoined();
        helper.SyncReadyToStart();
        helper.StartContentShare();
        helper.SyncStarted();
        helper.WaitStart();
        helper.WaitSuspendedForNeedTerminateApplication();

        nnt::ldn::PutSleep();
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));
        nnt::ldn::WakeUp();

        helper.ResumeContentShare();
        NNT_LDN_SYNC("PatchShareFailure.Sleep.Wait");
    }
}
#endif

//
// テストのエントリポイントです。
//
extern "C" void nnMain()
{
    // コマンドライン引数に関する設定です。
    nnt::lcs::CommandLineParserSetting setting = { };
    setting.flag = static_cast<nn::Bit32>(
        nnt::lcs::CommandLineOptionFlag_NodeCount |
        nnt::lcs::CommandLineOptionFlag_NodeIndex |
        nnt::lcs::CommandLineOptionFlag_SceneId
        );
    setting.nodeCountMin = 2;
    setting.nodeCountMax = nnt::ldn::SynchronizationClientCountMax;

    // コマンドライン引数を解析します。
    int argc = nn::os::GetHostArgc();
    char **argv = nn::os::GetHostArgv();
    ::testing::InitGoogleTest(&argc, argv);
    nnt::lcs::Parse(&g_TestConfig, setting, argc, argv);

    // socket ライブラリと ns ライブラリを初期化しておきます。
    nn::socket::Initialize(g_SocketConfig);
    nnt::lcs::NsInitializer nsInitiailzer;

    // テストの準備です。
    nnt::lcs::Setup(g_TestConfig.sceneId);

    // 他のインスタンスと同期するためにテストグループに参加します。
    int result;
    if (g_TestConfig.nodeIndex == 0)
    {
        g_pServer = new nnt::ldn::HtcsSynchronizationServer(
            g_SynchronizationBuffer, sizeof(g_SynchronizationBuffer));
        result = g_pServer->CreateServer(
            "nn::lcs::Integration::PatchShareFailure", g_TestConfig.nodeCount - 1);
        g_pSync = g_pServer;
    }
    else
    {
        g_pClient = new nnt::ldn::HtcsSynchronizationClient(
            g_SynchronizationBuffer, sizeof(g_SynchronizationBuffer));
        result = g_pClient->Connect("nn::lcs::Integration::PatchShareFailure");
        if (g_TestConfig.nodeIndex == 0)
        {
            g_TestConfig.nodeIndex = static_cast<int8_t>(g_pClient->GetClientIndex());
        }
        if (g_TestConfig.nodeCount == 0)
        {
            g_TestConfig.nodeCount = static_cast<int8_t>(g_pClient->GetClientCount() + 1);
        }
        g_pSync = g_pClient;
    }

    // テストを実行します。
    int exitCode;
    if (result == nnt::ldn::SynchronizationResult_Success)
    {
        exitCode = RUN_ALL_TESTS();
    }
    else
    {
        exitCode = result;
    }
    delete g_pSync;
    nn::socket::Finalize();
    nnt::Exit(exitCode);
}
