﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nn/lcs.h>
#include <nn/lcs/lcs_DebugApi.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/socket.h>
#include <nnt.h>
#include <nnt/lcs/testLcs_Utility.h>
#include <nnt/ldn/testLdn_HtcsSynchronization.h>
#include <nnt/ldn/testLdn_NifmUtility.h>
#include <nnt/ldn/testLdn_Utility.h>
#include <nnt/result/testResult_Assert.h>
#include "../Common/Params.h"
#include "../Common/Precondition.h"

namespace
{
    // テストの設定値です。
    nnt::lcs::TestConfig g_TestConfig;

    // LCS/Socket の初期化用バッファです。
    NN_ALIGNAS(nn::lcs::RequiredBufferAlignment) char g_Buffer[4 * 1024 * 1024];
    NN_STATIC_ASSERT(nn::lcs::RequiredBufferSize < sizeof(g_Buffer));
    nn::socket::ConfigDefaultWithMemory g_SocketConfig;

    // EULA 取得用のバッファです。
    char g_EulaBuffer[1 * 1024 * 1024];

    // 同期に使用するオブジェクトとバッファです。
    nnt::ldn::ISynchronizationServer* g_pServer;
    char g_SynchronizationBuffer[16 * 1024];

    // 同期用のマクロです。
    #define NNT_LDN_SYNC(keyword)\
        ASSERT_EQ(::nnt::ldn::SynchronizationResult_Success, g_pServer->Synchronize(keyword))

    // テストを実施する条件を指定するためのマクロです。
    #define NNT_LDN_REQUIRES_NODE_COUNT(count) \
        if (g_TestConfig.nodeCount < (count))\
        {\
            return;\
        }

} // namespace <unnamed>

//
// ホストしかセッションに参加していない状態でスキャンします。
//
TEST(Scan, OnlyHost)
{
    // テストケース開始前の同期です。
    NNT_LDN_REQUIRES_NODE_COUNT(2);
    NNT_LDN_SYNC("Scan.OnlyHost.Start");

    // NS, LDN, LCS ライブラリを初期化します。
    nn::lcs::Config config = nnt::lcs::CreateDefaultHostConfig();
    nnt::lcs::NsInitializer nsInitializer;
    nnt::ldn::SystemInitializer ldnInitializer;
    nnt::lcs::Initializer lcsInitializer(g_Buffer, sizeof(g_Buffer), config);

    // 複数のアプリケーションに対してテストを繰り返します。
    for (uint64_t id = nnt::lcs::Application::Matching01.value;
         id <= nnt::lcs::Application::Matching05.value; id += 0x1000)
    {
        // 対象のアプリケーションを指定します。
        const auto settings = nnt::lcs::CreateSessionSettings(nn::lcs::NodeCountMax, id);

        // セッションを構築します。
        NNT_ASSERT_RESULT_SUCCESS(nn::lcs::OpenSession(settings));
        NNT_LDN_SYNC("Scan.OnlyHost.SessionOpened");

        // クライアントによるスキャンが完了するまで待機します。
        NNT_LDN_SYNC("Scan.OnlyHost.ScanCompleted");

        // セッションを破棄します。
        NNT_ASSERT_RESULT_SUCCESS(nn::lcs::LeaveSession());
    }
}

//
// ホストとクライアント 1 台がセッションに参加している状態でスキャンします。
//
TEST(Scan, HostAndClient)
{
    // テストケース開始前の同期です。
    NNT_LDN_REQUIRES_NODE_COUNT(3);
    NNT_LDN_SYNC("Scan.HostAndClient.Start");

    // NS, LDN, LCS ライブラリを初期化します。
    nn::lcs::Config config = nnt::lcs::CreateDefaultHostConfig();
    nnt::lcs::NsInitializer nsInitializer;
    nnt::ldn::SystemInitializer ldnInitializer;
    nnt::lcs::Initializer lcsInitializer(g_Buffer, sizeof(g_Buffer), config);

    // イベントを取得します。
    nn::os::SystemEventType stateChangeEvent;
    nn::lcs::AttachStateChangeEvent(&stateChangeEvent);
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

    // 複数のアプリケーションに対してテストを繰り返します。
    for (uint64_t id = nnt::lcs::Application::Matching01.value;
         id <= nnt::lcs::Application::Matching04.value; id += 0x1000)
    {
        // 対象のアプリケーションを指定します。
        const auto& cond = GetPrecondition("Scan", "HostAndClient", id);
        const auto settings = nnt::lcs::CreateSessionSettings(nn::lcs::NodeCountMax, id);

        // ホストとクライアントのうち、最新のバージョンを計算します。
        char displayVersion[16];
        const int version = std::max(
            cond.nodes[0].application.version, cond.nodes[1].application.version);
        nnt::lcs::ConvertToDisplayVersion(displayVersion, sizeof(displayVersion), version);

        // セッションを構築します。
        NNT_ASSERT_RESULT_SUCCESS(nn::lcs::OpenSession(settings));
        NNT_LDN_SYNC("Scan.HostAndClient.SessionOpened");

        // クライアントによるスキャンが完了するまで待機します。
        NNT_LDN_SYNC("Scan.HostAndClient.ScanCompletedBeforeJoin");

        // クライアントの接続を待ち受けます。
        nn::os::TimedWaitSystemEvent(&stateChangeEvent, JoinTimeout);
        NNT_LDN_SYNC("Scan.HostAndClient.Joined");

        // クライアントによるスキャンが完了するまで待機します。
        NNT_LDN_SYNC("Scan.HostAndClient.ScanCompletedAfterJoin");

        // セッションを破棄します。
        NNT_ASSERT_RESULT_SUCCESS(nn::lcs::LeaveSession());
    }
}

//
// 複数のセッションが存在する状態で、バッファ数 1 でスキャンします。
// このテストだけホスト、クライアントの役割が逆転します。
//
TEST(Scan, BufferSizeMin)
{
    // テストケース開始前の同期です。
    NNT_LDN_REQUIRES_NODE_COUNT(3);
    NNT_LDN_SYNC("Scan.BufferSizeMin.Start");

    // NS, LDN, LCS ライブラリを初期化します。
    nn::lcs::Config config = nnt::lcs::CreateDefaultClientConfig();
    nnt::lcs::NsInitializer nsInitializer;
    nnt::ldn::SystemInitializer ldnInitializer;
    nnt::lcs::Initializer lcsInitializer(g_Buffer, sizeof(g_Buffer), config);

    // セッションの構築を待機します。
    NNT_LDN_SYNC("Scan.BufferSizeMin.SessionOpened");

    // セッションのスキャンを何回か繰り返して、常に 1 つセッションを発見できることを確認します。
    const int repeatCount = 2;
    for (int i = 0; i < repeatCount; ++i)
    {
        int count;
        nn::lcs::SessionInfo session;
        NNT_ASSERT_RESULT_SUCCESS(nn::lcs::Scan(&session, &count, 1));
        ASSERT_EQ(1, count);
    }
    NNT_LDN_SYNC("Scan.BufferSizeMin.ScanCompleted");
}

//
// 最大端末数による接続制限を確認します。
//
TEST(Restrict, NodeCountMax)
{
    // テストケース開始前の同期です。
    NNT_LDN_REQUIRES_NODE_COUNT(2);
    NNT_LDN_SYNC("Restrict.NodeCountMax.Start");

    // NS, LDN, LCS ライブラリを初期化します。
    nn::lcs::Config config = nnt::lcs::CreateDefaultHostConfig();
    nnt::lcs::NsInitializer nsInitializer;
    nnt::ldn::SystemInitializer ldnInitializer;
    nnt::lcs::Initializer lcsInitializer(g_Buffer, sizeof(g_Buffer), config);

    // ホストとしてセッションを構築します。
    const auto id = DefaultApplicationId;
    const int nodeCountMax = std::min<int>(nn::lcs::NodeCountMax, g_TestConfig.nodeCount) - 1;
    const auto settings = nnt::lcs::CreateSessionSettings(nodeCountMax, id);
    nnt::lcs::HostStarter starter(settings);
    NNT_LDN_SYNC("Restrict.NodeCountMax.SessionOpened");

    // クライアントの接続完了まで待機します。
    NNT_LDN_SYNC("Restrict.NodeCountMax.ScanCompleted");
    NNT_LDN_SYNC("Restrict.NodeCountMax.Joined");

    // 接続制限が有効に作用していることを確認します。
    nn::lcs::SessionInfo session;
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetSessionInfo(&session));
    ASSERT_EQ(nodeCountMax, session.nodeCount);

    // ノード情報の正当性を検証します。
    nn::lcs::NodeInfo nodes[nn::lcs::NodeCountMax];
    int count;
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetNodes(nodes, &count, nn::lcs::NodeCountMax));
    ASSERT_EQ(nodeCountMax, count);
    ASSERT_STREQ(nnt::lcs::DefaultHostUserName, nodes[0].userName);
    for (int i = 1; i < nodeCountMax; ++i)
    {
        ASSERT_STREQ(nnt::lcs::DefaultClientUserName, nodes[i].userName);
    }
    NNT_LDN_SYNC("Restrict.NodeCountMax.SessionVerified");
    NNT_LDN_SYNC("Restrict.NodeCountMax.Finished");
}

//
// Accept Policy による接続制限を確認します。
//
TEST(Restrict, AcceptPolicyAlwaysReject)
{
    // テストケース開始前の同期です。
    NNT_LDN_REQUIRES_NODE_COUNT(2);
    NNT_LDN_SYNC("Restrict.AcceptPolicyAlwaysReject.Start");

    // NS, LDN, LCS ライブラリを初期化します。
    nn::lcs::Config config = nnt::lcs::CreateDefaultHostConfig();
    nnt::lcs::NsInitializer nsInitializer;
    nnt::ldn::SystemInitializer ldnInitializer;
    nnt::lcs::Initializer lcsInitializer(g_Buffer, sizeof(g_Buffer), config);

    // ホストとしてセッションを構築します。
    const auto id = DefaultApplicationId;
    const auto settings = nnt::lcs::CreateSessionSettings(nn::lcs::NodeCountMax, id);
    nnt::lcs::HostStarter starter(settings);

    // ポリシーを変更します。
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::SetClientAcceptPolicy(nn::lcs::AcceptPolicy_AlwaysReject));
    NNT_LDN_SYNC("Restrict.AcceptPolicyAlwaysReject.SessionOpened");

    // クライアントの接続完了まで待機します。
    NNT_LDN_SYNC("Restrict.AcceptPolicyAlwaysReject.ScanCompleted");
    NNT_LDN_SYNC("Restrict.AcceptPolicyAlwaysReject.Joined");

    // 接続制限が有効に作用していることを確認します。
    nn::lcs::SessionInfo session;
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetSessionInfo(&session));
    ASSERT_EQ(1, session.nodeCount);

    // ノード情報の正当性を検証します。
    nn::lcs::NodeInfo nodes[nn::lcs::NodeCountMax];
    int count;
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetNodes(nodes, &count, nn::lcs::NodeCountMax));
    ASSERT_EQ(1, count);
    NNT_LDN_SYNC("Restrict.AcceptPolicyAlwaysReject.Finished");
}

//
// ホストのフライトモードによるマッチング失敗を確認します。
//
TEST(MatchingFailure, FlightMode)
{
    // テストケース開始前の同期です。
    NNT_LDN_REQUIRES_NODE_COUNT(2);
    NNT_LDN_SYNC("MatchingFailure.FlightMode.Start");

    // NS, LDN, LCS ライブラリを初期化します。
    nn::lcs::Config config = nnt::lcs::CreateDefaultHostConfig();
    nnt::lcs::NsInitializer nsInitializer;
    nnt::ldn::SystemInitializer ldnInitializer;
    nnt::lcs::Initializer lcsInitializer(g_Buffer, sizeof(g_Buffer), config);

    // イベントを取得します。
    nn::os::SystemEventType stateChangeEvent;
    nn::lcs::AttachStateChangeEvent(&stateChangeEvent);
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

    // ホストとしてセッションを構築します。
    const auto id = DefaultApplicationId;
    const auto settings = nnt::lcs::CreateSessionSettings(nn::lcs::NodeCountMax, id);
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::OpenSession(settings));
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    nn::os::ClearSystemEvent(&stateChangeEvent);
    NNT_LDN_SYNC("MatchingFailure.FlightMode.SessionOpened");

    // クライアントの接続完了まで待機します。
    NNT_LDN_SYNC("MatchingFailure.FlightMode.ScanCompleted");
    NNT_LDN_SYNC("MatchingFailure.FlightMode.Joined");

    // フライトモードに入ります。
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    nn::os::ClearSystemEvent(&stateChangeEvent);
    nnt::ldn::ScopedWirelessCommunicationSwitchSetter wifiSwitch(false);
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    nn::os::ClearSystemEvent(&stateChangeEvent);

    // 失敗理由を正しく取得できることを確認します。
    ASSERT_EQ(nn::lcs::State_ContentsShareFailed, nn::lcs::GetState());
    ASSERT_EQ(nn::lcs::ContentsShareFailureReason_FlightMode,
              nn::lcs::GetContentsShareFailureReason());
    NNT_LDN_SYNC("MatchingFailure.FlightMode.Destroyed");
}

//
// ホストのセッション破棄によるマッチング失敗を確認します。
//
TEST(MatchingFailure, DestroySession)
{
    // テストケース開始前の同期です。
    NNT_LDN_REQUIRES_NODE_COUNT(2);
    NNT_LDN_SYNC("MatchingFailure.DestroySession.Start");

    // NS, LDN, LCS ライブラリを初期化します。
    nn::lcs::Config config = nnt::lcs::CreateDefaultHostConfig();
    nnt::lcs::NsInitializer nsInitializer;
    nnt::ldn::SystemInitializer ldnInitializer;
    nnt::lcs::Initializer lcsInitializer(g_Buffer, sizeof(g_Buffer), config);

    // イベントを取得します。
    nn::os::SystemEventType stateChangeEvent;
    nn::lcs::AttachStateChangeEvent(&stateChangeEvent);
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

    // クライアントの接続を待ち受けてからセッションを破棄します。
    {
        // ホストとしてセッションを構築します。
        const auto id = DefaultApplicationId;
        const auto settings = nnt::lcs::CreateSessionSettings(nn::lcs::NodeCountMax, id);
        nnt::lcs::HostStarter starter(settings);
        ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
        nn::os::ClearSystemEvent(&stateChangeEvent);
        NNT_LDN_SYNC("MatchingFailure.DestroySession.SessionOpened");

        // クライアントの接続完了まで待機します。
        NNT_LDN_SYNC("MatchingFailure.DestroySession.ScanCompleted");
        NNT_LDN_SYNC("MatchingFailure.DestroySession.Joined");
        ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
        nn::os::ClearSystemEvent(&stateChangeEvent);

        // セッションを破棄します。
    }
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    nn::os::ClearSystemEvent(&stateChangeEvent);
    ASSERT_EQ(nn::lcs::State_Initialized, nn::lcs::GetState());
    NNT_LDN_SYNC("MatchingFailure.DestroySession.Destroyed");
}

//
// ホストの Finalize() によるマッチング失敗を確認します。
//
TEST(MatchingFailure, HostFinalize)
{
    // テストケース開始前の同期です。
    NNT_LDN_REQUIRES_NODE_COUNT(2);
    NNT_LDN_SYNC("MatchingFailure.HostFinalize.Start");

    // NS, LDN, LCS ライブラリを初期化します。
    nn::lcs::Config config = nnt::lcs::CreateDefaultHostConfig();
    nnt::lcs::NsInitializer nsInitializer;
    nnt::ldn::SystemInitializer ldnInitializer;
    nnt::lcs::Initializer lcsInitializer(g_Buffer, sizeof(g_Buffer), config);

    // イベントを取得します。
    nn::os::SystemEventType stateChangeEvent;
    nn::lcs::AttachStateChangeEvent(&stateChangeEvent);
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

    // ホストとしてセッションを構築します。
    const auto id = DefaultApplicationId;
    const auto settings = nnt::lcs::CreateSessionSettings(nn::lcs::NodeCountMax, id);
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::OpenSession(settings));
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    nn::os::ClearSystemEvent(&stateChangeEvent);
    NNT_LDN_SYNC("MatchingFailure.HostFinalize.SessionOpened");

    // クライアントの接続完了まで待機します。
    NNT_LDN_SYNC("MatchingFailure.HostFinalize.ScanCompleted");
    NNT_LDN_SYNC("MatchingFailure.HostFinalize.Joined");
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    nn::os::ClearSystemEvent(&stateChangeEvent);

    // セッションを破棄しないまま Finalize() します。
}

//
// LeaveSession によるクライアントの離脱を検証します。
//
TEST(ClientLeave, LeaveSession)
{
    // テストケース開始前の同期です。
    NNT_LDN_REQUIRES_NODE_COUNT(2);
    NNT_LDN_SYNC("ClientLeave.LeaveSession.Start");

    // NS, LDN, LCS ライブラリを初期化します。
    nn::lcs::Config config = nnt::lcs::CreateDefaultHostConfig();
    nnt::lcs::NsInitializer nsInitializer;
    nnt::ldn::SystemInitializer ldnInitializer;
    nnt::lcs::Initializer lcsInitializer(g_Buffer, sizeof(g_Buffer), config);

    // イベントを取得します。
    nn::os::SystemEventType stateChangeEvent;
    nn::lcs::AttachStateChangeEvent(&stateChangeEvent);
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

    // ホストとしてセッションを構築します。
    const auto id = DefaultApplicationId;
    const auto settings = nnt::lcs::CreateSessionSettings(nn::lcs::NodeCountMax, id);
    nnt::lcs::HostStarter starter(settings);
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    nn::os::ClearSystemEvent(&stateChangeEvent);
    NNT_LDN_SYNC("ClientLeave.LeaveSession.SessionOpened");
    NNT_LDN_SYNC("ClientLeave.LeaveSession.ScanCompleted");

    // クライアントの接続完了まで待機します。
    nn::lcs::NodeInfo nodes[nn::lcs::NodeCountMax];
    int count;
    do
    {
        ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&stateChangeEvent, JoinTimeout));
        nn::os::ClearSystemEvent(&stateChangeEvent);
        NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetNodes(nodes, &count, nn::lcs::NodeCountMax));
    } while (count < g_TestConfig.nodeCount);
    NNT_LDN_SYNC("ClientLeave.LeaveSession.Joined");

    // クライアントの離脱完了まで待機します。
    do
    {
        ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&stateChangeEvent, JoinTimeout));
        nn::os::ClearSystemEvent(&stateChangeEvent);
        NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetNodes(nodes, &count, nn::lcs::NodeCountMax));
    } while (1 < count);
    NNT_LDN_SYNC("ClientLeave.LeaveSession.Disconnected");
}

//
// Finalize によるクライアントの離脱を検証します。
//
TEST(ClientLeave, Finalize)
{
    // テストケース開始前の同期です。
    NNT_LDN_REQUIRES_NODE_COUNT(2);
    NNT_LDN_SYNC("ClientLeave.Finalize.Start");

    // NS, LDN, LCS ライブラリを初期化します。
    nn::lcs::Config config = nnt::lcs::CreateDefaultHostConfig();
    nnt::lcs::NsInitializer nsInitializer;
    nnt::ldn::SystemInitializer ldnInitializer;
    nnt::lcs::Initializer lcsInitializer(g_Buffer, sizeof(g_Buffer), config);

    // イベントを取得します。
    nn::os::SystemEventType stateChangeEvent;
    nn::lcs::AttachStateChangeEvent(&stateChangeEvent);
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

    // ホストとしてセッションを構築します。
    const auto id = DefaultApplicationId;
    const auto settings = nnt::lcs::CreateSessionSettings(nn::lcs::NodeCountMax, id);
    nnt::lcs::HostStarter starter(settings);
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    nn::os::ClearSystemEvent(&stateChangeEvent);
    NNT_LDN_SYNC("ClientLeave.Finalize.SessionOpened");
    NNT_LDN_SYNC("ClientLeave.Finalize.ScanCompleted");

    // クライアントの接続完了まで待機します。
    nn::lcs::NodeInfo nodes[nn::lcs::NodeCountMax];
    int count;
    do
    {
        ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&stateChangeEvent, JoinTimeout));
        nn::os::ClearSystemEvent(&stateChangeEvent);
        NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetNodes(nodes, &count, nn::lcs::NodeCountMax));
    } while (count < g_TestConfig.nodeCount);
    NNT_LDN_SYNC("ClientLeave.Finalize.Joined");

    // クライアントの離脱完了まで待機します。
    do
    {
        ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&stateChangeEvent, JoinTimeout));
        nn::os::ClearSystemEvent(&stateChangeEvent);
        NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetNodes(nodes, &count, nn::lcs::NodeCountMax));
    } while (1 < count);
    NNT_LDN_SYNC("ClientLeave.Finalize.Disconnected");
}

//
// フライトモード によるクライアントの離脱を検証します。
//
TEST(ClientLeave, FlightMode)
{
    // テストケース開始前の同期です。
    NNT_LDN_REQUIRES_NODE_COUNT(2);
    NNT_LDN_SYNC("ClientLeave.FlightMode.Start");

    // NS, LDN, LCS ライブラリを初期化します。
    nn::lcs::Config config = nnt::lcs::CreateDefaultHostConfig();
    nnt::lcs::NsInitializer nsInitializer;
    nnt::ldn::SystemInitializer ldnInitializer;
    nnt::lcs::Initializer lcsInitializer(g_Buffer, sizeof(g_Buffer), config);

    // イベントを取得します。
    nn::os::SystemEventType stateChangeEvent;
    nn::lcs::AttachStateChangeEvent(&stateChangeEvent);
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

    // ホストとしてセッションを構築します。
    const auto id = DefaultApplicationId;
    const auto settings = nnt::lcs::CreateSessionSettings(nn::lcs::NodeCountMax, id);
    nnt::lcs::HostStarter starter(settings);
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    nn::os::ClearSystemEvent(&stateChangeEvent);
    NNT_LDN_SYNC("ClientLeave.FlightMode.SessionOpened");
    NNT_LDN_SYNC("ClientLeave.FlightMode.ScanCompleted");

    // クライアントの接続完了まで待機します。
    nn::lcs::NodeInfo nodes[nn::lcs::NodeCountMax];
    int count;
    do
    {
        ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&stateChangeEvent, JoinTimeout));
        nn::os::ClearSystemEvent(&stateChangeEvent);
        NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetNodes(nodes, &count, nn::lcs::NodeCountMax));
    } while (count < g_TestConfig.nodeCount);
    NNT_LDN_SYNC("ClientLeave.FlightMode.Joined");

    // クライアントの離脱完了まで待機します。
    do
    {
        ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&stateChangeEvent, JoinTimeout));
        nn::os::ClearSystemEvent(&stateChangeEvent);
        NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetNodes(nodes, &count, nn::lcs::NodeCountMax));
    } while (1 < count);
    NNT_LDN_SYNC("ClientLeave.FlightMode.Disconnected");
}

//
// GetNodes() が正しく機能することを検証します。
//
TEST(Monitor, GetNodes)
{
    // テストケース開始前の同期です。
    NNT_LDN_REQUIRES_NODE_COUNT(2);
    NNT_LDN_SYNC("Monitor.GetNodes.Start");

    // NS, LDN, LCS ライブラリを初期化します。
    nn::lcs::Config config = nnt::lcs::CreateDefaultHostConfig();
    nnt::lcs::NsInitializer nsInitializer;
    nnt::ldn::SystemInitializer ldnInitializer;
    nnt::lcs::Initializer lcsInitializer(g_Buffer, sizeof(g_Buffer), config);

    // イベントを取得します。
    nn::os::SystemEventType stateChangeEvent;
    nn::lcs::AttachStateChangeEvent(&stateChangeEvent);
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

    // ホストとしてセッションを構築します。
    const auto id = DefaultApplicationId;
    const auto settings = nnt::lcs::CreateSessionSettings(nn::lcs::NodeCountMax, id);
    nnt::lcs::HostStarter starter(settings);
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    nn::os::ClearSystemEvent(&stateChangeEvent);
    NNT_LDN_SYNC("Monitor.GetNodes.SessionOpened");
    NNT_LDN_SYNC("Monitor.GetNodes.ScanCompleted");

    // クライアントを一台ずつ接続させて、GetNodes() の動作を確認します。
    for (int expectedCount = 2; expectedCount <= g_TestConfig.nodeCount; ++expectedCount)
    {
        // クライアントの接続を待機します。
        ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&stateChangeEvent, JoinTimeout));
        nn::os::ClearSystemEvent(&stateChangeEvent);

        // 現在接続中のクライアントを確認します。
        nn::lcs::NodeInfo nodes[nn::lcs::NodeCountMax];
        int actualCount;
        NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetNodes(nodes, &actualCount, nn::lcs::NodeCountMax));
        ASSERT_EQ(expectedCount, actualCount);
        ASSERT_EQ(nn::lcs::GetMyIndex(), nodes[0].index);
        ASSERT_STREQ(nnt::lcs::DefaultHostUserName, nodes[0].userName);
        for (int i = 1; i < actualCount; ++i)
        {
            ASSERT_NE(0, nodes[i].index);
            ASSERT_STREQ(nnt::lcs::DefaultClientUserName, nodes[i].userName);
        }
        NNT_LDN_SYNC("Monitor.GetNodes.Joined");
    }

    // クライアントを一台ずつ離脱させて、GetNodes() の動作を確認します。
    for (int expectedCount = g_TestConfig.nodeCount - 1; expectedCount >= 1; --expectedCount)
    {
        // クライアントの離脱を待機します。
        ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&stateChangeEvent, JoinTimeout));
        nn::os::ClearSystemEvent(&stateChangeEvent);
        ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

        // 現在接続中のクライアントを確認します。
        nn::lcs::NodeInfo nodes[nn::lcs::NodeCountMax];
        int actualCount;
        NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetNodes(nodes, &actualCount, nn::lcs::NodeCountMax));
        ASSERT_EQ(expectedCount, actualCount);
        ASSERT_EQ(nn::lcs::GetMyIndex(), nodes[0].index);
        ASSERT_STREQ(nnt::lcs::DefaultHostUserName, nodes[0].userName);
        for (int i = 1; i < actualCount; ++i)
        {
            ASSERT_NE(0, nodes[i].index);
            ASSERT_STREQ(nnt::lcs::DefaultClientUserName, nodes[i].userName);
        }
        NNT_LDN_SYNC("Monitor.GetNodes.Disconnected");
    }
}

//
// GetRequiredStorageSize() が正しく機能することを検証します。
//
TEST(Monitor, GetRequiredStorageSize)
{
    // テストケース開始前の同期です。
    NNT_LDN_REQUIRES_NODE_COUNT(3);
    NNT_LDN_SYNC("Monitor.GetRequiredStorageSize.Start");

    // NS, LDN, LCS ライブラリを初期化します。
    nn::lcs::Config config = nnt::lcs::CreateDefaultHostConfig();
    nnt::lcs::NsInitializer nsInitializer;
    nnt::ldn::SystemInitializer ldnInitializer;
    nnt::lcs::Initializer lcsInitializer(g_Buffer, sizeof(g_Buffer), config);

    // イベントを取得します。
    nn::os::SystemEventType stateChangeEvent;
    nn::lcs::AttachStateChangeEvent(&stateChangeEvent);
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

    // 複数のアプリケーションに対してテストを繰り返します。
    for (uint64_t id = nnt::lcs::Application::Matching01.value;
         id <= nnt::lcs::Application::Matching04.value; id += 0x1000)
    {
        // テストの事前条件を取得します。
        const auto& cond = GetPrecondition("Monitor", "GetRequiredStorageSize", id);
        const auto& app = cond.application;

        // 対象のアプリケーションを指定します。
        const auto settings = nnt::lcs::CreateSessionSettings(nn::lcs::NodeCountMax, id);

        // ホストとしてセッションを構築します。
        nnt::lcs::HostStarter starter(settings);
        ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
        nn::os::ClearSystemEvent(&stateChangeEvent);
        NNT_LDN_SYNC("Monitor.GetRequiredStorageSize.SessionOpened");

        // この時点では必要サイズは 0 です。
        size_t requiredSize;
        NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetRequiredStorageSize(&requiredSize));
        ASSERT_EQ(0U, requiredSize);
        NNT_LDN_SYNC("Monitor.GetRequiredStorageSize.ScanCompleted");

        // クライアントの接続を待ち受けます。
        NNT_LDN_SYNC("Monitor.GetRequiredStorageSize.Joined");
        ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
        nn::os::ClearSystemEvent(&stateChangeEvent);

        // 自身が最新のパッチをもっていない場合には必要サイズが 1 以上になります。
        NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetRequiredStorageSize(&requiredSize));
        if (cond.nodes[0].application.version == app.latestVersion)
        {
            ASSERT_EQ(0U, requiredSize);
        }
        else
        {
            ASSERT_GT(requiredSize, 0U);
        }
        NNT_LDN_SYNC("Monitor.GetRequiredStorageSize.ReadyToDestroy");
    }
}

//
// Joined 状態で実行できない API を実行して ResultInvalidState が返ることを確認します。
//
TEST(InvalidState, Joined)
{
    // テストケース開始前の同期です。
    NNT_LDN_REQUIRES_NODE_COUNT(2);
    NNT_LDN_SYNC("InvalidState.Joined.Start");

    // NS, LDN, LCS ライブラリを初期化します。
    nn::lcs::Config config = nnt::lcs::CreateDefaultHostConfig();
    nnt::lcs::NsInitializer nsInitializer;
    nnt::ldn::SystemInitializer ldnInitializer;
    nnt::lcs::Initializer lcsInitializer(g_Buffer, sizeof(g_Buffer), config);

    // イベントを取得します。
    nn::os::SystemEventType stateChangeEvent;
    nn::lcs::AttachStateChangeEvent(&stateChangeEvent);
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

    // 対象のアプリケーションを指定します。
    const auto id = DefaultApplicationId;
    const auto settings = nnt::lcs::CreateSessionSettings(nn::lcs::NodeCountMax, id);

    // ホストとしてセッションを構築します。
    nnt::lcs::HostStarter starter(settings);
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    nn::os::ClearSystemEvent(&stateChangeEvent);
    NNT_LDN_SYNC("InvalidState.Joined.SessionOpened");
    NNT_LDN_SYNC("InvalidState.Joined.ScanCompleted");

    // クライアントの接続を待ち受けます。
    NNT_LDN_SYNC("InvalidState.Joined.Joined");
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    nn::os::ClearSystemEvent(&stateChangeEvent);

    // Opened 状態では GetProgress() は実行できません。
    nn::lcs::Progress progress;
    NNT_ASSERT_RESULT_FAILURE(nn::lcs::ResultInvalidState, nn::lcs::GetProgress(&progress));

    // Opened 状態では GetNodeProgress() は実行できません。
    nn::lcs::NodeProgress nodeProgress;
    NNT_ASSERT_RESULT_FAILURE(
        nn::lcs::ResultInvalidState,
        nn::lcs::GetNodeProgress(&nodeProgress, nn::lcs::GetMyIndex()));

    // Opened 状態では GetDownloadedContents() は実行できません。
    nn::lcs::ContentsInfo content;
    int contentCount;
    NNT_ASSERT_RESULT_FAILURE(
        nn::lcs::ResultInvalidState,
        nn::lcs::GetDownloadedContents(&content, &contentCount, 1));

    // Opened 状態では GetDownloadedEulaDataSize() は実行できません。
    size_t eulaSize;
    const char* path = "JPja/Eula.msbt.szs";
    NNT_ASSERT_RESULT_FAILURE(
        nn::lcs::ResultInvalidState,
        nn::lcs::GetDownloadedEulaDataSize(&eulaSize, path));

    // Opened 状態では GetDownloadedEulaData() は実行できません。
    NNT_ASSERT_RESULT_FAILURE(
        nn::lcs::ResultInvalidState,
        nn::lcs::GetDownloadedEulaData(&eulaSize, g_EulaBuffer, sizeof(g_EulaBuffer), path));

    // Opened 状態では GetSessionContext() は実行できません。
    nn::lcs::SessionContext context;
    NNT_ASSERT_RESULT_FAILURE(nn::lcs::ResultInvalidState, nn::lcs::GetSessionContext(&context));

    // Opened 状態では ResumeContentsShare() は実行できません。
    NNT_ASSERT_RESULT_FAILURE(nn::lcs::ResultInvalidState, nn::lcs::ResumeContentsShare());

    // Opened 状態では ResumeSession() は実行できません。
    NNT_ASSERT_RESULT_FAILURE(nn::lcs::ResultInvalidState, nn::lcs::ResumeSession(context));

    // Opened 状態では GetContentsShareFailureReason() は実行できません。
    ASSERT_EQ(nn::lcs::ContentsShareFailureReason_None, nn::lcs::GetContentsShareFailureReason());

    // Opened 状態では GetSuspendedReason() は実行できません。
    ASSERT_EQ(nn::lcs::SuspendedReason_None, nn::lcs::GetSuspendedReason());

    // Opened 状態では OpenSession() は実行できません。
    NNT_ASSERT_RESULT_FAILURE(nn::lcs::ResultInvalidState, nn::lcs::OpenSession(settings));

    // Opened 状態では Scan() は実行できません。
    nn::lcs::SessionInfo session;
    int scanResultCount;
    NNT_ASSERT_RESULT_FAILURE(
        nn::lcs::ResultInvalidState,
        nn::lcs::Scan(&session, &scanResultCount, 1));

    // Opened 状態では JoinSession() は実行できません。
    NNT_ASSERT_RESULT_FAILURE(nn::lcs::ResultInvalidState, nn::lcs::JoinSession(session));

    NNT_LDN_SYNC("InvalidState.Joined.ReadyToDestroy");
}

//
// ホストのプロトコルに互換性がある場合の挙動を確認します。
//
TEST(Join, CompatibleProtocolVersion)
{
    // テストケース開始前の同期です。
    NNT_LDN_REQUIRES_NODE_COUNT(2);
    NNT_LDN_SYNC("Join.CompatibleProtocolVersion.Start");

    // プロトコルバージョンを設定します。
    nnt::lcs::ProtocolVersionSetter protocolVersionSetter(2, 1);

    // NS, LDN, LCS ライブラリを初期化します。
    nn::lcs::Config config = nnt::lcs::CreateDefaultHostConfig();
    nnt::lcs::NsInitializer nsInitializer;
    nnt::ldn::SystemInitializer ldnInitializer;
    nnt::lcs::Initializer lcsInitializer(g_Buffer, sizeof(g_Buffer), config);

    // イベントを取得します。
    nn::os::SystemEventType stateChangeEvent;
    nn::lcs::AttachStateChangeEvent(&stateChangeEvent);
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

    // 対象のアプリケーションを指定します。
    const auto id = DefaultApplicationId;
    const auto settings = nnt::lcs::CreateSessionSettings(nn::lcs::NodeCountMax, id);

    // ホストとしてセッションを構築します。
    nnt::lcs::HostStarter starter(settings);
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    nn::os::ClearSystemEvent(&stateChangeEvent);
    NNT_LDN_SYNC("Join.CompatibleProtocolVersion.SessionOpened");
    NNT_LDN_SYNC("Join.CompatibleProtocolVersion.ReadyToDestroy");
}

//
// ホストのプロトコルに互換性が無い場合の挙動を確認します。
//
TEST(Join, IncompatibleProtocolVersion)
{
    // テストケース開始前の同期です。
    NNT_LDN_REQUIRES_NODE_COUNT(2);
    NNT_LDN_SYNC("Join.IncompatibleProtocolVersion.Start");

    // プロトコルバージョンを設定します。
    nnt::lcs::ProtocolVersionSetter protocolVersionSetter(2, 1);

    // NS, LDN, LCS ライブラリを初期化します。
    nn::lcs::Config config = nnt::lcs::CreateDefaultHostConfig();
    nnt::lcs::NsInitializer nsInitializer;
    nnt::ldn::SystemInitializer ldnInitializer;
    nnt::lcs::Initializer lcsInitializer(g_Buffer, sizeof(g_Buffer), config);

    // イベントを取得します。
    nn::os::SystemEventType stateChangeEvent;
    nn::lcs::AttachStateChangeEvent(&stateChangeEvent);
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

    // 対象のアプリケーションを指定します。
    const auto id = DefaultApplicationId;
    const auto settings = nnt::lcs::CreateSessionSettings(nn::lcs::NodeCountMax, id);

    // ホストとしてセッションを構築します。
    nnt::lcs::HostStarter starter(settings);
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    nn::os::ClearSystemEvent(&stateChangeEvent);
    NNT_LDN_SYNC("Join.IncompatibleProtocolVersion.SessionOpened");
    NNT_LDN_SYNC("Join.IncompatibleProtocolVersion.ReadyToDestroy");
}

//
// クライアントがアプリケーションを所有していない場合の挙動を確認します。
//
TEST(Join, ClientDoesNotHaveApplication)
{
    // テストケース開始前の同期です。
    NNT_LDN_REQUIRES_NODE_COUNT(2);
    NNT_LDN_SYNC("Join.ClientDoesNotHaveApplication.Start");

    // NS, LDN, LCS ライブラリを初期化します。
    nn::lcs::Config config = nnt::lcs::CreateDefaultHostConfig();
    nnt::lcs::NsInitializer nsInitializer;
    nnt::ldn::SystemInitializer ldnInitializer;
    nnt::lcs::Initializer lcsInitializer(g_Buffer, sizeof(g_Buffer), config);

    // イベントを取得します。
    nn::os::SystemEventType stateChangeEvent;
    nn::lcs::AttachStateChangeEvent(&stateChangeEvent);
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

    // 対象のアプリケーションを指定します。
    const auto& app = nnt::lcs::Application::Matching05;
    const auto settings = nnt::lcs::CreateSessionSettings(nn::lcs::NodeCountMax, app);

    // ホストとしてセッションを構築します。
    nnt::lcs::HostStarter starter(settings);
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    nn::os::ClearSystemEvent(&stateChangeEvent);
    NNT_LDN_SYNC("Join.ClientDoesNotHaveApplication.SessionOpened");
    NNT_LDN_SYNC("Join.ClientDoesNotHaveApplication.ReadyToDestroy");
}

//
// ホストが複数存在する状態で Join します。
//
TEST(Join, MultipleHost)
{
    // テストケース開始前の同期です。
    NNT_LDN_REQUIRES_NODE_COUNT(3);
    NNT_LDN_SYNC("Join.MultipleHost.Start");

    // NS, LDN, LCS ライブラリを初期化します。
    nn::lcs::Config config = nnt::lcs::CreateDefaultHostConfig();
    nnt::lcs::NsInitializer nsInitializer;
    nnt::ldn::SystemInitializer ldnInitializer;
    nnt::lcs::Initializer lcsInitializer(g_Buffer, sizeof(g_Buffer), config);

    // イベントを取得します。
    nn::os::SystemEventType stateChangeEvent;
    nn::lcs::AttachStateChangeEvent(&stateChangeEvent);
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

    // 対象のアプリケーションを指定します。
    const auto& app = nnt::lcs::Application::Matching04;
    const auto settings = nnt::lcs::CreateSessionSettings(nn::lcs::NodeCountMax, app);

    // ホストとしてセッションを構築し、接続を待ち受けます。
    nnt::lcs::HostStarter starter(settings);
    nn::lcs::SessionInfo session;
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    nn::os::ClearSystemEvent(&stateChangeEvent);
    NNT_LDN_SYNC("Join.MultipleHost.SessionOpened");
    NNT_LDN_SYNC("Join.MultipleHost.ReadyToJoin1");
    NNT_LDN_SYNC("Join.MultipleHost.Joined1");
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetSessionInfo(&session));
    ASSERT_EQ(g_TestConfig.nodeCount - 1, session.nodeCount);

    // テストの完了まで待機します。
    NNT_LDN_SYNC("Join.MultipleHost.ReadyToJoin2");
    NNT_LDN_SYNC("Join.MultipleHost.Joined2");
    NNT_LDN_SYNC("Join.MultipleHost.ReadyToDestroy");
}

//
// テストのエントリポイントです。
//
extern "C" void nnMain()
{
    // コマンドライン引数に関する設定です。
    nnt::lcs::CommandLineParserSetting setting = { };
    setting.flag = static_cast<nn::Bit32>(
        nnt::lcs::CommandLineOptionFlag_NodeCount |
        nnt::lcs::CommandLineOptionFlag_NodeIndex |
        nnt::lcs::CommandLineOptionFlag_SceneId);
    setting.nodeCountMin = 2;
    setting.nodeCountMax = nnt::ldn::SynchronizationClientCountMax;

    // コマンドライン引数を解析します。
    int argc = nn::os::GetHostArgc();
    char **argv = nn::os::GetHostArgv();
    ::testing::InitGoogleTest(&argc, argv);
    nnt::lcs::Parse(&g_TestConfig, setting, argc, argv);
    ASSERT_EQ(0, g_TestConfig.nodeIndex);

    // socket ライブラリと ns ライブラリを初期化しておきます。
    nn::socket::Initialize(g_SocketConfig);
    nnt::lcs::NsInitializer nsInitiailzer;

    // テストの準備です。
    nnt::lcs::Setup(g_TestConfig.sceneId);

    // 他のインスタンスと同期するためにテストサーバを生成します。
    nnt::ldn::HtcsSynchronizationServer server(
        g_SynchronizationBuffer, sizeof(g_SynchronizationBuffer));
    int result = server.CreateServer("nn::lcs::Integration::Matching", g_TestConfig.nodeCount - 1);

    // テストを実行します。
    int exitCode;
    if (result == nnt::ldn::SynchronizationResult_Success)
    {
        g_pServer = &server;
        exitCode = RUN_ALL_TESTS();
        server.DestroyServer();
    }
    else
    {
        exitCode = result;
    }
    nn::socket::Finalize();
    nnt::Exit(exitCode);
}
