﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/album/album_Result.h>
#include <nn/album/album_SaveScreenshot.h>
#include <nn/lcs/lcs_PrivateDebugApi.h>
#include <nn/ns/ns_Result.h>
#include <nn/ns/ns_SystemUpdateApi.h>
#include <nn/os.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_ScopeExit.h>
#include <nnt/lcs/testLcs_Assert.h>
#include <nnt/lcs/testLcs_Context.h>
#include <nnt/lcs/testLcs_Helper.h>
#include <nnt/lcs/testLcs_Initializer.h>
#include <nnt/lcs/testLcs_Request.h>
#include <nnt/lcs/testLcs_Setup.h>
#include <nnt/lcs/testLcs_TimedAction.h>
#include <nnt/ldn/testLdn_HtcsSynchronization.h>
#include <nnt/ldn/testLdn_Utility.h>
#include <nnt.h>

namespace nnt { namespace lcs { namespace
{
    // LCS の初期化用バッファです。
    NN_ALIGNAS(nn::lcs::RequiredBufferAlignment) char g_Buffer[4 * 1024 * 1024];
    NN_STATIC_ASSERT(nn::lcs::RequiredBufferSize < sizeof(g_Buffer));

    // 同期に使用するバッファです。
    NN_ALIGNAS(4096) char g_SynchronizationBuffer[16 * 1024];

    // 接続待ちのタイムアウト時間です。
    const nn::TimeSpan JoinTimeout = nn::TimeSpan::FromSeconds(5);

    // スキャン待ちのタイムアウト時間です。
    const nn::TimeSpan ScanTimeout = nn::TimeSpan::FromSeconds(4);

    // 状態変化待ちのタイムアウト時間です。
    const nn::TimeSpan StateChangeTimeout = nn::TimeSpan::FromSeconds(2);

    // 配信完了待ちのタイムアウト時間です。最悪時はこの時間内に 7 台が受信を完了する必要があります。
    const nn::TimeSpan ShareTimeout = nn::TimeSpan::FromSeconds(300);

    // コンテンツの最大サイズです。
    // const size_t ContentSizeMax = 128 * 1024U * 1024U;

    // EULA データのパスです。
    const char EulaDataPath[] = "dummy.txt";

}}} // namespace nnt::lcs::<unnamed>

namespace nnt { namespace lcs
{
    TestHelper::TestHelper(const TestConfig& config, const char* group) NN_NOEXCEPT
        : m_pSync(nullptr),
          m_TestConfig(config)
    {
        NN_ASSERT_NOT_NULL(group);
        std::strncpy(m_Group, group, nnt::ldn::SynchronizationGroupNameLengthMax);
        m_Group[nnt::ldn::SynchronizationGroupNameLengthMax] = 0;
    }

    void TestHelper::RunTest(
        const Precondition& cond, const char* prefix, int requiredNodeCount) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(prefix);
        std::strncpy(m_Prefix, prefix, PrefixLengthMax);
        m_Prefix[PrefixLengthMax] = 0;

        // 他のインスタンスと同期するためにテストグループに参加します。
        if (m_TestConfig.nodeIndex == cond.masterIndex)
        {
            auto* pServer = new nnt::ldn::HtcsSynchronizationServer(
                g_SynchronizationBuffer, sizeof(g_SynchronizationBuffer));
            ASSERT_EQ(nnt::ldn::SynchronizationResult_Success, pServer->CreateServer(
                m_Group, m_TestConfig.nodeCount - 1));
            m_pSync = pServer;
        }
        else
        {
            auto* pClient = new nnt::ldn::HtcsSynchronizationClient(
                g_SynchronizationBuffer, sizeof(g_SynchronizationBuffer));
            ASSERT_EQ(nnt::ldn::SynchronizationResult_Success, pClient->Connect(m_Group));
            m_pSync = pClient;
        }
        m_pSync->SetTimeout(ShareTimeout);

        // クライアントの端末数の指定は必須ではないため、未入力時は LdnTestBridge から取得します。
        if (m_TestConfig.nodeCount == 0)
        {
            m_TestConfig.nodeCount = static_cast<int8_t>(m_pSync->GetClientCount() + 1);
        }

        // テストを終える際にテストグループから抜けます。
        NN_UTIL_SCOPE_EXIT
        {
            delete m_pSync;
            m_pSync = nullptr;

            // FIXME: 同じグループ名を使い回している場合に、
            // 誤って破棄される直前の古いグループに再度加入してしまうことを防ぎます。
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        };

        // 端末数が必要数に達している場合にテストを実行します。
        if (requiredNodeCount <= m_TestConfig.nodeCount)
        {
            RunTestImpl(cond);
        }
    }

    void TestHelper::RunTestImpl(const Precondition& cond) NN_NOEXCEPT
    {
        // NS, LDN, LCS ライブラリを初期化します。
        nn::lcs::Config config = m_TestConfig.nodeIndex == 0 ?
            nnt::lcs::CreateDefaultHostConfig() : nnt::lcs::CreateDefaultClientConfig();
        nnt::lcs::NsInitializer nsInitializer;
        nnt::ldn::SystemInitializer ldnInitializer;
        nnt::lcs::Initializer lcsInitializer(g_Buffer, sizeof(g_Buffer), config);

        // SystemUpdateControl は占有されています。
        nn::ns::SystemUpdateControl control;
        NNT_ASSERT_RESULT_FAILURE(nn::ns::ResultAlreadyOccupied, control.Occupy());

        // イベントを取得します。
        nn::os::SystemEventType stateChangeEvent;
        nn::lcs::AttachStateChangeEvent(&stateChangeEvent);
        ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

        // ローカルパッチ配信を開始します。
        if (m_TestConfig.resume)
        {
            Resume(cond, &stateChangeEvent);
        }
        else if (m_TestConfig.nodeIndex == 0)
        {
            StartHost(cond, &stateChangeEvent);
        }
        else
        {
            StartClient(cond, &stateChangeEvent);
        }

        // SystemUpdateControl は占有されています。
        NNT_ASSERT_RESULT_FAILURE(nn::ns::ResultAlreadyOccupied, control.Occupy());

        // システム配信、パッチ配信を行います。
        const auto& master = cond.nodes[cond.masterIndex];
        const auto& self = cond.nodes[m_TestConfig.nodeIndex];
        if (self.system.isLupRequired && !m_TestConfig.resume)
        {
            if (self.system.isLupEnabled)
            {
                ReceiveSystem(cond, &stateChangeEvent);

                if (cond.scenario == Scenario_LeaveAtSuspendedReasonEulaRequired)
                {
                    // EULA 同意の中断中に離脱するシナリオの場合はセッションを終了します。
                }
                else if (cond.scenario == Scenario_LeaveAtSuspendedReasonRebootRequired)
                {
                    // 再起動要求の中断時に離脱するシナリオの場合は離脱してセッションを終了します。
                    LeaveSession(&stateChangeEvent);
                }
                else
                {
                    // セッションコンテキストを保存して一時離脱します。
                    nn::lcs::SessionContext context;
                    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetSessionContext(&context));
                    NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::SaveSessionContext(context));
                    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::SuspendSession());

                    if (cond.scenario == Scenario_ResumeWithoutReboot)
                    {
                        // 再起動せずに再合流するシナリオの場合は再起動せずに再合流します。
                        NNT_ASSERT_RESULT_SUCCESS(nn::lcs::ResumeSession(context));
                        ASSERT_EQ(nn::lcs::State_Suspended, nn::lcs::GetState());
                        ASSERT_EQ(nn::lcs::SuspendedReason_RebootRequired, nn::lcs::GetSuspendedReason());

                        // 再び一時離脱します。
                        NNT_ASSERT_RESULT_SUCCESS(nn::lcs::SuspendSession());
                    }

                    // テスト終了後に再起動してテストを再実行させます。
                    if (!master.system.hasApplicationDeliveryProtocolBackwardCompatibility)
                    {
                        nnt::lcs::RequestApplicationDeliveryProtocolVersionUpdate(
                            master.system.applicationDeliveryProtocolVersion);
                    }
                    nnt::lcs::RequestReboot();
                    nnt::lcs::RequestResume();
                }
            }
            else
            {
                TimedWaitFailure(&stateChangeEvent,
                    nn::lcs::ContentsShareFailureReason_CannotUpdateSystem);
            }
        }
        else if ((!cond.isPatchShareEnabled) ||
            (cond.application.latestVersion < self.application.requiredVersion))
        {
            if (self.application.version < cond.application.targetVersion)
            {
                // 送信者のシステム更新ができないためか、
                // 起動必須バージョンを満たしていないためパッチを受信できません。
                TimedWaitFailure(&stateChangeEvent,
                    nn::lcs::ContentsShareFailureReason_NotExistDownloadContents);
            }
            else
            {
                TimedWaitComplete(cond, &stateChangeEvent);
            }
        }
        else if (self.application.version < cond.application.latestVersion)
        {
            // ストレージ空き領域が不足しているため中断します。
            if (cond.nodes[m_TestConfig.nodeIndex].isBuiltInUserStorageNotEnough && !m_TestConfig.resume)
            {
                // ストレージ空き領域が不足している時の Suspended 状態になるまで待機します。
                nnt::lcs::TimedWaitTransition(nn::lcs::State_Suspended, &stateChangeEvent, ShareTimeout);
                ASSERT_EQ(nn::lcs::SuspendedReason_StorageSpaceNotEnough,
                    nn::lcs::GetSuspendedReason());

                // セッションコンテキストを保存して一時離脱します。
                nn::lcs::SessionContext context;
                NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetSessionContext(&context));
                NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::SaveSessionContext(context));
                NNT_ASSERT_RESULT_SUCCESS(nn::lcs::SuspendSession());

                // 空き領域確保用テストアプリケーションが指定されている場合、
                // テスト終了後に空き領域確保用テストアプリケーションを削除してテストを再実行させます。
                if (cond.applicationForClean.id != nn::ncm::ApplicationId::GetInvalidId())
                {
                    char argument[256];
                    nn::util::SNPrintf(argument, sizeof(argument), "application uninstall 0x%llx", cond.applicationForClean.id.value);
                    nnt::lcs::RequestDevMenuCommandSystem(argument);
                    nnt::lcs::RequestResume();
                }

                return;
            }

            ReceivePatch(cond, &stateChangeEvent);

            // 配信を開始してからストレージの空き領域を不足させるシナリオを実行します、
            if (cond.scenario == Scenario_StorageSpaceNotEnoughAfterStartShare)
            {
                nn::album::Initialize();
                NN_UTIL_SCOPE_EXIT
                {
                    nn::album::Finalize();
                };

                // スクリーンショットのファイルサイズが小さくなり過ぎないように適当な画像データを生成します。
                static struct Pixel { uint8_t r; uint8_t g; uint8_t b; uint8_t a; } s_Image[1280 * 720];
                for (int y = 0; y < 720; ++y)
                {
                    for (int x = 0; x < 1280; ++x)
                    {
                        s_Image[y * 1280 + x] = {
                            static_cast<uint8_t>(y *  13 + x * 251),
                            static_cast<uint8_t>(y *  97 + x *   3),
                            static_cast<uint8_t>(y * 163 * x *  15),
                            static_cast<uint8_t>(y *   7 + x * 137)
                        };
                    }
                }

                // パッチの受信ができなくなる程度にストレージを埋めます。
                for (int i = 10; i--;)
                {
                    auto result = nn::album::SaveScreenshot(s_Image, sizeof(s_Image), nn::album::ImageSize_1280x720, nn::album::AlbumReportOption_ReportAlways);
                    if (nn::album::ResultAlbumFull::Includes(result))
                    {
                        break;
                    }
                    NNT_ASSERT_RESULT_SUCCESS(result);
                }

                TimedWaitFailure(&stateChangeEvent, nn::lcs::ContentsShareFailureReason_NotEnoughStorageSpace);
            }
            else
            {
                TimedWaitComplete(cond, &stateChangeEvent);
            }
        }
        else
        {
            TimedWaitComplete(cond, &stateChangeEvent);
        }

        // SystemUpdateControl は占有されています。
        NNT_ASSERT_RESULT_FAILURE(nn::ns::ResultAlreadyOccupied, control.Occupy());
    }  // NOLINT(impl/function_size)

    void TestHelper::Resume(
        const nnt::lcs::Precondition& cond, nn::os::SystemEventType* pEvent) NN_NOEXCEPT
    {
        // 前回のセッションコンテキストを読み込みます。
        nn::lcs::SessionContext context;
        NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::LoadSessionContext(&context));

        // セッションを再開します。
        ASSERT_FALSE(nn::os::TryWaitSystemEvent(pEvent));
        NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::TimedResumeSession(context, JoinTimeout));
        ASSERT_TRUE(nn::os::TryWaitSystemEvent(pEvent));
        nn::os::ClearSystemEvent(pEvent);
    }

    //
    // ホストとしてローカルコンテンツ配信を開始します。
    //
    void TestHelper::StartHost(
        const nnt::lcs::Precondition& cond, nn::os::SystemEventType* pEvent) NN_NOEXCEPT
    {
        auto& stateChangeEvent = *pEvent;

        // セッションを構築します。
        const auto settings = nnt::lcs::CreateSessionSettings(
            nn::lcs::NodeCountMax, cond.application.id);
        ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
        NNT_ASSERT_RESULT_SUCCESS(nn::lcs::OpenSession(settings));
        ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
        nn::os::ClearSystemEvent(&stateChangeEvent);
        ASSERT_EQ(nn::lcs::State_Opened, nn::lcs::GetState());
        Sync("SessionOpened");

        // クライアントの接続を待機します。
        Sync("Joined");
        ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
        nn::os::ClearSystemEvent(&stateChangeEvent);
        Sync("ReadyToStart");

        // コンテンツの配信を開始します。
        ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
        NNT_ASSERT_RESULT_SUCCESS(nn::lcs::StartContentsShare());
        ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
        nn::os::ClearSystemEvent(&stateChangeEvent);
        Sync("Started");
    }

    //
    // クライアントとしてローカルコンテンツ配信を開始します。
    //
    void TestHelper::StartClient(
        const nnt::lcs::Precondition& cond, nn::os::SystemEventType* pEvent) NN_NOEXCEPT
    {
        auto& stateChangeEvent = *pEvent;

        // セッションの構築を待機します。
        Sync("SessionOpened");

        // インデックスが若い順に、1 人ずつセッションに参加します。
        int retryCount = m_TestConfig.nodeCount * 10;
        for (int i = 0; i < retryCount; ++i)
        {
            // セッションをスキャンします。
            nn::lcs::SessionInfo sessions[nn::lcs::ScanResultCountMax];
            int count;
            int index;
            NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::TimedScan(
                sessions, &count, &index, nn::lcs::ScanResultCountMax,
                nnt::lcs::DefaultHostUserName, ScanTimeout));

            // 現在の端末数を確認し、自分の順番が回ってきていたら接続します。
            const auto& session = sessions[index];
            if (session.nodeCount == m_TestConfig.nodeIndex)
            {
                ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
                NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::TimedJoin(session, JoinTimeout));
                ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
                nn::os::ClearSystemEvent(&stateChangeEvent);
                break;
            }
        }
        ASSERT_EQ(nn::lcs::State_Joined, nn::lcs::GetState());
        Sync("Joined");

        // 他のクライアントの接続がすぐに反映されない可能性があるため、一定時間待機します。
        nn::os::SleepThread(StateChangeTimeout);
        nn::os::ClearSystemEvent(&stateChangeEvent);
        ASSERT_EQ(nn::lcs::State_Joined, nn::lcs::GetState());
        Sync("ReadyToStart");

        // コンテンツ配信の開始まで待機します。
        Sync("Started");
        ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&stateChangeEvent, StateChangeTimeout));
        nn::os::ClearSystemEvent(&stateChangeEvent);
    }

    //
    // システムの受信を待機します。
    //
    void TestHelper::ReceiveSystem(
        const nnt::lcs::Precondition& cond, nn::os::SystemEventType* pEvent) NN_NOEXCEPT
    {
        const auto& master = cond.nodes[cond.masterIndex];

        // Application Delivery Info に互換性が無い場合、Suspended 状態になるまで待機します。
        if (!master.system.hasApplicationDeliveryProtocolBackwardCompatibility)
        {
            nnt::lcs::TimedWaitTransition(nn::lcs::State_Suspended, pEvent, ShareTimeout);
            ASSERT_EQ(nn::lcs::SuspendedReason_IncompatibleContentsInfo,
                nn::lcs::GetSuspendedReason());
            NNT_ASSERT_RESULT_SUCCESS(nn::lcs::ResumeContentsShare());
        }

        // アプリケーションの終了を要求する Suspended 状態になるまで待機します。
        nnt::lcs::TimedWaitTransition(nn::lcs::State_Suspended, pEvent, ShareTimeout);
        ASSERT_EQ(nn::lcs::SuspendedReason_NeedTerminateApplication,
            nn::lcs::GetSuspendedReason());
        NNT_ASSERT_RESULT_SUCCESS(nn::lcs::ResumeContentsShare());

        // Eula の確認を要求する Suspended 状態になるまで待機します。
        nnt::lcs::TimedWaitTransition(nn::lcs::State_Suspended, pEvent, ShareTimeout);
        ASSERT_EQ(nn::lcs::SuspendedReason_EulaRequired, nn::lcs::GetSuspendedReason());
        ValidateEula(master.system.hasEula);
        // EULA　確認中に離脱するシナリオの場合は離脱してシステムの受信を終了します。
        if (cond.scenario == Scenario_LeaveAtSuspendedReasonEulaRequired)
        {
            LeaveSession(pEvent);
            return;
        }
        NNT_ASSERT_RESULT_SUCCESS(nn::lcs::ResumeContentsShare());

        // 再起動を要求する Suspended 状態になるまで待機します。
        nnt::lcs::TimedWaitTransition(nn::lcs::State_Suspended, pEvent, ShareTimeout);
        ASSERT_EQ(nn::lcs::SuspendedReason_RebootRequired, nn::lcs::GetSuspendedReason());

        // SystemUpdateControl は占有されています。
        nn::ns::SystemUpdateControl control;
        NNT_ASSERT_RESULT_FAILURE(nn::ns::ResultAlreadyOccupied, control.Occupy());

        // この時点でシステムの受信が完了しています。
        nn::lcs::ContentsInfo contents[nn::lcs::DownloadableContentsCountMax];
        int downloadedContentsCount;
        NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetDownloadedContents(
            contents, &downloadedContentsCount, nn::lcs::DownloadableContentsCountMax));
        ASSERT_EQ(1, downloadedContentsCount);
        ASSERT_TRUE(contents[0].contentsFlag.Test<nn::lcs::ContentsType::System>());
        ASSERT_FALSE(contents[0].contentsFlag.Test<nn::lcs::ContentsType::Application>());
        ASSERT_FALSE(contents[0].contentsFlag.Test<nn::lcs::ContentsType::Patch>());
    }

    //
    // パッチの受信を待機します。
    //
    void TestHelper::ReceivePatch(
        const nnt::lcs::Precondition& cond, nn::os::SystemEventType* pEvent) NN_NOEXCEPT
    {
        // アプリケーションの終了を要求する Suspended 状態になるまで待機します。
        nnt::lcs::TimedWaitTransition(nn::lcs::State_Suspended, pEvent, ShareTimeout);
        ASSERT_EQ(nn::lcs::SuspendedReason_NeedTerminateApplication,
            nn::lcs::GetSuspendedReason());
        NNT_ASSERT_RESULT_SUCCESS(nn::lcs::ResumeContentsShare());
    }

    //
    // コンテンツ配信の完了まで待機します。
    //
    void TestHelper::TimedWaitComplete(
        const nnt::lcs::Precondition& cond, nn::os::SystemEventType* pEvent) NN_NOEXCEPT
    {
        // コンテンツ配信の完了まで待機します。
        nnt::lcs::TimedWaitTransition(nn::lcs::State_Completed, pEvent, ShareTimeout);
        nn::os::ClearSystemEvent(pEvent);

        // SystemUpdateControl は占有されています。
        nn::ns::SystemUpdateControl control;
        NNT_ASSERT_RESULT_FAILURE(nn::ns::ResultAlreadyOccupied, control.Occupy());

        // この端末が受信したコンテンツの一覧を取得します。
        nn::lcs::ContentsInfo contents[nn::lcs::DownloadableContentsCountMax];
        int downloadedContentsCount;
        NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetDownloadedContents(
            contents, &downloadedContentsCount, nn::lcs::DownloadableContentsCountMax));

        // 要求した通り、コンテンツを受信できていることを確認します。
        const auto& self = cond.nodes[m_TestConfig.nodeIndex];
        const nn::lcs::ContentsInfo* pSystem = nullptr;
        const nn::lcs::ContentsInfo* pApp = nullptr;
        if (cond.isPatchShareEnabled)
        {
            if (self.system.isLupRequired &&
                self.application.version < cond.application.latestVersion)
            {
                ASSERT_EQ(2, downloadedContentsCount);
                pSystem = &contents[0];
                pApp = &contents[1];
            }
            else if (self.system.isLupRequired)
            {
                ASSERT_EQ(1, downloadedContentsCount);
                pSystem = &contents[0];
            }
            else if (self.application.version < cond.application.latestVersion)
            {
                ASSERT_EQ(1, downloadedContentsCount);
                pApp = &contents[0];
            }
        }
        else
        {
            ASSERT_EQ(0, downloadedContentsCount);
        }
        if (pSystem != nullptr)
        {
            ASSERT_TRUE(pSystem->contentsFlag.Test<nn::lcs::ContentsType::System>());
            ASSERT_FALSE(pSystem->contentsFlag.Test<nn::lcs::ContentsType::Application>());
            ASSERT_FALSE(pSystem->contentsFlag.Test<nn::lcs::ContentsType::Patch>());
        }
        if (pApp != nullptr)
        {
            ASSERT_FALSE(pApp->contentsFlag.Test<nn::lcs::ContentsType::System>());
            ASSERT_FALSE(pApp->contentsFlag.Test<nn::lcs::ContentsType::Application>());
            ASSERT_TRUE(pApp->contentsFlag.Test<nn::lcs::ContentsType::Patch>());
            ASSERT_EQ(0U, pApp->attributeFlag);
            ASSERT_EQ(cond.application.id, pApp->applicationId);
            ASSERT_STREQ(cond.application.latestDisplayVersion, pApp->displayVersion);
        }

        // 自身の役割を確認しておきます。
        if (m_TestConfig.nodeIndex == cond.masterIndex)
        {
            ASSERT_EQ(nn::lcs::InnerRole_Master, nn::lcs::GetInnerRole());
        }
        else
        {
            ASSERT_EQ(nn::lcs::InnerRole_Slave, nn::lcs::GetInnerRole());
        }

        // セッションを破棄します。
        ASSERT_EQ(nn::lcs::State_Completed, nn::lcs::GetState());
        ASSERT_FALSE(nn::os::TryWaitSystemEvent(pEvent));
        NNT_ASSERT_RESULT_SUCCESS(nn::lcs::LeaveSession());
        ASSERT_TRUE(nn::os::TryWaitSystemEvent(pEvent));
        nn::os::ClearSystemEvent(pEvent);
        ASSERT_EQ(nn::lcs::State_Initialized, nn::lcs::GetState());
        Sync("Destroyed");
    }

    //
    // コンテンツ配信の完了まで待機します。
    //
    void TestHelper::TimedWaitFailure(
        nn::os::SystemEventType* pEvent, nn::lcs::ContentsShareFailureReason reason) NN_NOEXCEPT
    {
        // コンテンツ配信が失敗するまで待機します。
        nnt::lcs::TimedWaitTransition(
            nn::lcs::State_ContentsShareFailed, pEvent, ShareTimeout);
        nn::os::ClearSystemEvent(pEvent);
        ASSERT_EQ(reason, nn::lcs::GetContentsShareFailureReason());

        // SystemUpdateControl は占有されています。
        nn::ns::SystemUpdateControl control;
        NNT_ASSERT_RESULT_FAILURE(nn::ns::ResultAlreadyOccupied, control.Occupy());

        // セッションを破棄します。
        ASSERT_FALSE(nn::os::TryWaitSystemEvent(pEvent));
        NNT_ASSERT_RESULT_SUCCESS(nn::lcs::LeaveSession());
        ASSERT_TRUE(nn::os::TryWaitSystemEvent(pEvent));
        nn::os::ClearSystemEvent(pEvent);
        ASSERT_EQ(nn::lcs::State_Initialized, nn::lcs::GetState());
        Sync("Destroyed");
    }

    //
    // セッションから離脱してコンテンツ配信を終了します。
    //
    void TestHelper::LeaveSession(
        nn::os::SystemEventType* pEvent) NN_NOEXCEPT
    {
        // SystemUpdateControl は占有されています。
        nn::ns::SystemUpdateControl control;
        NNT_ASSERT_RESULT_FAILURE(nn::ns::ResultAlreadyOccupied, control.Occupy());

        // セッションを破棄します。
        ASSERT_FALSE(nn::os::TryWaitSystemEvent(pEvent));
        NNT_ASSERT_RESULT_SUCCESS(nn::lcs::LeaveSession());
        ASSERT_TRUE(nn::os::TryWaitSystemEvent(pEvent));
        nn::os::ClearSystemEvent(pEvent);
        ASSERT_EQ(nn::lcs::State_Initialized, nn::lcs::GetState());
        Sync("Destroyed");
    }

    //
    // EULA の検証を行います。
    //
    void TestHelper::ValidateEula(bool hasEula) NN_NOEXCEPT
    {
        if (hasEula)
        {
            size_t dataSize;

            // 存在しないデータパスを指定すると失敗します。
            NNT_ASSERT_RESULT_FAILURE(nn::lcs::ResultBadDataPath, nn::lcs::GetDownloadedEulaDataSize(&dataSize, "neko.cat"));

            // Eula コンテンツのデータサイズが取得できることを確認します。
            NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetDownloadedEulaDataSize(&dataSize, EulaDataPath));

            auto bufferSize = dataSize + 10;
            auto buffer = new uint8_t[bufferSize];

            size_t readSize;

            // バッファサイズがデータサイズより小さいと失敗します。
            NNT_ASSERT_RESULT_FAILURE(nn::lcs::ResultBufferNotEnough, nn::lcs::GetDownloadedEulaData(&readSize, buffer, dataSize - 1, EulaDataPath));

            // 存在しないデータパスを指定すると失敗します。
            NNT_ASSERT_RESULT_FAILURE(nn::lcs::ResultBadDataPath, nn::lcs::GetDownloadedEulaData(&readSize, buffer, bufferSize, "nya.meow"));

            // EULA データが正しく読めることを確認します。
            NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetDownloadedEulaData(&readSize, buffer, bufferSize, EulaDataPath));
            ASSERT_EQ(readSize, dataSize);

            delete[] buffer;
        }
        else
        {
            size_t dataSize;
            // Eula を受信していないため失敗します。
            NNT_ASSERT_RESULT_FAILURE(nn::lcs::ResultNotExsistEula, nn::lcs::GetDownloadedEulaDataSize(&dataSize, EulaDataPath));

            char buffer[1];
            size_t readSize;
            // Eula を受信していないため失敗します。
            NNT_ASSERT_RESULT_FAILURE(nn::lcs::ResultNotExsistEula, nn::lcs::GetDownloadedEulaData(&readSize, buffer, sizeof(buffer), EulaDataPath));
        }
    }

    void TestHelper::Sync(const char* keyword) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(keyword);
        char buffer[nnt::ldn::SynchronizationKeywordLengthMax + 1];
        nn::util::SNPrintf(buffer, sizeof(buffer), "%s.%s", m_Prefix, keyword);
        ASSERT_EQ(nnt::ldn::SynchronizationResult_Success, m_pSync->Synchronize(buffer));
    }

}} // namespace nnt::lcs
