﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/lmem/lmem_Common.h>
#include <nn/lmem/lmem_FrameHeap.h>
#include "testLMem_Util.h"

namespace {

const size_t g_ThreadStackSize = 8192;
NN_ALIGNAS(4096) char g_ThreadStack[g_ThreadStackSize];
nn::os::ThreadType g_Thread;
nn::os::EventType g_Event;

/**
 * @brief   FrameHeap を繰り返し作成します。
 */
void CreateFrameHeapConstantly(void* arg)
{
    NN_UNUSED(arg);

    const size_t HeapSize = nn::os::MemoryBlockUnitSize;
    void* heapAddr;
    nn::lmem::HeapHandle heapHandle;

    heapAddr = std::malloc(HeapSize);

    // 繰り返しヒープを作成
    for(int i = 0; i < 1024; ++i)
    {
        heapHandle = nn::lmem::CreateFrameHeap(heapAddr, HeapSize, nn::lmem::CreationOption_NoOption);
        ASSERT_NOT_NULL(heapHandle);     // ヒープ作成に成功したか
    }

    nn::lmem::DestroyFrameHeap(heapHandle);
    std::free(heapAddr);

    nn::os::SignalEvent(&g_Event);
}

/**
 * @brief   フレームヒープのテストで利用するテストフィクスチャです。
 */
class FrameHeapBasicTest :  public ::testing::TestWithParam<bool>
{
protected:

    const size_t m_HeapSize;                // グローバルヒープのサイズ
    const int m_MinAlignment;               // ユニットヒープの最小アライメント
    uintptr_t m_Heap;                       // グローバルヒープのポインタ
    size_t allocateSize;                    // 確保するメモリサイズ
    nn::lmem::HeapHandle m_HeapHandle;      // ユニットヒープのヒープハンドル
    nn::lmem::HeapCommonHead m_HeapHead;    // ヒープ管理領域（ヒープ共通ヘッダ）
    bool m_HasHeapHeadInternally;           // ヒープの管理領域をヒープ内部に持つかどうか
    size_t m_HeadSize;                      // ヒープ内部に確保されている管理領域のサイズ

    FrameHeapBasicTest() : m_HeapSize(nn::os::MemoryBlockUnitSize), m_MinAlignment(4), allocateSize(128){}

    /**
     * @brief テスト開始時に毎回呼び出される関数です。
     */
    virtual void SetUp()
    {
        m_Heap = reinterpret_cast<uintptr_t>(std::malloc(m_HeapSize));
    }

    /**
     * @brief パラメータ化されたテストで用いられるクラス変数を初期化します。
     */
    void SetUpParam()
    {
        m_HasHeapHeadInternally = GetParam();

        if(m_HasHeapHeadInternally)
        {
            m_HeadSize = sizeof(nn::lmem::HeapCommonHead);
        }
        else
        {
            // ヒープ管理領域を外部に確保するため、ヒープ内部には管理領域はない
            m_HeadSize = 0;
        }
    }

    /**
     * @brief テスト終了時に毎回呼び出される関数です。
     */
    virtual void TearDown()
    {
        std::free(reinterpret_cast<void*>(m_Heap));    // グローバルヒープへの返却
    }

};

}   // unnamed namespace

// =============
// テスト
// =============

// 単純なヒープの作成
// ヒープを作成します。ヒープの作成が成功（ハンドルが nullptr でない）なら OK
TEST_F(FrameHeapBasicTest, SimpleCreateHeap)
{
    // ヒープの作成
    m_HeapHandle = nn::lmem::CreateFrameHeap(reinterpret_cast<void*>(m_Heap), m_HeapSize, nn::lmem::CreationOption_NoOption);
    ASSERT_NOT_NULL(m_HeapHandle);     // ヒープ作成に成功したか

    // ヒープの破棄
    nn::lmem::DestroyFrameHeap(m_HeapHandle);
}

// ヒープの作成とメモリブロックの確保
// メモリブロックの確保が、正しいアドレス領域に意図した状態で行われていれば、 OK
TEST_P(FrameHeapBasicTest, CreateAndAllocate)
{
    void* pMemoryAddress;

    SetUpParam();

    // ヒープの作成とメモリ確保
    if(m_HasHeapHeadInternally)
    {
        // ヒープ管理領域をヒープ内部に持つ場合
        m_HeapHandle = nn::lmem::CreateFrameHeap(reinterpret_cast<void*>(m_Heap), m_HeapSize, nn::lmem::CreationOption_NoOption);
    }
    else
    {
        // ヒープ管理領域を静的に確保した場合
        m_HeapHandle = nn::lmem::CreateFrameHeap(reinterpret_cast<void*>(m_Heap), m_HeapSize, nn::lmem::CreationOption_NoOption, &m_HeapHead);
    }
    ASSERT_NOT_NULL(m_HeapHandle);

    pMemoryAddress = nn::lmem::AllocateFromFrameHeap(m_HeapHandle, allocateSize, m_MinAlignment);
    ASSERT_NOT_NULL(pMemoryAddress);  // メモリ確保に成功したか
    EXPECT_TRUE(nn::lmem::HasAddress(m_HeapHandle, pMemoryAddress));

    // 割り当てられたメモリに値を正常に書き込めるか確認
    EXPECT_TRUE(WriteNumericalSequence(pMemoryAddress, allocateSize));
    EXPECT_TRUE(HasNumericalSequence(pMemoryAddress, allocateSize));

    // サイズチェック
    EXPECT_EQ(nn::lmem::GetFrameHeapAllocatableSize(m_HeapHandle, m_MinAlignment), m_HeapSize - allocateSize - m_HeadSize);
    ASSERT_NE(nn::lmem::AllocateFromFrameHeap(m_HeapHandle, allocateSize, m_MinAlignment), nullptr);
    EXPECT_EQ(nn::lmem::GetFrameHeapAllocatableSize(m_HeapHandle, m_MinAlignment), m_HeapSize - (allocateSize * 2) - m_HeadSize);

    nn::lmem::FreeToFrameHeap(m_HeapHandle, nn::lmem::FreeMode_All);
    nn::lmem::DestroyFrameHeap(m_HeapHandle);
}

// メモリブロックのリサイズ
// メモリブロックを正しく拡張、縮小できれば、OK
TEST_F(FrameHeapBasicTest, ResizeBlock)
{
    allocateSize = 64;
    void* pMemoryAddress;

    // ヒープの作成とメモリ確保
    m_HeapHandle = nn::lmem::CreateFrameHeap(reinterpret_cast<void*>(m_Heap), m_HeapSize, nn::lmem::CreationOption_NoOption);
    ASSERT_NOT_NULL(m_HeapHandle);

    pMemoryAddress = nn::lmem::AllocateFromFrameHeap(m_HeapHandle, allocateSize);
    ASSERT_NOT_NULL(pMemoryAddress);  // メモリ確保に成功したか

    // 割り当てられたメモリに値を正常に書き込めるか確認
    EXPECT_TRUE(WriteNumericalSequence(pMemoryAddress, allocateSize));
    EXPECT_TRUE(HasNumericalSequence(pMemoryAddress, allocateSize));

    // リサイズ（縮小）
    EXPECT_EQ(nn::lmem::ResizeFrameHeapBlock(m_HeapHandle, pMemoryAddress, allocateSize / 2), allocateSize / 2);

    // リサイズ後も値が読めるか
    EXPECT_TRUE(HasNumericalSequence(pMemoryAddress, allocateSize / 2));

    // サイズを変えないリサイズ
    EXPECT_EQ(nn::lmem::ResizeFrameHeapBlock(m_HeapHandle, pMemoryAddress, allocateSize / 2), allocateSize / 2);
    // 元の大きさに戻す
    EXPECT_EQ(nn::lmem::ResizeFrameHeapBlock(m_HeapHandle, pMemoryAddress, allocateSize), allocateSize);

    // 再度 Read/Write テスト
    EXPECT_TRUE(WriteNumericalSequence(pMemoryAddress, allocateSize));
    EXPECT_TRUE(HasNumericalSequence(pMemoryAddress, allocateSize));

    // リサイズ（拡張）
    EXPECT_EQ(nn::lmem::ResizeFrameHeapBlock(m_HeapHandle, pMemoryAddress, allocateSize * 2), allocateSize * 2);

    // リサイズ後も値が読めるか
    EXPECT_TRUE(HasNumericalSequence(pMemoryAddress, allocateSize));

    // サイズを変えないリサイズ
    EXPECT_EQ(nn::lmem::ResizeFrameHeapBlock(m_HeapHandle, pMemoryAddress, allocateSize * 2), allocateSize * 2);
    // 元の大きさに戻す
    EXPECT_EQ(nn::lmem::ResizeFrameHeapBlock(m_HeapHandle, pMemoryAddress, allocateSize), allocateSize);

    // 再度 Read/Write テスト
    EXPECT_TRUE(WriteNumericalSequence(pMemoryAddress, allocateSize));
    EXPECT_TRUE(HasNumericalSequence(pMemoryAddress, allocateSize));

    nn::lmem::FreeToFrameHeap(m_HeapHandle, nn::lmem::FreeMode_All);
    nn::lmem::DestroyFrameHeap(m_HeapHandle);
}

// ヒープの縮小
// 前方／後方からの Adjust が正常に行われれば、OK
TEST_P(FrameHeapBasicTest, Adjust)
{
    void* pMemoryAddress;
    nn::lmem::MemoryRange range;

    SetUpParam();

    // ヒープの作成
    if(m_HasHeapHeadInternally)
    {
        // ヒープ管理領域をヒープ内部に持つ場合
        m_HeapHandle = nn::lmem::CreateFrameHeap(reinterpret_cast<void*>(m_Heap), m_HeapSize, nn::lmem::CreationOption_NoOption);
    }
    else
    {
        // ヒープ管理領域を静的に確保した場合
        m_HeapHandle = nn::lmem::CreateFrameHeap(reinterpret_cast<void*>(m_Heap), m_HeapSize, nn::lmem::CreationOption_NoOption, &m_HeapHead);
    }
    ASSERT_NOT_NULL(m_HeapHandle);

    if(m_HasHeapHeadInternally)
    {
        // ヒープ管理領域があるため、前方からの縮小は失敗するはず
        range = nn::lmem::AdjustFrameHeap(m_HeapHandle, nn::lmem::AdjustMode_Head);
        EXPECT_EQ(range.size, 0);
        // 前方からのメモリ確保
        pMemoryAddress = nn::lmem::AllocateFromFrameHeap(m_HeapHandle, allocateSize, m_MinAlignment);
    }
    else
    {
        // 後方からのメモリ確保
        pMemoryAddress = nn::lmem::AllocateFromFrameHeap(m_HeapHandle, allocateSize, -m_MinAlignment);
    }
    ASSERT_NOT_NULL(pMemoryAddress);

    // 確保ブロックに書き込み
    EXPECT_TRUE(WriteNumericalSequence(pMemoryAddress, allocateSize));

    // 縮小失敗ケースと成功ケースを試す
    if(m_HasHeapHeadInternally)
    {
        // 前方からの縮小は失敗するはず
        range = nn::lmem::AdjustFrameHeap(m_HeapHandle, nn::lmem::AdjustMode_Head);
        EXPECT_EQ(range.size, 0);
        // ヒープの後方からの縮小
        range = nn::lmem::AdjustFrameHeap(m_HeapHandle, nn::lmem::AdjustMode_Tail);
    }
    else
    {
        range = nn::lmem::AdjustFrameHeap(m_HeapHandle, nn::lmem::AdjustMode_Tail);
        EXPECT_EQ(range.size, 0);
        // ヒープの前方からの縮小
        range = nn::lmem::AdjustFrameHeap(m_HeapHandle, nn::lmem::AdjustMode_Head);
    }

    // 縮小後のサイズが算出結果と等しい
    EXPECT_EQ(range.size, m_HeapSize - allocateSize - m_HeadSize);
    EXPECT_EQ(nn::lmem::GetTotalSize(m_HeapHandle), allocateSize + m_HeadSize);

    // Read テスト
    EXPECT_TRUE(HasNumericalSequence(pMemoryAddress, allocateSize));

    // 縮小後は空き領域がなくなる
    EXPECT_EQ(nn::lmem::GetFrameHeapAllocatableSize(m_HeapHandle, m_MinAlignment), 0);

    nn::lmem::FreeToFrameHeap(m_HeapHandle, nn::lmem::FreeMode_All);

    // 何も確保していない状況で Adjust するとヒープ領域がなくなる
    if(m_HasHeapHeadInternally)
    {
        range = nn::lmem::AdjustFrameHeap(m_HeapHandle, nn::lmem::AdjustMode_Tail);
        EXPECT_EQ(nn::lmem::GetTotalSize(m_HeapHandle), m_HeadSize);
    }
    else
    {
        range = nn::lmem::AdjustFrameHeap(m_HeapHandle, nn::lmem::AdjustMode_Head);
        EXPECT_EQ(nn::lmem::GetTotalSize(m_HeapHandle), 0);
    }
    EXPECT_NE(range.size, static_cast<size_t>(0));
    EXPECT_EQ(nn::lmem::GetFrameHeapAllocatableSize(m_HeapHandle, m_MinAlignment), 0);

    nn::lmem::DestroyFrameHeap(m_HeapHandle);
}

// ヒープ状態の保存
// ヒープの状態を正しく保存でき、正しく復元ができれば、OK
TEST_P(FrameHeapBasicTest, State)
{
    allocateSize = 64;
    void* pMemoryAddress;
    nn::lmem::FrameHeapState state;

    SetUpParam();

    // ヒープの作成とメモリ確保
    if(m_HasHeapHeadInternally)
    {
        m_HeapHandle = nn::lmem::CreateFrameHeap(reinterpret_cast<void*>(m_Heap), m_HeapSize, nn::lmem::CreationOption_NoOption);
    }
    else
    {
        m_HeapHandle = nn::lmem::CreateFrameHeap(reinterpret_cast<void*>(m_Heap), m_HeapSize, nn::lmem::CreationOption_NoOption, &m_HeapHead);
    }
    ASSERT_NOT_NULL(m_HeapHandle);
    pMemoryAddress = nn::lmem::AllocateFromFrameHeap(m_HeapHandle, allocateSize, m_MinAlignment);
    ASSERT_NOT_NULL(pMemoryAddress);
    nn::lmem::AllocateFromFrameHeap(m_HeapHandle, allocateSize, m_MinAlignment);

    // 確保ブロックに書き込み
    EXPECT_TRUE(WriteNumericalSequence(pMemoryAddress, allocateSize));

    state = nn::lmem::GetFrameHeapState(m_HeapHandle);    // 状態を保存

    nn::lmem::AllocateFromFrameHeap(m_HeapHandle, allocateSize, m_MinAlignment);
    nn::lmem::AllocateFromFrameHeap(m_HeapHandle, allocateSize, m_MinAlignment);

    // Read テスト
    EXPECT_TRUE(HasNumericalSequence(pMemoryAddress, allocateSize));

    nn::lmem::RestoreFrameHeapState(m_HeapHandle, state); // 状態を復元

    // Read テスト
    EXPECT_TRUE(HasNumericalSequence(pMemoryAddress, allocateSize));

    // サイズチェック
    // 保存した状態に戻れているか
    EXPECT_EQ(nn::lmem::GetFrameHeapAllocatableSize(m_HeapHandle, m_MinAlignment), m_HeapSize - 2 * allocateSize - m_HeadSize);

    nn::lmem::FreeToFrameHeap(m_HeapHandle, nn::lmem::FreeMode_All);
    nn::lmem::DestroyFrameHeap(m_HeapHandle);
}

// サイズ取得関数のテスト
// 取得したサイズと、計算で求めたサイズが一致していれば、OK
TEST_P(FrameHeapBasicTest, Size)
{
    void* pMemoryAddress;

    SetUpParam();

    // ヒープの作成
    if(m_HasHeapHeadInternally)
    {
        m_HeapHandle = nn::lmem::CreateFrameHeap(reinterpret_cast<void*>(m_Heap), m_HeapSize, nn::lmem::CreationOption_ZeroClear);
    }
    else
    {
        m_HeapHandle = nn::lmem::CreateFrameHeap(reinterpret_cast<void*>(m_Heap), m_HeapSize, nn::lmem::CreationOption_ZeroClear, &m_HeapHead);
    }
    ASSERT_NOT_NULL(m_HeapHandle);

    // サイズのテスト
    size_t allocatableSize = nn::lmem::GetFrameHeapAllocatableSize(m_HeapHandle, m_MinAlignment);
    size_t totalSize = nn::lmem::GetTotalSize(m_HeapHandle);
    EXPECT_EQ(allocatableSize, m_HeapSize - m_HeadSize);
    EXPECT_EQ(totalSize, m_HeapSize);

    // メモリブロックの確保
    pMemoryAddress = nn::lmem::AllocateFromFrameHeap(m_HeapHandle, allocateSize, m_MinAlignment);
    ASSERT_NOT_NULL(pMemoryAddress);

    // サイズのテスト
    allocatableSize = nn::lmem::GetFrameHeapAllocatableSize(m_HeapHandle, m_MinAlignment);
    EXPECT_EQ(totalSize, allocatableSize + allocateSize + m_HeadSize);

    nn::lmem::FreeToFrameHeap(m_HeapHandle, nn::lmem::FreeMode_All);
    nn::lmem::DestroyFrameHeap(m_HeapHandle);
}

// 解放モード
// 前方領域の解放、後方領域の解放が正しく行われていれば、OK
TEST_P(FrameHeapBasicTest, FreeMode)
{
    void* pMemoryAddress1;
    void* pMemoryAddress2;

    SetUpParam();

    // ヒープの作成とメモリブロックの確保
    if(m_HasHeapHeadInternally)
    {
        m_HeapHandle = nn::lmem::CreateFrameHeap(reinterpret_cast<void*>(m_Heap), m_HeapSize, nn::lmem::CreationOption_ZeroClear);
    }
    else
    {
        m_HeapHandle = nn::lmem::CreateFrameHeap(reinterpret_cast<void*>(m_Heap), m_HeapSize, nn::lmem::CreationOption_ZeroClear, &m_HeapHead);
    }
    ASSERT_NOT_NULL(m_HeapHandle);
    pMemoryAddress1 = nn::lmem::AllocateFromFrameHeap(m_HeapHandle, allocateSize, m_MinAlignment);
    ASSERT_NOT_NULL(pMemoryAddress1);
    pMemoryAddress2 = nn::lmem::AllocateFromFrameHeap(m_HeapHandle, allocateSize, -m_MinAlignment);
    ASSERT_NOT_NULL(pMemoryAddress2);

    // Read/Write テスト
    EXPECT_TRUE(WriteNumericalSequence(pMemoryAddress1, allocateSize));
    EXPECT_TRUE(HasNumericalSequence(pMemoryAddress1, allocateSize));
    EXPECT_TRUE(WriteNumericalSequence(pMemoryAddress2, allocateSize));
    EXPECT_TRUE(HasNumericalSequence(pMemoryAddress2, allocateSize));

    nn::lmem::FreeToFrameHeap(m_HeapHandle, nn::lmem::FreeMode_Front);
    nn::lmem::FreeToFrameHeap(m_HeapHandle, nn::lmem::FreeMode_Rear);
    nn::lmem::DestroyFrameHeap(m_HeapHandle);

    // 前方から確保している領域を先に解放
    if(m_HasHeapHeadInternally)
    {
        m_HeapHandle = nn::lmem::CreateFrameHeap(reinterpret_cast<void*>(m_Heap), m_HeapSize, nn::lmem::CreationOption_ZeroClear);
    }
    else
    {
        m_HeapHandle = nn::lmem::CreateFrameHeap(reinterpret_cast<void*>(m_Heap), m_HeapSize, nn::lmem::CreationOption_ZeroClear, &m_HeapHead);
    }
    ASSERT_NOT_NULL(m_HeapHandle);
    pMemoryAddress1 = nn::lmem::AllocateFromFrameHeap(m_HeapHandle, allocateSize, m_MinAlignment);
    ASSERT_NOT_NULL(pMemoryAddress1);
    pMemoryAddress2 = nn::lmem::AllocateFromFrameHeap(m_HeapHandle, allocateSize, -m_MinAlignment);
    ASSERT_NOT_NULL(pMemoryAddress2);

    EXPECT_EQ(m_HeapSize - allocateSize * 2 - m_HeadSize, nn::lmem::GetFrameHeapAllocatableSize(m_HeapHandle, m_MinAlignment));

    nn::lmem::FreeToFrameHeap(m_HeapHandle, nn::lmem::FreeMode_Front);
    EXPECT_EQ(m_HeapSize - allocateSize - m_HeadSize, nn::lmem::GetFrameHeapAllocatableSize(m_HeapHandle, m_MinAlignment));

    nn::lmem::FreeToFrameHeap(m_HeapHandle, nn::lmem::FreeMode_Rear);
    EXPECT_EQ(m_HeapSize - m_HeadSize, nn::lmem::GetFrameHeapAllocatableSize(m_HeapHandle, m_MinAlignment));

    nn::lmem::DestroyFrameHeap(m_HeapHandle);

    // 後方から確保している領域を先に解放
    if(m_HasHeapHeadInternally)
    {
        m_HeapHandle = nn::lmem::CreateFrameHeap(reinterpret_cast<void*>(m_Heap), m_HeapSize, nn::lmem::CreationOption_ZeroClear);
    }
    else
    {
        m_HeapHandle = nn::lmem::CreateFrameHeap(reinterpret_cast<void*>(m_Heap), m_HeapSize, nn::lmem::CreationOption_ZeroClear, &m_HeapHead);
    }
    ASSERT_NOT_NULL(m_HeapHandle);
    pMemoryAddress1 = nn::lmem::AllocateFromFrameHeap(m_HeapHandle, allocateSize, m_MinAlignment);
    ASSERT_NOT_NULL(pMemoryAddress1);
    pMemoryAddress2 = nn::lmem::AllocateFromFrameHeap(m_HeapHandle, allocateSize, -m_MinAlignment);
    ASSERT_NOT_NULL(pMemoryAddress2);

    EXPECT_EQ(m_HeapSize - allocateSize * 2 - m_HeadSize, nn::lmem::GetFrameHeapAllocatableSize(m_HeapHandle, m_MinAlignment));

    nn::lmem::FreeToFrameHeap(m_HeapHandle, nn::lmem::FreeMode_Rear);
    EXPECT_EQ(m_HeapSize - allocateSize - m_HeadSize, nn::lmem::GetFrameHeapAllocatableSize(m_HeapHandle, m_MinAlignment));

    nn::lmem::FreeToFrameHeap(m_HeapHandle, nn::lmem::FreeMode_Front);
    EXPECT_EQ(m_HeapSize - m_HeadSize, nn::lmem::GetFrameHeapAllocatableSize(m_HeapHandle, m_MinAlignment));

    nn::lmem::DestroyFrameHeap(m_HeapHandle);
}

// メモリブロックの確保失敗
// 現在のフリー領域より大きなメモリの確保に失敗すれば、OK
TEST_F(FrameHeapBasicTest, AllocationFailure)
{
    const int BlockNum = 64;
    allocateSize = m_HeapSize / (BlockNum / 2);
    void* pMemoryAddresses[BlockNum];   // 少し余裕を持って配列を確保

    // ヒープの作成
    m_HeapHandle = nn::lmem::CreateFrameHeap(reinterpret_cast<void*>(m_Heap), m_HeapSize, nn::lmem::CreationOption_NoOption);
    ASSERT_NOT_NULL(m_HeapHandle);

    // allocateSize で確保できなくなるまで確保する
    int i = 0;
    while(nn::lmem::GetFrameHeapAllocatableSize(m_HeapHandle, m_MinAlignment) > allocateSize)
    {
        pMemoryAddresses[i] = nn::lmem::AllocateFromFrameHeap(m_HeapHandle, allocateSize, m_MinAlignment);
        ASSERT_NOT_NULL(pMemoryAddresses[i]);
        i++;
    }

    NN_LOG("Allocate %d times.\n", i);

    pMemoryAddresses[i] = nn::lmem::AllocateFromFrameHeap(m_HeapHandle, allocateSize, m_MinAlignment);
    ASSERT_NULL(pMemoryAddresses[i]);

    nn::lmem::FreeToFrameHeap(m_HeapHandle, nn::lmem::FreeMode_All);
    nn::lmem::DestroyFrameHeap(m_HeapHandle);

}

// ヒープ作成の繰り返し
// ヒープを同じアドレスで繰り返し呼んでも問題ないことを確認する
TEST(FrameHeapCreateTest, CreateHeapConstantly)
{
    int threadPriority = nn::os::GetThreadCurrentPriority(nn::os::GetCurrentThread());
    if(threadPriority < nn::os::LowestThreadPriority)
    {
        threadPriority++;
    }
    auto result = nn::os::CreateThread(&g_Thread, CreateFrameHeapConstantly, NULL, g_ThreadStack, g_ThreadStackSize, threadPriority);
    ASSERT_TRUE(result.IsSuccess());

    nn::os::InitializeEvent(&g_Event, false, nn::os::EventClearMode_AutoClear);

    const int timeOut = 20;     // 20 秒でタイムアウトとする
    nn::os::StartThread(&g_Thread);

    ASSERT_TRUE(nn::os::TimedWaitEvent(&g_Event, nn::TimeSpan::FromSeconds(timeOut)));

    nn::os::WaitThread(&g_Thread);
    nn::os::DestroyThread(&g_Thread);
    nn::os::FinalizeEvent(&g_Event);
}

// ヒープの繰り返し前の確保状況のクリア
// Create 後に以前のヒープ状況がクリアされていることを確認する
TEST_F(FrameHeapBasicTest, CreateHeapClear)
{
    // ヒープの作成
    m_HeapHandle = nn::lmem::CreateFrameHeap(reinterpret_cast<void*>(m_Heap), m_HeapSize, nn::lmem::CreationOption_NoOption);
    ASSERT_NOT_NULL(m_HeapHandle);
    size_t allocatableSize = nn::lmem::GetFrameHeapAllocatableSize(m_HeapHandle, nn::lmem::DefaultAlignment);

    void* ptr = nn::lmem::AllocateFromFrameHeap(m_HeapHandle, 32);
    ASSERT_NOT_NULL(ptr);
    EXPECT_GT(allocatableSize, nn::lmem::GetFrameHeapAllocatableSize(m_HeapHandle, nn::lmem::DefaultAlignment));

    // Destroy せずに再度 Create
    m_HeapHandle = nn::lmem::CreateFrameHeap(reinterpret_cast<void*>(m_Heap), m_HeapSize, nn::lmem::CreationOption_NoOption);
    ASSERT_NOT_NULL(m_HeapHandle);
    // ヒープは初期化されているはず
    EXPECT_EQ(allocatableSize, nn::lmem::GetFrameHeapAllocatableSize(m_HeapHandle, nn::lmem::DefaultAlignment));

    nn::lmem::DestroyFrameHeap(m_HeapHandle);
}

// =============
// テスト（Debug ビルド、Develop ビルドでのみ有効）
// =============

#if defined(NN_TEST_MEM_DEBUG)

// ダンプのテスト
// メモリの状況が正しくダンプできていれば、OK
TEST_P(FrameHeapBasicTest, Dump)
{
    void* pMemoryAddress;

    SetUpParam();

    // ヒープの作成とメモリ確保
    if(m_HasHeapHeadInternally)
    {
        m_HeapHandle = nn::lmem::CreateFrameHeap(reinterpret_cast<void*>(m_Heap), m_HeapSize, nn::lmem::CreationOption_NoOption);
    }
    else
    {
        m_HeapHandle = nn::lmem::CreateFrameHeap(reinterpret_cast<void*>(m_Heap), m_HeapSize, nn::lmem::CreationOption_NoOption, &m_HeapHead);
    }
    ASSERT_NOT_NULL(m_HeapHandle);
    pMemoryAddress = nn::lmem::AllocateFromFrameHeap(m_HeapHandle, allocateSize, m_MinAlignment);
    ASSERT_NOT_NULL(pMemoryAddress);

    // Read/Write テスト
    EXPECT_TRUE(WriteNumericalSequence(pMemoryAddress, allocateSize));
    EXPECT_TRUE(HasNumericalSequence(pMemoryAddress, allocateSize));

    // サイズチェック
    EXPECT_EQ(nn::lmem::GetFrameHeapAllocatableSize(m_HeapHandle, m_MinAlignment), m_HeapSize - allocateSize - m_HeadSize);

    ASSERT_NOT_NULL(nn::lmem::AllocateFromFrameHeap(m_HeapHandle, allocateSize, -m_MinAlignment));

    EXPECT_EQ(nn::lmem::GetFrameHeapAllocatableSize(m_HeapHandle, m_MinAlignment), m_HeapSize - allocateSize * 2 - m_HeadSize);

    nn::lmem::DumpFrameHeap(m_HeapHandle);

    NN_LOG("[FrameHeapBasicTest - Dump]\n");
    NN_LOG("Visual Check: 256bytes used?\n");
    NN_LOG("Visual Check: allocated 128bytes from head?\n");
    NN_LOG("Visual Check: allocated 128bytes from tail?\n\n");

    nn::lmem::FreeToFrameHeap(m_HeapHandle, nn::lmem::FreeMode_All);
    nn::lmem::DestroyFrameHeap(m_HeapHandle);
}

// デバッグ時のメモリフィル
// 未使用、確保、解放それぞれにおいて、指定の値がフィルできているなら、OK
TEST_P(FrameHeapBasicTest, DebugFill)
{
    const uint32_t FillUnallocated = 0x01234567;
    const uint32_t FillAllocate = 0x456789ab;
    const uint32_t FillFree  = 0x89abcdef;
    void* pMemoryAddress;

    SetUpParam();

    // フィルする値を設定
    nn::lmem::SetFillValue(nn::lmem::FillType_Unallocated, FillUnallocated);
    nn::lmem::SetFillValue(nn::lmem::FillType_Allocate, FillAllocate);
    nn::lmem::SetFillValue(nn::lmem::FillType_Free, FillFree);

    // フィルする値が正しく設定できているか
    EXPECT_EQ(nn::lmem::GetFillValue(nn::lmem::FillType_Unallocated), FillUnallocated);
    EXPECT_EQ(nn::lmem::GetFillValue(nn::lmem::FillType_Allocate), FillAllocate);
    EXPECT_EQ(nn::lmem::GetFillValue(nn::lmem::FillType_Free), FillFree);

    // ヒープの作成とメモリ確保
    if(m_HasHeapHeadInternally)
    {
        m_HeapHandle = nn::lmem::CreateFrameHeap(reinterpret_cast<void*>(m_Heap), m_HeapSize, nn::lmem::CreationOption_DebugFill);
    }
    else
    {
        m_HeapHandle = nn::lmem::CreateFrameHeap(reinterpret_cast<void*>(m_Heap), m_HeapSize, nn::lmem::CreationOption_DebugFill, &m_HeapHead);
    }
    ASSERT_NOT_NULL(m_HeapHandle);

    // DebugFill (未使用時) テスト
    pMemoryAddress = reinterpret_cast<void*>(nn::lmem::GetStartAddress(m_HeapHandle));
    EXPECT_TRUE(HasEqualNum(pMemoryAddress, m_HeapSize - m_HeadSize, FillUnallocated));

    // DebugFill (確保時) テスト
    pMemoryAddress = nn::lmem::AllocateFromFrameHeap(m_HeapHandle, allocateSize, m_MinAlignment);
    ASSERT_NOT_NULL(pMemoryAddress);
    EXPECT_TRUE(HasEqualNum(pMemoryAddress, allocateSize, FillAllocate));

    // メモリブロックをリサイズした場合
    // メモリブロックを拡大した場合は、拡大した部分は確保時と同様の Fill が行われる
    EXPECT_EQ(nn::lmem::ResizeFrameHeapBlock(m_HeapHandle, pMemoryAddress, allocateSize * 2), allocateSize * 2);
    EXPECT_TRUE(HasEqualNum(reinterpret_cast<void*>(reinterpret_cast<uintptr_t>(pMemoryAddress) + allocateSize), allocateSize, FillAllocate));

    // メモリブロックを縮小した場合は、縮小した部分は解放時と同様の Fill が行われる
    EXPECT_EQ(nn::lmem::ResizeFrameHeapBlock(m_HeapHandle, pMemoryAddress, allocateSize), allocateSize);
    EXPECT_TRUE(HasEqualNum(reinterpret_cast<void*>(reinterpret_cast<uintptr_t>(pMemoryAddress) + allocateSize), allocateSize, FillFree));

    // DebugFill (解放時) テスト
    nn::lmem::FreeToFrameHeap(m_HeapHandle, nn::lmem::FreeMode_All);
    EXPECT_TRUE(HasEqualNum(pMemoryAddress, allocateSize, FillFree));

    nn::lmem::DestroyFrameHeap(m_HeapHandle);
}

// メモリ確保時のゼロクリア
// メモリ確保時とリサイズ時に、適切にメモリがゼロクリアされていれば、OK
TEST_P(FrameHeapBasicTest, ZeroClear)
{
    void* pMemoryAddress;

    SetUpParam();

    // ヒープの作成
    if(m_HasHeapHeadInternally)
    {
        m_HeapHandle = nn::lmem::CreateFrameHeap(reinterpret_cast<void*>(m_Heap), m_HeapSize, nn::lmem::CreationOption_ZeroClear);
    }
    else
    {
        m_HeapHandle = nn::lmem::CreateFrameHeap(reinterpret_cast<void*>(m_Heap), m_HeapSize, nn::lmem::CreationOption_ZeroClear, &m_HeapHead);
    }
    ASSERT_NOT_NULL(m_HeapHandle);

    WriteNumericalSequence(nn::lmem::GetStartAddress(m_HeapHandle), m_HeapSize - m_HeadSize);

    // ゼロクリアテスト（確保時）
    pMemoryAddress = nn::lmem::AllocateFromFrameHeap(m_HeapHandle, allocateSize, m_MinAlignment);
    ASSERT_NOT_NULL(pMemoryAddress);
    EXPECT_TRUE(HasEqualNum(pMemoryAddress, allocateSize, 0));

    // Read/Write テスト
    EXPECT_TRUE(WriteNumericalSequence(pMemoryAddress, allocateSize));
    EXPECT_TRUE(HasNumericalSequence(pMemoryAddress, allocateSize));

    // ゼロクリアテスト（リサイズ時）
    // リサイズで増加した領域もゼロクリアされる
    EXPECT_EQ(nn::lmem::ResizeFrameHeapBlock(m_HeapHandle, pMemoryAddress, allocateSize * 2), allocateSize * 2);
    void* pMemoryAddressForCheck = reinterpret_cast<void*>(reinterpret_cast<uintptr_t>(pMemoryAddress) + allocateSize);
    EXPECT_TRUE(HasEqualNum(pMemoryAddressForCheck, allocateSize, 0));

    nn::lmem::FreeToFrameHeap(m_HeapHandle, nn::lmem::FreeMode_All);
    nn::lmem::DestroyFrameHeap(m_HeapHandle);
}
#endif

INSTANTIATE_TEST_CASE_P(HeapHeadPosition, FrameHeapBasicTest, testing::Bool());
