﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>

#include <nn/jit/jit_Api.h>
#include <nn/fs.h>
#include <nn/sf/sf_HipcClient.h>
#include <nn/sf/sf_ProxyObjectAllocator.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/init.h>
#include <nn/svc/svc_Base.h>
#include <nn/util/util_BitUtil.h>
#include <nn/os/os_TransferMemory.h>
#include "TestVmJitAarch64.h"

namespace nn { namespace jit { namespace testvm {

class OtherProcessJitRunner
    : public JitRunner
{
private:

    nn::jit::JitHandle m_Handle;
    uint64_t m_RxGeneratedOffset = 0;
    size_t m_RxSize;

public:

    OtherProcessJitRunner(nn::jit::JitHandle handle, MachineContext* pContext, size_t rxSize) NN_NOEXCEPT
        : JitRunner(pContext)
        , m_Handle(handle)
        , m_RxSize(rxSize)
    {
    }

    virtual void* GenerateCode(size_t* pByteSize, uint32_t* pCodeCount, int32_t pc) NN_NOEXCEPT NN_OVERRIDE
    {
        auto pContext = GetMachineContext();
        const auto& program = pContext->program;

        nn::jit::CodeRange outRx;
        nn::jit::CodeRange outRo;
        uint64_t tag = 0;
        const void* source = program.instructions;
        size_t sourceSize = program.instructionCount * sizeof(*program.instructions);
        nn::jit::CodeRange rxBuffer = {};
        rxBuffer.offset = m_RxGeneratedOffset;
        rxBuffer.size = m_RxSize - m_RxGeneratedOffset;
        nn::jit::CodeRange roBuffer = {};

        int dummy;
        auto ret = nn::jit::GenerateCode(&outRx, &outRo, m_Handle, tag, source, sourceSize, rxBuffer, roBuffer, &pc, sizeof(pc), &dummy, 0);
        NN_ABORT_UNLESS(ret == 0);
        m_RxGeneratedOffset += outRx.size;

        auto f = reinterpret_cast<void*>(nn::jit::GetJitEnvironmentInfo(m_Handle).rxCodeAddress + outRx.offset);
        *pByteSize = outRx.size;
        *pCodeCount = 0;
        return f;
    }

};

Result RunTestVmByJitImpl(MachineContext* pContext) NN_NOEXCEPT
{
    NN_RESULT_DO(nn::jit::Initialize());
    NN_UTIL_SCOPE_EXIT
    {
        nn::jit::Finalize();
    };

    static char nrrBuffer[1024 * 1024];
    int64_t nrrSize;
    static char nroBuffer[1024 * 1024];
    int64_t nroSize;

    {
        const auto FsCacheSize = 1024 * 1024;
        std::unique_ptr<char[]> fsCache(new char[FsCacheSize]);
        NN_ABORT_UNLESS_RESULT_SUCCESS(fs::MountRom("rom", fsCache.get(), FsCacheSize));
        {
            fs::FileHandle nrrFile;
            NN_ABORT_UNLESS_RESULT_SUCCESS(fs::OpenFile(&nrrFile, "rom:/.nrr/JitTestSimpleVmMain.nrr", fs::OpenMode_Read));
            NN_ABORT_UNLESS_RESULT_SUCCESS(fs::GetFileSize(&nrrSize, nrrFile));
            NN_ABORT_UNLESS_RESULT_SUCCESS(fs::ReadFile(nrrFile, 0, nrrBuffer, nrrSize));
            fs::CloseFile(nrrFile);
        }
        {
            fs::FileHandle nroFile;
            NN_ABORT_UNLESS_RESULT_SUCCESS(fs::OpenFile(&nroFile, "rom:/nro/JitTestSimpleVmPlugin.nro", fs::OpenMode_Read));
            NN_ABORT_UNLESS_RESULT_SUCCESS(fs::GetFileSize(&nroSize, nroFile));
            NN_ABORT_UNLESS_RESULT_SUCCESS(fs::ReadFile(nroFile, 0, nroBuffer, nroSize));
            fs::CloseFile(nroFile);
        }
        fs::Unmount("rom");
    }

    const size_t WorkBufferSize = 512 * 1024 * 1024;
    auto workBuffer = nn::init::GetAllocator()->Allocate(WorkBufferSize, 2 * 1024 * 1024);
    NN_ABORT_UNLESS_NOT_NULL(workBuffer);
    NN_UTIL_SCOPE_EXIT
    {
        nn::init::GetAllocator()->Free(workBuffer);
    };

    nn::jit::JitEnvironmentConfiguration configuration;
    configuration.rxCodeSize = 128 * 1024 * 1024;
    configuration.roCodeSize = 128 * 1024 * 1024;
    nn::jit::JitPluginInfo pluginInfo;
    pluginInfo.nrr = nrrBuffer;
    pluginInfo.nrrSize = nrrSize;
    pluginInfo.nro = nroBuffer;
    pluginInfo.nroSize = nroSize;
    pluginInfo.pluginWorkMemorySize = 128 * 1024 * 1024;
    nn::jit::JitHandle handle;
    NN_RESULT_DO(nn::jit::CreateJitEnvironment(&handle, configuration, pluginInfo, workBuffer, WorkBufferSize));
    NN_UTIL_SCOPE_EXIT
    {
        nn::jit::DestroyJitEnvironment(handle);
    };

    OtherProcessJitRunner runner(handle, pContext, configuration.rxCodeSize);
    runner.Run();

    NN_RESULT_SUCCESS;
}

void RunTestVmByJit(MachineContext* pContext) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(RunTestVmByJitImpl(pContext));
}

}}}
