﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "TestOnProcessJitRunner.h"


#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Log.h>
#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

#include <nn/os.h>
#include <nn/init.h>
#include <nn/svc/svc_Base.h>
#include <nn/svc/svc_Tcb.h>
#include <nn/svc/svc_Result.h>

#include "TestVmJitAarch64.h"
#include "Utility.h"

namespace nn { namespace jit { namespace testvm {

class JitOneProcessEnvironment
{
private:

    static uint64_t MapProcessCodeMemory(svc::Handle processHandle, uint64_t bufferAddr, uint64_t codeSize) NN_NOEXCEPT
    {
        jit::AslrAllocator aslrAllocator(processHandle);
    retry:
        auto toAddr = aslrAllocator.GetAslrRegion(codeSize);
        /*
            svc::MapProcessCodeMemory
            - 疑似ハンドルは渡せない → DuplicateHandle で本ハンドルに修正 (実装は内部で自己 IPC)
            - 本番では別プロセスになるため、問題ないはず
            - fromAddr は MemoryState_Normal を要求するため BSS 領域は指定できない → ヒープから確保
        */
        NN_RESULT_TRY(svc::MapProcessCodeMemory(processHandle, toAddr, bufferAddr, codeSize))
            NN_RESULT_CATCH(svc::ResultInvalidCurrentMemory)
            {
                goto retry;
            }
        NN_RESULT_END_TRY
        return toAddr;
    }

    static void UnmapProcessCodeMemory(svc::Handle processHandle, uint64_t codeAddress, uint64_t bufferAddr, uint64_t codeSize) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(svc::UnmapProcessCodeMemory(processHandle, codeAddress, bufferAddr, codeSize));
    }

    static uintptr_t MapProcessMemory(svc::Handle processHandle, uint64_t codeAddr, uint64_t codeSize) NN_NOEXCEPT
    {
        jit::AslrAllocator aslrAllocator(processHandle);
    retry:
        auto toAddr = aslrAllocator.GetAslrRegion(codeSize);
        NN_RESULT_TRY(svc::MapProcessMemory(toAddr, processHandle, codeAddr, codeSize))
            NN_RESULT_CATCH(svc::ResultInvalidCurrentMemory)
            {
                goto retry;
            }
        NN_RESULT_END_TRY
        return toAddr;
    }

    static void UnmapProcessMemory(uintptr_t toAddr, svc::Handle processHandle, uint64_t codeAddr, uint64_t codeSize) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(svc::UnmapProcessMemory(toAddr, processHandle, codeAddr, codeSize));
    }

private:

    bool m_Initialized = false;
    size_t m_CodeSize;
    svc::Handle m_ProcessHandle;
    void* m_Buffer;
    uint64_t m_CodeAddr;
    uintptr_t m_MappedCodeAddr;

    Result Initialize() NN_NOEXCEPT
    {
        auto success = false;

        auto processHandle = jit::DuplicateHandle(svc::PSEUDO_HANDLE_CURRENT_PROCESS);
        NN_UTIL_SCOPE_EXIT
        {
            if (!success)
            {
                svc::CloseHandle(processHandle);
            }
        };

        auto buffer = nn::init::GetAllocator()->Allocate(m_CodeSize, 4096);
        NN_UTIL_SCOPE_EXIT
        {
            if (!success)
            {
                nn::init::GetAllocator()->Free(buffer);
            }
        };
        auto bufferAddr = reinterpret_cast<uintptr_t>(buffer);

        auto codeAddr = MapProcessCodeMemory(processHandle, bufferAddr, m_CodeSize);
        NN_UTIL_SCOPE_EXIT
        {
            if (!success)
            {
                UnmapProcessCodeMemory(processHandle, codeAddr, bufferAddr, m_CodeSize);
            }
        };

        auto mappedCodeAddr = MapProcessMemory(processHandle, codeAddr, m_CodeSize);
        NN_UTIL_SCOPE_EXIT
        {
            if (!success)
            {
                UnmapProcessMemory(mappedCodeAddr, processHandle, codeAddr, m_CodeSize);
            }
        };
        std::memset(reinterpret_cast<void*>(mappedCodeAddr), 0, m_CodeSize);

        NN_ABORT_UNLESS_RESULT_SUCCESS(svc::SetProcessMemoryPermission(processHandle, codeAddr, m_CodeSize, svc::MemoryPermission_ReadExecute));
        NN_UTIL_SCOPE_EXIT
        {
            if (!success)
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(svc::SetProcessMemoryPermission(processHandle, codeAddr, m_CodeSize, svc::MemoryPermission_None));
            }
        };

        success = true;
        this->m_ProcessHandle = processHandle;
        this->m_Buffer = buffer;
        this->m_CodeAddr = codeAddr;
        this->m_MappedCodeAddr = mappedCodeAddr;
        this->m_Initialized = true;

        NN_RESULT_SUCCESS;
    }

    template <typename F>
    void IterateCacheLine(uintptr_t addr, size_t size, F f) NN_NOEXCEPT
    {
        uintptr_t cacheTypeRegister;
        asm volatile ("mrs %0, ctr_el0" : "=r" (cacheTypeRegister));
        size_t cacheLineSize = 4 << ((cacheTypeRegister >> 16) & 0xF);
        auto startAddr = reinterpret_cast<uintptr_t>(addr) & ~(cacheLineSize - 1);
        auto endAddr = reinterpret_cast<uintptr_t>(addr) + size;
        for (auto p = startAddr; p < endAddr; p += cacheLineSize)
        {
            f(p);
        }
    }

public:

    explicit JitOneProcessEnvironment(size_t codeSize) NN_NOEXCEPT
        : m_CodeSize(codeSize)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(Initialize());
    }

    ~JitOneProcessEnvironment() NN_NOEXCEPT
    {
        if (!m_Initialized)
        {
            return;
        }
        NN_ABORT_UNLESS_RESULT_SUCCESS(svc::SetProcessMemoryPermission(m_ProcessHandle, m_CodeAddr, m_CodeSize, svc::MemoryPermission_None));
        UnmapProcessMemory(m_MappedCodeAddr, m_ProcessHandle, m_CodeAddr, m_CodeSize);
        auto bufferAddr = reinterpret_cast<uintptr_t>(m_Buffer);
        UnmapProcessCodeMemory(m_ProcessHandle, m_CodeAddr, bufferAddr, m_CodeSize);
        nn::init::GetAllocator()->Free(m_Buffer);
        svc::CloseHandle(m_ProcessHandle);
    }

    uint64_t GetCodeAddr() const NN_NOEXCEPT
    {
        return m_CodeAddr;
    }

    uintptr_t GetMappedCodeAddr() const NN_NOEXCEPT
    {
        return m_MappedCodeAddr;
    }

};

class OneProcessJitRunner
    : private JitOneProcessEnvironment
    , public nn::jit::testvm::JitRunner
{
private:

    typedef JitOneProcessEnvironment Environment;
    int64_t m_CodeCurrentOffset = 0;

public:

    explicit OneProcessJitRunner(nn::jit::testvm::MachineContext* pContext) NN_NOEXCEPT
        : Environment(1024 * 1024)
        , JitRunner(pContext)
    {
    }

    virtual void* GenerateCode(size_t* pByteSize, uint32_t* pInstructionCount, int32_t pc) NN_NOEXCEPT NN_OVERRIDE
    {
        auto ret = m_CodeCurrentOffset;
        auto codeBuffer = reinterpret_cast<char*>(Environment::GetMappedCodeAddr()) + m_CodeCurrentOffset;
        size_t generatedByteSize;
        GenerateCodeImpl(&generatedByteSize, pInstructionCount, codeBuffer, pc);
        StoreDataCacheAndMemoryBarrier(reinterpret_cast<uintptr_t>(codeBuffer), generatedByteSize);
        *pByteSize = generatedByteSize;
        this->m_CodeCurrentOffset += generatedByteSize;
        return reinterpret_cast<void*>(Environment::GetCodeAddr() + ret);
    }

};

void RunOnProcessJit(nn::jit::testvm::MachineContext* pContext) NN_NOEXCEPT
{
    OneProcessJitRunner(pContext).Run();
}

}}}
