﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <cstring>
#include <mutex>

#include <nn/fs.h>
#include <nn/fs/fs_Debug.h>

#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/util/util_FormatString.h>

#include "AttachmentLogger.h"

const char* AttachmentLogger::Format = "AttachmentType, AttachmentId, CameraId, OutmostXMin, OutmostYMin, OutmostXMax, OutmostYMax, CenterX, CenterY,";

const char* ClusterFormat = "clusterId[%d], validity[%d], obj[%d].aveInt, obj[%d].centroidX, obj[%d].centoridY, obj[%d].pixCnt, obj[%d].boundX, obj[%d].boundY, obj[%d].boundW, obj[%d].boundH, ";
const char* ClusterRawFormat = "obj[%d].aveInt, obj[%d].centroidX, obj[%d].centoridY, obj[%d].pixCnt, obj[%d].boundX, obj[%d].boundY, obj[%d].boundW, obj[%d].boundH, ";
const char* SixAxisFormat = "accelX, accelY, accelZ, accelX[deg], accelY[deg], accelZ[deg], ObjectCount,";

const char* AttachmentLogger::MountName = "sd";

AttachmentLogger::AttachmentLogger() NN_NOEXCEPT
    : m_FileHandle()
    , m_FileOffset(0)
    , m_IsSdCardInserted(false)
    , m_StartTick(nn::os::GetSystemTick())
    , m_LineQueue()
{
    memset(m_FileName, 0, FileNameCountMax);
    memset(m_DirName, 0, FileNameCountMax);
}

void AttachmentLogger::Initialize(const char* pDirName) NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_SPEC_NX)
    NN_ABORT_UNLESS_NOT_NULL(pDirName);

    strncpy(m_DirName, MountName, FileNameCountMax);
    strncat(m_DirName, ":/", FileNameCountMax);
    strncat(m_DirName, pDirName, FileNameCountMax);

    nn::Result result;

    NN_LOG("Mount SD card as \"%s:/\"\n", MountName);

    // SD カードをマウント名 MountName でマウントします。
    result = nn::fs::MountSdCardForDebug(MountName);
    if (nn::fs::ResultSdCardAccessFailed::Includes(result))
    {
        // SD カードが挿入されていません。
        NN_LOG("SD card not found.");
        return;
    }
    // 上記以外の失敗は実装ミスのため、必ずアボートする
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // SD カードの挿入を確認
    m_IsSdCardInserted = true;

    // ディレクトリが存在しない場合は作成します。
    {
        nn::fs::DirectoryEntryType directoryEntryType;
        result = nn::fs::GetEntryType(&directoryEntryType, m_DirName);
        if (nn::fs::ResultPathNotFound().Includes(result))
        {
            // 対象ファイルが存在しません。
            // 対象ファイルを作成します。
            NN_LOG("Create \"%s\"\n", m_DirName);
            result = nn::fs::CreateDirectory(m_DirName);
            if (nn::fs::ResultPathNotFound::Includes(result))
            {
                // 対象ディレクトリが存在しません。
                // 親ディレクトリが必ず存在する場合は、このエラーハンドリングは不要です。
            }
            else if (nn::fs::ResultPathAlreadyExists::Includes(result) ||
                     nn::fs::ResultTargetLocked::Includes(result))
            {
                // 対象ディレクトリが既に存在しています。
                // ファイルが既に存在していても構わない場合は、このエラーハンドリングは不要です。
                // エラーハンドリングしない場合、ファイルのサイズが FileSize である保証が無いことに注意してください。
                // 必要であれば対象ファイルを削除してから再度作成してください。
            }
            else if (nn::fs::ResultUsableSpaceNotEnough::Includes(result))
            {
                // 空き領域が不足しています。
                // SD カードの空き領域を確認してください。
                NN_ABORT("Usable space not enough.\n");
                return;
            }
            // 上記以外の原因で失敗した場合はライブラリ内でアボートするため、
            // これ以上のエラーハンドリングは不要です。
        }
        // 上記以外の原因で失敗した場合はライブラリ内でアボートするため、
        // これ以上のエラーハンドリングは不要です。
    }
#else
    NN_UNUSED(pDirName);
#endif
}

void AttachmentLogger::Finalize() NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_SPEC_NX)
    if (m_IsSdCardInserted == false)
    {
        // SD カードが挿入されていないので何もしません。
        // NN_LOG("Do nothing since sd card isn't inserted.\n");
        return;
    }

    NN_LOG("Close \"%s\"\n", m_FileName);
    nn::fs::CloseFile(m_FileHandle);

    // アンマウントします。
    NN_LOG("Unmount \"%s:/\"\n", MountName);
    nn::fs::Unmount(MountName);
#endif
}

void AttachmentLogger::AppendLineFormat(int indexCountMax) NN_NOEXCEPT
{
    const int BufferSize = 1024 * 8;
    char writeBuffer[BufferSize] = {'\0'};

    int length = nn::util::SNPrintf(writeBuffer,
                                    sizeof(writeBuffer),
                                    Format);

    // 6軸データのフォーマット
    strncat(writeBuffer, SixAxisFormat, BufferSize);

    // 出力フォーマット(index順)
    for (auto i = 0; i < indexCountMax; i++)
    {
        char clusterBuffer[BufferSize] = {'\0'};
        length += nn::util::SNPrintf(clusterBuffer,
            sizeof(clusterBuffer),
            ClusterFormat, i, i, i, i, i, i, i, i, i, i);
        strncat(writeBuffer, clusterBuffer, BufferSize);
    }

    // 生データのフォーマット
    strncat(writeBuffer, ", clusterRawCount, ", BufferSize);
    for (auto i = 0; i < nn::irsensor::ClusteringProcessorObjectCountMax; i++)
    {
        char clusterBuffer[BufferSize] = {'\0'};
        length += nn::util::SNPrintf(clusterBuffer,
            sizeof(clusterBuffer),
            ClusterRawFormat, i, i, i, i, i, i, i, i);
        strncat(writeBuffer, clusterBuffer, BufferSize);
    }
    strncat(writeBuffer, "\n", BufferSize);

    AppendLine(writeBuffer, length);
}

void AttachmentLogger::AppendLine(const AttachmentOutput& output, const RecordInfo& info, int indexCountMax) NN_NOEXCEPT
{
    const int BufferSize = 1024 * 8;
    char writeBuffer[BufferSize] = { '\0' };

    int length = nn::util::SNPrintf(writeBuffer,
        sizeof(writeBuffer),
        "%d, %d, %d, %d, %d, %d, %d, %0.1f, %0.1f,",
        info.attachmentVersionId,
        info.attachmentId,
        info.measuredCameraId,
        output.outMostFrame.x,
        output.outMostFrame.y,
        output.outMostFrame.x + output.outMostFrame.width - 1,
        output.outMostFrame.y + output.outMostFrame.height - 1,
        output.centerPoint.x,
        output.centerPoint.y
        );

    // 6軸データのフォーマット
    char sixAxisBuffer[BufferSize] = { '\0' };
    length += nn::util::SNPrintf(sixAxisBuffer,
        sizeof(sixAxisBuffer),
        "%8.5f, %8.5f, %8.5f, %0.2f, %0.2f, %0.2f, %d, ",
        output.sixAxisData.acceleration.x,
        output.sixAxisData.acceleration.y,
        output.sixAxisData.acceleration.z,
        output.sixAxisDegreeData.GetX(),
        output.sixAxisDegreeData.GetY(),
        output.sixAxisDegreeData.GetZ(),
        output.objectCount
        );
    strncat(writeBuffer, sixAxisBuffer, BufferSize);

    // 出力フォーマット(index順)
    for (auto i = 0; i < indexCountMax; i++)
    {
        char clusterBuffer[BufferSize] = {'\0'};
        length += nn::util::SNPrintf(clusterBuffer,
            sizeof(clusterBuffer),
            "%d, %d, %f, %f, %f, %d, %d, %d, %d, %d, ",
            output.clusters[i].clusterId,
            output.clusters[i].isValid,
            output.clusters[i].measuredClusteringData.averageIntensity,
            output.clusters[i].measuredClusteringData.centroid.x,
            output.clusters[i].measuredClusteringData.centroid.y,
            output.clusters[i].measuredClusteringData.pixelCount,
            output.clusters[i].measuredClusteringData.bound.x,
            output.clusters[i].measuredClusteringData.bound.y,
            output.clusters[i].measuredClusteringData.bound.width,
            output.clusters[i].measuredClusteringData.bound.height
        );
        strncat(writeBuffer, clusterBuffer, BufferSize);
    }

    // 生データのフォーマット
    char rawBuffer[BufferSize] = { '\0' };
    length += nn::util::SNPrintf(rawBuffer,
        sizeof(rawBuffer),
        ", %d, ", output.objectRawCount);
    strncat(writeBuffer, rawBuffer, BufferSize);
    for (auto i = 0; i < output.objectRawCount; i++)
    {
        char clusterBuffer[BufferSize] = {'\0'};
        length += nn::util::SNPrintf(clusterBuffer,
            sizeof(clusterBuffer),
            "%f, %f, %f, %d, %d, %d, %d, %d, ",
            output.rawClusterData[i].averageIntensity,
            output.rawClusterData[i].centroid.x,
            output.rawClusterData[i].centroid.y,
            output.rawClusterData[i].pixelCount,
            output.rawClusterData[i].bound.x,
            output.rawClusterData[i].bound.y,
            output.rawClusterData[i].bound.width,
            output.rawClusterData[i].bound.height
        );
        strncat(writeBuffer, clusterBuffer, BufferSize);
    }
    strncat(writeBuffer, "\n", BufferSize);
    AppendLine(writeBuffer, length);
}

void AttachmentLogger::AppendLine(const char* str, int length) NN_NOEXCEPT
{
    NN_UNUSED(length);
    if (m_IsSdCardInserted == false)
    {
        // SD カードが挿入されていないので何もしません。
        // NN_LOG("Do nothing since sd card isn't inserted.\n");
        return;
    }

    m_LineQueue.push(str);
}

void AttachmentLogger::Write(const char* pFileName) NN_NOEXCEPT
{
    if (m_LineQueue.empty())
    {
        // キューが空なので何もしません。
        return;
    }

    // ファイルパス生成
    strncpy(m_FileName, m_DirName, FileNameCountMax);
    strncat(m_FileName, "/", FileNameCountMax);
    strncat(m_FileName, pFileName, FileNameCountMax);
    {
        nn::fs::DirectoryEntryType directoryEntryType;
        nn::Result result = nn::fs::GetEntryType(&directoryEntryType, m_FileName);
        if (nn::fs::ResultPathNotFound().Includes(result))
        {
            // 対象ファイルが存在しません。
            // 対象ファイルを作成します。
            NN_LOG("Create \"%s\"\n", m_FileName);
            result = nn::fs::CreateFile(m_FileName, 0);
            if (nn::fs::ResultPathNotFound::Includes(result))
            {
                // 対象ディレクトリが存在しません。
                // 親ディレクトリが必ず存在する場合は、このエラーハンドリングは不要です。
            }
            else if (nn::fs::ResultPathAlreadyExists::Includes(result) ||
                     nn::fs::ResultTargetLocked::Includes(result))
            {
                // 対象ディレクトリが既に存在しています。
                // ファイルが既に存在していても構わない場合は、このエラーハンドリングは不要です。
                // エラーハンドリングしない場合、ファイルのサイズが FileSize である保証が無いことに注意してください。
                // 必要であれば対象ファイルを削除してから再度作成してください。
            }
            else if (nn::fs::ResultUsableSpaceNotEnough::Includes(result))
            {
                // 空き領域が不足しています。
                // SD カードの空き領域を確認してください。
                NN_ABORT("Usable space not enough.\n");
                return;
            }
            // 上記以外の原因で失敗した場合はライブラリ内でアボートするため、
            // これ以上のエラーハンドリングは不要です。
        }
        // 上記以外の原因で失敗した場合はライブラリ内でアボートするため、
        // これ以上のエラーハンドリングは不要です。
    }
    // ファイル末尾に追記します。
    {
        NN_LOG("Open \"%s\"\n", m_FileName);
        nn::Result result = nn::fs::OpenFile(&m_FileHandle,
                                  m_FileName,
                                  nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend);

        if (nn::fs::ResultPathNotFound::Includes(result))
        {
            // 対象ファイルが存在しません。
            // 存在するファイルしか開かない場合は、このエラーハンドリングは不要です。
        }
        else if (nn::fs::ResultTargetLocked::Includes(result))
        {
            // 対象ファイルが既にオープンされています。
            // ファイルが既にオープンされている可能性が無い場合は、このエラーハンドリングは不要です。
        }
        // 上記以外の原因で失敗した場合はライブラリ内でアボートするため、
        // これ以上のエラーハンドリングは不要です。

        // ファイルサイズの取得に失敗した場合はライブラリ内でアボートするため、エラーハンドリングは不要です。
        (void)nn::fs::GetFileSize(&m_FileOffset, m_FileHandle);
    }

    // キューに溜まった文字列を抽出します。
    ::std::string str;
    int strLength = 0;

    while (!m_LineQueue.empty())
    {
        auto temp = m_LineQueue.front().c_str();
        str += temp;
        strLength += static_cast<int>(::std::strlen(temp));
        m_LineQueue.pop();
    }

    // 書き込み処理
    auto result = nn::fs::WriteFile(m_FileHandle,
                                    m_FileOffset,
                                    str.c_str(),
                                    strLength,
                                    nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush));

    if (nn::fs::ResultUsableSpaceNotEnough::Includes(result))
    {
        // 空き領域が不足しています。
        // 書き込みサイズが想定範囲内の値となっているか、SD カードの空き領域は十分かを確認してください。
        NN_ABORT("Usable space not enough.\n");
        return;
    }
    // 上記以外の原因で失敗した場合はライブラリ内でアボートするため、
    // これ以上のエラーハンドリングは不要です。

    // 確認用にコンソールに SD カードに書き込まれるデータをログ出力します。
    // NN_LOG("%d: %s", strLength, str.c_str());

    m_FileOffset += strLength;
}

void AttachmentLogger::WritePng(const char* pFileName, GraphicsSystem* pGraphicsSystem) NN_NOEXCEPT
{
    // ファイルパス生成
    strncpy(m_FileName, m_DirName, FileNameCountMax);
    strncat(m_FileName, "/", FileNameCountMax);
    strncat(m_FileName, pFileName, FileNameCountMax);
    pGraphicsSystem->GetCapturedFrame(m_FileName);
}
