﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <vector>

#include <nn/nn_Log.h>
#include <nn/fs/fs_MemoryManagement.h>
#include <nn/fs/fs_Rom.h>
#include <nn/fs/fs_File.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_Mount.h>
#include <nn/nn_Assert.h>
#include <nn/gfx/util/gfx_DebugFontTextWriter.h>
#include <nns/gfx/gfx_PrimitiveRenderer.h>
#include <nn/init/init_Malloc.h>
#include <nn/os.h>
#include <nn/vi.h>

#include "HandAnalysisUtil.h"

struct float2 {
    float x, y;
    float2() NN_NOEXCEPT{};
    float2(float _x, float _y) NN_NOEXCEPT : x(_x), y(_y) {}
    NN_IMPLICIT float2(float val) NN_NOEXCEPT : x(val), y(val) {}
    NN_IMPLICIT float2(nn::util::Float2 val) NN_NOEXCEPT : x(val.x), y(val.y) {}
    inline float2 operator + (const float2& value) const NN_NOEXCEPT{
        float2 result;
        result.x = this->x + value.x;
        result.y = this->y + value.y;
        return result;
    };
    inline float2 operator - (const float2& value) const NN_NOEXCEPT{
        float2 result;
        result.x = this->x - value.x;
        result.y = this->y - value.y;
        return result;
    };
    inline friend float2 operator * (float scalar, const float2& vector) NN_NOEXCEPT
    {
        float2 result;
        result.x = vector.x * scalar;
        result.y = vector.y * scalar;
        return result;
    }
    float2& operator = (nn::util::Float2& p)
    {
        this->x = p.x;
        this->y = p.y;
        return *this;
    }
};

void ComputePolygonProperties(PolygonProperties* pPolygonProperties, nn::util::Float2 const* pPolygonPoints, size_t polygonPointCount) NN_NOEXCEPT
{
    nn::util::Float2 Zero2f = {};

    if (polygonPointCount < 3)
    {
        pPolygonProperties->perimeter = 0;
        pPolygonProperties->signedArea = 0;
        pPolygonProperties->areaCentroid = Zero2f;
        pPolygonProperties->perimeterCentroid = Zero2f;
        return;
    }
    float sumLength = 0.0f;
    float sumPreArea = 0.0f;
    float2 sumBorderPreCentroid = float2(0.0f);
    float2 sumAreaPreCentroid = float2(0.0f);
    float2 const center = float2(0.0f);
    float2 A = float2(pPolygonPoints[polygonPointCount - 1]) - center;
    size_t const size = polygonPointCount - 1;
    for (size_t i = 0; i < size; ++i)
    {
        float2 const B = float2(pPolygonPoints[i]) - center;
        float2 const AB = B - A;
        float const length = sqrtf( AB.x * AB.x + AB.y * AB.y );
        NN_ASSERT(length == length);
        float const preArea = A.x * B.y - A.y * B.x;
        float2 const areaPreCentroid = preArea * (A + B);
        sumLength += length;
        sumPreArea += preArea;
        sumBorderPreCentroid.x += length * 0.5f * (A.x + B.x);
        sumBorderPreCentroid.y += length * 0.5f * (A.y + B.y);
        sumAreaPreCentroid.x += areaPreCentroid.x;
        sumAreaPreCentroid.y += areaPreCentroid.y;
        A = B;
    }
    NN_ASSERT(sumLength > 0);
    pPolygonProperties->perimeter = sumLength;
    pPolygonProperties->perimeterCentroid.x = (1.f / sumLength) * sumBorderPreCentroid.x + center.x;
    pPolygonProperties->perimeterCentroid.y = (1.f / sumLength) * sumBorderPreCentroid.y + center.y;
    if (sumPreArea > 0)
    {
        pPolygonProperties->signedArea = 0.5f * sumPreArea;
        pPolygonProperties->areaCentroid.x = sumAreaPreCentroid.x * (1.f / (3 * sumPreArea)) + center.x;
        pPolygonProperties->areaCentroid.y = sumAreaPreCentroid.y * (1.f / (3 * sumPreArea)) + center.y;
    }
    else
    {
        pPolygonProperties->signedArea = 0.f;
        pPolygonProperties->areaCentroid = pPolygonProperties->perimeterCentroid;
    }
}

void ExtractConvexHull( std::vector<size_t>& convexHullIndices, const std::vector<nn::util::Float2>& oneTurnSilhouette ) NN_NOEXCEPT
{
    NN_ASSERT(convexHullIndices.empty());
    size_t const MAX_POINT_COUNT = 2048;
    NN_ASSERT(oneTurnSilhouette.size() <= MAX_POINT_COUNT);
    size_t const size = std::min(oneTurnSilhouette.size(), MAX_POINT_COUNT);

    if (size < 4)
    {
        for (size_t i = 0; i < size; ++i)
            convexHullIndices.push_back(i);
        return;
    }
    size_t firstIndex = size_t(-1);
    float2 firstPoint = float2(std::numeric_limits<float>::max());
    for (size_t i = 0; i < size; ++i)
    {
        float2 const P = oneTurnSilhouette[i];
        if (P.x < firstPoint.x || (P.x == firstPoint.x && P.y <= firstPoint.y))
        {
            firstPoint = P;
            firstIndex = i;
        }
    }
    std::vector<size_t> sortedIndices;
    {
        std::vector<float2> dirsToCurrentPoint;
        for (size_t i = 0; i < size; ++i)
        {
            if (firstIndex == i) {
                dirsToCurrentPoint.push_back(float2(0.f));
                continue;
            }
            float2 const dirToCurrentPoint = float2(oneTurnSilhouette[i]) - firstPoint;
            NN_ASSERT(0 <= dirToCurrentPoint.x);
            dirsToCurrentPoint.push_back(dirToCurrentPoint);
            sortedIndices.push_back(i);
        }
        auto CompareAngles = [&dirsToCurrentPoint](size_t a, size_t b) -> bool{
            float const axby = dirsToCurrentPoint[a].x * dirsToCurrentPoint[b].y;
            float const aybx = dirsToCurrentPoint[a].y * dirsToCurrentPoint[b].x;
            if(aybx < axby)
                return true;
            else if(aybx > axby)
                return false;
            // having farthest after is good for rejecting the nearest
            // points because of the order they will be tested for the
            // det > 0 condition.
            return dirsToCurrentPoint[a].x * dirsToCurrentPoint[a].x + dirsToCurrentPoint[a].y * dirsToCurrentPoint[a].y < dirsToCurrentPoint[b].x * dirsToCurrentPoint[b].x + dirsToCurrentPoint[b].y * dirsToCurrentPoint[b].y;
        };
        std::sort(sortedIndices.begin(), sortedIndices.end(), CompareAngles);
    }
    std::vector<size_t> tmpConvexHullIndices;
    tmpConvexHullIndices.push_back(firstIndex);
    size_t sortSize = sortedIndices.size();
    for (size_t i = 0; i < sortSize; ++i)
    {
        size_t const index = sortedIndices[i];
        float2 const point = oneTurnSilhouette[index];
        while (tmpConvexHullIndices.size() >= 2)
        {
            size_t const prevIndex = tmpConvexHullIndices.back();
            float2 const prevPoint = oneTurnSilhouette[prevIndex];
            size_t const m2Index = *(tmpConvexHullIndices.rbegin() + 1);
            float2 const m2Point = oneTurnSilhouette[m2Index];
            float2 const prevSegment = prevPoint - m2Point;
            float2 const segment = point - prevPoint;
            float const d = prevSegment.x * segment.y - prevSegment.y * segment.x;
            if (d > 0)
            {
                break;
            }
            tmpConvexHullIndices.pop_back();
        }
        tmpConvexHullIndices.push_back(index);
    }
    size_t const convexHullSize = tmpConvexHullIndices.size();
    for (size_t i = 0; i < convexHullSize; ++i)
    {
        convexHullIndices.push_back(tmpConvexHullIndices[i]);
    }
    convexHullIndices.push_back(tmpConvexHullIndices[0]);
}

