﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/vfx/vfx_Random.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os.h>
#include <nn/os/os_Thread.h>
#include <nn/os/os_Tick.h>
#include <nn/irsensor.h>
#include <nnt.h>
#include "../Common/testIrsensor_Util.h"

using namespace ::nn::irsensor;

namespace nnt { namespace irsensor {
    IrCameraHandle IrSensorTest::s_Handles[NpadIdCountMax];
    int IrSensorTest::s_HandleCount;
}}

namespace
{
// ランダム呼び出し回数
const auto RepeatCountMax = 10;
// ランダム呼び出しインターバルの上限値 (ms)
const auto RepeatIntervalMaxTime = 100;
// NotReady を待つ回数 (無線環境が悪い場合を考慮して 5秒 まで待つ)
const auto WaitCountForNotReady = 333;
// コマンド呼び出しの定常インターバル
const auto CommandInterval = nn::TimeSpan::FromMilliSeconds(15);
// 起動直後のインターバル
const auto InitialInterval = nn::TimeSpan::FromSeconds(2);

// 80x60 サイズのバッファを用意しておく
NN_ALIGNAS(0x1000)
uint8_t s_WorkBuffer[ImageTransferProcessorWorkBufferSize80x60];

NN_ALIGNAS(0x1000)
uint8_t s_ImageBuffer[ImageTransferProcessorImageSize80x60];


// 接続完了フラグ
bool g_IsConnected = false;

// スレッド設定
const int ThreadNumMax = 9;
const size_t ThreadStackSize = 32 * 1024;
nn::os::ThreadType  g_Thread[ThreadNumMax];
NN_OS_ALIGNAS_THREAD_STACK char g_ThreadStack[ThreadNumMax][ThreadStackSize];
bool g_IsThreadRunning[ThreadNumMax] = {};
nn::os::TimerEventType g_TimerEvent[ThreadNumMax];

struct Parameter
{
    int threadId;
    nn::irsensor::IrCameraHandle handle;
}g_Param[ThreadNumMax];

void StartThread(nn::os::ThreadFunction threadFunc, int i, nn::irsensor::IrCameraHandle handle) NN_NOEXCEPT
{
    if (!g_IsThreadRunning[i])
    {
        NN_LOG("Create Thread[%d]\n", i);
        // タイマを起動(1frameごと)
        nn::os::InitializeTimerEvent(&g_TimerEvent[i], nn::os::EventClearMode_AutoClear);
        nn::os::StartPeriodicTimerEvent(&g_TimerEvent[i], nn::TimeSpan::FromMilliSeconds(0), nn::TimeSpan::FromMicroSeconds(16666));

        g_IsThreadRunning[i] = true;
        g_Param[i].handle = handle;
        g_Param[i].threadId = i;
        nn::os::CreateThread(
            &g_Thread[i], threadFunc, &g_Param[i], g_ThreadStack[i],
            sizeof(g_ThreadStack[i]), nn::os::DefaultThreadPriority);
        nn::os::StartThread(&g_Thread[i]);
    }
    return;
}

void StopThread(int i) NN_NOEXCEPT
{
    if (g_IsThreadRunning[i])
    {
        g_IsThreadRunning[i] = false;
        nn::os::WaitThread(&g_Thread[i]);
        nn::os::DestroyThread(&g_Thread[i]);
        // タイマを停止
        nn::os::StopTimerEvent(&g_TimerEvent[i]);
        nn::os::FinalizeTimerEvent(&g_TimerEvent[i]);
        NN_LOG("Destroy Thread[%d]\n", i);
    }
    return;
}

template <typename T>
class ThreadSafeWithConnectionTest : public ::nnt::irsensor::IrSensorTest
{
};

typedef ::testing::Types<MomentProcessorConfig, ClusteringProcessorConfig, ImageTransferProcessorConfig, HandAnalysisConfig> ProcessorConfigTypes;
TYPED_TEST_CASE(ThreadSafeWithConnectionTest, ProcessorConfigTypes);

template <typename T>
struct ImageProcessorState;

template <>
struct ImageProcessorState<MomentProcessorConfig>
{
    typedef MomentProcessorState type;
};

template <>
struct ImageProcessorState<ClusteringProcessorConfig>
{
    typedef ClusteringProcessorState type;
};

template <>
struct ImageProcessorState<ImageTransferProcessorConfig>
{
    typedef ImageTransferProcessorState type;
};

template <>
struct ImageProcessorState<HandAnalysisConfig>
{
    typedef HandAnalysisSilhouetteState type;
};

template <typename T>
struct ImageProcessorSet
{
    T config;
    typename ImageProcessorState<T>::type state;
};

template <typename T>
void GetImageProcessorSet(ImageProcessorSet<T>* pSet) NN_NOEXCEPT;

template <>
void GetImageProcessorSet(ImageProcessorSet<MomentProcessorConfig>* pSet) NN_NOEXCEPT
{
    GetMomentProcessorDefaultConfig(&pSet->config);
}

template <>
void GetImageProcessorSet(ImageProcessorSet<ClusteringProcessorConfig>* pSet) NN_NOEXCEPT
{
    GetClusteringProcessorDefaultConfig(&pSet->config);
}

template <>
void GetImageProcessorSet(ImageProcessorSet<ImageTransferProcessorConfig>* pSet) NN_NOEXCEPT
{
    GetImageTransferProcessorDefaultConfig(&pSet->config);

    pSet->config.format = ImageTransferProcessorFormat_80x60;
}

template <>
void GetImageProcessorSet(ImageProcessorSet<HandAnalysisConfig>* pSet) NN_NOEXCEPT
{
    pSet->config.mode = ::nn::irsensor::HandAnalysisMode_Silhouette;
}

void RunImageProcessor(const IrCameraHandle& handle, const MomentProcessorConfig& config) NN_NOEXCEPT
{
    RunMomentProcessor(handle, config);
}

void RunImageProcessor(const IrCameraHandle& handle, const ClusteringProcessorConfig& config) NN_NOEXCEPT
{
    RunClusteringProcessor(handle, config);
}

void RunImageProcessor(const IrCameraHandle& handle, const ImageTransferProcessorConfig& config) NN_NOEXCEPT
{
    auto buffer = s_WorkBuffer;
    RunImageTransferProcessor(handle, config, buffer, ImageTransferProcessorWorkBufferSize80x60);
}

void RunImageProcessor(const IrCameraHandle& handle, const HandAnalysisConfig& config) NN_NOEXCEPT
{
    RunHandAnalysis(handle, config);
}

nn::Result GetImageProcessorData(const IrCameraHandle& handle, ImageProcessorState<MomentProcessorConfig>::type state) NN_NOEXCEPT
{
    return GetMomentProcessorState(&state, handle);
}

nn::Result GetImageProcessorData(const IrCameraHandle& handle, ImageProcessorState<ClusteringProcessorConfig>::type state) NN_NOEXCEPT
{
    return GetClusteringProcessorState(&state, handle);
}

nn::Result GetImageProcessorData(const IrCameraHandle& handle, ImageProcessorState<ImageTransferProcessorConfig>::type state) NN_NOEXCEPT
{
    return GetImageTransferProcessorState(&state, s_ImageBuffer, ImageTransferProcessorImageSize80x60, handle);
}

nn::Result GetImageProcessorData(const IrCameraHandle& handle, ImageProcessorState<HandAnalysisConfig>::type state) NN_NOEXCEPT
{
    int maxCount = 1;
    int returnCount = 0;
    int64_t samplingNum = 0;
    return GetHandAnalysisSilhouetteState(&state, &returnCount, maxCount, samplingNum, handle);
}

template <typename T>
bool DoTestRandomApiCall(nn::irsensor::IrCameraHandle handle) NN_NOEXCEPT
{
    auto status = GetIrCameraStatus(handle);
    if (status == IrCameraStatus_Available)
    {
        nn::vfx::detail::RandomGenerator rand;
        nn::os::Tick tick = nn::os::GetSystemTick();
        uint32_t seed = static_cast<uint32_t>(tick.GetInt64Value());
        rand.Initialize(seed);

        ImageProcessorSet<T> processorSet;
        GetImageProcessorSet(&processorSet);

        // コマンド間の IntervalTime を設定 (0ms ~ 100ms までの乱数)
        auto interval = rand.GetUnsignedInteger() % (RepeatIntervalMaxTime + 1);
        auto intervalTime = nn::TimeSpan::FromMilliSeconds(interval);

        for (auto i = 0; i < RepeatCountMax; i++)
        {
            // ランダムに API を発行する
            const int ApiTypeCount = 4;
            int val = rand.GetUnsignedInteger() % ApiTypeCount;
            switch (val)
            {
            case 0:
                {
                    NN_LOG("Run [%d] thread:%d core:%d\n",
                        handle,
                        nn::os::GetThreadId(nn::os::GetCurrentThread()),
                        nn::os::GetCurrentCoreNumber());
                    RunImageProcessor(handle, processorSet.config);
                }
                break;
            case 1:
                {
                    NN_LOG("StopAsync [%d] thread:%d core:%d\n",
                        handle,
                        nn::os::GetThreadId(nn::os::GetCurrentThread()),
                        nn::os::GetCurrentCoreNumber());
                    StopImageProcessorAsync(handle);
                }
                break;
            case 2:
                {
                    NN_LOG("Stop [%d] thread:%d core:%d\n",
                        handle,
                        nn::os::GetThreadId(nn::os::GetCurrentThread()),
                        nn::os::GetCurrentCoreNumber());
                    StopImageProcessor(handle);
                }
                break;
            case 3:
                {
                    NN_LOG("Get [%d] thread:%d core:%d\n",
                        handle,
                        nn::os::GetThreadId(nn::os::GetCurrentThread()),
                        nn::os::GetCurrentCoreNumber());
                    GetImageProcessorData(handle, processorSet.state);
                }
                break;
            default:
                NN_UNEXPECTED_DEFAULT;
            }
            nn::os::SleepThread(intervalTime);
        }
        return true;
    }
    else
    {
        return false;
    }
}

template <typename T>
bool DoTestContinuousDynamicRun(const IrCameraHandle& handle) NN_NOEXCEPT
{
    auto status = GetIrCameraStatus(handle);
    if (status == IrCameraStatus_Available)
    {
        ImageProcessorSet<T> processorSet;
        GetImageProcessorSet(&processorSet);

        // (SIGLO-70354 不具合回避のため、初回は少し待つ)
        RunImageProcessor(handle, processorSet.config);
        nn::os::SleepThread(CommandInterval);

        for (auto i = 0; i < RepeatCountMax; i++)
        {
            // Run 関数を連続で呼び出す
            RunImageProcessor(handle, processorSet.config);
        }
        // データをチェックする
        nn::Result result;
        int counter = 0;
        while(NN_STATIC_CONDITION(true))
        {
            result = GetImageProcessorData(handle, processorSet.state);
            if (result.IsSuccess())
            {
                break;
            }
            nn::os::SleepThread(CommandInterval);
            counter++;
            if (counter > WaitCountForNotReady)
            {
                // テストは失敗にする
                NNT_EXPECT_RESULT_SUCCESS(result);
                return true;
            }
        }
        EXPECT_LE(0, processorSet.state.samplingNumber);

        // 停止する
        StopImageProcessor(handle);
        return true;
    }
    else
    {
        return false;
    }
}

template <typename T>
void RandomApiCallThread(void* args) NN_NOEXCEPT
{
    Parameter* pParam = reinterpret_cast<Parameter*>(args);
    auto handle = pParam->handle;
    auto id = pParam->threadId;

    while (g_IsThreadRunning[id])
    {
        g_IsConnected |= DoTestRandomApiCall<T>(handle);
        nn::os::WaitTimerEvent(&g_TimerEvent[id]);
    }
    return;
}

template <typename T>
void ContinuousDynamicRunThread(void* args) NN_NOEXCEPT
{
    Parameter* pParam = reinterpret_cast<Parameter*>(args);
    auto handle = pParam->handle;
    auto id = pParam->threadId;

    while (g_IsThreadRunning[id])
    {
        g_IsConnected |= DoTestContinuousDynamicRun<T>(handle);
        nn::os::WaitTimerEvent(&g_TimerEvent[id]);
    }
    return;
}

// 同ハンドルに対して複数スレッドから API をランダムに呼び出して
// 停止しないことを確認する
TYPED_TEST(ThreadSafeWithConnectionTest, RandomApiCall)
{
    EXPECT_LT(0, TestFixture::s_HandleCount);

    nn::os::SleepThread(InitialInterval);

    const int ThreadCountMax = 2;

    for (auto j = 0; j < TestFixture::s_HandleCount; ++j)
    {
        const auto& handle = TestFixture::s_Handles[j];

        for (auto i = 0; i < ThreadCountMax; i++)
        {
            StartThread(RandomApiCallThread<TypeParam>, i, handle);
        }
        // テスト時間分だけ待つ
        nn::os::SleepThread(::nn::TimeSpan::FromSeconds(10));
        for (auto i = 0; i < ThreadCountMax; i++)
        {
            StopThread(i);
        }
    }

    // 1台も接続がなかった場合は失敗
    ASSERT_TRUE(g_IsConnected);
    g_IsConnected = false;
    return;
}

// ハンドルごとにスレッドを立てて、動的Runテストが問題なく動作することを確認する
TYPED_TEST(ThreadSafeWithConnectionTest, ParallelHandle)
{
    EXPECT_LT(0, TestFixture::s_HandleCount);

    nn::os::SleepThread(InitialInterval);

    for (auto j = 0; j < TestFixture::s_HandleCount; ++j)
    {
        const auto& handle = TestFixture::s_Handles[j];
        StartThread(ContinuousDynamicRunThread<TypeParam>, j, handle);
    }
    // テスト時間分だけ待つ
    nn::os::SleepThread(::nn::TimeSpan::FromSeconds(10));
    for (auto j = 0; j < TestFixture::s_HandleCount; ++j)
    {
        StopThread(j);
    }

    // 1台も接続がなかった場合は失敗
    ASSERT_TRUE(g_IsConnected);
    g_IsConnected = false;
    return;
}

} // namespacev
