﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/vfx/vfx_Random.h>
#include <nn/nn_Common.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os/os_Thread.h>
#include <nn/os/os_Tick.h>
#include <nn/irsensor.h>
#include <nnt.h>
#include "../Common/testIrsensor_Util.h"

using namespace ::nn::irsensor;

namespace nnt { namespace irsensor {
    IrCameraHandle IrSensorTest::s_Handles[NpadIdCountMax];
    int IrSensorTest::s_HandleCount;
}}

namespace
{
// ランダム呼び出し回数
const auto RepeatCountMax = 10;
// ランダム呼び出しインターバルの上限値 (ms)
const auto RepeatIntervalMaxTime = 100;
// NotReady を待つ回数 (無線環境が悪い場合を考慮して 5秒 まで待つ)
const auto WaitCountForNotReady = 333;
// コマンド呼び出しの定常インターバル
const auto CommandInterval = nn::TimeSpan::FromMilliSeconds(15);
// 起動直後のインターバル
const auto InitialInterval = nn::TimeSpan::FromSeconds(2);

// 80x60 サイズのバッファを用意しておく
NN_ALIGNAS(0x1000)
uint8_t s_WorkBuffer[ImageTransferProcessorWorkBufferSize80x60];

NN_ALIGNAS(0x1000)
uint8_t s_ImageBuffer[ImageTransferProcessorImageSize80x60];


template <typename T>
class ModeTransitionWithConnectionTest : public ::nnt::irsensor::IrSensorTest
{
};

typedef ::testing::Types<MomentProcessorConfig, ClusteringProcessorConfig, ImageTransferProcessorConfig, HandAnalysisConfig> ProcessorConfigTypes;
TYPED_TEST_CASE(ModeTransitionWithConnectionTest, ProcessorConfigTypes);

template <typename T>
struct ImageProcessorState;

template <>
struct ImageProcessorState<MomentProcessorConfig>
{
    typedef MomentProcessorState type;
};

template <>
struct ImageProcessorState<ClusteringProcessorConfig>
{
    typedef ClusteringProcessorState type;
};

template <>
struct ImageProcessorState<ImageTransferProcessorConfig>
{
    typedef ImageTransferProcessorState type;
};

template <>
struct ImageProcessorState<HandAnalysisConfig>
{
    typedef HandAnalysisSilhouetteState type;
};

template <typename T>
struct ImageProcessorSet
{
    T config;
    typename ImageProcessorState<T>::type state;
};

template <typename T>
void GetImageProcessorSet(ImageProcessorSet<T>* pSet) NN_NOEXCEPT;

template <>
void GetImageProcessorSet(ImageProcessorSet<MomentProcessorConfig>* pSet) NN_NOEXCEPT
{
    GetMomentProcessorDefaultConfig(&pSet->config);
}

template <>
void GetImageProcessorSet(ImageProcessorSet<ClusteringProcessorConfig>* pSet) NN_NOEXCEPT
{
    GetClusteringProcessorDefaultConfig(&pSet->config);
}

template <>
void GetImageProcessorSet(ImageProcessorSet<ImageTransferProcessorConfig>* pSet) NN_NOEXCEPT
{
    GetImageTransferProcessorDefaultConfig(&pSet->config);

    pSet->config.format = ImageTransferProcessorFormat_80x60;
}

template <>
void GetImageProcessorSet(ImageProcessorSet<HandAnalysisConfig>* pSet) NN_NOEXCEPT
{
    pSet->config.mode = ::nn::irsensor::HandAnalysisMode_Silhouette;
}

void RunImageProcessor(const IrCameraHandle& handle, const MomentProcessorConfig& config) NN_NOEXCEPT
{
    RunMomentProcessor(handle, config);
}

void RunImageProcessor(const IrCameraHandle& handle, const ClusteringProcessorConfig& config) NN_NOEXCEPT
{
    RunClusteringProcessor(handle, config);
}

void RunImageProcessor(const IrCameraHandle& handle, const ImageTransferProcessorConfig& config) NN_NOEXCEPT
{
    auto buffer = s_WorkBuffer;
    RunImageTransferProcessor(handle, config, buffer, ImageTransferProcessorWorkBufferSize80x60);
}

void RunImageProcessor(const IrCameraHandle& handle, const HandAnalysisConfig& config) NN_NOEXCEPT
{
    RunHandAnalysis(handle, config);
}

nn::Result GetImageProcessorData(const IrCameraHandle& handle, ImageProcessorState<MomentProcessorConfig>::type state) NN_NOEXCEPT
{
    return GetMomentProcessorState(&state, handle);
}

nn::Result GetImageProcessorData(const IrCameraHandle& handle, ImageProcessorState<ClusteringProcessorConfig>::type state) NN_NOEXCEPT
{
    return GetClusteringProcessorState(&state, handle);
}

nn::Result GetImageProcessorData(const IrCameraHandle& handle, ImageProcessorState<ImageTransferProcessorConfig>::type state) NN_NOEXCEPT
{
    return GetImageTransferProcessorState(&state, s_ImageBuffer, ImageTransferProcessorImageSize80x60, handle);
}

nn::Result GetImageProcessorData(const IrCameraHandle& handle, ImageProcessorState<HandAnalysisConfig>::type state) NN_NOEXCEPT
{
    int maxCount = 1;
    int returnCount = 0;
    int64_t samplingNum = 0;
    return GetHandAnalysisSilhouetteState(&state, &returnCount, maxCount, samplingNum, handle);
}

template <typename T>
bool DoTestDynamicRandomTransition(const IrCameraHandle& handle) NN_NOEXCEPT
{
    auto status = GetIrCameraStatus(handle);
    if (status == IrCameraStatus_Available)
    {
        ImageProcessorSet<T> processorSet;
        GetImageProcessorSet(&processorSet);

        nn::vfx::detail::RandomGenerator rand;
        nn::os::Tick tick = nn::os::GetSystemTick();
        uint32_t seed = static_cast<uint32_t>(tick.GetInt64Value());
        rand.Initialize(seed);

        // コマンド間の IntervalTime を設定 (0ms ~ 100ms までの乱数)
        auto interval = rand.GetUnsignedInteger() % (RepeatIntervalMaxTime + 1);
        auto intervalTime = nn::TimeSpan::FromMilliSeconds(interval);

        for (auto i = 0; i < RepeatCountMax; i++)
        {
            if (rand.GetBool())
            {
                RunImageProcessor(handle, processorSet.config);
            }
            else
            {
                StopImageProcessorAsync(handle);
            }

            nn::os::SleepThread(intervalTime);
        }
        // 最後に対象のモードを呼び出す
        RunImageProcessor(handle, processorSet.config);

        // データをチェックする
        nn::Result result;
        int counter = 0;
        while(NN_STATIC_CONDITION(true))
        {
            result = GetImageProcessorData(handle, processorSet.state);
            if (result.IsSuccess())
            {
                break;
            }
            nn::os::SleepThread(CommandInterval);
            counter++;
            if (counter > WaitCountForNotReady)
            {
                // テストは失敗にする
                NNT_EXPECT_RESULT_SUCCESS(result);
                return true;
            }
        }

        EXPECT_LE(0, processorSet.state.samplingNumber);

        // 停止する
        StopImageProcessorAsync(handle);
        return true;
    }
    else
    {
        return false;
    }
}

template <typename T>
bool DoTestContinuousDynamicRun(const IrCameraHandle& handle) NN_NOEXCEPT
{
    auto status = GetIrCameraStatus(handle);
    if (status == IrCameraStatus_Available)
    {
        ImageProcessorSet<T> processorSet;
        GetImageProcessorSet(&processorSet);

        // (SIGLO-70354 不具合回避のため、初回は少し待つ)
        RunImageProcessor(handle, processorSet.config);
        nn::os::SleepThread(CommandInterval);

        for (auto i = 0; i < RepeatCountMax; i++)
        {
            // Run 関数を連続で呼び出す
            RunImageProcessor(handle, processorSet.config);
        }
        // データをチェックする
        nn::Result result;
        int counter = 0;
        while(NN_STATIC_CONDITION(true))
        {
            result = GetImageProcessorData(handle, processorSet.state);
            if (result.IsSuccess())
            {
                break;
            }
            nn::os::SleepThread(CommandInterval);
            counter++;
            if (counter > WaitCountForNotReady)
            {
                // テストは失敗にする
                NNT_EXPECT_RESULT_SUCCESS(result);
                return true;
            }
        }
        EXPECT_LE(0, processorSet.state.samplingNumber);

        // 停止する
        StopImageProcessor(handle);
        return true;
    }
    else
    {
        return false;
    }
}

template <typename T>
bool DoTestContinuousStopAsync(const IrCameraHandle& handle) NN_NOEXCEPT
{
    auto status = GetIrCameraStatus(handle);
    if (status == IrCameraStatus_Available)
    {
        ImageProcessorSet<T> processorSet;
        GetImageProcessorSet(&processorSet);

        // (SIGLO-70354 不具合回避のため、初回は少し待つ)
        RunImageProcessor(handle, processorSet.config);
        nn::os::SleepThread(CommandInterval);

        for (auto i = 0; i < RepeatCountMax; i++)
        {
            // StopAsync 関数を連続で呼び出す
            StopImageProcessorAsync(handle);
        }
        // その直後に Run を行う
        RunImageProcessor(handle, processorSet.config);
        // データをチェックする
        nn::Result result;
        int counter = 0;
        while(NN_STATIC_CONDITION(true))
        {
            result = GetImageProcessorData(handle, processorSet.state);
            if (result.IsSuccess())
            {
                break;
            }
            nn::os::SleepThread(CommandInterval);
            counter++;
            if (counter > WaitCountForNotReady)
            {
                // テストは失敗にする
                NNT_EXPECT_RESULT_SUCCESS(result);
                return true;
            }
        }
        EXPECT_LE(0, processorSet.state.samplingNumber);

        // 停止する
        StopImageProcessor(handle);
        return true;
    }
    else
    {
        return false;
    }
}

template <typename T>
bool DoTestContinuousTransition(const IrCameraHandle& handle) NN_NOEXCEPT
{
    auto status = GetIrCameraStatus(handle);
    if (status == IrCameraStatus_Available)
    {
        ImageProcessorSet<T> processorSet;
        GetImageProcessorSet(&processorSet);

        // (SIGLO-70354 不具合回避のため、初回は少し待つ)
        RunImageProcessor(handle, processorSet.config);
        nn::os::SleepThread(CommandInterval);

        for (auto i = 0; i < RepeatCountMax; i++)
        {
            // Run 関数と StopAsync関数を連続で呼び出す
            StopImageProcessorAsync(handle);
            RunImageProcessor(handle, processorSet.config);
        }
        // データをチェックする
        nn::Result result;
        int counter = 0;
        while(NN_STATIC_CONDITION(true))
        {
            result = GetImageProcessorData(handle, processorSet.state);
            if (result.IsSuccess())
            {
                break;
            }
            nn::os::SleepThread(CommandInterval);
            counter++;
            if (counter > WaitCountForNotReady)
            {
                // テストは失敗にする
                NNT_EXPECT_RESULT_SUCCESS(result);
                return true;
            }
        }
        EXPECT_LE(0, processorSet.state.samplingNumber);

        // 停止する
        StopImageProcessor(handle);
        return true;
    }
    else
    {
        return false;
    }
}

TYPED_TEST(ModeTransitionWithConnectionTest, RandomTransition)
{
    EXPECT_LT(0, TestFixture::s_HandleCount);

    nn::os::SleepThread(InitialInterval);

    // 少なくとも1台1以上のIR センサー使用可能なデバイスが接続されたかどうか。
    bool isConnected = false;

    for (auto j = 0; j < TestFixture::s_HandleCount; ++j)
    {
        const auto& handle = TestFixture::s_Handles[j];

        if (DoTestDynamicRandomTransition<TypeParam>(handle))
        {
            isConnected = true;
        }
    }

    // 1台も接続がなかった場合は失敗
    ASSERT_TRUE(isConnected);
}

TYPED_TEST(ModeTransitionWithConnectionTest, ContinuousDynamicRun)
{
    EXPECT_LT(0, TestFixture::s_HandleCount);

    nn::os::SleepThread(InitialInterval);

    // 少なくとも1台1以上のIR センサー使用可能なデバイスが接続されたかどうか。
    bool isConnected = false;

    for (auto j = 0; j < TestFixture::s_HandleCount; ++j)
    {
        const auto& handle = TestFixture::s_Handles[j];

        if (DoTestContinuousDynamicRun<TypeParam>(handle))
        {
            isConnected = true;
        }
    }

    // 1台も接続がなかった場合は失敗
    ASSERT_TRUE(isConnected);
}

TYPED_TEST(ModeTransitionWithConnectionTest, ContinuousStopAsync)
{
    EXPECT_LT(0, TestFixture::s_HandleCount);

    nn::os::SleepThread(InitialInterval);

    // 少なくとも1台1以上のIR センサー使用可能なデバイスが接続されたかどうか。
    bool isConnected = false;

    for (auto j = 0; j < TestFixture::s_HandleCount; ++j)
    {
        const auto& handle = TestFixture::s_Handles[j];

        if (DoTestContinuousStopAsync<TypeParam>(handle))
        {
            isConnected = true;
        }
    }

    // 1台も接続がなかった場合は失敗
    ASSERT_TRUE(isConnected);
}

TYPED_TEST(ModeTransitionWithConnectionTest, ContinuousTransition)
{
    EXPECT_LT(0, TestFixture::s_HandleCount);

    nn::os::SleepThread(InitialInterval);

    // 少なくとも1台1以上のIR センサー使用可能なデバイスが接続されたかどうか。
    bool isConnected = false;

    for (auto j = 0; j < TestFixture::s_HandleCount; ++j)
    {
        const auto& handle = TestFixture::s_Handles[j];

        if (DoTestContinuousTransition<TypeParam>(handle))
        {
            isConnected = true;
        }
    }

    // 1台も接続がなかった場合は失敗
    ASSERT_TRUE(isConnected);
}

} // namespace
