﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/irsensor.h>
#include <nn/vfx/vfx_Random.h>
#include <nn/nn_Macro.h>
#include <nn/os/os_MemoryHeapCommon.h>
#include <nn/os/os_Tick.h>
#include <nnt.h>
#include "../Common/testIrsensor_Util.h"

using namespace ::nn::irsensor;

namespace nnt { namespace irsensor {
    IrCameraHandle IrSensorTest::s_Handles[NpadIdCountMax];
    int IrSensorTest::s_HandleCount;
}}

namespace
{

// 最大サイズのバッファを用意しておく
NN_ALIGNAS(0x1000)
uint8_t s_WorkBuffer[ImageTransferProcessorWorkBufferSize320x240];

template <typename T>
class ModeTransitionTest : public ::nnt::irsensor::IrSensorTest
{
};

typedef ::testing::Types<MomentProcessorConfig, ClusteringProcessorConfig, ImageTransferProcessorConfig, HandAnalysisConfig> ProcessorConfigTypes;
TYPED_TEST_CASE(ModeTransitionTest, ProcessorConfigTypes);

template <typename T>
struct ImageProcessorState;

template <>
struct ImageProcessorState<MomentProcessorConfig>
{
    typedef MomentProcessorState type;
};

template <>
struct ImageProcessorState<ClusteringProcessorConfig>
{
    typedef ClusteringProcessorState type;
};

template <>
struct ImageProcessorState<ImageTransferProcessorConfig>
{
    typedef ImageTransferProcessorState type;
};

template <>
struct ImageProcessorState<HandAnalysisConfig>
{
    typedef HandAnalysisSilhouetteState type;
};

template <typename T>
struct ImageProcessorSet
{
    T config;
    typename ImageProcessorState<T>::type state;
};

template <typename T>
void GetImageProcessorSet(ImageProcessorSet<T>* pSet) NN_NOEXCEPT;

template <>
void GetImageProcessorSet(ImageProcessorSet<MomentProcessorConfig>* pSet) NN_NOEXCEPT
{
    GetMomentProcessorDefaultConfig(&pSet->config);
}

template <>
void GetImageProcessorSet(ImageProcessorSet<ClusteringProcessorConfig>* pSet) NN_NOEXCEPT
{
    GetClusteringProcessorDefaultConfig(&pSet->config);
}

template <>
void GetImageProcessorSet(ImageProcessorSet<ImageTransferProcessorConfig>* pSet) NN_NOEXCEPT
{
    GetImageTransferProcessorDefaultConfig(&pSet->config);

    pSet->config.format = ImageTransferProcessorFormat_320x240;
}

template <>
void GetImageProcessorSet(ImageProcessorSet<HandAnalysisConfig>* pSet) NN_NOEXCEPT
{
    pSet->config.mode = ::nn::irsensor::HandAnalysisMode_Silhouette;
}

void RunImageProcessor(const IrCameraHandle& handle, const MomentProcessorConfig& config) NN_NOEXCEPT
{
    RunMomentProcessor(handle, config);
}

void RunImageProcessor(const IrCameraHandle& handle, const ClusteringProcessorConfig& config) NN_NOEXCEPT
{
    RunClusteringProcessor(handle, config);
}

void RunImageProcessor(const IrCameraHandle& handle, const ImageTransferProcessorConfig& config) NN_NOEXCEPT
{
    auto buffer = s_WorkBuffer;
    RunImageTransferProcessor(handle, config, buffer, ImageTransferProcessorWorkBufferSize320x240);
}

void RunImageProcessor(const IrCameraHandle& handle, const HandAnalysisConfig& config) NN_NOEXCEPT
{
    RunHandAnalysis(handle, config);
}

template <typename T>
void DoTestDynamicTransitionToMoment(const IrCameraHandle& handle) NN_NOEXCEPT
{
    ImageProcessorSet<T> processorSet;
    GetImageProcessorSet(&processorSet);

    RunImageProcessor(handle, processorSet.config);

    ImageProcessorSet<MomentProcessorConfig> momentConfig;
    GetImageProcessorSet(&momentConfig);

    RunImageProcessor(handle, momentConfig.config);
    StopImageProcessor(handle);

    NNT_IRSENSOR_EXIT_SUCCESS;
}

template <typename T>
void DoTestDynamicTransitionToClustering(const IrCameraHandle& handle) NN_NOEXCEPT
{
    ImageProcessorSet<T> processorSet;
    GetImageProcessorSet(&processorSet);

    RunImageProcessor(handle, processorSet.config);

    ImageProcessorSet<ClusteringProcessorConfig> clusteringConfig;
    GetImageProcessorSet(&clusteringConfig);

    RunImageProcessor(handle, clusteringConfig.config);
    StopImageProcessor(handle);

    NNT_IRSENSOR_EXIT_SUCCESS;
}

template <typename T>
void DoTestDynamicTransitionToImageTransfer(const IrCameraHandle& handle) NN_NOEXCEPT
{
    ImageProcessorSet<T> processorSet;
    GetImageProcessorSet(&processorSet);

    RunImageProcessor(handle, processorSet.config);

    ImageProcessorSet<ImageTransferProcessorConfig> imageTransferConfig;
    GetImageProcessorSet(&imageTransferConfig);

    RunImageProcessor(handle, imageTransferConfig.config);
    StopImageProcessor(handle);

    NNT_IRSENSOR_EXIT_SUCCESS;
}

template <typename T>
void DoTestDynamicTransitionToHandAnalysis(const IrCameraHandle& handle) NN_NOEXCEPT
{
    ImageProcessorSet<T> processorSet;
    GetImageProcessorSet(&processorSet);

    RunImageProcessor(handle, processorSet.config);

    ImageProcessorSet<HandAnalysisConfig> handAnalysisConfig;
    GetImageProcessorSet(&handAnalysisConfig);

    RunImageProcessor(handle, handAnalysisConfig.config);
    StopImageProcessor(handle);

    NNT_IRSENSOR_EXIT_SUCCESS;
}

TYPED_TEST(ModeTransitionTest, DynamicTransitionBetweenTwoModes)
{
    EXPECT_LT(0, TestFixture::s_HandleCount);

    for (auto i = 0; i < TestFixture::s_HandleCount; ++i)
    {
        const auto& handle = TestFixture::s_Handles[i];

        ImageProcessorSet<TypeParam> processorSet;
        GetImageProcessorSet(&processorSet);

        NNT_IRSENSOR_EXPECT_EXIT(DoTestDynamicTransitionToMoment<TypeParam>(handle), NNT_IRSENSOR_EXIT_0, "");
        NNT_IRSENSOR_EXPECT_EXIT(DoTestDynamicTransitionToClustering<TypeParam>(handle), NNT_IRSENSOR_EXIT_0, "");
        NNT_IRSENSOR_EXPECT_EXIT(DoTestDynamicTransitionToImageTransfer<TypeParam>(handle), NNT_IRSENSOR_EXIT_0, "");
        NNT_IRSENSOR_EXPECT_EXIT(DoTestDynamicTransitionToHandAnalysis<TypeParam>(handle), NNT_IRSENSOR_EXIT_0, "");
    }
}

template <typename T>
void DoTestContinuousDynamicTransitionToSameMode(const IrCameraHandle& handle) NN_NOEXCEPT
{
    ImageProcessorSet<T> processorSet;
    GetImageProcessorSet(&processorSet);

    const int LOOP_COUNT_MAX = 10;
    for (auto i = 0; i < LOOP_COUNT_MAX; i++)
    {
        RunImageProcessor(handle, processorSet.config);
    }
    StopImageProcessor(handle);

    NNT_IRSENSOR_EXIT_SUCCESS;
}

TYPED_TEST(ModeTransitionTest, ContinuousDynamicTransitionToSameMode)
{
    EXPECT_LT(0, TestFixture::s_HandleCount);

    for (auto i = 0; i < TestFixture::s_HandleCount; ++i)
    {
        const auto& handle = TestFixture::s_Handles[i];

        ImageProcessorSet<TypeParam> processorSet;
        GetImageProcessorSet(&processorSet);

        NNT_IRSENSOR_EXPECT_EXIT(DoTestContinuousDynamicTransitionToSameMode<TypeParam>(handle), NNT_IRSENSOR_EXIT_0, "");
    }
}

template <typename T>
void DoTestRandomTransition(const IrCameraHandle& handle) NN_NOEXCEPT
{
    ImageProcessorSet<T> processorSet;
    GetImageProcessorSet(&processorSet);

    nn::vfx::detail::RandomGenerator rand;
    nn::os::Tick tick = nn::os::GetSystemTick();
    uint32_t seed = static_cast<uint32_t>(tick.GetInt64Value());
    rand.Initialize(seed);

    const int LOOP_COUNT_MAX = 30;
    for (auto i = 0; i < LOOP_COUNT_MAX; i++)
    {
        if (rand.GetBool())
        {
            RunImageProcessor(handle, processorSet.config);
        }
        else
        {
            StopImageProcessor(handle);
        }
    }
    StopImageProcessor(handle);

    NNT_IRSENSOR_EXIT_SUCCESS;
}

TYPED_TEST(ModeTransitionTest, RandomTransition)
{
    EXPECT_LT(0, TestFixture::s_HandleCount);

    for (auto i = 0; i < TestFixture::s_HandleCount; ++i)
    {
        const auto& handle = TestFixture::s_Handles[i];

        ImageProcessorSet<TypeParam> processorSet;
        GetImageProcessorSet(&processorSet);

        NNT_IRSENSOR_EXPECT_EXIT(DoTestRandomTransition<TypeParam>(handle), NNT_IRSENSOR_EXIT_0, "");
    }
}

} // namespace
