﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nn_Log.h>
#include <nn/socket.h>
#include <nn/nn_Assert.h>
#include <nn/nifm.h>
#include <nn/nn_Macro.h>
#include <nn/TargetConfigs/build_Base.h>
#include <nnt/nnt_Argument.h>

extern "C"
{

// Socket memory
namespace
{
    // using the "contrained" version here just to test it
    nn::socket::ConfigDefaultWithConstrainedMemory<
      8,    // max TCP sockets
      8     // max UDP sockets
    >g_SocketConfigWithMemory(4);
}

#if !defined(NN_BUILD_CONFIG_OS_WIN32)

const int  MemoryPoolSize = 10 * 1024 * 1024;
uint8_t    mempool[MemoryPoolSize];
size_t     mempoolSize = MemoryPoolSize;
#endif

int testMain(int argc, char **argv);

void* aligned_alloc(size_t, size_t)
{
    return nullptr;
}


void nninitStartup()
{
    nn::os::SetMemoryHeapSize(nn::socket::DefaultSocketMemoryPoolSize);
}

void nnMain()
{
    nn::Result result;

    int    i;
    int    argC;
    char** argV;
    const  char* defaultArguments[] = { "iperf", "-s", "-i", "10" };

    argC = nnt::GetHostArgc();

    if (argC > 1)
    {
        argV  = nnt::GetHostArgv();
    } else {
        /* Use default arguments if command line is empty */
        argV = const_cast<char**>(defaultArguments);
        argC = sizeof(defaultArguments) / sizeof(defaultArguments[0]);
    }

    NN_LOG("\n");

    /* Print command line to the console */
    for (i = 0; i < argC; i++)
    {
        NN_LOG("%s ", argV[i]);
    }

    NN_LOG("\n");

    /* if socket library is used to communicate with external nodes, we must initialize
       network interface manager (NIFM) first */
    result = nn::nifm::Initialize();
    if(result.IsSuccess())
    {
        NN_LOG("Nifm interface Initialized \n");
    }
    else
    {
        NN_LOG("Nifm interface Initialization failed %08x\n", result.GetInnerValueForDebug());
        #ifndef NN_BUILD_CONFIG_OS_WIN32
            return;
        #else
            // TODO: allow failure when SIGLONTD-2371 is fixed
            NN_LOG("WARNING: Ignoring NIFM init failure for windows platform\n" );
        #endif
    }

    /* this application is now requesting to use network interface */
    nn::nifm::SubmitNetworkRequest();
    NN_LOG("Nifm Network Request complete \n");

    /* wait for network interface availability, while providing status */
    while( nn::nifm::IsNetworkRequestOnHold() )
    {
        NN_LOG("Waiting for network interface availability...\n");
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }

    if( !nn::nifm::IsNetworkAvailable() )
    {
        NN_LOG("\nError: network is not available.\n");
        #ifndef NN_BUILD_CONFIG_OS_WIN32
            return;
        #else
            // TODO: allow failure when SIGLONTD-2371 is fixed
            NN_LOG("WARNING: Ignoring NIFM unavailable state for windows platform\n" );
        #endif
    }

    NN_LOG("Nifm interface up\n");

     // Uncommenting these lines will disable FreeBSD TCP buffer auto tune
     //g_SocketConfigWithMemory.SetTcpAutoSendBufferSizeMax(0);
     //g_SocketConfigWithMemory.SetTcpAutoReceiveBufferSizeMax(0);

    /* Initialize network stack, run the test */
    result = nn::socket::Initialize(g_SocketConfigWithMemory);

    NN_ASSERT(result.IsSuccess());

    NN_LOG("Socket Initialized\n");

    testMain(argC, argV);

    NN_LOG("Main Complete\n");

    nn::socket::Finalize();

    NN_LOG("Socket Finalized\n");

    /* this application no longer requires use of network interface */
    nn::nifm::CancelNetworkRequest();

    NN_LOG("Nifm Finalized\n");

    return;
}

}
