﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/os/os_MultipleWait.h>

#include <nn/nn_SdkAssert.h>
#include <nn/nn_TimeSpan.h>

namespace nnt { namespace sfutil {

class MultiWaiter;

class MultiWaitHodler
    : public nn::os::MultiWaitHolderType
{
private:

    bool m_Attached;

public:

    MultiWaitHodler() NN_NOEXCEPT
        : m_Attached(false)
    {
    }

    template <typename T>
    NN_IMPLICIT MultiWaitHodler(T* p) NN_NOEXCEPT
        : m_Attached(false)
    {
        this->Attach(p);
    }

    ~MultiWaitHodler() NN_NOEXCEPT
    {
        if (m_Attached)
        {
            nn::os::FinalizeMultiWaitHolder(this);
        }
    }

    nn::os::MultiWaitHolderType* GetBase() NN_NOEXCEPT
    {
        return this;
    }

    nn::os::MultiWaitHolderType* GetBaseForAttach() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(!m_Attached);
        this->m_Attached = true;
        return this;
    }

    template <typename T>
    void Attach(T* p) NN_NOEXCEPT
    {
        InitializeMultiWaitHolder(GetBaseForAttach(), p);
    }

};

class MultiWaiter
{
private:

    nn::os::MultiWaitType m_Base;

public:

    MultiWaiter() NN_NOEXCEPT
    {
        nn::os::InitializeMultiWait(&m_Base);
    }

    ~MultiWaiter() NN_NOEXCEPT
    {
        nn::os::FinalizeMultiWait(&m_Base);
    }

    nn::os::MultiWaitType* GetBase() NN_NOEXCEPT
    {
        return &m_Base;
    }

    void Link(nn::os::MultiWaitHolderType* pHolder) NN_NOEXCEPT
    {
        nn::os::LinkMultiWaitHolder(GetBase(), pHolder);
    }

    void Unlink(nn::os::MultiWaitHolderType* pHolder) NN_NOEXCEPT
    {
        nn::os::UnlinkMultiWaitHolder(pHolder);
    }

    nn::os::MultiWaitHolderType* WaitAny(bool unlink) NN_NOEXCEPT
    {
        auto ret = nn::os::WaitAny(GetBase());
        if (unlink)
        {
            nn::os::UnlinkMultiWaitHolder(ret);
        }
        return ret;
    }

    nn::os::MultiWaitHolderType* TryWaitAny(bool unlink) NN_NOEXCEPT
    {
        auto ret = nn::os::TryWaitAny(GetBase());
        if (!ret)
        {
            return nullptr;
        }
        if (unlink)
        {
            nn::os::UnlinkMultiWaitHolder(ret);
        }
        return ret;
    }

    nn::os::MultiWaitHolderType* TimedWaitAny(nn::TimeSpan timeout, bool unlink) NN_NOEXCEPT
    {
        auto ret = nn::os::TimedWaitAny(GetBase(), timeout);
        if (!ret)
        {
            return nullptr;
        }
        if (unlink)
        {
            nn::os::UnlinkMultiWaitHolder(ret);
        }
        return ret;
    }

};

}}
