﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
    @file
    @brief   Result のテストアサートを定義しています。
*/

#include <nnt/nntest.h>
#include <nn/nn_Result.h>
#include <iostream>

/**
    @brief     Result の成功を確認します。

    @param[in] actualResult 成功が期待される Result を指定します。
    @details   Result の成功を確認し、失敗していた場合は、Result を出力します。
               このとき、テストの実行は継続されます。

               すべてのビルドコンフィギュレーションで有効です。

               TODO: Result の表示は未実装です。
*/
#define NNT_EXPECT_RESULT_SUCCESS(actualResult) \
    EXPECT_PRED_FORMAT1(::nnt::result::detail::IsResultSuccess, actualResult)

/**
    @brief     Result の成功を確認します。

    @param[in] actualResult 成功が期待される Result を指定します。
    @details   Result の成功を確認し、失敗していた場合は、Result を出力します。
               このとき、テストは中断され、関数から戻ります。

               すべてのビルドコンフィギュレーションで有効です。

               TODO: Result の表示は未実装です。
*/
#define NNT_ASSERT_RESULT_SUCCESS(actualResult) \
    ASSERT_PRED_FORMAT1(::nnt::result::detail::IsResultSuccess, actualResult)

/**
    @brief     特定のエラーによる Result の失敗を確認します。

    @param[in] expectedResultClass 期待されるエラーの Result クラスを指定します。
    @param[in] actualResult        失敗が期待される Result を指定します。

    @details   特定のエラーによる Result の失敗を確認し、
               成功、または異なる種類のエラーにより失敗していた場合は、
               期待されるエラーと指定されたエラーを出力します。
               このとき、テストの実行は継続されます。

               すべてのビルドコンフィギュレーションで有効です。

               TODO: Result の表示は未実装です。
*/
#define NNT_EXPECT_RESULT_FAILURE(expectedResultClass, actualResult) \
    EXPECT_PRED_FORMAT2(::nnt::result::detail::IncludeResult, expectedResultClass(), actualResult)

/**
    @brief     特定のエラーによる Result の失敗をアサートします。

    @param[in] expectedResultClass 期待されるエラーの Result クラスを指定します。
    @param[in] actualResult        失敗が期待される Result を指定します。

    @details   特定のエラーによる Result の失敗を確認し、
               成功、または異なる種類のエラーにより失敗していた場合は、
               期待されるエラーと指定されたエラーを出力し、テストを失敗させます。
               このとき、テストは中断され、関数から戻ります。

               すべてのビルドコンフィギュレーションで有効です。

               TODO: Result の表示は未実装です。
*/
#define NNT_ASSERT_RESULT_FAILURE(expectedResultClass, actualResult) \
    ASSERT_PRED_FORMAT2(::nnt::result::detail::IncludeResult, expectedResultClass(), actualResult)


namespace nn {

    /**
        @brief gtest での Result のストリーム出力に対応するインタフェースです。

        ::testing::PrintToString の内部で使用されます。
        ストリーム出力を行う型と同じ名前空間に配置します。
    */
#if defined(NN_BUILD_CONFIG_COMPILER_VC)
    void PrintTo(const nn::Result& result, ::std::ostream* os);
#elif defined(NN_BUILD_CONFIG_COMPILER_GCC) || defined(NN_BUILD_CONFIG_COMPILER_CLANG)
    void PrintTo(const nn::Result& result, ::nnt::testing::detail::StringStream* os);
#endif

}

namespace nnt { namespace result { namespace detail {

    /**
        @page テストアサートの述語関数

        ::nnt::result::detail::IsResultSuccess 等はテストアサートのマクロの実装に用いられるもので、直接使用することはありません。
        このような述語関数を作成することで、テストアサートに失敗した時に親切な専用のメッセージを出力することができます。

        ::nnt::result::detail::IsResultSuccess の場合は、 NNT_EXPECT_RESULT_SUCCESS 等で使用されています。
    */

    /**
        @brief テストアサートの NNT_*_RESULT_SUCCESS に使用する述語関数です。
    */
    ::testing::AssertionResult IsResultSuccess(const char* expr, nn::Result result);

    /**
        @brief テストアサートの NNT_*_RESULT_FAILURE に使用する述語関数です。
    */
    template<typename ResultType>
    ::testing::AssertionResult IncludeResult(const char* expectedExpression, const char* resultExpression, ResultType expectedResult, nn::Result result);


    ::testing::AssertionResult IncludeResultImpl(const char* expectedExpression, const char* resultExpression, bool (*isExpected)(nn::Result), nn::Result expectedResult, nn::Result result);

    template<typename ResultType>
    ::testing::AssertionResult IncludeResult(const char* expectedExpression, const char* resultExpression, ResultType expectedResult, nn::Result result)
    {
        return IncludeResultImpl(expectedExpression, resultExpression, ResultType::Includes, expectedResult, result);
    }
}}}
