﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_TimeSpan.h>
#include <nn/ldn/ldn_Ipv4Address.h>
#include <nn/ldn/ldn_PrivateTypes.h>
#include <nn/ldn/ldn_Settings.h>

namespace nnt { namespace ldn
{

    /**
     * @brief       LDN ライブラリの初期化を RAII で実施するヘルパです。
     */
    class Initializer
    {
    public:

        /**
         * @brief       LDN ライブラリを初期化します。
         */
        Initializer() NN_NOEXCEPT;

        /**
         * @brief       LDN ライブラリを解放します。
         */
        ~Initializer() NN_NOEXCEPT;
    };

    /**
     * @brief       LDN ライブラリの初期化を RAII で実施するヘルパです。
     */
    class SystemInitializer
    {
    public:

        /**
         * @brief       LDN ライブラリを初期化します。
         */
        SystemInitializer() NN_NOEXCEPT;

        /**
         * @brief       LDN ライブラリを解放します。
         */
        ~SystemInitializer() NN_NOEXCEPT;
    };

    /**
     * @brief       LDN ライブラリの初期化を RAII で実施するヘルパです。
     */
    class MonitorInitializer
    {
    public:

        /**
         * @brief       LDN ライブラリを初期化します。
         */
        MonitorInitializer() NN_NOEXCEPT;

        /**
         * @brief       LDN ライブラリを解放します。
         */
        ~MonitorInitializer() NN_NOEXCEPT;
    };

    /**
     * @brief       AP モードの開始を RAII で実施するヘルパです。
     */
    class AccessPointStarter
    {
    public:

        /**
         * @brief       OpenAccessPoint 関数を実行し、 AccessPoint としての動作を開始します。
         */
        AccessPointStarter() NN_NOEXCEPT;

        /**
         * @brief       CloseAccessPoint 関数を実行し、 AccessPoint としての動作を終了します。
         */
        ~AccessPointStarter() NN_NOEXCEPT;
    };

    /**
     * @brief       STA モードの開始を RAII で実施するヘルパです。
     */
    class StationStarter
    {
    public:

        /**
         * @brief       OpenStation 関数を実行し、 Station としての動作を開始します。
         */
        StationStarter() NN_NOEXCEPT;

        /**
         * @brief       CloseStation 関数を実行し、 Station としての動作を終了します。
         */
        ~StationStarter() NN_NOEXCEPT;
    };

    /**
     * @brief       適当なパスフレーズを生成します。
     * @param[out]  pOut            パスフレーズの出力先です。
     * @param[in]   keyword         パスワードの基となるキーワードです。
     */
    void CreatePassphrase(nn::ldn::SecurityConfig* pOut, const char* keyword) NN_NOEXCEPT;

    /**
     * @brief       アクセスポイントの IPv4 アドレスが有効であることを検証します。
     * @param[in]   ipv4Address     対象の IPv4 アドレスです。
     * @return      有効であれば true です。
     */
    bool IsValidAccessPointIpv4Address(nn::ldn::Ipv4Address ipv4Address) NN_NOEXCEPT;

    /**
     * @brief       ステーションの IPv4 アドレスが有効であることを検証します。
     * @param[in]   apAddress       AP の IPv4 アドレスです。
     * @param[in]   staAddress      STA の IPv4 アドレスです。
     * @return      有効であれば true です。
     */
    bool IsValidStationIpv4Address(
        nn::ldn::Ipv4Address apAddress, nn::ldn::Ipv4Address staAddress) NN_NOEXCEPT;

    /**
     * @brief       タイムアウト付でネットワークを探索します。
     * @param[out]  pOutNetwork     見つかったネットワークの出力先です。
     * @param[out]  pOutCount       見つかったネットワークの数です。
     * @param[in]   bufferCount     pOutNetwork に格納できる NetworkInfo の数です。
     * @param[in]   filter          探索するネットワークを指定するフィルタです。
     * @param[in]   channel         無線チャンネルです。
     * @param[in]   timeout         タイムアウト時間です。
     * @return      処理の成否です。
     */
    nn::Result TimedScan(
        nn::ldn::NetworkInfo* pOutNetwork, int* pOutCount, int bufferCount,
        nn::ldn::ScanFilter filter, int channel, nn::TimeSpan timeout) NN_NOEXCEPT;

    /**
     * @brief       タイムアウト付でネットワークに接続します。
     * @param[in]   network         対象のネットワークです。
     * @param[in]   security        セキュリティの設定です。
     * @param[in]   user            ユーザの設定です。
     * @param[in]   version         ローカル通信バージョンです。
     * @param[in]   timeout         タイムアウト時間です。
     * @return      処理の成否です。
     */
    nn::Result TimedConnect(
        const nn::ldn::NetworkInfo& network,
        const nn::ldn::SecurityConfig& security,
        const nn::ldn::UserConfig& user,
        int version,
        nn::TimeSpan timeout) NN_NOEXCEPT;

    /**
     * @brief       タイムアウト付でネットワークに接続します。
     * @param[in]   network         対象のネットワークです。
     * @param[in]   security        セキュリティの設定です。
     * @param[in]   securityParam   セキュリティパラメータです。
     * @param[in]   user            ユーザの設定です。
     * @param[in]   version         ローカル通信バージョンです。
     * @param[in]   timeout         タイムアウト時間です。
     * @return      処理の成否です。
     */
    nn::Result TimedConnectPrivate(
        const nn::ldn::NetworkConfig& network,
        const nn::ldn::SecurityConfig& security,
        const nn::ldn::SecurityParameter& securityParam,
        const nn::ldn::UserConfig& user,
        int version,
        nn::TimeSpan timeout) NN_NOEXCEPT;

}} // namespace nnt::ldn
