﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_TimeSpan.h>

namespace nnt { namespace ldn
{
    /**
     * @brief       処理の結果を表す定数です。
     */
    enum SynchronizationResult
    {
        //! 成功しました。
        SynchronizationResult_Success,

        //! 未知のエラーが発生しました。
        SynchronizationResult_Unknown,

        //! 状態が正しくありません。
        SynchronizationResult_InvalidState,

        //! 処理がタイムアウトしました。
        SynchronizationResult_Timeout,

        //! バッファサイズが不足しています。
        SynchronizationResult_InsufficientBuffer
    };

    /**
     * @brief       グループ名の最大長です。
     */
    const size_t SynchronizationGroupNameLengthMax = 64;

    /**
     * @brief       同期に使用するキーワードの最大長です。
     */
    const size_t SynchronizationKeywordLengthMax = 128;

    /**
     * @brief       テストで共有できるデータの最大サイズです。
     */
    const size_t SynchronizationDataSizeMax = 256;

    /**
     * @brief       クライアントの最大数です。
     */
    const int SynchronizationClientCountMax = 15;

    /**
     * @brief       テストで同期するための共通インタフェースです。
     */
    struct ISynchronization
    {
        /**
         * @brief       デストラクタです。
         */
        virtual ~ISynchronization() NN_NOEXCEPT
        {
        }

        /**
         * @brief       タイムアウト時間を設定します。
         * @param[in]   timeout         タイムアウト時間です。
         * @return      処理の結果です。SynchronizationResult を参照してください。
         */
        virtual int SetTimeout(nn::TimeSpan timeout) NN_NOEXCEPT = 0;

        /**
         * @brief       同期します。
         * @param[in]   keyword         同期に使用するキーワードです。ASCII で指定してください。
         * @return      処理の結果です。SynchronizationResult を参照してください。
         */
        virtual int Synchronize(const char* keyword) NN_NOEXCEPT = 0;

        /**
         * @brief       クライアント数を取得します。
         * @return      クライアント数です。サーバーは含みません。
         */
        virtual int GetClientCount() const NN_NOEXCEPT = 0;
    };

    /**
     * @brief       テストの同期でサーバーの役割を果たすインタフェースです。
     */
    struct ISynchronizationServer : public ISynchronization
    {
        /**
         * @brief       デストラクタです。
         */
        virtual ~ISynchronizationServer() NN_NOEXCEPT
        {
        }

        /**
         * @brief       テストサーバを生成します。
         * @param[in]   groupName       作成するグループの名前です。ASCII で指定してください。
         * @param[in]   clientCount     クライアントの数です。
         * @return      処理の結果です。SynchronizationResult を参照してください。
         */
        virtual int CreateServer(const char* groupName, int clientCount) NN_NOEXCEPT = 0;

        /**
         * @brief       テストサーバを破棄します。
         * @return      処理の結果です。SynchronizationResult を参照してください。
         */
        virtual int DestroyServer() NN_NOEXCEPT = 0;

        /**
         * @brief       クライアントと同期するデータを設定します。
         * @param[in]   data            同期するデータです。
         * @param[in]   dataSize        同期するデータのバイトサイズです。
         * @return      処理の結果です。SynchronizationResult を参照してください。
         */
        virtual int SetData(const void* data, size_t dataSize) NN_NOEXCEPT = 0;
    };

    /**
     * @brief       テストの同期でクライアントの役割を果たすインタフェースです。
     */
    struct ISynchronizationClient : public ISynchronization
    {
        /**
         * @brief       デストラクタです。
         */
        virtual ~ISynchronizationClient() NN_NOEXCEPT
        {
        }

        /**
         * @brief       テストサーバに接続します。
         * @param[in]   groupName       参加するグループの名前です。ASCII で指定してください。
         * @return      処理の結果です。SynchronizationResult を参照してください。
         */
        virtual int Connect(const char* groupName) NN_NOEXCEPT = 0;

        /**
         * @brief       テストサーバから切断します。
         * @return      処理の結果です。SynchronizationResult を参照してください。
         */
        virtual int Disconnect() NN_NOEXCEPT = 0;

        /**
         * @brief       前回の同期でサーバから受信したデータを取得します。
         * @param[out]  buffer          サーバから受信したデータの出力先です。
         * @param[out]  pOutSize        サーバから受信したデータサイズの出力先です。
         * @param[in]   bufferSize      バッファサイズです。
         * @return      処理の結果です。SynchronizationResult を参照してください。
         */
        virtual int GetData(
            void* buffer, size_t* pOutSize, size_t bufferSize) const NN_NOEXCEPT = 0;

        /**
         * @brief       自身のクライアントインデックスを取得します。
         * @return      クライアントインデックスです。1 から始まる整数です。
         */
        virtual int GetClientIndex() const NN_NOEXCEPT = 0;
    };

}} // namespace nnt::ldn
