﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nnt/ldn/testLdn_Synchronization.h>

namespace nnt { namespace ldn { namespace detail
{
    struct HtcsSynchronizationServerBuffer
    {
        char htcs[4096];
        char send[4096];
        char recv[4096];
    };

    struct HtcsSynchronizationClientBuffer
    {
        char htcs[4096];
        char send[4096];
        char recv[4096];
    };

}}} // namespace nnt::ldn::detail

namespace nnt { namespace ldn
{
    /**
     * @brief       HTCS によるテストサーバーの実装です。
     * @details
     * HTCS を使用して同期するため、事前にホスト PC で LdnTestBridge を実行してください。
     * HtcsSynchronizationServer と HtcsSynchronizationClient は、
     * どちらか一方を 1 つだけ使用できます。
     */
    class HtcsSynchronizationServer : public ISynchronizationServer
    {
    public:

        /**
         * @brief       コントラクタです。
         * @param[in]   buffer          HTCS で使用するバッファです。
         * @param[in]   bufferSize      HTCS で使用するバッファのサイズです。
         */
        HtcsSynchronizationServer(void* buffer, size_t bufferSize) NN_NOEXCEPT;

        /**
         * @brief       デストラクタです。
         */
        virtual ~HtcsSynchronizationServer() NN_NOEXCEPT;

        virtual int SetTimeout(nn::TimeSpan timeout) NN_NOEXCEPT NN_OVERRIDE;

        virtual int CreateServer(
            const char* groupName, int clientCount) NN_NOEXCEPT NN_OVERRIDE;

        virtual int DestroyServer() NN_NOEXCEPT NN_OVERRIDE;

        virtual int SetData(const void* data, size_t dataSize) NN_NOEXCEPT NN_OVERRIDE;

        virtual int Synchronize(const char* keyword) NN_NOEXCEPT NN_OVERRIDE;

        virtual int GetClientCount() const NN_NOEXCEPT NN_OVERRIDE;

    private:

        int SendCreateGroupRequest() NN_NOEXCEPT;
        int SendSyncRequest(const char* keyword) NN_NOEXCEPT;
        int SendLeaveRequest() NN_NOEXCEPT;

        detail::HtcsSynchronizationServerBuffer* m_Buffer;
        int m_Socket;
        char m_GroupName[SynchronizationGroupNameLengthMax];
        int m_ClientCount;
        char m_Data[SynchronizationDataSizeMax];
        size_t m_DataSize;
        nn::TimeSpan m_Timeout;
        bool m_IsRunning;
    };

    /**
     * @brief       HTCS によるテストクライアントの実装です。
     * @details
     * HTCS を使用して同期するため、事前にホスト PC で LdnTestBridge を実行してください。
     * HtcsSynchronizationServer と HtcsSynchronizationClient は、
     * どちらか一方を 1 つだけ使用できます。
     */
    class HtcsSynchronizationClient : public ISynchronizationClient
    {
    public:

        /**
         * @brief       コントラクタです。
         * @param[in]   intentId        ネットワークの目的を指定する識別子です。
         * @param[in]   channel         無線チャンネルです。
         */
        HtcsSynchronizationClient(void* buffer, size_t bufferSize) NN_NOEXCEPT;

        /**
         * @brief       デストラクタです。
         */
        virtual ~HtcsSynchronizationClient() NN_NOEXCEPT;

        virtual int SetTimeout(nn::TimeSpan timeout) NN_NOEXCEPT NN_OVERRIDE;

        virtual int Connect(const char* groupName) NN_NOEXCEPT NN_OVERRIDE;

        virtual int Disconnect() NN_NOEXCEPT NN_OVERRIDE;

        virtual int Synchronize(const char* keyword) NN_NOEXCEPT NN_OVERRIDE;

        virtual int GetData(
            void* buffer, size_t* pOutSize, size_t bufferSize) const NN_NOEXCEPT NN_OVERRIDE;

        virtual int GetClientCount() const NN_NOEXCEPT NN_OVERRIDE;

        virtual int GetClientIndex() const NN_NOEXCEPT NN_OVERRIDE;

    private:

        int SendJoinGroupRequest(nn::TimeSpan timeout) NN_NOEXCEPT;
        int SendSyncRequest(const char* keyword) NN_NOEXCEPT;
        int SendLeaveRequest() NN_NOEXCEPT;

        detail::HtcsSynchronizationClientBuffer* m_Buffer;
        int m_Socket;
        char m_GroupName[SynchronizationGroupNameLengthMax];
        int m_ClientCount;
        int m_ClientIndex;
        char m_Data[SynchronizationDataSizeMax];
        size_t m_DataSize;
        nn::TimeSpan m_Timeout;
        bool m_IsRunning;
    };

}} // namespace nnt::ldn
