﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>

namespace nnt { namespace lcs
{
    /**
     * @brief       カスタムオプションのパーサーです。
     */
    typedef bool(*CustomCommandLineOptionParser)(int* pOutIndex, char* argv[], int argc, int index);

    /**
     * @brief       テストの設定です。
     */
    struct TestConfig
    {
        //! アクセスポイントを含むノードの数です。
        int8_t nodeCount;

        //! アクセスポイントを含むノードのインデックスです。
        int8_t nodeIndex;

        //! 再起動時に設定するフラグです。
        bool resume;
        NN_PADDING1;

        //! エイジング時間を秒単位で指定します。既定値は 1 年 (31,536,000) です。
        int32_t duration;

        //! テストのシード値です。指定しなかった場合はランダムです。
        int32_t seed;

        //! テスト成功・失敗の閾値（パーセント）です。
        float threshold;

        //! シーン識別子です。
        uint16_t sceneId;
        NN_PADDING2;
    };

    /**
     * @brief       コマンドライン引数で設定可能にするパラメータを指定するフラグです。
     */
    enum CommandLineOptionFlag
    {
        //! アクセスポイントを含むノードの数です。
        CommandLineOptionFlag_NodeCount = 1 << 0,

        //! アクセスポイントを含むノードのインデックスです。
        CommandLineOptionFlag_NodeIndex = 1 << 1,

        //! エイジング時間です。
        CommandLineOptionFlag_Duration = 1 << 2,

        //! テストのシード値です。
        CommandLineOptionFlag_Seed = 1 << 3,

        //! テストの成功・失敗の判定に使用する閾値です。
        CommandLineOptionFlag_Threshold = 1 << 4,

        //! 再起動時に指定するフラグです。
        CommandLineOptionFlag_Resume = 1 << 5,

        //! シーン識別子を指定するフラグです。
        CommandLineOptionFlag_SceneId = 1 << 6,

        //! カスタムオプションです。
        CommandLineOptionFlag_Custom = 1 << 7,
    };

    /**
     * @brief       CommandLineParser の設定です。
     */
    struct CommandLineParserSetting
    {
        //! コマンド引数で設定可能な項目を CommandLineOptionFlag の論理和で指定します。
        nn::Bit32 flag;

        //! アクセスポイントを含むノードの最小数です。
        int8_t nodeCountMin;

        //! アクセスポイントを含むノードの最大数です。
        int8_t nodeCountMax;
        NN_PADDING2;

        //! カスタムオプションのパーサーです。
        CustomCommandLineOptionParser parser;
    };

    /**
     * @brief       コマンドライン引数を解析し、テスト設定を生成します。
     * @param[out]  pOutConfig      テスト設定の出力先です。
     * @param[in]   setting         CommandLineParser の設定です。
     * @param[in]   argc            コマンドライン引数の数です。
     * @param[in]   argv            コマンドライン引数です。
     */
    void Parse(TestConfig* pOutConfig, const CommandLineParserSetting& setting,
               int argc, char* argv[]) NN_NOEXCEPT;

}} // namespace nnt::lcs
