﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstdlib>
#include <array>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nnt/nntest.h>

namespace
{
    class AllocatedMemoryList
    {
    public:
        struct MemoryInfo
        {
            void* p;
            size_t size;
        };

        AllocatedMemoryList()
            : m_TotalAllocatedSize(0)
        {
            MemoryInfo tmp = { nullptr, 0 };
            m_MemoryArray.fill(tmp);
        }

        ~AllocatedMemoryList()
        {
        }

        MemoryInfo* Find(void* p)
        {
            for (auto it = m_MemoryArray.begin(); it != m_MemoryArray.end(); it++)
            {
                if (it->p == p)
                {
                    return &(*it);
                }
            }
            return nullptr;
        }

        void Add(void* p, size_t size)
        {
            MemoryInfo* info = Find(nullptr);
            if (!info)
            {
                ASSERT_TRUE(false);
            }
            info->p = p;
            info->size = size;
            m_TotalAllocatedSize += size;
        }

        void Remove(void* p)
        {
            MemoryInfo* info = Find(p);
            if (!info)
            {
                ASSERT_TRUE(false);
            }
            info->p = nullptr;
            m_TotalAllocatedSize -= info->size;
        }

        size_t GetAllocatedSize()
        {
            return m_TotalAllocatedSize;
        }

    private:
        std::array<MemoryInfo, 128> m_MemoryArray; // 適当なサイズにしているため、必要に応じて変更が必要
        size_t m_TotalAllocatedSize;
    } g_AllocatedMemoryList;

    class ScopedMemoryLeakDetector
    {
    public:
        ScopedMemoryLeakDetector()
        {
            Begin();
        }

        ~ScopedMemoryLeakDetector()
        {
            bool success = End();
            EXPECT_EQ(true, success);
        }

    private:
        void Begin()
        {
            m_InitialSize = g_AllocatedMemoryList.GetAllocatedSize();
        }

        bool End()
        {
            m_FinalSize = g_AllocatedMemoryList.GetAllocatedSize();
            if (m_InitialSize != m_FinalSize)
            {
                NN_LOG("Detected Memory Leaks! %u bytes\n", m_FinalSize - m_InitialSize);
                return false;
            }

            return true;
        }

        size_t m_InitialSize;
        size_t m_FinalSize;
    };
}

void * operator new (std::size_t size)
{
    void* p = malloc(size);
    g_AllocatedMemoryList.Add(p, size);
    return p;
}

void operator delete (void * ptr) NN_NOEXCEPT
{
    g_AllocatedMemoryList.Remove(ptr);
    free(ptr);
}
