﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

#include <memory>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/fs.h>
#include <nn/fs/fsa/fs_IFileSystem.h>
#include <nn/fs/fsa/fs_IFile.h>
#include <nn/fs/fsa/fs_IDirectory.h>
#include <nn/fs/fsa/fs_Registrar.h>
#include <nn/fs/detail/fs_Newable.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_StringUtil.h>

#include <fsa/fs_FileAccessor.h>

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/fsUtil/testFs_util_function.h>

namespace nnt { namespace fs { namespace util {

// FileHandle を IFile にするアダプタ
class FileHandleFile : public nn::fs::fsa::IFile, public nn::fs::detail::Newable
{
    NN_DISALLOW_COPY(FileHandleFile);

public:
    explicit FileHandleFile(nn::fs::FileHandle handle) NN_NOEXCEPT
        : m_Handle(handle)
    {
    }

    ~FileHandleFile() NN_NOEXCEPT
    {
        nn::fs::CloseFile(m_Handle);
    }

    virtual nn::Result DoRead(size_t* outValue, int64_t offset, void* buffer, size_t size, const nn::fs::ReadOption& option) NN_NOEXCEPT NN_OVERRIDE
    {
        return nn::fs::ReadFile(outValue, m_Handle, offset, buffer, size, option);
    }
    virtual nn::Result DoWrite(int64_t offset, const void* buffer, size_t size, const nn::fs::WriteOption& option) NN_NOEXCEPT NN_OVERRIDE
    {
        return nn::fs::WriteFile(m_Handle, offset, buffer, size, option);
    }
    virtual nn::Result DoFlush() NN_NOEXCEPT NN_OVERRIDE
    {
        return nn::fs::FlushFile(m_Handle);
    }
    virtual nn::Result DoSetSize(int64_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        return nn::fs::SetFileSize(m_Handle, size);
    }
    virtual nn::Result DoGetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE
    {
        return nn::fs::GetFileSize(outValue, m_Handle);
    }
    virtual nn::Result DoOperateRange(void* outBuffer, size_t outBufferSize, nn::fs::OperationId operationId, int64_t offset, int64_t size, const void* inBuffer, size_t inBufferSize) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_RESULT_DO(GetAccessor().OperateRange(
            outBuffer, outBufferSize, operationId, offset, size, inBuffer, inBufferSize
        ));
        NN_RESULT_SUCCESS;
    }

private:
    nn::fs::detail::FileAccessor& GetAccessor() NN_NOEXCEPT
    {
        return *reinterpret_cast<nn::fs::detail::FileAccessor*>(m_Handle.handle);
    }

private:
    const nn::fs::FileHandle m_Handle;
};


// DirectoryHandle を IDirectoyr にするアダプタ
class DirectoryHandleDirectory : public nn::fs::fsa::IDirectory, public nn::fs::detail::Newable
{
    NN_DISALLOW_COPY(DirectoryHandleDirectory);

public:
    explicit DirectoryHandleDirectory(nn::fs::DirectoryHandle handle) NN_NOEXCEPT
        : m_Handle(handle)
    {
    }
    ~DirectoryHandleDirectory() NN_NOEXCEPT
    {
        nn::fs::CloseDirectory(m_Handle);
    }

    virtual nn::Result DoRead(int64_t* outValue, nn::fs::DirectoryEntry* outEntries, int64_t count) NN_NOEXCEPT NN_OVERRIDE
    {
        return nn::fs::ReadDirectory(outValue, outEntries, m_Handle, count);
    }
    virtual nn::Result DoGetEntryCount(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE
    {
        return nn::fs::GetDirectoryEntryCount(outValue, m_Handle);
    }

private:
    const nn::fs::DirectoryHandle m_Handle;
};



// fsa 経由で他ファイルシステムのサブディレクトリを指す SubdirFs
class SubdirectoryFileSystem : public nn::fs::fsa::IFileSystem, public nn::fs::detail::Newable
{
    NN_DISALLOW_COPY(SubdirectoryFileSystem);

public:
    static const int EntryNameLengthMax = 768;

    explicit SubdirectoryFileSystem(const char* pRootPath) NN_NOEXCEPT
        : m_RootPath(pRootPath)
    {
    }
    // virtual ~SubdirectoryFileSystem() NN_NOEXCEPT NN_OVERRIDE = default;

    virtual nn::Result DoOpenFile(std::unique_ptr<nn::fs::fsa::IFile>* outValue, const char* path, nn::fs::OpenMode mode) NN_NOEXCEPT NN_OVERRIDE
    {
        auto fullPath = ResolveFullPath(path);
        nn::fs::FileHandle handle;
        NN_RESULT_DO(nn::fs::OpenFile(&handle, fullPath.c_str(), mode));

        std::unique_ptr<FileHandleFile> file(new FileHandleFile(handle));
        if (file == nullptr)
        {
            nn::fs::CloseFile(handle);
            return nn::fs::ResultAllocationMemoryFailedNew();
        }

        *outValue = std::move(file);
        NN_RESULT_SUCCESS;
    }

    virtual nn::Result DoOpenDirectory(std::unique_ptr<nn::fs::fsa::IDirectory>* outValue, const char* path, nn::fs::OpenDirectoryMode mode) NN_NOEXCEPT NN_OVERRIDE
    {
        auto fullPath = ResolveFullPath(path);
        nn::fs::DirectoryHandle handle;
        NN_RESULT_DO(nn::fs::OpenDirectory(&handle, fullPath.c_str(), mode));

        std::unique_ptr<DirectoryHandleDirectory> dir(new DirectoryHandleDirectory(handle));
        if (dir == nullptr)
        {
            nn::fs::CloseDirectory(handle);
            return nn::fs::ResultAllocationMemoryFailedNew();
        }

        *outValue = std::move(dir);
        NN_RESULT_SUCCESS;
    }

    virtual nn::Result DoGetEntryType(nn::fs::DirectoryEntryType* outValue, const char* path) NN_NOEXCEPT NN_OVERRIDE
    {
        auto fullPath = ResolveFullPath(path);
        return nn::fs::GetEntryType(outValue, fullPath.c_str());
    }

    virtual nn::Result DoGetFreeSpaceSize(int64_t* outValue, const char* path) NN_NOEXCEPT NN_OVERRIDE
    {
        auto fullPath = ResolveFullPath(path);
        return nn::fs::GetFreeSpaceSize(outValue, fullPath.c_str());
    }
    virtual nn::Result DoGetTotalSpaceSize(int64_t* outValue, const char* path) NN_NOEXCEPT NN_OVERRIDE
    {
        auto fullPath = ResolveFullPath(path);
        return nn::fs::GetTotalSpaceSize(outValue, fullPath.c_str());
    }
    virtual nn::Result DoCreateFile(const char *path, int64_t size, int option) NN_NOEXCEPT NN_OVERRIDE
    {
        auto fullPath = ResolveFullPath(path);
        return nn::fs::CreateFile(fullPath.c_str(), size, option);
    }
    virtual nn::Result DoDeleteFile(const char *path) NN_NOEXCEPT NN_OVERRIDE
    {
        auto fullPath = ResolveFullPath(path);
        return nn::fs::DeleteFile(fullPath.c_str());
    }
    virtual nn::Result DoCreateDirectory(const char *path) NN_NOEXCEPT NN_OVERRIDE
    {
        auto fullPath = ResolveFullPath(path);
        return nn::fs::CreateDirectory(fullPath.c_str());
    }
    virtual nn::Result DoDeleteDirectory(const char *path) NN_NOEXCEPT NN_OVERRIDE
    {
        auto fullPath = ResolveFullPath(path);
        return nn::fs::DeleteDirectory(fullPath.c_str());
    }
    virtual nn::Result DoDeleteDirectoryRecursively(const char *path) NN_NOEXCEPT NN_OVERRIDE
    {
        auto fullPath = ResolveFullPath(path);
        return nn::fs::DeleteDirectoryRecursively(fullPath.c_str());
    }
    virtual nn::Result DoCleanDirectoryRecursively(const char* path) NN_NOEXCEPT NN_OVERRIDE
    {
        auto fullPath = ResolveFullPath(path);
        return nn::fs::CleanDirectoryRecursively(fullPath.c_str());
    }
    virtual nn::Result DoRenameFile(const char *currentPath, const char *newPath) NN_NOEXCEPT NN_OVERRIDE
    {
        auto currentFullPath = ResolveFullPath(currentPath);
        auto newFullPath = ResolveFullPath(newPath);
        return nn::fs::RenameFile(currentFullPath.c_str(), newFullPath.c_str());
    }
    virtual nn::Result DoRenameDirectory(const char *currentPath, const char *newPath) NN_NOEXCEPT NN_OVERRIDE
    {
        auto currentFullPath = ResolveFullPath(currentPath);
        auto newFullPath = ResolveFullPath(newPath);
        return nn::fs::RenameDirectory(currentFullPath.c_str(), newFullPath.c_str());
    }
    virtual nn::Result DoCommit() NN_NOEXCEPT NN_OVERRIDE
    {
        char filePath[768];
        nn::util::Strlcpy(filePath, m_RootPath.c_str(), static_cast<int>(m_RootPath.find(":/") + 1));
        return nn::fs::CommitSaveData(filePath);
    }

private:
    nnt::fs::util::String m_RootPath;

protected:
    nnt::fs::util::String ResolveFullPath(const char* offsetPath) const NN_NOEXCEPT
    {
        nnt::fs::util::String path = m_RootPath;
        path += offsetPath;
        return path;
    }
};

// 全て pBaseFileSystem に転送するファイルシステム
class ThruFs : public nn::fs::fsa::IFileSystem, public nn::fs::detail::Newable
{
public:
    explicit ThruFs(nn::fs::fsa::IFileSystem* pBaseFileSystem) NN_NOEXCEPT
        : m_BaseFileSystem(pBaseFileSystem)
    {
    }
    virtual nn::Result DoCreateFile(const char* path, int64_t size, int option) NN_NOEXCEPT NN_OVERRIDE
    {
        return m_BaseFileSystem->CreateFile(path, size, option);
    }
    virtual nn::Result DoDeleteFile(const char* path) NN_NOEXCEPT NN_OVERRIDE
    {
        return m_BaseFileSystem->DeleteFile(path);
    }
    virtual nn::Result DoCreateDirectory(const char* path) NN_NOEXCEPT NN_OVERRIDE
    {
        return m_BaseFileSystem->CreateDirectory(path);
    }
    virtual nn::Result DoDeleteDirectory(const char* path) NN_NOEXCEPT NN_OVERRIDE
    {
        return m_BaseFileSystem->DeleteDirectory(path);
    }
    virtual nn::Result DoDeleteDirectoryRecursively(const char* path) NN_NOEXCEPT NN_OVERRIDE
    {
        return m_BaseFileSystem->DeleteDirectoryRecursively(path);
    }
    virtual nn::Result DoCleanDirectoryRecursively(const char* path) NN_NOEXCEPT NN_OVERRIDE
    {
        return m_BaseFileSystem->CleanDirectoryRecursively(path);
    }
    virtual nn::Result DoRenameFile(const char* currentPath, const char* newPath) NN_NOEXCEPT NN_OVERRIDE
    {
        return m_BaseFileSystem->RenameFile(currentPath, newPath);
    }
    virtual nn::Result DoRenameDirectory(const char* currentPath, const char* newPath) NN_NOEXCEPT NN_OVERRIDE
    {
        return m_BaseFileSystem->RenameDirectory(currentPath, newPath);
    }
    virtual nn::Result DoGetEntryType(nn::fs::DirectoryEntryType* outValue, const char* path) NN_NOEXCEPT NN_OVERRIDE
    {
        return m_BaseFileSystem->GetEntryType(outValue, path);
    }
    virtual nn::Result DoOpenFile(std::unique_ptr<nn::fs::fsa::IFile>* outValue, const char* path, nn::fs::OpenMode mode) NN_NOEXCEPT NN_OVERRIDE
    {
        return m_BaseFileSystem->OpenFile(outValue, path, mode);
    }
    virtual nn::Result DoOpenDirectory(std::unique_ptr<nn::fs::fsa::IDirectory>* outValue, const char* path, nn::fs::OpenDirectoryMode mode) NN_NOEXCEPT NN_OVERRIDE
    {
        return m_BaseFileSystem->OpenDirectory(outValue, path, mode);
    }
    virtual nn::Result DoCommit() NN_NOEXCEPT NN_OVERRIDE
    {
        return m_BaseFileSystem->Commit();
    }
private:
    nn::fs::fsa::IFileSystem * m_BaseFileSystem;
};



}}}
