﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <memory>
#include <cstring>

#include <nn/nn_Common.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultPrivate.h>

#include <nn/fs/fs_IStorage.h>
#include <nn/fs/fs_QueryRange.h>
#include <nn/result/result_HandlingUtility.h>

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>

namespace nnt { namespace fs { namespace util {

/**
* @brief 範囲外アクセスチェック機能つきメモリストレージです。
*/
class SafeMemoryStorage : public nn::fs::IStorage
{
    NN_DISALLOW_COPY(SafeMemoryStorage);

public:
    //!< コンストラクタ
    explicit SafeMemoryStorage(int64_t storageSize) NN_NOEXCEPT
    : m_Size(0)
    {
        Initialize(storageSize);
    }

    //!< コンストラクタ
    SafeMemoryStorage() NN_NOEXCEPT
    : m_Size(0)
    {
    }

    //!< デストラクタ
    virtual ~SafeMemoryStorage() NN_NOEXCEPT
    {
        if( m_Buffer.size() > 0 )
        {
            EXPECT_EQ(true, CheckValid());
        }
    }

    /**
    * @brief        ストレージ用にバッファを確保します。
    *
    * @param[in]    storageSize ストレージのサイズ
    */
    void Initialize(int64_t storageSize) NN_NOEXCEPT
    {
        m_Size = storageSize;
        m_Buffer.resize(static_cast<size_t>(storageSize + VERIFY_AREA_SIZE * 2));

        uint8_t* p0 = static_cast<uint8_t*>(&m_Buffer[0]);
        uint8_t* p1 = static_cast<uint8_t*>(&m_Buffer[0]) + VERIFY_AREA_SIZE + m_Size;
        memset(p0, 0xFE, VERIFY_AREA_SIZE);
        memset(p1, 0xFE, VERIFY_AREA_SIZE);
        uint8_t* pb = static_cast<uint8_t*>(&m_Buffer[VERIFY_AREA_SIZE]);
        memset(pb, 0xFE, static_cast<size_t>(m_Size));
    }

    /**
    * @brief        ストレージの内容をバッファに読み込みます。
    *
    * @param[in]    offset  読み込み開始位置
    * @param[out]   buffer  読み込んだ内容をコピーするバッファ
    * @param[in]    size    読み込むデータサイズ
    *
    * @return       関数の処理結果を返します。
    */
    virtual nn::Result Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        if( (offset < 0) || (m_Size < static_cast<int64_t>(offset)) )
        {
            return nn::fs::ResultInvalidOffset();
        }

        const auto copySize = std::min(size, static_cast<size_t>(m_Size - offset));
        const uint8_t* p = static_cast<const uint8_t*>(&m_Buffer[VERIFY_AREA_SIZE]);
        std::memcpy(buffer, p + offset, copySize);

        NN_RESULT_SUCCESS;
    }

    /**
    * @brief        バッファの内容をストレージに書き込みます。
    *
    * @param[in]    offset  書き込み開始位置
    * @param[in]    buffer  書き込むデータ
    * @param[in]    size    書き込むデータサイズ
    *
    * @return       関数の処理結果を返します。
    */
    virtual nn::Result Write(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        if( (offset < 0) || (m_Size < static_cast<int64_t>(offset)) )
        {
            return nn::fs::ResultInvalidOffset();
        }

        const auto copySize = std::min(size, static_cast<size_t>(m_Size - offset));
        uint8_t* p = static_cast<uint8_t*>(&m_Buffer[VERIFY_AREA_SIZE]);
        std::memcpy(p + offset, buffer, copySize);

        NN_RESULT_SUCCESS;
    }

    /**
    * @brief        フラッシュします。
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess   必ず成功します。
    */
    virtual nn::Result Flush() NN_NOEXCEPT NN_OVERRIDE
    {
        NN_RESULT_SUCCESS;
    }

    virtual nn::Result OperateRange(
                           void* outBuffer,
                           size_t outBufferSize,
                           nn::fs::OperationId operationId,
                           int64_t offset,
                           int64_t size,
                           const void* inBuffer,
                           size_t inBufferSize
                       ) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(outBuffer);
        NN_UNUSED(outBufferSize);
        NN_UNUSED(inBuffer);
        NN_UNUSED(inBufferSize);

        switch( operationId )
        {
        case nn::fs::OperationId::FillZero:
            {
                if( (offset < 0) || (m_Size < static_cast<int64_t>(offset)) )
                {
                    return nn::fs::ResultInvalidOffset();
                }
                const auto offsetFill = static_cast<size_t>(VERIFY_AREA_SIZE + offset);
                const auto sizeFill = static_cast<size_t>(std::min(size, m_Size - offset));
                std::memset(&m_Buffer[offsetFill], 0, sizeFill);
                NN_RESULT_SUCCESS;
            }
            break;

        case nn::fs::OperationId::Invalidate:
            NN_RESULT_SUCCESS;

        case nn::fs::OperationId::QueryRange:
            NN_RESULT_THROW_UNLESS(outBuffer != nullptr, nn::fs::ResultNullptrArgument());
            NN_RESULT_THROW_UNLESS(outBufferSize == sizeof(nn::fs::QueryRangeInfo), nn::fs::ResultInvalidSize());

            reinterpret_cast<nn::fs::QueryRangeInfo*>(outBuffer)->Clear();
            NN_RESULT_SUCCESS;

        default:
            return nn::fs::ResultUnsupportedOperation();
        }
    }

    using IStorage::OperateRange;

    /**
    * @brief        ストレージのサイズを取得します。
    *
    * @param[out]   outValue    取得したサイズ格納先
    *
    * @return       関数の処理結果を返します。
    */
    virtual nn::Result GetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE
    {
        *outValue = m_Size;
        NN_RESULT_SUCCESS;
    }

    /**
    * @brief        この関数ではストレージのサイズを設定できません。
    *
    * @details      ストレージのサイズは Initialize で設定してください。
    */
    virtual nn::Result SetSize(int64_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(size);
        return nn::fs::ResultUnsupportedOperation();
    }

public:
    /**
    * @brief        バッファへのポインタを取得します。
    */
    void* GetBuffer() NN_NOEXCEPT
    {
        return &m_Buffer[VERIFY_AREA_SIZE];
    }

    /**
    * @brief        バッファへのポインタを取得します。
    */
    const void* GetBuffer() const NN_NOEXCEPT
    {
        return &m_Buffer[VERIFY_AREA_SIZE];
    }

    /**
    * @brief        バッファのサイズを取得します。
    */
    int64_t GetSize() const NN_NOEXCEPT
    {
        return m_Size;
    }

    /**
    * @brief        バッファをコピーします。
    */
    void Copy(const SafeMemoryStorage& rhs) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(this->GetSize() == rhs.GetSize());
        std::memcpy(
            &this->m_Buffer[0],
            &rhs.m_Buffer[0],
            static_cast<size_t>(m_Size + VERIFY_AREA_SIZE * 2)
        );
    }

    /**
    * @brief        ストレージの指定した範囲を指定バイトで埋めます。
    *
    * @param[in]    offset  書き込み開始位置
    * @param[in]    size    書き込むデータサイズ
    * @param[in]    c       書き込むバイトデータ
    */
    void FillBuffer(int64_t offset, size_t size, uint8_t c) NN_NOEXCEPT
    {
        uint8_t* p = static_cast<uint8_t*>(&m_Buffer[VERIFY_AREA_SIZE]);
        std::memset(p + offset, c, size);
    }

    /**
    * @brief        ストレージの指定した範囲が指定バイトで埋めまっているか検証します。
    *
    * @param[in]    offset  確認開始位置
    * @param[in]    size    確認するデータサイズ
    * @param[in]    c       確認するバイトデータ
    *
    * @return       ストレージの範囲が指定バイトで埋まっているかどうかを返します。
    * @retval       true    ストレージの範囲が指定バイトで埋まっていた
    * @retval       false   ストレージの範囲に指定バイトでない値が存在した
    */
    bool VerifyBuffer(int64_t offset, size_t size, uint8_t c) NN_NOEXCEPT
    {
        const uint8_t* p = static_cast<const uint8_t*>(&m_Buffer[VERIFY_AREA_SIZE]);
        for( size_t i = 0; i < size; ++i )
        {
            if( p[i + offset] != c )
            {
                return false;
            }
        }
        return true;
    }

    /**
    * @brief        ストレージの範囲外アクセスが発生していないか検証します。
    *
    * @return       ストレージの範囲外アクセスが発生していないかどうかを返します。
    * @retval       true    ストレージの範囲外アクセスは発生していない
    * @retval       false   ストレージの範囲外アクセスが発生した
    */
    bool CheckValid() const NN_NOEXCEPT
    {
        const uint8_t* p0 = &m_Buffer[0];
        const uint8_t* p1 = &m_Buffer[0] + VERIFY_AREA_SIZE + m_Size;
        for( int i = 0; i < VERIFY_AREA_SIZE; ++i )
        {
            if( p0[i] != 0xFE )
            {
                return false;
            }
            if( p1[i] != 0xFE )
            {
                return false;
            }
        }
        return true;
    }

    /**
    * @brief        この関数ではストレージが確保しているリソースを解放します
    */
    void Finalize() NN_NOEXCEPT
    {
        EXPECT_EQ(true, CheckValid());
        m_Buffer.clear();
        m_Buffer.shrink_to_fit();
    }

private:
    static const int64_t VERIFY_AREA_SIZE = 1024;
    Vector<uint8_t> m_Buffer;
    int64_t m_Size;
};

}}} // namespace nnt::fs::util

