﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <memory>

#include <nn/nn_Common.h>
#include <nn/fs.h>

namespace nnt { namespace fs { namespace util {

/**
* @brief ファイルデータキャッシュを保持するクラスです。
*/
class GlobalFileDataCache
{
    NN_DISALLOW_COPY(GlobalFileDataCache);

public:
    /**
    * @brief  コンストラクタです。
    */
    GlobalFileDataCache() NN_NOEXCEPT
        : m_pCache(nullptr, &nnt::fs::util::DeleterBuffer)
    {
    }

    GlobalFileDataCache(int argc, char** argv) NN_NOEXCEPT
        : m_pCache(nullptr, &nnt::fs::util::DeleterBuffer)
    {
        Initialize(argc, argv);
    }

    /**
    * @brief    デストラクタです。
    */
    virtual ~GlobalFileDataCache() NN_NOEXCEPT
    {
        Finalize();
    }

    void Initialize(int argc, char** argv) NN_NOEXCEPT
    {
        NN_ASSERT( !m_pCache );

        for(int i = 0; i < argc - 1; ++i)
        {
            if( strcmp(argv[i], "--filedatacache_size") == 0 )
            {
                const size_t cacheSize = atoi(argv[i + 1]) * 1024 * 1024;
                Initialize(cacheSize);
                break;
            }
        }
    }

    void Initialize(size_t size) NN_NOEXCEPT
    {
        NN_ASSERT( size > 0 );
        NN_ASSERT( !m_pCache );

        m_pCache = nnt::fs::util::AllocateBuffer(size);
        NN_ASSERT( m_pCache );
        nn::fs::EnableGlobalFileDataCache(m_pCache.get(), size);
    }

    void Finalize() NN_NOEXCEPT
    {
        if( m_pCache )
        {
            nn::fs::DisableGlobalFileDataCache();
            m_pCache.reset();
        }
    }

    bool IsGlobalCacheEnable() const NN_NOEXCEPT
    {
        return m_pCache ? true : false;
    }

private:
      std::unique_ptr<char, decltype(&DeleterBuffer)> m_pCache;
};

}}} // namespace nnt::fs::util
