﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nnt/fsApi/testFs_ApiCommon.h>

namespace nnt { namespace fs { namespace api {

    class FsApiUnitTestFile : public ITestFile, public nn::fs::detail::Newable
    {
    public:
        explicit FsApiUnitTestFile(std::unique_ptr<nn::fs::fsa::IFile>&& pFile) NN_NOEXCEPT
            : m_pFile(std::move(pFile))
        {
        }
        virtual ~FsApiUnitTestFile() NN_NOEXCEPT NN_OVERRIDE
        {
        }
        virtual void Close() NN_NOEXCEPT NN_OVERRIDE
        {
            m_pFile.reset();
        }
        virtual Result Read(int64_t offset, void* buffer, size_t size, const nn::fs::ReadOption& option) NN_NOEXCEPT NN_OVERRIDE
        {
            size_t readSize;
            NN_RESULT_DO(m_pFile->Read(&readSize, offset, buffer, size, option));
            NN_RESULT_SUCCESS;
        }
        virtual Result Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
        {
            size_t readSize;
            NN_RESULT_DO(m_pFile->Read(&readSize, offset, buffer, size, nn::fs::ReadOption()));
            NN_RESULT_SUCCESS;
        }
        virtual nn::Result Read(size_t* outValue, int64_t offset, void* buffer, size_t size, const nn::fs::ReadOption& option) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_RESULT_DO(m_pFile->Read(outValue, offset, buffer, size, option));
            NN_RESULT_SUCCESS;
        }
        virtual Result Read(size_t* outValue, int64_t offset, void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_RESULT_DO(m_pFile->Read(outValue, offset, buffer, size, nn::fs::ReadOption()));
            NN_RESULT_SUCCESS;
        }
        virtual nn::Result Write(int64_t offset, const void* buffer, size_t size, const nn::fs::WriteOption& option) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_RESULT_DO(m_pFile->Write(offset, buffer, size, option));
            NN_RESULT_SUCCESS;
        }
        virtual nn::Result Flush() NN_NOEXCEPT NN_OVERRIDE
        {
            NN_RESULT_DO(m_pFile->Flush());
            NN_RESULT_SUCCESS;
        }
        virtual nn::Result SetSize(int64_t size) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_RESULT_DO(m_pFile->SetSize(size));
            NN_RESULT_SUCCESS;
        }
        virtual nn::Result GetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_RESULT_DO(m_pFile->GetSize(outValue));
            NN_RESULT_SUCCESS;
        }
        virtual nn::Result OperateRange(void* outBuffer, size_t outBufferSize, nn::fs::OperationId operationId, int64_t offset, int64_t size, const void* inBuffer, size_t inBufferSize) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_RESULT_DO(m_pFile->OperateRange(outBuffer, outBufferSize, operationId, offset, size, inBuffer, inBufferSize));
            NN_RESULT_SUCCESS;
        }

        virtual nn::Result OperateRange(
            nn::fs::OperationId operationId,
            int64_t offset,
            int64_t size) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_RESULT_DO(m_pFile->OperateRange(operationId, offset, size));
            NN_RESULT_SUCCESS;
        }

        virtual nn::Result QueryRange(nn::fs::QueryRangeInfo* outValue, int64_t offset, int64_t size) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_RESULT_DO(m_pFile->OperateRange(outValue, sizeof(*outValue), nn::fs::OperationId::QueryRange, offset, size, nullptr, 0));
            NN_RESULT_SUCCESS;
        }

    private:
        std::unique_ptr<nn::fs::fsa::IFile> m_pFile;
    };

    class FsApiUnitTestDirectory : public ITestDirectory, public nn::fs::detail::Newable
    {
    public:
        explicit FsApiUnitTestDirectory(std::unique_ptr<nn::fs::fsa::IDirectory>&& pDirectory) NN_NOEXCEPT
            : m_pDirectory(std::move(pDirectory))
        {
        }
        virtual ~FsApiUnitTestDirectory() NN_NOEXCEPT NN_OVERRIDE
        {
        }
        virtual void Close() NN_NOEXCEPT NN_OVERRIDE
        {
            m_pDirectory.reset();
        }
        virtual nn::Result Read(int64_t* outValue, nn::fs::DirectoryEntry* outEntries, int64_t count) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_RESULT_DO(m_pDirectory->Read(outValue, outEntries, count));
            NN_RESULT_SUCCESS;
        }
        virtual nn::Result GetEntryCount(int64_t* outValue) NN_NOEXCEPT  NN_OVERRIDE
        {
            NN_RESULT_DO(m_pDirectory->GetEntryCount(outValue));
            NN_RESULT_SUCCESS;
        }
    private:
        std::unique_ptr<nn::fs::fsa::IDirectory> m_pDirectory;
    };

    class FsApiUnitTestFileSystem : public ITestFileSystem, public nn::fs::detail::Newable
    {
    public:
        FsApiUnitTestFileSystem() NN_NOEXCEPT {}
        explicit FsApiUnitTestFileSystem(std::unique_ptr<nn::fs::fsa::IFileSystem>&& pFileSystem) NN_NOEXCEPT
            : m_pFileSystem(std::move(pFileSystem))
        {
        }
        virtual ~FsApiUnitTestFileSystem() NN_NOEXCEPT NN_OVERRIDE {}
        virtual nn::Result CreateFile(const char* path, int64_t size, int option) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_RESULT_DO(m_pFileSystem->CreateFile(path, size, option));
            NN_RESULT_SUCCESS;
        }
        virtual nn::Result CreateFile(const char* path, int64_t size) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_RESULT_DO(m_pFileSystem->CreateFile(path, size));
            NN_RESULT_SUCCESS;
        }
        virtual nn::Result DeleteFile(const char* path) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_RESULT_DO(m_pFileSystem->DeleteFile(path));
            NN_RESULT_SUCCESS;
        }
        virtual nn::Result CreateDirectory(const char* path) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_RESULT_DO(m_pFileSystem->CreateDirectory(path));
            NN_RESULT_SUCCESS;
        }
        virtual nn::Result DeleteDirectory(const char* path) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_RESULT_DO(m_pFileSystem->DeleteDirectory(path));
            NN_RESULT_SUCCESS;
        }
        virtual nn::Result DeleteDirectoryRecursively(const char* path) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_RESULT_DO(m_pFileSystem->DeleteDirectoryRecursively(path));
            NN_RESULT_SUCCESS;
        }
        virtual nn::Result CleanDirectoryRecursively(const char* path) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_RESULT_DO(m_pFileSystem->CleanDirectoryRecursively(path));
            NN_RESULT_SUCCESS;
        }
        virtual nn::Result RenameFile(const char* currentPath, const char* newPath) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_RESULT_DO(m_pFileSystem->RenameFile(currentPath, newPath));
            NN_RESULT_SUCCESS;
        }
        virtual nn::Result RenameDirectory(const char* currentPath, const char* newPath) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_RESULT_DO(m_pFileSystem->RenameDirectory(currentPath, newPath));
            NN_RESULT_SUCCESS;
        }
        virtual nn::Result GetEntryType(nn::fs::DirectoryEntryType* outValue, const char* path) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_RESULT_DO(m_pFileSystem->GetEntryType(outValue, path));
            NN_RESULT_SUCCESS;
        }
        virtual nn::Result GetFreeSpaceSize(int64_t* outValue, const char* path) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_RESULT_DO(m_pFileSystem->GetFreeSpaceSize(outValue, path));
            NN_RESULT_SUCCESS;
        }
        virtual nn::Result GetTotalSpaceSize(int64_t* outValue, const char* path) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_RESULT_DO(m_pFileSystem->GetTotalSpaceSize(outValue, path));
            NN_RESULT_SUCCESS;
        }
        virtual nn::Result GetFileTimeStampRaw(nn::fs::FileTimeStampRaw* outTimeStamp, const char* path) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_RESULT_DO(m_pFileSystem->GetFileTimeStampRaw(outTimeStamp, path));
            NN_RESULT_SUCCESS;
        }
        virtual nn::Result OpenFile(std::unique_ptr<ITestFile>* outValue, const char* path, nn::fs::OpenMode mode) NN_NOEXCEPT NN_OVERRIDE
        {
            if( outValue == nullptr )
            {
                NN_RESULT_DO(m_pFileSystem->OpenFile(nullptr, path, mode));
            }
            else
            {
                std::unique_ptr<nn::fs::fsa::IFile> pFile;
                NN_RESULT_DO(m_pFileSystem->OpenFile(&pFile, path, mode));
                *outValue = std::unique_ptr<ITestFile>(new FsApiUnitTestFile(std::move(pFile)));
            }
            NN_RESULT_SUCCESS;
        }
        virtual nn::Result OpenDirectory(std::unique_ptr<ITestDirectory>* outValue, const char* path, nn::fs::OpenDirectoryMode mode) NN_NOEXCEPT NN_OVERRIDE
        {
            if( outValue == nullptr )
            {
                NN_RESULT_DO(m_pFileSystem->OpenDirectory(nullptr, path, mode));
            }
            else
            {
                std::unique_ptr<nn::fs::fsa::IDirectory> pDirectory;
                NN_RESULT_DO(m_pFileSystem->OpenDirectory(&pDirectory, path, mode));
                *outValue = std::unique_ptr<ITestDirectory>(new FsApiUnitTestDirectory(std::move(pDirectory)));
            }
            NN_RESULT_SUCCESS;
        }
        virtual nn::Result Commit() NN_NOEXCEPT NN_OVERRIDE
        {
            NN_RESULT_DO(m_pFileSystem->Commit());
            NN_RESULT_SUCCESS;
        }
        virtual nn::Result CommitSaveData() NN_NOEXCEPT NN_OVERRIDE
        {
            NN_RESULT_DO(m_pFileSystem->Commit());
            NN_RESULT_SUCCESS;
        }
        virtual nn::Result Flush() NN_NOEXCEPT NN_OVERRIDE
        {
            NN_RESULT_DO(m_pFileSystem->Flush());
            NN_RESULT_SUCCESS;
        }
        virtual nn::Result QueryEntry(char* outBuffer, size_t outBufferSize, const char* inBuffer, size_t inBufferSize, nn::fs::fsa::QueryId queryId, const char* path) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_RESULT_DO(m_pFileSystem->QueryEntry(outBuffer, outBufferSize, inBuffer, inBufferSize, queryId, path));
            NN_RESULT_SUCCESS;
        }

    private:
        std::unique_ptr<nn::fs::fsa::IFileSystem> m_pFileSystem;
    };

}}}
